# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/04_env.ipynb.

# %% auto 0
__all__ = ['TextEnv']

# %% ../nbs/04_env.ipynb 2
from typing import List, Callable, Tuple, Iterable

import torch
from torchtyping import TensorType
import gymnasium as gym
from gymnasium import Env

# %% ../nbs/04_env.ipynb 3
class TextEnv(gym.Env):
    def __init__(
        self, model: Callable, tokenizer: Callable,
        observation_input: List[int], context_length: int=1024
    ):
        """Initialize the environment.

        Args:
            model (_type_): The RL-based language model
            tokenizer (_type_): The tokenizer of the RL-based language model
            observation_input (list, optional): The list of all possible prompts.
            context_length (int, optional): The max context length of the RL-based language model . Defaults to 1024.
        """

        self.model = model
        self.context_length = context_length
        self.tokenizer = tokenizer
        
        vocab_ids = [token_id for text, token_id in tokenizer.vocab.items()]
        self.action_space = gym.spaces.Discrete(len(vocab_ids))
        self.actions = vocab_ids
        
        # TODO: add support a batch of observation space
        # currently only support one prompt
        self.observation_input = observation_input
        self.input_token_ids: List[int] = []
        self.predicted_token_ids: List[int] = []
    
    def is_in_action_space(self, action: int) -> bool:
        return True if action in self.actions else False

    def is_token_end(self, token_id: int) -> bool:
        end_tokens = [self.tokenizer.eos_token_id, self.tokenizer.sep_token_id]
        return True if token_id in end_tokens else False

    def is_max_context_length(self, tokens) -> bool:
        n_tokens = len(tokens)
        return True if n_tokens >= self.context_length else False

    def _add_predicted_token(self, token_id: int):
        """Add the token_id to the list of current predicted token

        Args:
            token_id (int): A token
        """
        self.predicted_token_ids.append(token_id)
        
    def step(self, action: int) -> Tuple[
        TensorType["seq_len", "n_embd"],
        int, bool, bool, dict, bool
    ]:
        assert self.is_in_action_space(action) == True
        
        terminated = False
        truncated = False
        info = {}
        
        if self.is_token_end(action) or self.is_max_context_length(self.input_token_ids):
            done = True
            reward = self._get_reward(self.predicted_token_ids)
        else:
            done = False
            reward = 0
            self._add_predicted_token(action)
        
        next_observation = self._get_obs()
        
        return next_observation, reward, terminated, truncated, info, done
    
    def _get_reward(self, action: int):
        raise Exception("Write your own reward function!")
    
    def _get_obs(self):
        """Return the current observation of the environment.

        Returns:
            _type_: _description_
        """
        input_token_ids = self.input_token_ids
        predicted_token_ids = self.predicted_token_ids
        
        if isinstance(self.input_token_ids, torch.Tensor) is False:
            input_token_ids = torch.tensor(self.input_token_ids, dtype=torch.int)
            
        if isinstance(self.predicted_token_ids, torch.Tensor) is False:
            predicted_token_ids = torch.tensor(self.predicted_token_ids, dtype=torch.int)
        
        # `transformer` do inference in batch, so add an extra dim for batch
        input_ids = torch.cat([input_token_ids, predicted_token_ids]).unsqueeze(0)
        
        # output = self.model(
        #     input_ids=input_ids,
        #     output_hidden_states=True,
        # )
        # last_hidden_state = output.hidden_states[-1]
        # return last_hidden_state
    
        return input_ids
    
    def reset(self) -> TensorType["seq_len"]:
        # the current input prompts's token_ids
        self.input_token_ids: List[int] = self.tokenizer(self.observation_input, return_tensors="pt")["input_ids"].tolist()[0]

        # the current generated token_ids
        self.predicted_token_ids: List[int] = []

        return self._get_obs()
