"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToIot = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const api = require("aws-cdk-lib/aws-apigateway");
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
/**
 * @summary The ApiGatewayIot class.
 */
class ApiGatewayToIot extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayIot class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {ApiGatewayToIotProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // IoT Core topic nesting. A topic in a publish or subscribe request can have no more than 7 forward slashes (/).
        // This excludes the first 3 slashes in the mandatory segments for Basic Ingest
        // Refer IoT Limits - https://docs.aws.amazon.com/general/latest/gr/iot-core.html#limits_iot
        this.topicNestingLevel = 7;
        defaults.CheckProps(props);
        // Assignment to local member variables to make these available to all member methods of the class.
        // (Split the string just in case user supplies fully qualified endpoint eg. ab123cdefghij4l-ats.iot.ap-south-1.amazonaws.com)
        this.iotEndpoint = props.iotEndpoint.trim().split('.')[0];
        // Mandatory params check
        if (!this.iotEndpoint || this.iotEndpoint.length < 0) {
            throw new Error('specify a valid iotEndpoint');
        }
        // Add additional params to the apiGatewayProps
        let extraApiGwProps = {
            binaryMediaTypes: ['application/octet-stream'],
            defaultMethodOptions: {
                apiKeyRequired: props.apiGatewayCreateApiKey
            }
        };
        // If apiGatewayProps are specified override the extra Api Gateway properties
        extraApiGwProps = defaults.consolidateProps(extraApiGwProps, props.apiGatewayProps);
        // Check whether an API Gateway execution role is specified?
        if (props.apiGatewayExecutionRole) {
            this.apiGatewayRole = props.apiGatewayExecutionRole;
        }
        else {
            // JSON that will be used for policy document
            const policyJSON = {
                Version: "2012-10-17",
                Statement: [
                    {
                        Action: [
                            "iot:UpdateThingShadow"
                        ],
                        Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:thing/*`,
                        Effect: "Allow"
                    },
                    {
                        Action: [
                            "iot:Publish"
                        ],
                        Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:topic/*`,
                        Effect: "Allow"
                    }
                ]
            };
            // Create a policy document
            const policyDocument = iam.PolicyDocument.fromJson(policyJSON);
            // Props for IAM Role
            const iamRoleProps = {
                assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                path: '/',
                inlinePolicies: { awsapigatewayiotpolicy: policyDocument }
            };
            // Create a policy that overrides the default policy that gets created with the construct
            this.apiGatewayRole = new iam.Role(this, 'apigateway-iot-role', iamRoleProps);
        }
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole,
            this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, extraApiGwProps, props.logGroupProps);
        // Validate the Query Params
        const requestValidatorProps = {
            restApi: this.apiGateway,
            validateRequestBody: false,
            validateRequestParameters: true
        };
        this.requestValidator = new api.RequestValidator(this, `aws-apigateway-iot-req-val`, requestValidatorProps);
        // Create a resource for messages '/message'
        const msgResource = this.apiGateway.root.addResource('message');
        // Create resources from '/message/{topic-level-1}' through '/message/{topic-level-1}/..../{topic-level-7}'
        let topicPath = 'topics';
        let parentNode = msgResource;
        let integParams = {};
        let methodParams = {};
        for (let pathLevel = 1; pathLevel <= this.topicNestingLevel; pathLevel++) {
            const topicName = `topic-level-${pathLevel}`;
            const topicResource = parentNode.addResource(`{${topicName}}`);
            const integReqParam = JSON.parse(`{"integration.request.path.${topicName}": "method.request.path.${topicName}"}`);
            const methodReqParam = JSON.parse(`{"method.request.path.${topicName}": true}`);
            topicPath = `${topicPath}/{${topicName}}`;
            integParams = Object.assign(integParams, integReqParam);
            methodParams = Object.assign(methodParams, methodReqParam);
            this.addResourceMethod(topicResource, props, topicPath, integParams, methodParams);
            parentNode = topicResource;
        }
        // Create a resource for shadow updates '/shadow'
        const shadowResource = this.apiGateway.root.addResource('shadow');
        // Create resource '/shadow/{thingName}'
        const defaultShadowResource = shadowResource.addResource('{thingName}');
        const shadowReqParams = { 'integration.request.path.thingName': 'method.request.path.thingName' };
        const methodShadowReqParams = { 'method.request.path.thingName': true };
        this.addResourceMethod(defaultShadowResource, props, 'things/{thingName}/shadow', shadowReqParams, methodShadowReqParams);
        // Create resource '/shadow/{thingName}/{shadowName}'
        const namedShadowResource = defaultShadowResource.addResource('{shadowName}');
        const namedShadowReqParams = Object.assign({
            'integration.request.path.shadowName': 'method.request.path.shadowName'
        }, shadowReqParams);
        const methodNamedShadowReqParams = Object.assign({
            'method.request.path.shadowName': true
        }, methodShadowReqParams);
        this.addResourceMethod(namedShadowResource, props, 'things/{thingName}/shadow?name={shadowName}', namedShadowReqParams, methodNamedShadowReqParams);
    }
    /**
     * Adds a method to specified resource
     * @param resource API Gateway resource to which this method is added
     * @param resourcePath path of resource from root
     * @param integReqParams request paramters for the Integration method
     * @param methodReqParams request parameters at Method level
     */
    addResourceMethod(resource, props, resourcePath, integReqParams, methodReqParams) {
        const integResp = [
            {
                statusCode: "200",
                selectionPattern: "2\\d{2}",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            },
            {
                statusCode: "500",
                selectionPattern: "5\\d{2}",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            },
            {
                statusCode: "403",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            }
        ];
        // Method responses for the resource
        const methodResp = [
            {
                statusCode: "200"
            },
            {
                statusCode: "500"
            },
            {
                statusCode: "403"
            }
        ];
        // Override the default Integration Request Props
        const integrationReqProps = {
            subdomain: this.iotEndpoint,
            options: {
                requestParameters: integReqParams,
                integrationResponses: integResp,
                passthroughBehavior: api.PassthroughBehavior.WHEN_NO_MATCH
            }
        };
        // Override the default Method Options
        const resourceMethodOptions = {
            requestParameters: methodReqParams,
            methodResponses: methodResp,
        };
        const resourceMethodParams = {
            service: 'iotdata',
            path: resourcePath,
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: resource,
            requestTemplate: "$input.json('$')",
            requestValidator: this.requestValidator,
            awsIntegrationProps: integrationReqProps,
            methodOptions: resourceMethodOptions
        };
        const apiMethod = defaults.addProxyMethodToApiResource(resourceMethodParams);
        if (props.apiGatewayCreateApiKey === true) {
            // cfn Nag doesn't like having a HTTP Method with Authorization Set to None, supress the warning
            defaults.addCfnSuppressRules(apiMethod, [
                {
                    id: "W59",
                    reason: "When ApiKey is being created, we also set apikeyRequired to true, so techincally apiGateway still looks for apiKey even though user specified AuthorizationType to NONE",
                },
            ]);
        }
    }
}
exports.ApiGatewayToIot = ApiGatewayToIot;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToIot[_a] = { fqn: "@aws-solutions-constructs/aws-apigateway-iot.ApiGatewayToIot", version: "2.16.0" };
//# sourceMappingURL=data:application/json;base64,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