# -*- coding: utf-8 -*-
from beecolpy import *

from copy import deepcopy
import math as mt
import numpy as np
import numpy.testing as npt

def sphere(x): #Continuous and NaN benchmark
    #Sphere function
    #Min: x=(0,0)
    total = 0
    test = 0
    for i in range(len(x)):
        total += x[i]**2
        test += x[i]
    if (test<5):
        return total
    else:
        return np.nan

def translate_bin(b):
    return np.sum([(b[::-1][i])*mt.pow(2,i) for i in range(len(b))])

def squared_bin(b):
    #y=(x-1)*(x-3)*(x-11)
    #y = x^3 - 15 x^2 + 47 x - 33
    #Min: x=0 b=[0000 0000] (Local)
    #     x=8.055 b~[0000 1000] (Global)
    #NaN: x>128 b>[0111 1111]
    x = translate_bin(b)
    if (x > 128):
        return np.nan
    else:
        return mt.pow(x,3) - 15*mt.pow(x,2) + 47*(x) - 33

base_abc_obj = abc(function=sphere,
                   boundaries=[(-10,10) for _ in range(2)],
                   colony_size=10,
                   scouts=0.5,
                   iterations=10,
                   min_max='min',
                   nan_protection=True,
                   log_agents=True,
                   seed=42)

base_bin_abc_obj = bin_abc(function=squared_bin,
                           bits_count=4,
                           transfer_function='sigmoid',
                           colony_size=10,
                           scouts=0.5,
                           iterations=10,
                           min_max='min',
                           method='bin',
                           result_format='best',
                           nan_protection=3,
                           log_agents=True,
                           seed=42)

base_am_abc_obj = bin_abc(function=squared_bin,
                          bits_count=4,
                          colony_size=10,
                          scouts=0.5,
                          iterations=10,
                          min_max='min',
                          method='am',
                          nan_protection=True,
                          log_agents=True,
                          seed=42)

average_nan_bin_abc_obj = bin_abc(function=squared_bin,
                                  bits_count=8,
                                  transfer_function='sigmoid',
                                  colony_size=10,
                                  scouts=0.5,
                                  iterations=10,
                                  min_max='min',
                                  method='bin',
                                  result_format='average',
                                  nan_protection=3,
                                  log_agents=True,
                                  seed=42)

nan_bin_abc_obj = bin_abc(function=squared_bin,
                          bits_count=8,
                          transfer_function='sigmoid',
                          colony_size=60,
                          scouts=10,
                          iterations=30,
                          min_max='min',
                          method='bin',
                          result_format='average',
                          nan_protection=3,
                          log_agents=True,
                          seed=42)


def test_food_source_generation():
    # Test algorithm initialization
    global base_abc_obj
    out = [food.position for food in base_abc_obj.foods]
    ref = [[2.7885359691576745, -9.49978489554666],
           [-4.499413632617615, -5.5357852370235445],
           [7.843591354096908, -8.261223347411677],
           [-1.561563606294591, -9.404055611238594],
           [-5.627240503927933, 0.10710576206724731]]
    npt.assert_array_almost_equal(out, ref, decimal=6)


def test_fit_solution():
    # Test solver capability
    global base_abc_obj
    abc_obj = deepcopy(base_abc_obj)
    abc_obj.fit()
    out = abc_obj.get_solution()
    ref = [0.004621261711518124, -0.08296007192538377]
    npt.assert_array_almost_equal(out, ref, decimal=6)


def test_get_agents():
    # Verifies the process step-by-step
    global base_abc_obj
    abc_obj = deepcopy(base_abc_obj)
    abc_obj.fit()
    out = abc_obj.get_agents()
    ref = [[[2.7885359691576745, -9.49978489554666],
            [-4.499413632617615, -5.5357852370235445],
            [7.843591354096908, -8.261223347411677],
            [-1.561563606294591, -9.404055611238594],
            [-5.627240503927933, 0.10710576206724731]],
           [[2.7885359691576745, -8.233291099602813],
            [1.5832471812626028, -5.5357852370235445],
            [3.7358052227687066, -8.261223347411677],
            [-1.2017059867618722, -9.404055611238594],
            [-5.177686266266753, 0.10710576206724731]],
           [[2.7885359691576745, -8.233291099602813],
            [1.5832471812626028, -5.5357852370235445],
            [3.07809973415758, -3.1160692355609463],
            [-1.2017059867618722, -9.404055611238594],
            [-0.626259589706855, 0.10710576206724731]],
           [[2.7885359691576745, -5.5667304312754515],
            [1.5832471812626028, -4.793940503704757],
            [3.07809973415758, -3.1160692355609463],
            [-1.2017059867618722, -9.404055611238594],
            [-0.626259589706855, -0.08296007192538377]],
           [[2.7885359691576745, -5.5667304312754515],
            [1.5832471812626028, -4.793940503704757],
            [3.07809973415758, -2.9972606160578668],
            [-1.2017059867618722, -2.33874341261713],
            [0.004621261711518124, -0.08296007192538377]],
           [[2.7885359691576745, -5.5667304312754515],
            [1.5832471812626028, -1.4037521889161138],
            [3.07809973415758, -2.8781556401746196],
            [-0.23585290985708707, -0.35015979771326466],
            [0.004621261711518124, -0.08296007192538377]],
           [[2.7885359691576745, -5.5667304312754515],
            [0.18420441941726584, -1.4037521889161138],
            [3.07809973415758, -2.8781556401746196],
            [-0.23585290985708707, -0.35015979771326466],
            [7.585404849690857, -2.609458225222321]],
           [[2.7885359691576745, -0.8101232115728161],
            [0.18420441941726584, -1.4037521889161138],
            [3.07809973415758, -2.8781556401746196],
            [9.443925300345207, -9.202102437775636],
            [5.401373297801247, -2.609458225222321]],
           [[2.683096137661125, -0.8101232115728161],
            [-6.4286436765507275, 9.25068631523111],
            [-2.3425767195845317, -2.262486375815069],
            [9.443925300345207, -4.543441485273379],
            [5.401373297801247, -1.4635223212335464]],
           [[-1.2529852411980031, -0.019504875840150926],
            [-6.4286436765507275, 9.25068631523111],
            [0.36371179397453135, -2.262486375815069],
            [6.465646902426063, -4.543441485273379],
            [5.319323057174841, -1.4635223212335464]],
           [[-0.5647328495432691, -0.019504875840150926],
            [-6.4286436765507275, 9.25068631523111],
            [0.36371179397453135, -1.6127483532703213],
            [6.465646902426063, -4.543441485273379],
            [5.319323057174841, -1.4635223212335464]]]
    npt.assert_array_almost_equal(out, ref, decimal=6)


def test_get_status():
    # Test exploration and NaN protection
    global base_abc_obj
    abc_obj = deepcopy(base_abc_obj)
    abc_obj.fit()
    out = abc_obj.get_status()
    ref = (10, 3, 3)
    npt.assert_array_equal(out, ref)


def test_bin_food_source_generation():
    # Test algorithm initialization
    global base_bin_abc_obj
    out = [food.position for food in base_bin_abc_obj._bin_abc_object.foods]
    ref = [[2.7885359691576745, -9.49978489554666, -4.499413632617615, -5.5357852370235445],
           [-1.561563606294591, -9.404055611238594, -5.627240503927933, 0.10710576206724731],
           [-5.591187559186066, 1.7853136775181753, 6.1886091335565325, -9.87002480643878],
           [9.144261444135623, -3.2681090977474643, -8.145083132397042, -8.06567246333072],
           [0.7245618290940143, 9.462315279587411, -2.42931245583293, 1.0408126254645396]]
    npt.assert_array_almost_equal(out, ref, decimal=6)


def test_bin_fit_best_solution():
    # Test solver capability
    global base_bin_abc_obj
    bin_abc_obj = deepcopy(base_bin_abc_obj)
    bin_abc_obj.fit()
    out = bin_abc_obj.get_solution()
    ref = [True, False, False, False]
    npt.assert_array_equal(out, ref)


def test_bin_fit_average_solution():
    # Test solver capability
    global average_nan_bin_abc_obj
    bin_abc_obj = deepcopy(average_nan_bin_abc_obj)
    bin_abc_obj.fit()
    out = bin_abc_obj.get_solution()
    ref = [False, False, False, False, True, False, False, False]
    npt.assert_array_equal(out, ref)


def test_bin_fit_average_probability():
    # Test solver capability
    global average_nan_bin_abc_obj
    bin_abc_obj = deepcopy(average_nan_bin_abc_obj)
    bin_abc_obj.fit()
    out = bin_abc_obj.get_solution(probability_vector = True)
    ref = [0.7310473617198062,
           0.014879869353197546,
           0.017970912760133604,
           0.0010584454334110485,
           0.985098442266522,
           0.19047384206718057,
           0.0014741775024230605,
           0.05895833890958545]
    npt.assert_array_almost_equal(out, ref, decimal=6)


def test_bin_nan_protection():
    # Test NaN protection in BABC
    global nan_bin_abc_obj
    bin_abc_obj = deepcopy(nan_bin_abc_obj)
    bin_abc_obj.fit()
    out1 = bin_abc_obj.get_status()
    out2 = bin_abc_obj.get_solution(probability_vector = False)
    ref1 = (30, 28, 36)
    ref2 = [False, False, False, False, True, False, False, False]
    npt.assert_array_equal(out1, ref1)
    npt.assert_array_equal(out2, ref2)


def test_bin_get_agents():
    # Verifies the process step-by-step
    global base_bin_abc_obj
    bin_abc_obj = deepcopy(base_bin_abc_obj)
    bin_abc_obj.fit()
    out = bin_abc_obj.get_agents()
    ref = [[[2.7885359691576745, -9.49978489554666, -4.499413632617615, -5.5357852370235445],
            [-1.561563606294591, -9.404055611238594, -5.627240503927933, 0.10710576206724731],
            [-5.591187559186066, 1.7853136775181753, 6.1886091335565325, -9.87002480643878],
            [9.144261444135623, -3.2681090977474643, -8.145083132397042, -8.06567246333072],
            [0.7245618290940143, 9.462315279587411, -2.42931245583293, 1.0408126254645396]],
           [[2.7885359691576745, -9.49978489554666, -4.499413632617615, -5.5357852370235445],
            [-1.561563606294591, -9.404055611238594, -5.627240503927933, 0.10710576206724731],
            [-5.591187559186066, 1.7853136775181753, 6.1886091335565325, -9.87002480643878],
            [9.144261444135623, -3.2681090977474643, -8.145083132397042, -8.06567246333072],
            [0.7245618290940143, 9.462315279587411, -2.42931245583293, 1.0408126254645396]],
           [[2.7885359691576745, -9.49978489554666, -4.499413632617615, -5.5357852370235445],
            [-1.561563606294591, -9.404055611238594, -5.627240503927933, 0.10710576206724731],
            [-5.591187559186066, 1.7853136775181753, 6.1886091335565325, -9.87002480643878],
            [9.144261444135623, -3.2681090977474643, -8.145083132397042, -8.06567246333072],
            [0.7245618290940143, 9.462315279587411, -2.42931245583293, 1.0408126254645396]],
           [[2.7885359691576745, -9.49978489554666, -4.499413632617615, -5.5357852370235445],
            [-1.561563606294591, -9.404055611238594, -5.627240503927933, 0.10710576206724731],
            [-5.591187559186066, 1.7853136775181753, 6.1886091335565325, -9.87002480643878],
            [9.144261444135623, -3.2681090977474643, -8.145083132397042, -8.06567246333072],
            [0.7245618290940143, 9.462315279587411, -2.42931245583293, 1.0408126254645396]],
           [[2.7885359691576745, -9.49978489554666, -4.499413632617615, -5.5357852370235445],
            [-1.561563606294591, -9.404055611238594, -5.627240503927933, 0.10710576206724731],
            [-5.591187559186066, 1.7853136775181753, 6.1886091335565325, -9.87002480643878],
            [9.144261444135623, -3.2681090977474643, -8.145083132397042, -8.06567246333072],
            [0.8967570342570317, 9.462315279587411, -2.42931245583293, 1.0408126254645396]],
           [[2.7885359691576745, -9.49978489554666, -4.499413632617615, -5.5357852370235445],
            [-1.561563606294591, -9.404055611238594, -5.627240503927933, 0.10710576206724731],
            [-5.591187559186066, 1.7853136775181753, 6.1886091335565325, -9.87002480643878],
            [9.144261444135623, -3.2681090977474643, -8.145083132397042, -8.06567246333072],
            [0.8967570342570317, 9.462315279587411, -2.42931245583293, 1.0408126254645396]],
           [[2.7885359691576745, -9.49978489554666, -4.499413632617615, -5.5357852370235445],
            [-1.561563606294591, -9.404055611238594, -5.627240503927933, 0.10710576206724731],
            [-5.310128439707293, -3.2830492675737215, 7.860536655806335, -8.38844684684663],
            [9.144261444135623, -3.2681090977474643, -8.145083132397042, -8.06567246333072],
            [0.8967570342570317, 9.462315279587411, -2.42931245583293, 1.0408126254645396]],
           [[-7.838025006847868, -9.48643149005068, -3.7608551121060962, 3.5469457370081763],
            [-1.561563606294591, -9.404055611238594, -5.627240503927933, 0.10710576206724731],
            [-5.310128439707293, -3.2830492675737215, 7.860536655806335, -8.38844684684663],
            [9.144261444135623, -3.2681090977474643, -8.145083132397042, -8.06567246333072],
            [0.8967570342570317, 9.462315279587411, -2.42931245583293, 1.0408126254645396]],
           [[-7.838025006847868, -9.48643149005068, -3.7608551121060962, 3.5469457370081763],
            [-1.561563606294591, -9.404055611238594, -5.627240503927933, 0.10710576206724731],
            [-5.310128439707293, -3.2830492675737215, 7.860536655806335, -8.38844684684663],
            [-4.470007460626215, 1.7917381192158182, 5.204157907461653, 1.9358148102460344],
            [0.8967570342570317, 9.462315279587411, -2.42931245583293, 1.0408126254645396]],
           [[-7.838025006847868, -9.48643149005068, -3.7608551121060962, 3.5469457370081763],
            [-1.2448227285680407, 1.9607332408220728, 7.788457785849072, 4.690678465712306],
            [-5.310128439707293, -3.2830492675737215, 7.860536655806335, -8.38844684684663],
            [-4.470007460626215, 1.7917381192158182, 5.204157907461653, 1.9358148102460344],
            [0.8967570342570317, 9.462315279587411, -2.42931245583293, 1.0408126254645396]],
           [[-7.838025006847868, -9.48643149005068, -3.7608551121060962, 3.5469457370081763],
            [-1.2448227285680407, 1.9607332408220728, 7.788457785849072, 4.690678465712306],
            [-5.310128439707293, -3.2830492675737215, 7.860536655806335, -8.38844684684663],
            [-4.470007460626215, 1.7917381192158182, 5.204157907461653, 1.9358148102460344],
            [-4.343220334343467, -4.028883004564657, 1.7387544482832222, 9.978046664586774]]]
    npt.assert_array_almost_equal(out, ref, decimal=6)


def test_bin_get_status():
    # Test exploration
    global base_bin_abc_obj
    bin_abc_obj = deepcopy(base_bin_abc_obj)
    bin_abc_obj.fit()
    out = bin_abc_obj.get_status()
    ref = (10, 5, 0)
    npt.assert_array_equal(out, ref)


def test_am_food_source_generation():
    # Test algorithm initialization
    global base_am_abc_obj
    out = [food.position for food in base_am_abc_obj._bin_abc_object.foods]
    ref = [[0.557707193831535, -1.8999569791093323, -0.899882726523523, -1.107157047404709],
           [0.9458848566560496, 0.7067979496916452, 1.5687182708193816, -1.6522446694823354],
           [-0.3123127212589183, -1.8808111222477186, -1.1254481007855865, 0.02142115241344955],
           [-1.8938561212645455, -1.204649397253406, 0.5995377511180928, 0.1797659224128667],
           [-1.1182375118372132, 0.3570627355036349, 1.2377218267113066, -1.974004961287756]]
    npt.assert_array_almost_equal(out, ref, decimal=6)


def test_am_fit_solution():
    # Test solver capability
    global base_am_abc_obj
    am_abc_obj = deepcopy(base_am_abc_obj)
    am_abc_obj.fit()
    out = am_abc_obj.get_solution()
    ref = [True, False, False, False]
    npt.assert_array_equal(out, ref)


def test_am_get_agents():
    # Verifies the process step-by-step
    global base_am_abc_obj
    am_abc_obj = deepcopy(base_am_abc_obj)
    am_abc_obj.fit()
    out = am_abc_obj.get_agents()
    ref = [[[0.557707193831535, -1.8999569791093323, -0.899882726523523, -1.107157047404709],
            [0.9458848566560496, 0.7067979496916452, 1.5687182708193816, -1.6522446694823354],
            [-0.3123127212589183, -1.8808111222477186, -1.1254481007855865, 0.02142115241344955],
            [-1.8938561212645455, -1.204649397253406, 0.5995377511180928, 0.1797659224128667],
            [-1.1182375118372132, 0.3570627355036349, 1.2377218267113066, -1.974004961287756]],
           [[0.557707193831535, -1.8893582439328636, -0.899882726523523, -1.107157047404709],
            [1.0161392006272225, 0.7067979496916452, 1.5687182708193816, -1.6522446694823354],
            [-0.3123127212589183, -1.8808111222477186, -1.1254481007855865, 0.02142115241344955],
            [-2, -1.204649397253406, 0.5995377511180928, 0.1797659224128667],
            [-1.1182375118372132, 0.3570627355036349, 1.2377218267113066, -1.974004961287756]],
           [[0.557707193831535, -1.8893582439328636, 0.3286988892938725, -1.107157047404709],
            [1.0161392006272225, 0.7067979496916452, 1.5687182708193816, -1.6522446694823354],
            [-0.3123127212589183, -1.8808111222477186, -1.1254481007855865, 0.02142115241344955],
            [-2, -1.204649397253406, 0.5995377511180928, 0.3057522684219449],
            [-1.1182375118372132, 0.3570627355036349, 1.2377218267113066, -1.974004961287756]],
           [[0.557707193831535, -1.8893582439328636, 0.3286988892938725, -1.107157047404709],
            [1.0161392006272225, 0.7067979496916452, 1.5687182708193816, -1.6522446694823354],
            [-0.3123127212589183, -1.8808111222477186, -1.1254481007855865, 0.02142115241344955],
            [-2, -1.204649397253406, 0.5995377511180928, 0.3057522684219449],
            [-1.1182375118372132, 0.3570627355036349, 1.2377218267113066, -1.974004961287756]],
           [[0.557707193831535, -1.8893582439328636, 0.3286988892938725, -1.7642204917687523],
            [0.3537604576414042, -1.9712366440596258, 0.8313637861938163, -1.7645049324947064],
            [-0.3123127212589183, -1.8808111222477186, -1.1254481007855865, 0.02142115241344955],
            [-2, -1.204649397253406, 0.5995377511180928, 0.3057522684219449],
            [-1.1182375118372132, 0.3570627355036349, 1.2377218267113066, -1.974004961287756]],
           [[0.557707193831535, -1.8893582439328636, 0.3286988892938725, -1.7642204917687523],
            [0.3537604576414042, -1.9712366440596258, 0.8313637861938163, -0.7438010528471708],
            [-0.3123127212589183, -1.8808111222477186, -1.1254481007855865, 0.02142115241344955],
            [-2, -1.204649397253406, 0.5995377511180928, 0.3057522684219449],
            [-1.1182375118372132, 0.3570627355036349, 1.140180015064122, -1.974004961287756]],
           [[0.557707193831535, -1.8893582439328636, 0.3286988892938725, -1.7642204917687523],
            [0.3537604576414042, -1.9712366440596258, 0.8313637861938163, -0.7438010528471708],
            [-0.3123127212589183, -1.8808111222477186, -1.1254481007855865, 0.02142115241344955],
            [-2, -1.204649397253406, 0.5995377511180928, 0.3057522684219449],
            [-1.1182375118372132, 0.3570627355036349, 1.140180015064122, -1.974004961287756]],
           [[0.557707193831535, -1.8893582439328636, 0.3286988892938725, -1.7642204917687523],
            [0.3537604576414042, -1.9712366440596258, 0.8313637861938163, -0.7438010528471708],
            [-0.3123127212589183, -1.8808111222477186, -0.10276323247356367, 0.02142115241344955],
            [-2, -1.204649397253406, 0.5995377511180928, 0.3057522684219449],
            [-1.1182375118372132, 0.3570627355036349, 1.140180015064122, -1.974004961287756]],
           [[0.557707193831535, -1.8893582439328636, 0.3286988892938725, -1.7642204917687523],
            [0.3537604576414042, -1.9712366440596258, 0.8313637861938163, -0.7438010528471708],
            [-0.3123127212589183, -1.8808111222477186, -0.10276323247356367, 0.02142115241344955],
            [-0.5852854615415266, 0.23953407657547876, 1.4988508941643648, 1.895348090549156],
            [-1.1182375118372132, 0.3570627355036349, 1.140180015064122, -1.974004961287756]],
           [[-0.617251553280799, 1.1457127128147278, -0.25442446451756817, 1.9369433972663597],
            [0.3537604576414042, -1.9712366440596258, 0.8313637861938163, -0.7438010528471708],
            [-0.3123127212589183, -1.8808111222477186, -0.10276323247356367, 0.02142115241344955],
            [-0.5852854615415266, 0.23953407657547876, 1.4988508941643648, 1.895348090549156],
            [-1.1182375118372132, 0.3570627355036349, 1.140180015064122, -1.974004961287756]],
           [[-0.617251553280799, 1.1457127128147278, -0.25442446451756817, 1.9369433972663597],
            [0.044695987498348355, 1.174941596424008, 1.839328014732145, 0.9438457837608869],
            [-0.3123127212589183, -1.8808111222477186, -0.10276323247356367, 0.02142115241344955],
            [-0.5852854615415266, 0.23953407657547876, 1.4988508941643648, 1.895348090549156],
            [-1.1182375118372132, 0.3570627355036349, 1.140180015064122, -1.974004961287756]]]
    npt.assert_array_almost_equal(out, ref, decimal=6)


def test_am_get_status():
    # Test exploration
    global base_am_abc_obj
    am_abc_obj = deepcopy(base_am_abc_obj)
    am_abc_obj.fit()
    out = am_abc_obj.get_status()
    ref = (10, 4, 0)
    npt.assert_array_almost_equal(out, ref)
