# coding: utf-8
from __future__ import unicode_literals

import re

from .common import InfoExtractor


class RadioFranceIE(InfoExtractor):
    _VALID_URL = r"^https?://maison\.radiofrance\.fr/radiovisions/(?P<id>[^?#]+)"
    IE_NAME = "radiofrance"

    _TEST = {
        "url": "http://maison.radiofrance.fr/radiovisions/one-one",
        "md5": "bdbb28ace95ed0e04faab32ba3160daf",
        "info_dict": {
            "id": "one-one",
            "ext": "ogg",
            "title": "One to one",
            "description": "Plutôt que d'imaginer la radio de demain comme technologie ou comme création de contenu, je veux montrer que quelles que soient ses évolutions, j'ai l'intime conviction que la radio continuera d'être un grand média de proximité pour les auditeurs.",
            "uploader": "Thomas Hercouët",
        },
    }

    def _real_extract(self, url):
        m = re.match(self._VALID_URL, url)
        video_id = m.group("id")

        webpage = self._download_webpage(url, video_id)
        title = self._html_search_regex(r"<h1>(.*?)</h1>", webpage, "title")
        description = self._html_search_regex(
            r'<div class="bloc_page_wrapper"><div class="text">(.*?)</div>',
            webpage,
            "description",
            fatal=False,
        )
        uploader = self._html_search_regex(
            r'<div class="credit">&nbsp;&nbsp;&copy;&nbsp;(.*?)</div>',
            webpage,
            "uploader",
            fatal=False,
        )

        formats_str = self._html_search_regex(
            r'class="jp-jplayer[^"]*" data-source="([^"]+)">', webpage, "audio URLs"
        )
        formats = [
            {
                "format_id": fm[0],
                "url": fm[1],
                "vcodec": "none",
                "preference": i,
            }
            for i, fm in enumerate(
                re.findall(r"([a-z0-9]+)\s*:\s*'([^']+)'", formats_str)
            )
        ]
        self._sort_formats(formats)

        return {
            "id": video_id,
            "title": title,
            "formats": formats,
            "description": description,
            "uploader": uploader,
        }
