# coding: utf-8
from __future__ import unicode_literals

import re

from .common import InfoExtractor
from ..utils import js_to_json


class C56IE(InfoExtractor):
    _VALID_URL = r"https?://(?:(?:www|player)\.)?56\.com/(?:.+?/)?(?:v_|(?:play_album.+-))(?P<textid>.+?)\.(?:html|swf)"
    IE_NAME = "56.com"
    _TESTS = [
        {
            "url": "http://www.56.com/u39/v_OTM0NDA3MTY.html",
            "md5": "e59995ac63d0457783ea05f93f12a866",
            "info_dict": {
                "id": "93440716",
                "ext": "flv",
                "title": "网事知多少 第32期：车怒",
                "duration": 283.813,
            },
        },
        {
            "url": "http://www.56.com/u47/v_MTM5NjQ5ODc2.html",
            "md5": "",
            "info_dict": {
                "id": "82247482",
                "title": "爱的诅咒之杜鹃花开",
            },
            "playlist_count": 7,
            "add_ie": ["Sohu"],
        },
    ]

    def _real_extract(self, url):
        mobj = re.match(self._VALID_URL, url, flags=re.VERBOSE)
        text_id = mobj.group("textid")

        webpage = self._download_webpage(url, text_id)
        sohu_video_info_str = self._search_regex(
            r"var\s+sohuVideoInfo\s*=\s*({[^}]+});",
            webpage,
            "Sohu video info",
            default=None,
        )
        if sohu_video_info_str:
            sohu_video_info = self._parse_json(
                sohu_video_info_str, text_id, transform_source=js_to_json
            )
            return self.url_result(sohu_video_info["url"], "Sohu")

        page = self._download_json(
            "http://vxml.56.com/json/%s/" % text_id, text_id, "Downloading video info"
        )

        info = page["info"]

        formats = [
            {"format_id": f["type"], "filesize": int(f["filesize"]), "url": f["url"]}
            for f in info["rfiles"]
        ]
        self._sort_formats(formats)

        return {
            "id": info["vid"],
            "title": info["Subject"],
            "duration": int(info["duration"]) / 1000.0,
            "formats": formats,
            "thumbnail": info.get("bimg") or info.get("img"),
        }
