"""Script registry for managing template scripts."""

import logging
from typing import Any

from .base import TemplateScript
from .script_loader import ScriptLoader, TemplateScriptError

logger = logging.getLogger(__name__)


class ScriptRegistry:
    """Central registry for managing template scripts."""

    def __init__(self, config_service: Any = None) -> None:
        """Initialize the script registry.

        Args:
            config_service: Optional configuration service to pass to scripts
        """
        self.config_service = config_service
        self.loader = ScriptLoader()
        self._initialized = False

    def initialize(self) -> None:
        """Initialize the registry by discovering available scripts."""
        if self._initialized:
            return

        try:
            self.loader.discover_scripts()
            self._initialized = True
            logger.info("Script registry initialized successfully")
        except Exception as e:
            logger.error(f"Failed to initialize script registry: {e}")
            raise TemplateScriptError(f"Script registry initialization failed: {e}") from e

    def get_script(self, script_name: str) -> TemplateScript:
        """Get a script instance by name.

        Args:
            script_name: Name of the script to retrieve

        Returns:
            Instance of the requested script

        Raises:
            TemplateScriptError: If script is not found
        """
        if not self._initialized:
            self.initialize()

        return self.loader.get_script(script_name, config_service=self.config_service)

    def has_script(self, script_name: str) -> bool:
        """Check if a script exists in the registry.

        Args:
            script_name: Name of the script to check

        Returns:
            True if script exists, False otherwise
        """
        if not self._initialized:
            self.initialize()

        return script_name in self.loader._loaded_scripts

    def list_scripts(self) -> dict[str, str]:
        """List all available scripts with their descriptions.

        Returns:
            Dictionary mapping script names to descriptions
        """
        if not self._initialized:
            self.initialize()

        return self.loader.list_available_scripts()

    def execute_script(self, script_name: str, **kwargs: Any) -> dict[str, Any]:
        """Execute a script and return its generated context.

        Args:
            script_name: Name of the script to execute
            **kwargs: Parameters to pass to the script

        Returns:
            Context dictionary generated by the script

        Raises:
            TemplateScriptError: If script execution fails
        """
        try:
            script = self.get_script(script_name)

            # Validate required parameters
            required_params = script.get_required_parameters()
            missing_params = [param for param in required_params if param not in kwargs]
            if missing_params:
                raise TemplateScriptError(f"Script '{script_name}' missing required parameters: {missing_params}")

            # Generate context
            context = script.generate_context(**kwargs)

            # Validate context
            if not script.validate_context(context):
                raise TemplateScriptError(f"Script '{script_name}' generated invalid context")

            logger.debug(f"Script '{script_name}' executed successfully")
            return context

        except TemplateScriptError:
            raise
        except Exception as e:
            raise TemplateScriptError(f"Failed to execute script '{script_name}': {e}") from e

    def reload_scripts(self) -> None:
        """Reload all scripts from disk."""
        self.loader.reload_scripts()
        logger.info("Scripts reloaded in registry")

    def get_script_info(self, script_name: str) -> dict[str, Any]:
        """Get detailed information about a script.

        Args:
            script_name: Name of the script

        Returns:
            Dictionary with script information

        Raises:
            TemplateScriptError: If script is not found
        """
        script = self.get_script(script_name)

        return {
            "name": script.name,
            "description": script.description,
            "required_parameters": script.get_required_parameters(),
            "optional_parameters": script.get_optional_parameters(),
            "class_name": script.__class__.__name__,
            "module": script.__class__.__module__,
        }
