"""Template service implementation."""

import logging
import os
from typing import TYPE_CHECKING, Any

if TYPE_CHECKING:
    from printerm.services.interfaces import ConfigService
import yaml
from jinja2 import BaseLoader, Environment
from mistune import Markdown
from unidecode import unidecode

from printerm.exceptions import TemplateError
from printerm.services.markdown_renderer import PrinterRenderer
from printerm.templates.scripts import ScriptRegistry
from printerm.templates.scripts.script_loader import TemplateScriptError

logger = logging.getLogger(__name__)
PRINT_TEMPLATE_FOLDER = os.path.join(os.path.dirname(os.path.realpath(__file__)), "..", "templates", "print_templates")


class TemplateServiceImpl:
    """Template service implementation."""

    def __init__(
        self, template_dir: str = PRINT_TEMPLATE_FOLDER, config_service: "ConfigService | None" = None
    ) -> None:
        self.template_dir = template_dir
        self.config_service = config_service
        self.templates: dict[str, dict[str, Any]] = {}
        self.script_registry = ScriptRegistry(config_service=config_service)
        self.load_templates()

    def load_templates(self) -> None:
        """Load all templates from the template directory."""
        try:
            self.templates = {}
            if not os.path.exists(self.template_dir):
                raise TemplateError(f"Template directory not found: {self.template_dir}")
            for filename in os.listdir(self.template_dir):
                if filename.endswith(".yaml"):
                    path = os.path.join(self.template_dir, filename)
                    try:
                        with open(path, encoding="utf-8") as file:
                            template = yaml.safe_load(file)
                            key = os.path.splitext(filename)[0]
                            self.templates[key] = template
                    except Exception as e:
                        logger.warning(f"Failed to load template {filename}: {e}")
        except Exception as e:
            raise TemplateError("Failed to load templates", str(e)) from e

    def get_template(self, name: str) -> dict[str, Any]:
        """Get a template by name."""
        template = self.templates.get(name)
        if not template:
            raise TemplateError(f"Template '{name}' not found")
        return template

    def list_templates(self) -> list[str]:
        """List all available template names."""
        return list(self.templates.keys())

    def render_template(
        self, template_name: str, context: dict[str, Any] | None = None, **script_kwargs: Any
    ) -> list[dict[str, Any]]:
        """Render a template with the given context or auto-generated context to unwrapped segments.
        Args:
            template_name: Name of the template to render
            context: Optional manual context. If None, will try to generate using scripts
            **script_kwargs: Additional parameters to pass to template scripts
        Returns:
            List of rendered segments (unwrapped, may contain long lines)
        """
        # Generate context if not provided
        if context is None:
            context = self.generate_template_context(template_name, **script_kwargs)
        template = self.get_template(template_name)
        # Get configuration settings
        chars_per_line = 32
        enable_special_letters = False
        if self.config_service:
            chars_per_line = self.config_service.get_chars_per_line()
            enable_special_letters = self.config_service.get_enable_special_letters()
        segments = template.get("segments", [])
        rendered_segments = []
        env = Environment(loader=BaseLoader(), autoescape=True, keep_trailing_newline=True)
        try:
            for segment in segments:
                template_text = segment["text"]
                jinja_template = env.from_string(template_text)
                try:
                    text = jinja_template.render(**context)
                except Exception as e:
                    raise TemplateError(f"Error rendering template '{template_name}'", str(e)) from e
                if not enable_special_letters:
                    logger.debug("Transliterating text to ASCII")
                    text = unidecode(text)
                if segment.get("markdown", False):
                    # Use Mistune to parse markdown and render segments
                    renderer = PrinterRenderer(chars_per_line)
                    markdown = Markdown(renderer=renderer)
                    markdown(text)
                    # Process the segments generated by the renderer (no wrapping here)
                    for seg in renderer.segments:
                        text = seg["text"]
                        styles = seg.get("styles", {})
                        rendered_segments.append({"text": text, "styles": styles})
                else:
                    rendered_segments.append({"text": text, "styles": segment.get("styles", {})})
            return rendered_segments
        except Exception as e:
            if isinstance(e, TemplateError):
                raise
            raise TemplateError(f"Failed to render template '{template_name}'", str(e)) from e

    def generate_template_context(
        self, template_name: str, manual_context: dict[str, Any] | None = None, **script_kwargs: Any
    ) -> dict[str, Any]:
        """Generate context for template rendering using scripts or manual input.
        Args:
            template_name: Name of the template
            manual_context: Manually provided context variables (optional)
            **script_kwargs: Additional parameters to pass to template scripts
        Returns:
            Dictionary of context variables for template rendering
        """
        template = self.get_template(template_name)
        # Check if template has a script defined
        script_name = template.get("script")
        if script_name:
            try:
                # Use script to generate context
                logger.debug(f"Using script '{script_name}' for template '{template_name}'")
                context = self.script_registry.execute_script(script_name, **script_kwargs)
                # Merge with any manual context provided
                if manual_context:
                    context.update(manual_context)
                return context
            except TemplateScriptError as e:
                raise TemplateError(f"Script execution failed for template '{template_name}'", str(e)) from e
            except Exception as e:
                logger.warning(f"Script '{script_name}' failed, falling back to manual input: {e}")
        # Fall back to manual context if no script or script failed
        if manual_context is None:
            manual_context = {}
        return manual_context

    def get_template_variables(self, template_name: str) -> list[dict[str, Any]]:
        """Get variables required for a template.
        Args:
            template_name: Name of the template
        Returns:
            List of variable definitions with name, description, required, etc.
        """
        template = self.get_template(template_name)
        return template.get("variables", [])

    def has_script(self, template_name: str) -> bool:
        """Check if a template has an associated script.
        Args:
            template_name: Name of the template
        Returns:
            True if template has a script, False otherwise
        """
        template = self.get_template(template_name)
        script_name = template.get("script")
        return script_name is not None and self.script_registry.has_script(script_name)

    def get_script_info(self, template_name: str) -> dict[str, Any] | None:
        """Get information about a template's script.
        Args:
            template_name: Name of the template
        Returns:
            Script information dictionary or None if no script
        """
        template = self.get_template(template_name)
        script_name = template.get("script")
        if script_name and self.script_registry.has_script(script_name):
            return self.script_registry.get_script_info(script_name)
        return None

    def list_available_scripts(self) -> dict[str, str]:
        """List all available template scripts.
        Returns:
            Dictionary mapping script names to descriptions
        """
        return self.script_registry.list_scripts()
