from .. import polygon
from ..polygon import PolygonSection
import numpy as np


class _SimpleShape(PolygonSection):
    def __init__(self, exterior, interior, mesh_size=None, **kwds):
        self._mesh_kwds = {}

        super().__init__(
            exterior=exterior,
            interior=interior,
            mesh_size=mesh_size,
            **kwds)

    def translate(self, offset, **kwd):
        typ = type(self) 

        kwds = {k: getattr(self, k) for k in typ._parameters}
        new = typ(**kwds, centroid=offset+self.centroid, **kwd)
        new.material = self.material
        new.z = getattr(self, "z", 0)
        return new



class WideFlange(_SimpleShape):
    _parameters = ["d", "bf", "tf", "tw", "b2", "t2", "k", "mesh_scale"]

    def __init__(self, d, 
                 bf=None, 
                 tw=None, tf=None,
                 t=None, 
                 b=None,
                 b2=None, t2=None,
                 k = None,
                 centroid=None,
                 mesh_scale=None,
                 **kwds):

        if centroid is None:
            centroid = np.zeros(2)

        if t is None and tf is None and tw is None:
            raise TypeError("Must provide t, tf, or tw")
        if b is None and bf is None:
            raise TypeError("Must provide b or bf")

        if t is not None and tf is not None and t != tf:
            raise ValueError("Got both t and tf")

        if t is not None and tw is not None and t != tw:
            raise ValueError("Got both t and tw")
        elif tf is None:
            tf = t
        elif t is None:
            t = tf
        
        if b is not None and bf is not None and b != bf:
            raise ValueError("Got both b and bf")
        elif b is None:
            b = bf 
        elif bf is None:
            bf = b

        self.d  = d
        self.bf = bf
        if tf is None:
            tf = t
        self.tf = tf

        if b2 is None:
            b2 = b
        self.b2 = b2

        if t2 is None:
            t2 = tf
        self.t2 = t2

        if tw is None:
            tw = tf
        self.tw = tw

        self.k = k

        if mesh_scale is None:
            mesh_scale = 1/3
        self.mesh_scale = mesh_scale

        #
        #
        #
        bf = self.bf
        b2 = self.b2
        tf = self.tf
        t2 = self.t2
        tw = self.tw
        d = self.d

        y_top = d / 2.0
        y_bot = -d / 2.0

        exterior = np.array([
            [-bf/2,   y_top],         # 1  top-left flange
            [ bf/2,   y_top],         # 2
            [ bf/2,   y_top - tf],    # 3  step down into web
            [ tw/2,   y_top - tf],    # 4
            [ tw/2,   y_bot + t2],    # 5  down web
            [ b2/2,   y_bot + t2],    # 6  step into bottom flange
            [ b2/2,   y_bot],         # 7
            [-b2/2,   y_bot],         # 8
            [-b2/2,   y_bot + t2],    # 9
            [-tw/2,   y_bot + t2],    # 10 up web
            [-tw/2,   y_top - tf],    # 11
            [-bf/2,   y_top - tf],    # 12
        ], dtype=float)

        super().__init__(mesh_size=min(tf, tw)*mesh_scale,
                         exterior = exterior+polygon.centroid(exterior)-centroid,
                         interior = None,
                         **kwds)


#         # Area and moment of inertia
#         self.A  = tw*(d - tf - t2) + bf*tf + b2*t2

# #       Iy and Iz are wrong for tf !=  t2
# #       self.Iy = tw*(d - tf - t2)**3/12.0 + bf*tf*(0.5*(d - tf))**2 + b2*t2*(0.5*(d - t2)) ** 2
#         self.Iz = 2*tf*bf**3/12

    def shear_factor(self, nu=0.3):
        b  = self.bf
        tf = self.tf
        tw = self.tw
        d  = self.d

        m = 2*b*tf/(d*tw)
        n = b/d
        return (10*(1+nu)*(1+3*m)**2)/((12+72*m + 150*m**2 + 90*m**3) + nu*(11+66*m + 135*m**2 + 90*m**3) + 30*n**2*(m + m**2) + 5*nu*n**2*(8*m+9*m**2))


    def _create_patches(self, mesh_scale=None, mesh_kwds={}):

        from shps.frame import patch

        if mesh_scale is None:
            mesh_scale = 1/3
        bf = self.bf
        b2 = self.b2
        tf = self.tf
        t2 = self.t2
        tw = self.tw
        d  = self.d

        if self.k is None:
            quads = []
        else:
            k = self.k
            r = k - tf
            h = d / 2 - tf  # y coordinate of top of flange

            quads = [
                patch.quad(vertices=[
                    (-tw/2,     d/2- k),
                    (-tw/2,     h),
                    (-tw/2 - r, h),
                    (-tw/2 - r + r/2**0.5, d/2-k + r/2**0.5),
                ]),
                patch.quad(vertices=[
                    ( tw/2,     d/2-k),
                    ( tw/2,     h),
                    ( tw/2 + r, h),
                    ( tw/2 + r - r/2**0.5, d/2-k + r/2**0.5),
                ]),
                patch.quad(vertices=[
                    (-tw/2,    -h + r),
                    (-tw/2,    -h),
                    (-tw/2 - r, -h),
                    (-tw/2 - r + r/2**0.5, -d/2 + k - r/2**0.5),
                ]),
                patch.quad(vertices=[
                    ( tw/2,    -h + r),
                    ( tw/2,    -h),
                    ( tw/2 + r, -h),
                    ( tw/2 + r - r/2**0.5, -d/2 + k - r/2**0.5),
                ]),
            ]

        yoff = ( d - tf) / 2

        return [
            patch.rect(corners=[[-bf/2,        yoff-tf/2],[bf/2,  yoff+tf/2]]),# ,  divs=(nfl, nft), rule=int_typ),
            patch.rect(corners=[[-tw/2,       -yoff+tf/2],[tw/2,  yoff-tf/2]]),# ,  divs=(nwt, nwl), rule=int_typ),
            patch.rect(corners=[[-b2/2, -(d - t2)/2-t2/2],[bf/2, -yoff+tf/2]]),# ,  divs=(nfl, nft), rule=int_typ),
            *quads
        ]


    # def _create_model(self, mesh_scale=None):
    #     """
    #     Saritas and Filippou (2009) "Frame Element for Metallic Shear-Yielding Members under Cyclic Loading"
    #     """
    #     b  = self.bf
    #     tf = self.tf
    #     tw = self.tw
    #     d  = self.d

    #     # Shear from Saritas and Filippou (2009)
    #     # Ratio of total flange area to web area
    #     alpha = 2*b*tf/d/(2*tw)
    #     # NOTE: This is 1/beta_S where beta_S is Afsin's beta
    #     beta = (1+3*alpha)*(2/3)/((1+2*alpha)**2-2/3*(1+2*alpha)+1/5)
    #     def psi(y, z):
    #         # webs
    #         if abs(y) < (d/2-tf):
    #             return 0 #beta*((1+2*alpha) - (2*y/d)**2) - 1 #+ 1
    #         # flange
    #         else:
    #             return 0 #beta*(2*alpha)*(z/b) - 1

    #     mesh = self._create_mesh(mesh_scale=mesh_scale)

    #     return TriangleModel.from_meshio(mesh, warp_shear=psi)



class Rectangle(_SimpleShape):
    _parameters = ["b", "d", "mesh_scale"]
    def __init__(self, b, d, centroid=None, mesh_scale=None, **kwds):
        self.b = b
        self.d = d
        if centroid is None:
            centroid = np.zeros(2)

        self.mesh_scale = mesh_scale or 1/10

        mesh_size = min(b,d)*self.mesh_scale 

        super().__init__(mesh_size=mesh_size,
                         interior=None,
                         exterior = -centroid + np.array([
                            [-b / 2,  -d / 2],
                            [ b / 2,  -d / 2],
                            [ b / 2,   d / 2],
                            [-b / 2,   d / 2],
                        ]),
                        **kwds)

class Circle(_SimpleShape):
    _parameters = ["radius", "mesh_scale", "divisions"]
    def __init__(self, radius, centroid=None, mesh_scale=None, divisions=60, **kwds):
        self.r = self.radius = radius
        self.divisions = divisions
        if centroid is None:
            centroid = np.zeros(2)

        self.mesh_scale = mesh_scale or 1/10

        mesh_size = radius*self.mesh_scale
        exterior = np.array([
            [radius*np.cos(theta), radius*np.sin(theta)]
            for theta in np.linspace(0, 2*np.pi, divisions)
        ], dtype=float)

        super().__init__(mesh_size=mesh_size,
                         interior=None,
                         exterior = -centroid + exterior,
                        **kwds)

class HollowRectangle(_SimpleShape):
    _parameters = ["b", "d", "tf", "tw"]
    def __init__(self, b, d, tf, tw, t=None, **kwds):
        self.b = b
        self.d = d
        self.tf = tf 
        self.tw = tw

        exterior = np.array([
            [-b / 2, -d / 2],
            [ b / 2, -d / 2],
            [ b / 2,  d / 2],
            [-b / 2,  d / 2],
        ], dtype=float)

        tf, tw = self.tf, self.tw
        interior = [np.array([
                [-(b/2 - tw), -(d/2 - tf)],
                [ (b/2 - tw), -(d/2 - tf)],
                [ (b/2 - tw),  (d/2 - tf)],
                [-(b/2 - tw),  (d/2 - tf)],
            ], dtype=float)]
        super().__init__(mesh_size=min(tf, tw)/10,
                         exterior=exterior,
                         interior=interior, **kwds)

    

class Channel(_SimpleShape):
    """
    _  ___________
      |__________|
      | |
      | |
      |o|
      | |
      | |_________
      |__________|


    """
    _parameters = ["d", "b", "tf", "tw", "mesh_scale"]
    def __init__(self, d, b, tf, tw=None, mesh_scale=1/4, **kwds):
        self.tf = tf
        self.tw = tw if tw is not None else tf
        self.d = d
        self.b = b

        mesh_size = min(tf, tw)*mesh_scale
        self.mesh_scale = mesh_scale

        y_top =  d/2
        y_bot = -d/2

        exterior = np.array([
            [   - tw / 2, y_top],        # 1  top-left
            [ b - tw / 2, y_top],        # 2  top-right
            [ b - tw / 2, y_top - tf],    # 3  down into flange
            [     tw / 2, y_top - tf],    # 4  over to web
            [     tw / 2, y_bot + tf],    # 5  down web
            [ b - tw / 2, y_bot + tf],    # 6  out to bottom flange
            [ b - tw / 2, y_bot],        # 7  bottom-right
            [   - tw / 2, y_bot],        # 8  bottom-left
        ], dtype=float)
        super().__init__(exterior=exterior, mesh_size=mesh_size, **kwds)





class Angle(_SimpleShape):
    def __init__(self, t, b, d, **kwds):
        self.t = t
        self.b = b
        self.d = d
        super().__init__(mesh_size=min(t, b)/4, **kwds)

    def exterior(self):
        t = self.t
        b = self.b
        d = self.d

        pts = np.array([
            [ b - t / 2,  t / 2],          # 1  outer top-right
            [-t / 2,      t / 2],          # 2  outer top-left
            [-t / 2, -d + t / 2],          # 3  outer bottom-left
            [ t / 2, -d + t / 2],          # 4  inner bottom-left of vertical leg
            [ t / 2,     -t / 2],          # 5  inner corner
            [ b - t / 2, -t / 2],          # 6  outer bottom-right of horizontal leg
        ], dtype=float)

        return pts

