"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToSqs = void 0;
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
const cdk = require("@aws-cdk/core");
/**
 * @summary The ApiGatewayToSqs class.
 */
class ApiGatewayToSqs extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayToSqs class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {ApiGatewayToSqsProps} props - user provided props for the construct.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the dead letter queue, if applicable
        this.deadLetterQueue = defaults.buildDeadLetterQueue(this, {
            existingQueueObj: props.existingQueueObj,
            deployDeadLetterQueue: props.deployDeadLetterQueue,
            deadLetterQueueProps: props.deadLetterQueueProps,
            maxReceiveCount: props.maxReceiveCount
        });
        // Setup the queue
        [this.sqsQueue] = defaults.buildQueue(this, 'queue', {
            queueProps: props.queueProps,
            deadLetterQueue: this.deadLetterQueue
        });
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, props.apiGatewayProps);
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        // Setup the API Gateway resource
        const apiGatewayResource = this.apiGateway.root.addResource('message');
        // Create
        let createRequestTemplate = "Action=SendMessage&MessageBody=$util.urlEncode(\"$input.body\")";
        if (props.createRequestTemplate) {
            createRequestTemplate = props.createRequestTemplate;
        }
        if (props.allowCreateOperation && props.allowCreateOperation === true) {
            this.addActionToPolicy("sqs:SendMessage");
            defaults.addProxyMethodToApiResource({
                service: "sqs",
                path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "POST",
                apiResource: this.apiGateway.root,
                requestTemplate: createRequestTemplate,
                contentType: "'application/x-www-form-urlencoded'"
            });
        }
        // Read
        let readRequestTemplate = "Action=ReceiveMessage";
        if (props.readRequestTemplate) {
            readRequestTemplate = props.readRequestTemplate;
        }
        if (!props.allowReadOperation || props.allowReadOperation === true) {
            this.addActionToPolicy("sqs:ReceiveMessage");
            defaults.addProxyMethodToApiResource({
                service: "sqs",
                path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "GET",
                apiResource: this.apiGateway.root,
                requestTemplate: readRequestTemplate,
                contentType: "'application/x-www-form-urlencoded'"
            });
        }
        // Delete
        let deleteRequestTemplate = "Action=DeleteMessage&ReceiptHandle=$util.urlEncode($input.params('receiptHandle'))";
        if (props.deleteRequestTemplate) {
            deleteRequestTemplate = props.deleteRequestTemplate;
        }
        if (props.allowDeleteOperation && props.allowDeleteOperation === true) {
            this.addActionToPolicy("sqs:DeleteMessage");
            defaults.addProxyMethodToApiResource({
                service: "sqs",
                path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "DELETE",
                apiResource: apiGatewayResource,
                requestTemplate: deleteRequestTemplate,
                contentType: "'application/x-www-form-urlencoded'"
            });
        }
    }
    addActionToPolicy(action) {
        this.apiGatewayRole.addToPolicy(new iam.PolicyStatement({
            resources: [
                this.sqsQueue.queueArn
            ],
            actions: [`${action}`]
        }));
    }
}
exports.ApiGatewayToSqs = ApiGatewayToSqs;
//# sourceMappingURL=data:application/json;base64,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