/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import * as api from '@aws-cdk/aws-apigateway';
import * as sqs from '@aws-cdk/aws-sqs';
import * as iam from '@aws-cdk/aws-iam';
import { Construct } from '@aws-cdk/core';
import { LogGroup } from '@aws-cdk/aws-logs';
/**
 * @summary The properties for the ApiGatewayToSqs class.
 */
export interface ApiGatewayToSqsProps {
    /**
     * Optional user-provided props to override the default props for the API Gateway.
     *
     * @default - Default properties are used.
     */
    readonly apiGatewayProps?: api.RestApiProps | any;
    /**
     * Existing instance of SQS queue object, if this is set then the queueProps is ignored.
     *
     * @default - None
     */
    readonly existingQueueObj?: sqs.Queue;
    /**
     * User provided props to override the default props for the SQS queue.
     *
     * @default - Default props are used
     */
    readonly queueProps?: sqs.QueueProps;
    /**
     * Whether to deploy a secondary queue to be used as a dead letter queue.
     *
     * @default - required field.
     */
    readonly deployDeadLetterQueue?: boolean;
    /**
     * Optional user provided properties for the dead letter queue
     *
     * @default - Default props are used
     */
    readonly deadLetterQueueProps?: sqs.QueueProps;
    /**
     * The number of times a message can be unsuccessfully dequeued before being moved to the dead-letter queue.
     *
     * @default - required only if deployDeadLetterQueue = true.
     */
    readonly maxReceiveCount?: number;
    /**
     * Whether to deploy an API Gateway Method for Create operations on the queue (i.e. sqs:SendMessage).
     *
     * @default - false
     */
    readonly allowCreateOperation?: boolean;
    /**
     * API Gateway Request template for Create method, if allowCreateOperation set to true
     *
     * @default - None
     */
    readonly createRequestTemplate?: string;
    /**
     * Whether to deploy an API Gateway Method for Read operations on the queue (i.e. sqs:ReceiveMessage).
     *
     * @default - "Action=SendMessage&MessageBody=$util.urlEncode(\"$input.body\")"
     */
    readonly allowReadOperation?: boolean;
    /**
     * API Gateway Request template for Get method, if allowReadOperation set to true
     *
     * @default - "Action=ReceiveMessage"
     */
    readonly readRequestTemplate?: string;
    /**
     * Whether to deploy an API Gateway Method for Delete operations on the queue (i.e. sqs:DeleteMessage).
     *
     * @default - false
     */
    readonly allowDeleteOperation?: boolean;
    /**
     * API Gateway Request template for Delete method, if allowDeleteOperation set to true
     *
     * @default - "Action=DeleteMessage&ReceiptHandle=$util.urlEncode($input.params('receiptHandle'))"
     */
    readonly deleteRequestTemplate?: string;
}
/**
 * @summary The ApiGatewayToSqs class.
 */
export declare class ApiGatewayToSqs extends Construct {
    readonly apiGateway: api.RestApi;
    readonly apiGatewayRole: iam.Role;
    readonly apiGatewayCloudWatchRole: iam.Role;
    readonly apiGatewayLogGroup: LogGroup;
    readonly sqsQueue: sqs.Queue;
    readonly deadLetterQueue?: sqs.DeadLetterQueue;
    /**
     * @summary Constructs a new instance of the ApiGatewayToSqs class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {ApiGatewayToSqsProps} props - user provided props for the construct.
     * @since 0.8.0
     * @access public
     */
    constructor(scope: Construct, id: string, props: ApiGatewayToSqsProps);
    private addActionToPolicy;
}
