#!/usr/bin/env node

// ****************************************************
// unified - an ALPS-to-??? translator
//
// author:  @mamund
// date:    2020-04
//
// desc:    translates ALPS.[yaml\json] into:
//          - ALPS.json
//          - SDL
//          - protobuf
//          - openAPI
//          - asyncAPI
//          - WSDL
//
// notes    install as npm install -g .
//          proof-of-concept utility (needs work)
// ****************************************************

// modules
const chalk = require("chalk");
const boxen = require("boxen");
const yargs = require("yargs");
const YAML = require("yamljs");
const fs = require('fs');

// args
const options = yargs
 .usage("Usage: -f <alpsfile> -t <format type> -o <outfile>")
 .option("f", { alias: "file", 
    describe: "Input file (alps.[yaml|JSON])", 
    type: "string", demandOption: true })
 .option("t", { alias: "type", 
    describe: "Format Type \n([j]son, [p]roto, [s]dl, [a]syncapi, [o]penapi, [w]sdl)",
    type: "string", demandOption: false})
 .option("o", { alias: "out", 
    describe: "Output file", 
    type: "string", demandOption: false})
 .argv;

// cleanup regex
const rxHash = /#/g;
const rxQ = /\?\?/g;

// init vars
var alps_document = {};
var format = "json";
var rtn = "";

// **************************
// read incoming file
//
// first, assume incoming file is YAML
try {
  var file = `${process.cwd()}/${options.file}`;
  alps_document = YAML.load(file);
} 
catch(err) {
  // ok, assume incoming file is JSON
  try {
    var file = `${process.cwd()}/${options.file}`;
    var doc = fs.readFileSync(file);
    alps_document = JSON.parse(doc);
  }
  catch(err) {
    // not JSON, either!
    console.log(`ERROR: ${err}`);
  }
}

// selection translation
try {
  format = options.type.toLowerCase();
} 
catch {
  format = "json";
}

// process requested translation
switch (format) {
  case "s":
  case "sdl":
    rtn = toSDL(alps_document);
    break;
  case "a":
  case "async":
  case "asyncapi":
    rtn = toAsync(alps_document);
    break;
  case "o":		
  case "oas":
  case "open":
  case "openapi":
    rtn = toOAS(alps_document);
    break;
  case "p":
  case "proto":
    rtn = toProto(alps_document);
    break;
  case "j":
  case "json":
    rtn = toJSON(alps_document);
    break;		
  case "w":
  case "wsdl":
  case "soap":
    rtn = toWSDL(alps_document);
    break;		
  default:
    console.log(`ERROR: unknown format: ${format}`);
}

// output directly
if(options.out) {
  try {
    fs.writeFileSync(options.out, rtn);
  } 
  catch(err) {
    console.log(`ERROR: ${err}`);
  }
}
else {
  console.log(rtn);
}

// *******************************************
// translators
// *******************************************

// ****************************************************
// to WSDL
// ****************************************************
function toWSDL(doc) {
  var rtn = ""; 
  
  rtn += `<?xml version = '1.0' encoding = 'UTF-8'?>\n`;
  rtn += `<!-- generated by "unified" from ${options.file} -->\n`;
  rtn += `<!-- created: ${new Date()} -->\n`;
  rtn += '<!-- source: http://github.com/mamund/2020-11-unified -->\n';
  rtn += '\n';
  rtn += "<definitions>\n";
  rtn += "  <todo />\n";
  rtn += "</definitions>\n";  
  return rtn
}

// ****************************************************
// to ALPS JSON
// ****************************************************
function toJSON(doc) {
  var rtn = ""; 
  try {
    rtn = JSON.stringify(doc, null, 2);
  }
  catch(err) {
    console.log(`ERROR: ${err}`);
  }
  return rtn
}

// ****************************************************
// to proto file
// passes https://protogen.marcgravell.com/ validator
// ****************************************************
function toProto(doc) {
  var rtn = "";
  var obj;
  var coll;

  // preamble
  rtn += 'syntax = "proto3";\n';
  rtn += `package ${doc.alps.ext.filter(metadata_title)[0].value.replace(/ /g,'_')||"ALPS_API"};\n`;
  rtn += '\n';

  // signature
  rtn += '// *******************************************************************\n';
  rtn += `// generated by "unified" from ${options.file}\n`;
  rtn += `// date: ${new Date()}`;
  rtn += '\n';
  rtn += '// http://github.com/mamund/2020-11-unified\n';
  rtn += '// *******************************************************************\n';
  rtn += '\n';
  
  // params
  coll = doc.alps.descriptor.filter(semantic);
  coll.forEach(function(msg) {
    rtn += `message ${msg.id}Params {\n`;
    var c = 0;
    c++;
    rtn += `  string ${msg.id} = ${c};\n`;    
    rtn += '}\n';
  });
  rtn += '\n';

  // objects
  coll = doc.alps.descriptor.filter(groups);
  coll.forEach(function(msg) {
    rtn += `message ${msg.id} {\n`;
    var c = 0;
    msg.descriptor.forEach(function(prop) {
      c++;
      rtn += `  string ${prop.href} = ${c};\n`;    
    });
    rtn += '}\n';
    rtn += `message ${msg.id}Response {\n`;
    rtn += `  repeated ${msg.id} ${msg.id}Collection = 1;\n`
    rtn += '}\n';
    rtn += `message ${msg.id}Empty {}\n`;
  });
  rtn += '\n';

  // procedures
  rtn += `service ${doc.alps.ext.filter(metadata_title)[0].value.replace(/ /g,'_')||"ALPS_API"}_Service {\n`;
  
  coll = doc.alps.descriptor.filter(safe);
  coll.forEach(function(item) {
    rtn += `  rpc ${item.id}(`
    if(item.descriptor) {
      rtn += item.descriptor[0].href;      
    }
    else {
      rtn += `${item.rt}Empty`;
    }
    rtn += `) returns (${item.rt}Response) {};\n`;  
  });
  
  coll = doc.alps.descriptor.filter(unsafe);
  coll.forEach(function(item) {
    rtn += `  rpc ${item.id}(`
    if(item.descriptor) {
      rtn += item.descriptor[0].href;      
    }
    rtn += `) returns (${item.rt}Response) {};\n`;  
  });

  coll = doc.alps.descriptor.filter(idempotent);
  coll.forEach(function(item) {
    rtn += `  rpc ${item.id}(`
    if(item.descriptor) {
      rtn += item.descriptor[0].href;
      if(item.descriptor[0].href === "#id") {
        rtn += "Params";
      }      
    }
    rtn += `) returns (${item.rt}Response) {};\n`;  
  });
  
  rtn += '}\n';
 
  // clean up 
  rtn = rtn.replace(rxHash,"");
  rtn = rtn.replace(rxQ,"#");
   
  return rtn;
}

// *******************************************
// to graphql sdl
// passes https://app.graphqleditor.com/
// *******************************************
function toSDL(doc) {
  var rtn = "";
  var coll;

  // signature
  rtn += '?? *******************************************************************\n';
  rtn += `?? generated by "unified" from ${options.file}\n`;
  rtn += `?? date: ${new Date()}`;
  rtn += '\n';
  rtn += '?? http://github.com/mamund/2020-11-unified\n';
  rtn += '?? *******************************************************************\n';
  rtn += '\n';

  // types
  coll = doc.alps.descriptor.filter(groups);
  coll.forEach(function(item) {
    rtn += `type ${item.id} {\n`;
    item.descriptor.forEach(function(prop) {
      rtn += `  ${prop.href}: String!\n`;    
    });
    rtn += '}\n';
  }); 
  rtn += '\n';
  
  // query
  coll = doc.alps.descriptor.filter(safe);
  coll.forEach(function(item) {
    rtn += 'type Query {\n';
    rtn += `  ${item.id}: [${item.rt}]\n`;
    rtn += '}\n';
  });
  rtn += '\n';

  // mutations
  rtn += 'type Mutation {\n';
  coll = doc.alps.descriptor.filter(unsafe);
  coll.forEach(function(item) {
    rtn += `  ${item.id}(`;
    if(item.descriptor) {
      rtn += `${item.descriptor[0].href}: String!`;
    }  
    rtn += `): ${item.rt}\n`;
  });                       
  coll = doc.alps.descriptor.filter(idempotent);
  coll.forEach(function(item) {
    rtn += `  ${item.id}(`;
    if(item.descriptor) {
      rtn += `${item.descriptor[0].href}: String!`;
    }  
    rtn += `): ${item.rt}\n`;  
  });                       
  rtn += '}\n';

  // final schema declaration
  rtn += '\n';
  rtn += 'schema {\n';
  rtn += '  query: Query,\n';
  rtn += '  mutation: Mutation\n';
  rtn += '}\n';
  
  rtn = rtn.replace(rxHash,"");
  rtn = rtn.replace(rxQ,"#");
  
  return rtn;
}

// ***************************************************
// to OpenAPI document
// passes https://apitools.dev/swagger-parser/online/
// ***************************************************
function toOAS(doc) {
  var rtn = "";

  // preamble
  rtn += "openapi: 3.0.1\n";
  rtn += "\n";
  
  // signature
  rtn += '?? *******************************************************************\n';
  rtn += `?? generated by "unified" from ${options.file}\n`;
  rtn += `?? date: ${new Date()}`;
  rtn += '\n';
  rtn += '?? http://github.com/mamund/2020-11-unified\n';
  rtn += '?? *******************************************************************\n';
  rtn += '\n';
  
    
  // info section
  rtn += "info:\n";
  rtn += `  title: ${doc.alps.ext.filter(metadata_title)[0].value||"ALPS API"}\n`;
  rtn += `  description: ${doc.alps.doc.value||`Generated from ALPS file ${options.file}`}\n`;
  rtn += "  version: 1.0.0\n";
  rtn += "\n";
  
  if(doc.alps.ext.filter(metadata_root)) {
    rtn += "servers:\n"
    rtn += `- url: '${doc.alps.ext.filter(metadata_root)[0].value}'\n`;
    rtn += "\n";
  }
  
  // paths
  rtn += "paths:\n";
  
  // gets
  coll = doc.alps.descriptor.filter(safe);
  coll.forEach(function(item) {
    rtn += `  /${item.id}:\n`;
    rtn += "    get:\n";
    rtn += `      summary: '${item.text||item.id}'\n`;
    rtn += `      operationId: ${item.id}\n`;
    rtn += "      responses:\n";
    rtn += "        200:\n";
    rtn += `          description: ${item.id}\n`;
    rtn += "          content:\n";
    rtn += "            application/json:\n";
    rtn += "              schema:\n";
    rtn += "                type: array\n";
    rtn += "                items:\n";
    rtn += `                  $ref: '??/components/schemas/${item.rt||item.returns}'\n`;
  });
  
  // posts
  coll = doc.alps.descriptor.filter(unsafe);
  coll.forEach(function(item) {
    rtn += `  /${item.id}:\n`;
    rtn += "    post:\n";
    rtn += `      summary: '${item.text||item.id}'\n`;
    rtn += `      operationId: ${item.id}\n`;
    rtn += "      requestBody:\n";
    rtn += "        content:\n";
    rtn += "          application/json:\n";
    rtn += "            schema:\n";
    rtn += `              $ref: '??/components/schemas/${item.rt||item.returns}'\n`;
    rtn += "      responses:\n";
    rtn += "        200:\n";
    rtn += `          description: add ${item.id}\n`;
    rtn += "          content:\n";
    rtn += "            application/json:\n";
    rtn += "              schema:\n";
    rtn += "                type: array\n";
    rtn += "                items:\n";
    rtn += `                  $ref: '??/components/schemas/${item.rt||item.returns}'\n`;
  });

  // put
  coll = doc.alps.descriptor.filter(update);
  coll.forEach(function(item) {
    rtn += `  /${item.id}:\n`;
    rtn += "    put:\n";
    rtn += `      summary: '${item.text||item.id}'\n`;
    rtn += `      operationId: ${item.id}\n`;
    rtn += "      requestBody:\n";
    rtn += "        content:\n";
    rtn += "          application/json:\n";
    rtn += "            schema:\n";
    rtn += `              $ref: '??/components/schemas/${item.rt||item.returns}'\n`;
    rtn += "      responses:\n";
    rtn += "        200:\n";
    rtn += `          description: add ${item.id}\n`;
    rtn += "          content:\n";
    rtn += "            application/json:\n";
    rtn += "              schema:\n";
    rtn += "                type: array\n";
    rtn += "                items:\n";
    rtn += `                  $ref: '??/components/schemas/${item.rt||item.returns}'\n`;
  });

  // deletes
  coll = doc.alps.descriptor.filter(remove);
  coll.forEach(function(item) {
    rtn += `  /${item.id}/{id}:\n`;
    rtn += "    delete:\n";
    rtn += `      summary: '${item.text||item.id}'\n`;
    rtn += `      operationId: ${item.id}\n`;
    rtn += "      parameters:\n";
    item.descriptor.forEach(function(prop) {
      rtn += `        - name: ${prop.href}\n`;
      rtn += "          in: path\n";
      rtn += `          description: ${prop.href} of ${item.id}\n`;
      rtn += "          required: true\n";
      rtn += "          schema:\n";
      rtn += "            type: string\n";      
    });
    rtn += "      responses:\n";
    rtn += "        204:\n";
    rtn += `          description: delete ${item.id}\n`;
  });
  rtn += "\n";
    
  // components
  rtn += "components:\n";  
  rtn += "  schemas:\n";
  coll = doc.alps.descriptor.filter(groups);
  coll.forEach(function(item) {
    rtn += `    ${item.id}:\n`;
    if(item.text) {
      rtn += `      description: ${item.text}\n`;
    }
    rtn += "      type: object\n";
    rtn += "      properties:\n";
    item.descriptor.forEach(function(prop) {
      rtn += `          ${prop.href}:\n`;
      rtn += "            type: string\n";
      rtn += `            example: ${rString(prop.href)}\n`; 
    });      
  });
  
  // clean up doc
  rtn = rtn.replace(rxHash,"");
  rtn = rtn.replace(rxQ,"#");
  
  return rtn;
}

// ****************************************************
// to AsyncAPI document (incomplete)
// ****************************************************
function toAsync(doc) {
  var rtn = "";
  // preamble
  rtn += "async: 2.0.0\n";
  rtn += "\n";
  
  // signature
  rtn += '?? *******************************************************************\n';
  rtn += `?? generated by "unified" from ${options.file}\n`;
  rtn += `?? date: ${new Date()}`;
  rtn += '\n';
  rtn += '?? http://github.com/mamund/2020-11-unified\n';
  rtn += '?? *******************************************************************\n';
  rtn += '\n';

  rtn += `id: '${doc.alps.id}'\n`;
  rtn += '\n';

  // info section
  rtn += "info:\n";
  rtn += `  title: ${doc.alps.ext.filter(metadata_title)[0].value||"ALPS API"}\n`;
  rtn += `  description: ${doc.alps.doc.value||`Generated from ALPS file ${options.file}`}\n`;
  rtn += "  version: '1.0.0'\n";
  rtn += `  baseTopic: ${doc.alps.ext.filter(metadata_name)[0].value||""}\n`;
  rtn += `  host: ${doc.alps.ext.filter(metadata_root)[0].value||"http://localhost:8888/root"}\n`;
  rtn += "  schemes:\n";
  rtn += "    - 'amqp'\n";
  rtn += "    - 'mqtt'\n";
  rtn += "\n";
  
  rtn += "# topics:\n";
  rtn += "# **** TBD ****";
  
  // clean up doc
  rtn = rtn.replace(rxHash,"");
  rtn = rtn.replace(rxQ,"#");
  
  return rtn;
}

//*******************************************
// collection filters
//*******************************************
function semantic(doc) {
  return doc.type === "semantic";
}

function groups(doc) {
  return doc.type === "group";
}

function safe(doc) {
  return  doc.type === "safe";
}

function unsafe(doc) {
  return  doc.type === "unsafe";
}

function idempotent(doc) {
  return  doc.type === "idempotent";
}

function remove(doc) {
  return  (doc.type === "idempotent" && (doc.tags && doc.tags.indexOf("delete")!=-1));
}

function update(doc) {
  return  (doc.type === "idempotent" && (doc.tags && doc.tags.indexOf("update")!=-1));
}

function metadata_id(doc) {
  return (doc.type ==="metadata" && (doc.name && doc.name === ("id")));
}
function metadata_title(doc) {
  return (doc.type ==="metadata" && (doc.name && doc.name === ("title")));
}
function metadata_root(doc) {
  return (doc.type ==="metadata" && (doc.name && doc.name === ("root")));
}
function metadata_name(doc) {
  return (doc.type ==="metadata" && (doc.name && doc.name === ("name")));
}

function rString(id) {
  var rtn = "";
  if(id && id.indexOf("id")!=-1) {
    rtn = Math.random().toString(9).substring(2, 4) + Math.random().toString(9).substring(2, 4);
  }
  else {
    rtn = Math.random().toString(36).substring(2, 15) + Math.random().toString(36).substring(2, 15);
  }
  return rtn;
}
