# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['prometheus_http_sd']

package_data = \
{'': ['*'], 'prometheus_http_sd': ['templates/*']}

install_requires = \
['Flask>=2.1.3,<3.0.0',
 'PyYAML>=6.0,<7.0',
 'prometheus-client>=0.14.1,<0.15.0',
 'waitress>=2.1.2,<3.0.0']

entry_points = \
{'console_scripts': ['prometheus-http-sd = prometheus_http_sd.cli:main']}

setup_kwargs = {
    'name': 'prometheus-http-sd',
    'version': '1.1.1',
    'description': 'Prometheus HTTP SD framework.',
    'long_description': '# prometheus-http-sd\n\nThis is a\n[Prometheus HTTP SD](https://prometheus.io/docs/prometheus/latest/http_sd/)\nframework.\n\n[![Test](https://github.com/laixintao/prometheus-http-sd/actions/workflows/test.yaml/badge.svg)](https://github.com/laixintao/prometheus-http-sd/actions/workflows/test.yaml)\n\n<!-- vim-markdown-toc GFM -->\n\n- [Features](#features)\n- [Installation](#installation)\n- [Usage](#usage)\n  - [Manage prometheus-http-sd by systemd](#manage-prometheus-http-sd-by-systemd)\n  - [Admin Page](#admin-page)\n  - [Serve under a different root path](#serve-under-a-different-root-path)\n- [Define you targets](#define-you-targets)\n  - [Your target generator](#your-target-generator)\n  - [The Target Path](#the-target-path)\n  - [Overwriting `job_name` labels](#overwriting-job_name-labels)\n  - [Check and Validate your Targets](#check-and-validate-your-targets)\n  - [Script Dependencies](#script-dependencies)\n- [Update Your Scripts](#update-your-scripts)\n- [Best Practice](#best-practice)\n\n<!-- vim-markdown-toc -->\n\n## Features\n\n- Support static targets from Json file;\n- Support static targets from Yaml file;\n- Support generating target list using Python script;\n- Support `check` command, to testing the generated target is as expected, and\n  counting the targets;\n- You can monitoring your target generator via `/metrics`, see\n  [metrics](./docs/metrics.txt);\n- Admin page to list all target paths;\n- Auto reload when generator or targets changed;\n- Support managing targets in a hierarchy way;\n\n## Installation\n\n```shell\npip install prometheus-http-sd\n```\n\n## Usage\n\nFirst, you need a directory, everything in this directory will be used to\ngenerate targets for prometheus-http-sd.\n\n```shell\n$ mkdir targets\n```\n\nIn this directory, every file is called a target "generator":\n\n- Filename that ending with `.json` will be exposed directly\n- Filename that ending with `.yaml` will be exposed directly\n- Filename that ending with `.py` must include a `generate_targets()` function,\n  the function will be run, and it must return a `TargetList` (Type helper in\n  `prometheus_http_sd.targets.`)\n- Filename that starts with `_` will be ignored, so you can have some python\n  utils there, for e.g. `_utils/__init__.py` that you can import in you\n  `generate_targets()`\n- Filename that starts with `.` (hidden file in Linux) will also be ignored\n\nLet write our first target generator by yaml, put this into your\n`targets/first_target.yaml`:\n\n```yaml\n---\n- targets:\n    - "10.1.1.9:9100"\n    - "10.1.1.10:9100"\n  labels:\n    job: node\n    datacenter: nyc\n    group: g1\n- targets:\n    - "10.2.1.9:9100"\n    - "10.2.1.10:9100"\n  labels:\n    job: node\n    datacenter: sg\n    group: g2\n```\n\nIf you use json, the data structure is the same, just in Json format.\n\nLet\'s put another generator using Python:\n\nPut this into your `targets/by_python.py`:\n\n```python\ndef generate_targets():\n  return {"targets": "10.1.1.22:2379", "labels": {"app": "etcd"}}\n```\n\nThen you can run `prometheus-http-sd serve -h 0.0.0.0 -p 8080 /tmp/targets`,\nprometheus-http-sd will start to expose targets at: http://0.0.0.0:8080/targets\n\nThe `-h` and `-p` is optional, defaults to `127.0.0.1` and `8080`.\n\n```shell\n$ prometheus-http-sd /tmp/good_root\n[2022-07-24 00:52:03,896] {wasyncore.py:486} INFO - Serving on http://127.0.0.1:8080\n```\n\nFinally, you can tell your Prometheus to find targets under\nhttp://127.0.0.1:8080/targets, by adding this into your Prometheus config:\n\n```yaml\nscrape_configs:\n  - job_name: "etcd"\n    http_sd_config:\n      url: http://127.0.0.1:8080/targets/\n```\n\n### Manage prometheus-http-sd by systemd\n\nJust put this file under `/lib/systemd/system/http-sd.service` (remember to\nchange your installation path and root_dir path):\n\n```\n# /lib/systemd/system/http-sd.service\n[Unit]\nDescription=Prometheus HTTP SD Service\nWants=network-online.target\nAfter=network-online.target\n\n[Service]\nType=simple\nExecStart=/opt/httpsd_env/bin/prometheus-http-sd serve \\\n    -h 0.0.0.0                                         \\\n    -p 8080                                            \\\n    /opt/httpsd_targets\n\nRestart=always\nRestartSec=90\n\n[Install]\nWantedBy=multi-user.target\n```\n\n### Admin Page\n\nYou can open the root path, `http://127.0.0.1:8080/` in this example, and you\nwill see all of the available paths list in the admin page.\n\n![](./docs/admin1.png)\n\n### Serve under a different root path\n\nIf you put prometheus-http-sd behind a reverse proxy like Nginx, like this:\n\n```\nlocation /http_sd/ {\n      proxy_pass http://prometheus_http_sd;\n}\n```\n\nThen you need to tell prometheus_http_sd to serve all HTTP requests under this\npath, by using the `--url_prefix /http_sd` cli option, (or `-r /http_sd` for\nshort).\n\n## Define you targets\n\n### Your target generator\n\nPlease see the [Usage](#usage) to know how to define your generator.\n\n### The Target Path\n\nprometheus-http-sd support sub-pathes.\n\nFor example, if we use `prometheus-http-sd serve gateway`, and the `gateway`\ndirectory\'s structure is as follows:\n\n```shell\ngateway\n├── nginx\n│\xa0\xa0 ├── edge.py\n│\xa0\xa0 └── targets.json\n└── targets.json\n```\n\nThen:\n\n- `/targets/gateway` will return the targets from:\n  - `gateway/nginx/edge.py`\n  - `gateway/nginx/targets.json`\n  - `gateway/targets.json`\n- `/targets/gateway/nginx` will return the targets from:\n  - `gateway/nginx/edge.py`\n  - `gateway/nginx/targets.json`\n\nThis is very useful when you use vertical scaling. Say you have 5 Prometheus\ninstances, and you want each one of them scrape for different targets, then you\ncan use the sub-path feature of prometheus-http-sd.\n\nFor example, in one Prometheus\'s scrape config:\n\n```yaml\nscrape_configs:\n  - job_name: "nginx"\n    http_sd_config:\n      url: http://prometheus-http-sd:8080/targets/nginx\n\n  - job_name: "etcd"\n    http_sd_config:\n      url: http://prometheus-http-sd:8080/targets/etcd\n```\n\nAnd in another one:\n\n```yaml\nscrape_configs:\n  - job_name: "nginx"\n    http_sd_config:\n      url: http://prometheus-http-sd:8080/targets/database\n\n  - job_name: "etcd"\n    http_sd_config:\n      url: http://prometheus-http-sd:8080/targets/application\n```\n\n### Overwriting `job_name` labels\n\nYou may want to put all of etcd targets in one generator, including port 2379\nfor etcd metrics and 9100 for node_exporter metrics of the etcd server. But the\n`job_name` setting was based on per URL.\n\nThe trick is that, you can overwrite the `job` label in the target labels, like\nthis:\n\n```yaml\n---\n- targets:\n    - "10.1.1.9:9100"\n  labels:\n    job: node\n    datacenter: nyc\n    group: g1\n- targets:\n    - "10.1.1.9:2379"\n  labels:\n    job: etcd\n    datacenter: nyc\n    group: g1\n```\n\n### Check and Validate your Targets\n\nYou can use `prometheus-http-sd check` command to test your targets dir. It will\nrun all of you generators, validate the targets, and print the targets count\nthat each generator generates.\n\n```shell\n$ prometheus-http-sd check test/test_generator/root\n[2022-08-06 00:50:11,095] {validate.py:16} INFO - Run generator test/test_generator/root/json/target.json, took 0.0011398792266845703s, generated 1 targets.\n[2022-08-06 00:50:11,100] {validate.py:16} INFO - Run generator test/test_generator/root/yaml/target.yaml, took 0.0043718814849853516s, generated 2 targets.\n[2022-08-06 00:50:11,100] {validate.py:22} INFO - Done! Generated {total_targets} in total.\n```\n\nIt\'s a good idea to use `prometheus-http-sd check` in your CI system to validate\nyour targets generator scripts and target files.\n\n### Script Dependencies\n\nIf you want your scripts to use some other python library, just install them\ninto the **same virtualenv** that you install prometheus-http-sd, so that\nprometheus-http-sd can import them.\n\n## Update Your Scripts\n\nIf you want to update your script file or target json file, just upload and\noverwrite with your new version, it will take effect immediately after you\nmaking changes, **there is no need to restart** prometheus-http-sd,\nprometheus-http-sd will read the file (or reload the python script) every time\nserving a request.\n\nIt is worth noting that restarting is safe because if Prometheus failed to get\nthe target list via HTTP request, it won\'t update its current target list to\nempty, instead,\n[it will keep using the current list](https://prometheus.io/docs/prometheus/latest/http_sd/).\n\n> Prometheus caches target lists. If an error occurs while fetching an updated\n> targets list, Prometheus keeps using the current targets list.\n\nFor the same reason, if there are 3 scripts under `/targets/mysystem` and only\none failed for a request, prometheus-http-sd will return a HTTP 500 Error for\nthe whole request instead of returning the partial targets from the other two\nscripts.\n\nAlso for the same reason, if your script met any error, you should throw out\n`Exception` all the way to the top instead of catch it in your script and return\na null `TargetList`, if you return a null `TargetList`, prometheus-http-sd will\nthink that your script run successfully and empty the target list as well.\n\nYou can notice this error from stdout logs or `/metrics` from\nprometheus-http-sd.\n\n## Best Practice\n\nYou can use a git repository to manage your target generator.\n',
    'author': 'laixintao',
    'author_email': 'laixintaoo@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
