"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const certificatemanager = require("@aws-cdk/aws-certificatemanager");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
var HttpVersion;
(function (HttpVersion) {
    HttpVersion["HTTP1_1"] = "http1.1";
    HttpVersion["HTTP2"] = "http2";
})(HttpVersion = exports.HttpVersion || (exports.HttpVersion = {}));
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 */
var PriceClass;
(function (PriceClass) {
    PriceClass["PRICE_CLASS_100"] = "PriceClass_100";
    PriceClass["PRICE_CLASS_200"] = "PriceClass_200";
    PriceClass["PRICE_CLASS_ALL"] = "PriceClass_All";
})(PriceClass = exports.PriceClass || (exports.PriceClass = {}));
/**
 * How HTTPs should be handled with your distribution.
 */
var ViewerProtocolPolicy;
(function (ViewerProtocolPolicy) {
    ViewerProtocolPolicy["HTTPS_ONLY"] = "https-only";
    ViewerProtocolPolicy["REDIRECT_TO_HTTPS"] = "redirect-to-https";
    ViewerProtocolPolicy["ALLOW_ALL"] = "allow-all";
})(ViewerProtocolPolicy = exports.ViewerProtocolPolicy || (exports.ViewerProtocolPolicy = {}));
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *  which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *  multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 */
var SSLMethod;
(function (SSLMethod) {
    SSLMethod["SNI"] = "sni-only";
    SSLMethod["VIP"] = "vip";
})(SSLMethod = exports.SSLMethod || (exports.SSLMethod = {}));
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 */
var SecurityPolicyProtocol;
(function (SecurityPolicyProtocol) {
    SecurityPolicyProtocol["SSL_V3"] = "SSLv3";
    SecurityPolicyProtocol["TLS_V1"] = "TLSv1";
    SecurityPolicyProtocol["TLS_V1_2016"] = "TLSv1_2016";
    SecurityPolicyProtocol["TLS_V1_1_2016"] = "TLSv1.1_2016";
    SecurityPolicyProtocol["TLS_V1_2_2018"] = "TLSv1.2_2018";
})(SecurityPolicyProtocol = exports.SecurityPolicyProtocol || (exports.SecurityPolicyProtocol = {}));
var OriginSslPolicy;
(function (OriginSslPolicy) {
    OriginSslPolicy["SSL_V3"] = "SSLv3";
    OriginSslPolicy["TLS_V1"] = "TLSv1";
    OriginSslPolicy["TLS_V1_1"] = "TLSv1.1";
    OriginSslPolicy["TLS_V1_2"] = "TLSv1.2";
})(OriginSslPolicy = exports.OriginSslPolicy || (exports.OriginSslPolicy = {}));
var OriginProtocolPolicy;
(function (OriginProtocolPolicy) {
    OriginProtocolPolicy["HTTP_ONLY"] = "http-only";
    OriginProtocolPolicy["MATCH_VIEWER"] = "match-viewer";
    OriginProtocolPolicy["HTTPS_ONLY"] = "https-only";
})(OriginProtocolPolicy = exports.OriginProtocolPolicy || (exports.OriginProtocolPolicy = {}));
/**
 * An enum for the supported methods to a CloudFront distribution.
 */
var CloudFrontAllowedMethods;
(function (CloudFrontAllowedMethods) {
    CloudFrontAllowedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedMethods["GET_HEAD_OPTIONS"] = "GHO";
    CloudFrontAllowedMethods["ALL"] = "ALL";
})(CloudFrontAllowedMethods = exports.CloudFrontAllowedMethods || (exports.CloudFrontAllowedMethods = {}));
/**
 * Enums for the methods CloudFront can cache.
 */
var CloudFrontAllowedCachedMethods;
(function (CloudFrontAllowedCachedMethods) {
    CloudFrontAllowedCachedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedCachedMethods["GET_HEAD_OPTIONS"] = "GHO";
})(CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedCachedMethods || (exports.CloudFrontAllowedCachedMethods = {}));
var LambdaEdgeEventType;
(function (LambdaEdgeEventType) {
    /**
     * The origin-request specifies the request to the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_REQUEST"] = "origin-request";
    /**
     * The origin-response specifies the response from the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_RESPONSE"] = "origin-response";
    /**
     * The viewer-request specifies the incoming request
     */
    LambdaEdgeEventType["VIEWER_REQUEST"] = "viewer-request";
    /**
     * The viewer-response specifies the outgoing reponse
     */
    LambdaEdgeEventType["VIEWER_RESPONSE"] = "viewer-response";
})(LambdaEdgeEventType = exports.LambdaEdgeEventType || (exports.LambdaEdgeEventType = {}));
/**
 * Viewer certificate configuration class
 */
class ViewerCertificate {
    constructor(props, aliases = []) {
        this.props = props;
        this.aliases = aliases;
    }
    /**
     * Generate an AWS Certificate Manager (ACM) viewer certificate configuration
     *
     * @param certificate AWS Certificate Manager (ACM) certificate.
     *                    Your certificate must be located in the us-east-1 (US East (N. Virginia)) region to be accessed by CloudFront
     * @param options certificate configuration options
     */
    static fromAcmCertificate(certificate, options = {}) {
        const { sslMethod: sslSupportMethod = SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            acmCertificateArn: certificate.certificateArn, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate an IAM viewer certificate configuration
     *
     * @param iamCertificateId Identifier of the IAM certificate
     * @param options certificate configuration options
     */
    static fromIamCertificate(iamCertificateId, options = {}) {
        const { sslMethod: sslSupportMethod = SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            iamCertificateId, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate a viewer certifcate configuration using
     * the CloudFront default certificate (e.g. d111111abcdef8.cloudfront.net)
     * and a {@link SecurityPolicyProtocol.TLS_V1} security policy.
     *
     * @param aliases Alternative CNAME aliases
     *                You also must create a CNAME record with your DNS service to route queries
     */
    static fromCloudFrontDefaultCertificate(...aliases) {
        return new ViewerCertificate({ cloudFrontDefaultCertificate: true }, aliases);
    }
}
exports.ViewerCertificate = ViewerCertificate;
/**
 * Amazon CloudFront is a global content delivery network (CDN) service that securely delivers data, videos,
 * applications, and APIs to your viewers with low latency and high transfer speeds.
 * CloudFront fronts user provided content and caches it at edge locations across the world.
 *
 * Here's how you can use this construct:
 *
 * ```ts
 * import { CloudFrontWebDistribution } from '@aws-cdk/aws-cloudfront'
 *
 * const sourceBucket = new Bucket(this, 'Bucket');
 *
 * const distribution = new CloudFrontWebDistribution(this, 'MyDistribution', {
 *  originConfigs: [
 *    {
 *      s3OriginSource: {
 *      s3BucketSource: sourceBucket
 *      },
 *      behaviors : [ {isDefaultBehavior: true}]
 *    }
 *  ]
 * });
 * ```
 *
 * This will create a CloudFront distribution that uses your S3Bucket as it's origin.
 *
 * You can customize the distribution using additional properties from the CloudFrontWebDistributionProps interface.
 *
 *
 */
class CloudFrontWebDistribution extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Maps our methods to the string arrays they are
         */
        this.METHOD_LOOKUP_MAP = {
            GH: ['GET', 'HEAD'],
            GHO: ['GET', 'HEAD', 'OPTIONS'],
            ALL: ['DELETE', 'GET', 'HEAD', 'OPTIONS', 'PATCH', 'POST', 'PUT'],
        };
        /**
         * Maps for which SecurityPolicyProtocol are available to which SSLMethods
         */
        this.VALID_SSL_PROTOCOLS = {
            [SSLMethod.SNI]: [
                SecurityPolicyProtocol.TLS_V1, SecurityPolicyProtocol.TLS_V1_1_2016,
                SecurityPolicyProtocol.TLS_V1_2016, SecurityPolicyProtocol.TLS_V1_2_2018,
            ],
            [SSLMethod.VIP]: [SecurityPolicyProtocol.SSL_V3, SecurityPolicyProtocol.TLS_V1],
        };
        let distributionConfig = {
            comment: props.comment,
            enabled: true,
            defaultRootObject: props.defaultRootObject !== undefined ? props.defaultRootObject : 'index.html',
            httpVersion: props.httpVersion || HttpVersion.HTTP2,
            priceClass: props.priceClass || PriceClass.PRICE_CLASS_100,
            ipv6Enabled: (props.enableIpV6 !== undefined) ? props.enableIpV6 : true,
            // tslint:disable-next-line:max-line-length
            customErrorResponses: props.errorConfigurations,
            webAclId: props.webACLId,
        };
        const behaviors = [];
        const origins = [];
        let originIndex = 1;
        for (const originConfig of props.originConfigs) {
            const originId = `origin${originIndex}`;
            if (!originConfig.s3OriginSource && !originConfig.customOriginSource) {
                throw new Error('There must be at least one origin source - either an s3OriginSource or a customOriginSource');
            }
            if (originConfig.customOriginSource && originConfig.s3OriginSource) {
                throw new Error('There cannot be both an s3OriginSource and a customOriginSource in the same SourceConfiguration.');
            }
            const originHeaders = [];
            if (originConfig.originHeaders) {
                Object.keys(originConfig.originHeaders).forEach(key => {
                    const oHeader = {
                        headerName: key,
                        headerValue: originConfig.originHeaders[key],
                    };
                    originHeaders.push(oHeader);
                });
            }
            let s3OriginConfig;
            if (originConfig.s3OriginSource) {
                // first case for backwards compatibility
                if (originConfig.s3OriginSource.originAccessIdentity) {
                    // grant CloudFront OriginAccessIdentity read access to S3 bucket
                    originConfig.s3OriginSource.s3BucketSource.grantRead(originConfig.s3OriginSource.originAccessIdentity);
                    s3OriginConfig = {
                        originAccessIdentity: `origin-access-identity/cloudfront/${originConfig.s3OriginSource.originAccessIdentity.originAccessIdentityName}`,
                    };
                }
                else {
                    s3OriginConfig = {};
                }
            }
            const originProperty = {
                id: originId,
                domainName: originConfig.s3OriginSource
                    ? originConfig.s3OriginSource.s3BucketSource.bucketRegionalDomainName
                    : originConfig.customOriginSource.domainName,
                originPath: originConfig.originPath,
                originCustomHeaders: originHeaders.length > 0 ? originHeaders : undefined,
                s3OriginConfig,
                customOriginConfig: originConfig.customOriginSource
                    ? {
                        httpPort: originConfig.customOriginSource.httpPort || 80,
                        httpsPort: originConfig.customOriginSource.httpsPort || 443,
                        originKeepaliveTimeout: originConfig.customOriginSource.originKeepaliveTimeout
                            && originConfig.customOriginSource.originKeepaliveTimeout.toSeconds() || 5,
                        originReadTimeout: originConfig.customOriginSource.originReadTimeout
                            && originConfig.customOriginSource.originReadTimeout.toSeconds() || 30,
                        originProtocolPolicy: originConfig.customOriginSource.originProtocolPolicy || OriginProtocolPolicy.HTTPS_ONLY,
                        originSslProtocols: originConfig.customOriginSource.allowedOriginSSLVersions || [OriginSslPolicy.TLS_V1_2],
                    }
                    : undefined,
            };
            for (const behavior of originConfig.behaviors) {
                behaviors.push({ ...behavior, targetOriginId: originId });
            }
            origins.push(originProperty);
            originIndex++;
        }
        origins.forEach(origin => {
            if (!origin.s3OriginConfig && !origin.customOriginConfig) {
                throw new Error(`Origin ${origin.domainName} is missing either S3OriginConfig or CustomOriginConfig. At least 1 must be specified.`);
            }
        });
        distributionConfig = {
            ...distributionConfig,
            origins,
        };
        const defaultBehaviors = behaviors.filter(behavior => behavior.isDefaultBehavior);
        if (defaultBehaviors.length !== 1) {
            throw new Error('There can only be one default behavior across all sources. [ One default behavior per distribution ].');
        }
        distributionConfig = { ...distributionConfig, defaultCacheBehavior: this.toBehavior(defaultBehaviors[0], props.viewerProtocolPolicy) };
        const otherBehaviors = [];
        for (const behavior of behaviors.filter(b => !b.isDefaultBehavior)) {
            if (!behavior.pathPattern) {
                throw new Error('pathPattern is required for all non-default behaviors');
            }
            otherBehaviors.push(this.toBehavior(behavior, props.viewerProtocolPolicy));
        }
        distributionConfig = { ...distributionConfig, cacheBehaviors: otherBehaviors.length > 0 ? otherBehaviors : undefined };
        if (props.aliasConfiguration && props.viewerCertificate) {
            throw new Error([
                'You cannot set both aliasConfiguration and viewerCertificate properties.',
                'Please only use viewerCertificate, as aliasConfiguration is deprecated.',
            ].join(' '));
        }
        let _viewerCertificate = props.viewerCertificate;
        if (props.aliasConfiguration) {
            const { acmCertRef, securityPolicy, sslMethod, names: aliases } = props.aliasConfiguration;
            _viewerCertificate = ViewerCertificate.fromAcmCertificate(certificatemanager.Certificate.fromCertificateArn(this, 'AliasConfigurationCert', acmCertRef), { securityPolicy, sslMethod, aliases });
        }
        if (_viewerCertificate) {
            const { props: viewerCertificate, aliases } = _viewerCertificate;
            Object.assign(distributionConfig, { aliases, viewerCertificate });
            const { minimumProtocolVersion, sslSupportMethod } = viewerCertificate;
            if (minimumProtocolVersion != null && sslSupportMethod != null) {
                const validProtocols = this.VALID_SSL_PROTOCOLS[sslSupportMethod];
                if (validProtocols.indexOf(minimumProtocolVersion.toString()) === -1) {
                    // tslint:disable-next-line:max-line-length
                    throw new Error(`${minimumProtocolVersion} is not compabtible with sslMethod ${sslSupportMethod}.\n\tValid Protocols are: ${validProtocols.join(', ')}`);
                }
            }
        }
        else {
            distributionConfig = { ...distributionConfig,
                viewerCertificate: { cloudFrontDefaultCertificate: true },
            };
        }
        if (props.loggingConfig) {
            this.loggingBucket = props.loggingConfig.bucket || new s3.Bucket(this, 'LoggingBucket');
            distributionConfig = {
                ...distributionConfig,
                logging: {
                    bucket: this.loggingBucket.bucketRegionalDomainName,
                    includeCookies: props.loggingConfig.includeCookies || false,
                    prefix: props.loggingConfig.prefix,
                },
            };
        }
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig });
        this.node.defaultChild = distribution;
        this.domainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    toBehavior(input, protoPolicy) {
        let toReturn = {
            allowedMethods: this.METHOD_LOOKUP_MAP[input.allowedMethods || CloudFrontAllowedMethods.GET_HEAD],
            cachedMethods: this.METHOD_LOOKUP_MAP[input.cachedMethods || CloudFrontAllowedCachedMethods.GET_HEAD],
            compress: input.compress !== false,
            defaultTtl: input.defaultTtl && input.defaultTtl.toSeconds(),
            forwardedValues: input.forwardedValues || { queryString: false, cookies: { forward: 'none' } },
            maxTtl: input.maxTtl && input.maxTtl.toSeconds(),
            minTtl: input.minTtl && input.minTtl.toSeconds(),
            trustedSigners: input.trustedSigners,
            targetOriginId: input.targetOriginId,
            viewerProtocolPolicy: protoPolicy || ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        };
        if (!input.isDefaultBehavior) {
            toReturn = Object.assign(toReturn, { pathPattern: input.pathPattern });
        }
        if (input.lambdaFunctionAssociations) {
            toReturn = Object.assign(toReturn, {
                lambdaFunctionAssociations: input.lambdaFunctionAssociations
                    .map(fna => ({
                    eventType: fna.eventType,
                    lambdaFunctionArn: fna.lambdaFunction && fna.lambdaFunction.functionArn,
                })),
            });
            // allow edgelambda.amazonaws.com to assume the functions' execution role.
            for (const a of input.lambdaFunctionAssociations) {
                if (a.lambdaFunction.role && a.lambdaFunction.role instanceof iam.Role && a.lambdaFunction.role.assumeRolePolicy) {
                    a.lambdaFunction.role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                        actions: ['sts:AssumeRole'],
                        principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')],
                    }));
                }
            }
        }
        return toReturn;
    }
}
exports.CloudFrontWebDistribution = CloudFrontWebDistribution;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoid2ViX2Rpc3RyaWJ1dGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIndlYl9kaXN0cmlidXRpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxzRUFBc0U7QUFDdEUsd0NBQXdDO0FBRXhDLHNDQUFzQztBQUN0QyxxQ0FBcUM7QUFDckMsaUVBQXlEO0FBSXpELElBQVksV0FHWDtBQUhELFdBQVksV0FBVztJQUNyQixrQ0FBbUIsQ0FBQTtJQUNuQiw4QkFBZSxDQUFBO0FBQ2pCLENBQUMsRUFIVyxXQUFXLEdBQVgsbUJBQVcsS0FBWCxtQkFBVyxRQUd0QjtBQUVEOztHQUVHO0FBQ0gsSUFBWSxVQUlYO0FBSkQsV0FBWSxVQUFVO0lBQ3BCLGdEQUFrQyxDQUFBO0lBQ2xDLGdEQUFrQyxDQUFBO0lBQ2xDLGdEQUFrQyxDQUFBO0FBQ3BDLENBQUMsRUFKVyxVQUFVLEdBQVYsa0JBQVUsS0FBVixrQkFBVSxRQUlyQjtBQUVEOztHQUVHO0FBQ0gsSUFBWSxvQkFJWDtBQUpELFdBQVksb0JBQW9CO0lBQzlCLGlEQUF5QixDQUFBO0lBQ3pCLCtEQUF1QyxDQUFBO0lBQ3ZDLCtDQUF1QixDQUFBO0FBQ3pCLENBQUMsRUFKVyxvQkFBb0IsR0FBcEIsNEJBQW9CLEtBQXBCLDRCQUFvQixRQUkvQjtBQTRDRDs7Ozs7Ozs7Ozs7Ozs7O0dBZUc7QUFDSCxJQUFZLFNBR1g7QUFIRCxXQUFZLFNBQVM7SUFDbkIsNkJBQWdCLENBQUE7SUFDaEIsd0JBQVcsQ0FBQTtBQUNiLENBQUMsRUFIVyxTQUFTLEdBQVQsaUJBQVMsS0FBVCxpQkFBUyxRQUdwQjtBQUVEOzs7R0FHRztBQUNILElBQVksc0JBTVg7QUFORCxXQUFZLHNCQUFzQjtJQUNoQywwQ0FBZ0IsQ0FBQTtJQUNoQiwwQ0FBZ0IsQ0FBQTtJQUNoQixvREFBMEIsQ0FBQTtJQUMxQix3REFBOEIsQ0FBQTtJQUM5Qix3REFBOEIsQ0FBQTtBQUNoQyxDQUFDLEVBTlcsc0JBQXNCLEdBQXRCLDhCQUFzQixLQUF0Qiw4QkFBc0IsUUFNakM7QUF5SEQsSUFBWSxlQUtYO0FBTEQsV0FBWSxlQUFlO0lBQ3pCLG1DQUFnQixDQUFBO0lBQ2hCLG1DQUFnQixDQUFBO0lBQ2hCLHVDQUFvQixDQUFBO0lBQ3BCLHVDQUFvQixDQUFBO0FBQ3RCLENBQUMsRUFMVyxlQUFlLEdBQWYsdUJBQWUsS0FBZix1QkFBZSxRQUsxQjtBQUVELElBQVksb0JBSVg7QUFKRCxXQUFZLG9CQUFvQjtJQUM5QiwrQ0FBdUIsQ0FBQTtJQUN2QixxREFBNkIsQ0FBQTtJQUM3QixpREFBeUIsQ0FBQTtBQUMzQixDQUFDLEVBSlcsb0JBQW9CLEdBQXBCLDRCQUFvQixLQUFwQiw0QkFBb0IsUUFJL0I7QUFtQkQ7O0dBRUc7QUFDSCxJQUFZLHdCQUlYO0FBSkQsV0FBWSx3QkFBd0I7SUFDbEMsMkNBQWUsQ0FBQTtJQUNmLG9EQUF3QixDQUFBO0lBQ3hCLHVDQUFXLENBQUE7QUFDYixDQUFDLEVBSlcsd0JBQXdCLEdBQXhCLGdDQUF3QixLQUF4QixnQ0FBd0IsUUFJbkM7QUFFRDs7R0FFRztBQUNILElBQVksOEJBR1g7QUFIRCxXQUFZLDhCQUE4QjtJQUN4QyxpREFBZSxDQUFBO0lBQ2YsMERBQXdCLENBQUE7QUFDMUIsQ0FBQyxFQUhXLDhCQUE4QixHQUE5QixzQ0FBOEIsS0FBOUIsc0NBQThCLFFBR3pDO0FBMkdELElBQVksbUJBbUJYO0FBbkJELFdBQVksbUJBQW1CO0lBQzdCOzs7T0FHRztJQUNILHdEQUFpQyxDQUFBO0lBQ2pDOzs7T0FHRztJQUNILDBEQUFtQyxDQUFBO0lBQ25DOztPQUVHO0lBQ0gsd0RBQWlDLENBQUE7SUFDakM7O09BRUc7SUFDSCwwREFBbUMsQ0FBQTtBQUNyQyxDQUFDLEVBbkJXLG1CQUFtQixHQUFuQiwyQkFBbUIsS0FBbkIsMkJBQW1CLFFBbUI5QjtBQTZCRDs7R0FFRztBQUNILE1BQWEsaUJBQWlCO0lBa0Q1QixZQUNrQixLQUFnRCxFQUNoRCxVQUFvQixFQUFFO1FBRHRCLFVBQUssR0FBTCxLQUFLLENBQTJDO1FBQ2hELFlBQU8sR0FBUCxPQUFPLENBQWU7SUFBSSxDQUFDO0lBbkQ3Qzs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsa0JBQWtCLENBQUMsV0FBNEMsRUFBRSxVQUFvQyxFQUFFO1FBQ25ILE1BQU0sRUFDSixTQUFTLEVBQUUsZ0JBQWdCLEdBQUcsU0FBUyxDQUFDLEdBQUcsRUFDM0MsY0FBYyxFQUFFLHNCQUFzQixFQUN0QyxPQUFPLEdBQ1IsR0FBRyxPQUFPLENBQUM7UUFFWixPQUFPLElBQUksaUJBQWlCLENBQUM7WUFDM0IsaUJBQWlCLEVBQUUsV0FBVyxDQUFDLGNBQWMsRUFBRSxnQkFBZ0IsRUFBRSxzQkFBc0I7U0FDeEYsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQyxnQkFBd0IsRUFBRSxVQUFvQyxFQUFFO1FBQy9GLE1BQU0sRUFDSixTQUFTLEVBQUUsZ0JBQWdCLEdBQUcsU0FBUyxDQUFDLEdBQUcsRUFDM0MsY0FBYyxFQUFFLHNCQUFzQixFQUN0QyxPQUFPLEdBQ1IsR0FBRyxPQUFPLENBQUM7UUFFWixPQUFPLElBQUksaUJBQWlCLENBQUM7WUFDM0IsZ0JBQWdCLEVBQUUsZ0JBQWdCLEVBQUUsc0JBQXNCO1NBQzNELEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNJLE1BQU0sQ0FBQyxnQ0FBZ0MsQ0FBQyxHQUFHLE9BQWlCO1FBQ2pFLE9BQU8sSUFBSSxpQkFBaUIsQ0FBQyxFQUFFLDRCQUE0QixFQUFFLElBQUksRUFBRSxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQ2hGLENBQUM7Q0FLRjtBQXJERCw4Q0FxREM7QUF5R0Q7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBNkJHO0FBQ0gsTUFBYSx5QkFBMEIsU0FBUSxHQUFHLENBQUMsU0FBUztJQXVDMUQsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUFxQztRQUNqRixLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBckJuQjs7V0FFRztRQUNjLHNCQUFpQixHQUFHO1lBQ25DLEVBQUUsRUFBRSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUM7WUFDbkIsR0FBRyxFQUFFLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxTQUFTLENBQUM7WUFDL0IsR0FBRyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUUsS0FBSyxDQUFDO1NBQ2xFLENBQUM7UUFFRjs7V0FFRztRQUNjLHdCQUFtQixHQUF3QztZQUMxRSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsRUFBRTtnQkFDZixzQkFBc0IsQ0FBQyxNQUFNLEVBQUUsc0JBQXNCLENBQUMsYUFBYTtnQkFDbkUsc0JBQXNCLENBQUMsV0FBVyxFQUFFLHNCQUFzQixDQUFDLGFBQWE7YUFDekU7WUFDRCxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLE1BQU0sRUFBRSxzQkFBc0IsQ0FBQyxNQUFNLENBQUM7U0FDaEYsQ0FBQztRQUtBLElBQUksa0JBQWtCLEdBQStDO1lBQ25FLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTztZQUN0QixPQUFPLEVBQUUsSUFBSTtZQUNiLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxpQkFBaUIsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsWUFBWTtZQUNqRyxXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVcsSUFBSSxXQUFXLENBQUMsS0FBSztZQUNuRCxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVUsSUFBSSxVQUFVLENBQUMsZUFBZTtZQUMxRCxXQUFXLEVBQUUsQ0FBQyxLQUFLLENBQUMsVUFBVSxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxJQUFJO1lBQ3ZFLDJDQUEyQztZQUMzQyxvQkFBb0IsRUFBRSxLQUFLLENBQUMsbUJBQW1CO1lBQy9DLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTtTQUN6QixDQUFDO1FBRUYsTUFBTSxTQUFTLEdBQXlCLEVBQUUsQ0FBQztRQUUzQyxNQUFNLE9BQU8sR0FBcUMsRUFBRSxDQUFDO1FBRXJELElBQUksV0FBVyxHQUFHLENBQUMsQ0FBQztRQUNwQixLQUFLLE1BQU0sWUFBWSxJQUFJLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDOUMsTUFBTSxRQUFRLEdBQUcsU0FBUyxXQUFXLEVBQUUsQ0FBQztZQUN4QyxJQUFJLENBQUMsWUFBWSxDQUFDLGNBQWMsSUFBSSxDQUFDLFlBQVksQ0FBQyxrQkFBa0IsRUFBRTtnQkFDcEUsTUFBTSxJQUFJLEtBQUssQ0FBQyw2RkFBNkYsQ0FBQyxDQUFDO2FBQ2hIO1lBQ0QsSUFBSSxZQUFZLENBQUMsa0JBQWtCLElBQUksWUFBWSxDQUFDLGNBQWMsRUFBRTtnQkFDbEUsTUFBTSxJQUFJLEtBQUssQ0FBQyxrR0FBa0csQ0FBQyxDQUFDO2FBQ3JIO1lBRUQsTUFBTSxhQUFhLEdBQWlELEVBQUUsQ0FBQztZQUN2RSxJQUFJLFlBQVksQ0FBQyxhQUFhLEVBQUU7Z0JBQzlCLE1BQU0sQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLGFBQWEsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFBRTtvQkFDcEQsTUFBTSxPQUFPLEdBQStDO3dCQUMxRCxVQUFVLEVBQUUsR0FBRzt3QkFDZixXQUFXLEVBQUUsWUFBWSxDQUFDLGFBQWMsQ0FBQyxHQUFHLENBQUM7cUJBQzlDLENBQUM7b0JBQ0YsYUFBYSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztnQkFDOUIsQ0FBQyxDQUFDLENBQUM7YUFDSjtZQUVELElBQUksY0FBa0UsQ0FBQztZQUN2RSxJQUFJLFlBQVksQ0FBQyxjQUFjLEVBQUU7Z0JBQy9CLHlDQUF5QztnQkFDekMsSUFBSSxZQUFZLENBQUMsY0FBYyxDQUFDLG9CQUFvQixFQUFFO29CQUNwRCxpRUFBaUU7b0JBQ2pFLFlBQVksQ0FBQyxjQUFjLENBQUMsY0FBYyxDQUFDLFNBQVMsQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDLG9CQUFvQixDQUFDLENBQUM7b0JBRXZHLGNBQWMsR0FBRzt3QkFDZixvQkFBb0IsRUFDbEIscUNBQ0UsWUFBWSxDQUFDLGNBQWMsQ0FBQyxvQkFBb0IsQ0FBQyx3QkFDbkQsRUFBRTtxQkFDTCxDQUFDO2lCQUNIO3FCQUFNO29CQUNMLGNBQWMsR0FBRyxFQUFFLENBQUM7aUJBQ3JCO2FBQ0Y7WUFFRCxNQUFNLGNBQWMsR0FBbUM7Z0JBQ3JELEVBQUUsRUFBRSxRQUFRO2dCQUNaLFVBQVUsRUFBRSxZQUFZLENBQUMsY0FBYztvQkFDckMsQ0FBQyxDQUFDLFlBQVksQ0FBQyxjQUFjLENBQUMsY0FBYyxDQUFDLHdCQUF3QjtvQkFDckUsQ0FBQyxDQUFDLFlBQVksQ0FBQyxrQkFBbUIsQ0FBQyxVQUFVO2dCQUMvQyxVQUFVLEVBQUUsWUFBWSxDQUFDLFVBQVU7Z0JBQ25DLG1CQUFtQixFQUFFLGFBQWEsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLFNBQVM7Z0JBQ3pFLGNBQWM7Z0JBQ2Qsa0JBQWtCLEVBQUUsWUFBWSxDQUFDLGtCQUFrQjtvQkFDakQsQ0FBQyxDQUFDO3dCQUNBLFFBQVEsRUFBRSxZQUFZLENBQUMsa0JBQWtCLENBQUMsUUFBUSxJQUFJLEVBQUU7d0JBQ3hELFNBQVMsRUFBRSxZQUFZLENBQUMsa0JBQWtCLENBQUMsU0FBUyxJQUFJLEdBQUc7d0JBQzNELHNCQUFzQixFQUFFLFlBQVksQ0FBQyxrQkFBa0IsQ0FBQyxzQkFBc0I7K0JBQ3pFLFlBQVksQ0FBQyxrQkFBa0IsQ0FBQyxzQkFBc0IsQ0FBQyxTQUFTLEVBQUUsSUFBSSxDQUFDO3dCQUM1RSxpQkFBaUIsRUFBRSxZQUFZLENBQUMsa0JBQWtCLENBQUMsaUJBQWlCOytCQUMvRCxZQUFZLENBQUMsa0JBQWtCLENBQUMsaUJBQWlCLENBQUMsU0FBUyxFQUFFLElBQUksRUFBRTt3QkFDeEUsb0JBQW9CLEVBQUUsWUFBWSxDQUFDLGtCQUFrQixDQUFDLG9CQUFvQixJQUFJLG9CQUFvQixDQUFDLFVBQVU7d0JBQzdHLGtCQUFrQixFQUFFLFlBQVksQ0FBQyxrQkFBa0IsQ0FBQyx3QkFBd0IsSUFBSSxDQUFDLGVBQWUsQ0FBQyxRQUFRLENBQUM7cUJBQzNHO29CQUNELENBQUMsQ0FBQyxTQUFTO2FBQ2QsQ0FBQztZQUVGLEtBQUssTUFBTSxRQUFRLElBQUksWUFBWSxDQUFDLFNBQVMsRUFBRTtnQkFDN0MsU0FBUyxDQUFDLElBQUksQ0FBQyxFQUFFLEdBQUcsUUFBUSxFQUFFLGNBQWMsRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFDO2FBQzNEO1lBRUQsT0FBTyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztZQUM3QixXQUFXLEVBQUUsQ0FBQztTQUNmO1FBRUQsT0FBTyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRTtZQUN2QixJQUFJLENBQUMsTUFBTSxDQUFDLGNBQWMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxrQkFBa0IsRUFBRTtnQkFDeEQsTUFBTSxJQUFJLEtBQUssQ0FBQyxVQUFVLE1BQU0sQ0FBQyxVQUFVLHdGQUF3RixDQUFDLENBQUM7YUFDdEk7UUFDSCxDQUFDLENBQUMsQ0FBQztRQUNILGtCQUFrQixHQUFHO1lBQ25CLEdBQUcsa0JBQWtCO1lBQ3JCLE9BQU87U0FDUixDQUFDO1FBRUYsTUFBTSxnQkFBZ0IsR0FBRyxTQUFTLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsUUFBUSxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFDbEYsSUFBSSxnQkFBZ0IsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ2pDLE1BQU0sSUFBSSxLQUFLLENBQUMsdUdBQXVHLENBQUMsQ0FBQztTQUMxSDtRQUVELGtCQUFrQixHQUFHLEVBQUUsR0FBRyxrQkFBa0IsRUFBRSxvQkFBb0IsRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxFQUFFLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQyxFQUFFLENBQUM7UUFFdkksTUFBTSxjQUFjLEdBQTRDLEVBQUUsQ0FBQztRQUNuRSxLQUFLLE1BQU0sUUFBUSxJQUFJLFNBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFO1lBQ2xFLElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVyxFQUFFO2dCQUN6QixNQUFNLElBQUksS0FBSyxDQUFDLHVEQUF1RCxDQUFDLENBQUM7YUFDMUU7WUFDRCxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxvQkFBb0IsQ0FBMEMsQ0FBQyxDQUFDO1NBQ3JIO1FBRUQsa0JBQWtCLEdBQUcsRUFBRSxHQUFHLGtCQUFrQixFQUFFLGNBQWMsRUFBRSxjQUFjLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxTQUFTLEVBQUUsQ0FBQztRQUV2SCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsSUFBSSxLQUFLLENBQUMsaUJBQWlCLEVBQUU7WUFDdkQsTUFBTSxJQUFJLEtBQUssQ0FBQztnQkFDZCwwRUFBMEU7Z0JBQzFFLHlFQUF5RTthQUMxRSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1NBQ2Q7UUFFRCxJQUFJLGtCQUFrQixHQUFHLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQztRQUNqRCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsRUFBRTtZQUM1QixNQUFNLEVBQUMsVUFBVSxFQUFFLGNBQWMsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLE9BQU8sRUFBQyxHQUFHLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQztZQUV6RixrQkFBa0IsR0FBRyxpQkFBaUIsQ0FBQyxrQkFBa0IsQ0FDdkQsa0JBQWtCLENBQUMsV0FBVyxDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRSx3QkFBd0IsRUFBRSxVQUFVLENBQUMsRUFDN0YsRUFBRSxjQUFjLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxDQUN2QyxDQUFDO1NBQ0g7UUFFRCxJQUFJLGtCQUFrQixFQUFFO1lBQ3RCLE1BQU0sRUFBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUUsT0FBTyxFQUFDLEdBQUcsa0JBQWtCLENBQUM7WUFDL0QsTUFBTSxDQUFDLE1BQU0sQ0FBQyxrQkFBa0IsRUFBRSxFQUFDLE9BQU8sRUFBRSxpQkFBaUIsRUFBQyxDQUFDLENBQUM7WUFFaEUsTUFBTSxFQUFDLHNCQUFzQixFQUFFLGdCQUFnQixFQUFDLEdBQUcsaUJBQWlCLENBQUM7WUFFckUsSUFBSSxzQkFBc0IsSUFBSSxJQUFJLElBQUksZ0JBQWdCLElBQUksSUFBSSxFQUFFO2dCQUM5RCxNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsZ0JBQTZCLENBQUMsQ0FBQztnQkFFL0UsSUFBSSxjQUFjLENBQUMsT0FBTyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUU7b0JBQ3BFLDJDQUEyQztvQkFDM0MsTUFBTSxJQUFJLEtBQUssQ0FBQyxHQUFHLHNCQUFzQixzQ0FBc0MsZ0JBQWdCLDZCQUE2QixjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztpQkFDMUo7YUFDRjtTQUNGO2FBQU07WUFDTCxrQkFBa0IsR0FBRyxFQUFFLEdBQUcsa0JBQWtCO2dCQUMxQyxpQkFBaUIsRUFBRSxFQUFFLDRCQUE0QixFQUFFLElBQUksRUFBRTthQUMxRCxDQUFDO1NBQ0g7UUFFRCxJQUFJLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDdkIsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDLE1BQU0sSUFBSSxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLGVBQWUsQ0FBQyxDQUFDO1lBQ3hGLGtCQUFrQixHQUFHO2dCQUNuQixHQUFHLGtCQUFrQjtnQkFDckIsT0FBTyxFQUFFO29CQUNQLE1BQU0sRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLHdCQUF3QjtvQkFDbkQsY0FBYyxFQUFFLEtBQUssQ0FBQyxhQUFhLENBQUMsY0FBYyxJQUFJLEtBQUs7b0JBQzNELE1BQU0sRUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDLE1BQU07aUJBQ25DO2FBQ0YsQ0FBQztTQUNIO1FBRUQsTUFBTSxZQUFZLEdBQUcsSUFBSSxzQ0FBZSxDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRSxFQUFFLGtCQUFrQixFQUFFLENBQUMsQ0FBQztRQUN6RixJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksR0FBRyxZQUFZLENBQUM7UUFDdEMsSUFBSSxDQUFDLFVBQVUsR0FBRyxZQUFZLENBQUMsY0FBYyxDQUFDO1FBQzlDLElBQUksQ0FBQyxjQUFjLEdBQUcsWUFBWSxDQUFDLEdBQUcsQ0FBQztJQUN6QyxDQUFDO0lBRU8sVUFBVSxDQUFDLEtBQXlCLEVBQUUsV0FBa0M7UUFDOUUsSUFBSSxRQUFRLEdBQUc7WUFDYixjQUFjLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxjQUFjLElBQUksd0JBQXdCLENBQUMsUUFBUSxDQUFDO1lBQ2pHLGFBQWEsRUFBRSxJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLGFBQWEsSUFBSSw4QkFBOEIsQ0FBQyxRQUFRLENBQUM7WUFDckcsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRLEtBQUssS0FBSztZQUNsQyxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVUsSUFBSSxLQUFLLENBQUMsVUFBVSxDQUFDLFNBQVMsRUFBRTtZQUM1RCxlQUFlLEVBQUUsS0FBSyxDQUFDLGVBQWUsSUFBSSxFQUFFLFdBQVcsRUFBRSxLQUFLLEVBQUUsT0FBTyxFQUFFLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRSxFQUFFO1lBQzlGLE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFO1lBQ2hELE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFO1lBQ2hELGNBQWMsRUFBRSxLQUFLLENBQUMsY0FBYztZQUNwQyxjQUFjLEVBQUUsS0FBSyxDQUFDLGNBQWM7WUFDcEMsb0JBQW9CLEVBQUUsV0FBVyxJQUFJLG9CQUFvQixDQUFDLGlCQUFpQjtTQUM1RSxDQUFDO1FBQ0YsSUFBSSxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsRUFBRTtZQUM1QixRQUFRLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxRQUFRLEVBQUUsRUFBRSxXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUM7U0FDeEU7UUFDRCxJQUFJLEtBQUssQ0FBQywwQkFBMEIsRUFBRTtZQUNwQyxRQUFRLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxRQUFRLEVBQUU7Z0JBQ2pDLDBCQUEwQixFQUFFLEtBQUssQ0FBQywwQkFBMEI7cUJBQ3pELEdBQUcsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUM7b0JBQ1gsU0FBUyxFQUFFLEdBQUcsQ0FBQyxTQUFTO29CQUN4QixpQkFBaUIsRUFBRSxHQUFHLENBQUMsY0FBYyxJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUMsV0FBVztpQkFDeEUsQ0FBQyxDQUFDO2FBQ04sQ0FBQyxDQUFDO1lBRUgsMEVBQTBFO1lBQzFFLEtBQUssTUFBTSxDQUFDLElBQUksS0FBSyxDQUFDLDBCQUEwQixFQUFFO2dCQUNoRCxJQUFJLENBQUMsQ0FBQyxjQUFjLENBQUMsSUFBSSxJQUFJLENBQUMsQ0FBQyxjQUFjLENBQUMsSUFBSSxZQUFZLEdBQUcsQ0FBQyxJQUFJLElBQUksQ0FBQyxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUU7b0JBQ2hILENBQUMsQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLGFBQWEsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7d0JBQzNFLE9BQU8sRUFBRSxDQUFFLGdCQUFnQixDQUFFO3dCQUM3QixVQUFVLEVBQUUsQ0FBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQywwQkFBMEIsQ0FBQyxDQUFFO3FCQUNyRSxDQUFDLENBQUMsQ0FBQztpQkFDTDthQUNGO1NBQ0Y7UUFDRCxPQUFPLFFBQVEsQ0FBQztJQUNsQixDQUFDO0NBQ0Y7QUF0UEQsOERBc1BDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2VydGlmaWNhdGVtYW5hZ2VyIGZyb20gJ0Bhd3MtY2RrL2F3cy1jZXJ0aWZpY2F0ZW1hbmFnZXInO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgczMgZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IENmbkRpc3RyaWJ1dGlvbiB9IGZyb20gJy4vY2xvdWRmcm9udC5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSURpc3RyaWJ1dGlvbiB9IGZyb20gJy4vZGlzdHJpYnV0aW9uJztcbmltcG9ydCB7IElPcmlnaW5BY2Nlc3NJZGVudGl0eSB9IGZyb20gJy4vb3JpZ2luX2FjY2Vzc19pZGVudGl0eSc7XG5cbmV4cG9ydCBlbnVtIEh0dHBWZXJzaW9uIHtcbiAgSFRUUDFfMSA9ICdodHRwMS4xJyxcbiAgSFRUUDIgPSAnaHR0cDInXG59XG5cbi8qKlxuICogVGhlIHByaWNlIGNsYXNzIGRldGVybWluZXMgaG93IG1hbnkgZWRnZSBsb2NhdGlvbnMgQ2xvdWRGcm9udCB3aWxsIHVzZSBmb3IgeW91ciBkaXN0cmlidXRpb24uXG4gKi9cbmV4cG9ydCBlbnVtIFByaWNlQ2xhc3Mge1xuICBQUklDRV9DTEFTU18xMDAgPSAnUHJpY2VDbGFzc18xMDAnLFxuICBQUklDRV9DTEFTU18yMDAgPSAnUHJpY2VDbGFzc18yMDAnLFxuICBQUklDRV9DTEFTU19BTEwgPSAnUHJpY2VDbGFzc19BbGwnXG59XG5cbi8qKlxuICogSG93IEhUVFBzIHNob3VsZCBiZSBoYW5kbGVkIHdpdGggeW91ciBkaXN0cmlidXRpb24uXG4gKi9cbmV4cG9ydCBlbnVtIFZpZXdlclByb3RvY29sUG9saWN5IHtcbiAgSFRUUFNfT05MWSA9ICdodHRwcy1vbmx5JyxcbiAgUkVESVJFQ1RfVE9fSFRUUFMgPSAncmVkaXJlY3QtdG8taHR0cHMnLFxuICBBTExPV19BTEwgPSAnYWxsb3ctYWxsJ1xufVxuXG4vKipcbiAqIENvbmZpZ3VyYXRpb24gZm9yIGN1c3RvbSBkb21haW4gbmFtZXNcbiAqXG4gKiBDbG91ZEZyb250IGNhbiB1c2UgYSBjdXN0b20gZG9tYWluIHRoYXQgeW91IHByb3ZpZGUgaW5zdGVhZCBvZiBhXG4gKiBcImNsb3VkZnJvbnQubmV0XCIgZG9tYWluLiBUbyB1c2UgdGhpcyBmZWF0dXJlIHlvdSBtdXN0IHByb3ZpZGUgdGhlIGxpc3Qgb2ZcbiAqIGFkZGl0aW9uYWwgZG9tYWlucywgYW5kIHRoZSBBQ00gQ2VydGlmaWNhdGUgdGhhdCBDbG91ZEZyb250IHNob3VsZCB1c2UgZm9yXG4gKiB0aGVzZSBhZGRpdGlvbmFsIGRvbWFpbnMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQWxpYXNDb25maWd1cmF0aW9uIHtcbiAgLyoqXG4gICAqIEFSTiBvZiBhbiBBV1MgQ2VydGlmaWNhdGUgTWFuYWdlciAoQUNNKSBjZXJ0aWZpY2F0ZS5cbiAgICovXG4gIHJlYWRvbmx5IGFjbUNlcnRSZWY6IHN0cmluZztcblxuICAvKipcbiAgICogRG9tYWluIG5hbWVzIG9uIHRoZSBjZXJ0aWZpY2F0ZVxuICAgKlxuICAgKiBCb3RoIG1haW4gZG9tYWluIG5hbWUgYW5kIFN1YmplY3QgQWx0ZXJuYXRpdmUgTmFtZXMuXG4gICAqL1xuICByZWFkb25seSBuYW1lczogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIEhvdyBDbG91ZEZyb250IHNob3VsZCBzZXJ2ZSBIVFRQUyByZXF1ZXN0cy5cbiAgICpcbiAgICogU2VlIHRoZSBub3RlcyBvbiBTU0xNZXRob2QgaWYgeW91IHdpc2ggdG8gdXNlIG90aGVyIFNTTCB0ZXJtaW5hdGlvbiB0eXBlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgU1NMTWV0aG9kLlNOSVxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jbG91ZGZyb250L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1ZpZXdlckNlcnRpZmljYXRlLmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IHNzbE1ldGhvZD86IFNTTE1ldGhvZDtcblxuICAvKipcbiAgICogVGhlIG1pbmltdW0gdmVyc2lvbiBvZiB0aGUgU1NMIHByb3RvY29sIHRoYXQgeW91IHdhbnQgQ2xvdWRGcm9udCB0byB1c2UgZm9yIEhUVFBTIGNvbm5lY3Rpb25zLlxuICAgKlxuICAgKiBDbG91ZEZyb250IHNlcnZlcyB5b3VyIG9iamVjdHMgb25seSB0byBicm93c2VycyBvciBkZXZpY2VzIHRoYXQgc3VwcG9ydCBhdFxuICAgKiBsZWFzdCB0aGUgU1NMIHZlcnNpb24gdGhhdCB5b3Ugc3BlY2lmeS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBTU0x2MyBpZiBzc2xNZXRob2QgVklQLCBUTFN2MSBpZiBzc2xNZXRob2QgU05JXG4gICAqL1xuICByZWFkb25seSBzZWN1cml0eVBvbGljeT86IFNlY3VyaXR5UG9saWN5UHJvdG9jb2w7XG59XG5cbi8qKlxuICogVGhlIFNTTCBtZXRob2QgQ2xvdWRGcm9udCB3aWxsIHVzZSBmb3IgeW91ciBkaXN0cmlidXRpb24uXG4gKlxuICogU2VydmVyIE5hbWUgSW5kaWNhdGlvbiAoU05JKSAtIGlzIGFuIGV4dGVuc2lvbiB0byB0aGUgVExTIGNvbXB1dGVyIG5ldHdvcmtpbmcgcHJvdG9jb2wgYnkgd2hpY2ggYSBjbGllbnQgaW5kaWNhdGVzXG4gKiAgd2hpY2ggaG9zdG5hbWUgaXQgaXMgYXR0ZW1wdGluZyB0byBjb25uZWN0IHRvIGF0IHRoZSBzdGFydCBvZiB0aGUgaGFuZHNoYWtpbmcgcHJvY2Vzcy4gVGhpcyBhbGxvd3MgYSBzZXJ2ZXIgdG8gcHJlc2VudFxuICogIG11bHRpcGxlIGNlcnRpZmljYXRlcyBvbiB0aGUgc2FtZSBJUCBhZGRyZXNzIGFuZCBUQ1AgcG9ydCBudW1iZXIgYW5kIGhlbmNlIGFsbG93cyBtdWx0aXBsZSBzZWN1cmUgKEhUVFBTKSB3ZWJzaXRlc1xuICogKG9yIGFueSBvdGhlciBzZXJ2aWNlIG92ZXIgVExTKSB0byBiZSBzZXJ2ZWQgYnkgdGhlIHNhbWUgSVAgYWRkcmVzcyB3aXRob3V0IHJlcXVpcmluZyBhbGwgdGhvc2Ugc2l0ZXMgdG8gdXNlIHRoZSBzYW1lIGNlcnRpZmljYXRlLlxuICpcbiAqIENsb3VkRnJvbnQgY2FuIHVzZSBTTkkgdG8gaG9zdCBtdWx0aXBsZSBkaXN0cmlidXRpb25zIG9uIHRoZSBzYW1lIElQIC0gd2hpY2ggYSBsYXJnZSBtYWpvcml0eSBvZiBjbGllbnRzIHdpbGwgc3VwcG9ydC5cbiAqXG4gKiBJZiB5b3VyIGNsaWVudHMgY2Fubm90IHN1cHBvcnQgU05JIGhvd2V2ZXIgLSBDbG91ZEZyb250IGNhbiB1c2UgZGVkaWNhdGVkIElQcyBmb3IgeW91ciBkaXN0cmlidXRpb24gLSBidXQgdGhlcmUgaXMgYSBwcm9yYXRlZCBtb250aGx5IGNoYXJnZSBmb3JcbiAqIHVzaW5nIHRoaXMgZmVhdHVyZS4gQnkgZGVmYXVsdCwgd2UgdXNlIFNOSSAtIGJ1dCB5b3UgY2FuIG9wdGlvbmFsbHkgZW5hYmxlIGRlZGljYXRlZCBJUHMgKFZJUCkuXG4gKlxuICogU2VlIHRoZSBDbG91ZEZyb250IFNTTCBmb3IgbW9yZSBkZXRhaWxzIGFib3V0IHByaWNpbmcgOiBodHRwczovL2F3cy5hbWF6b24uY29tL2Nsb3VkZnJvbnQvY3VzdG9tLXNzbC1kb21haW5zL1xuICpcbiAqL1xuZXhwb3J0IGVudW0gU1NMTWV0aG9kIHtcbiAgU05JID0gJ3NuaS1vbmx5JyxcbiAgVklQID0gJ3ZpcCdcbn1cblxuLyoqXG4gKiBUaGUgbWluaW11bSB2ZXJzaW9uIG9mIHRoZSBTU0wgcHJvdG9jb2wgdGhhdCB5b3Ugd2FudCBDbG91ZEZyb250IHRvIHVzZSBmb3IgSFRUUFMgY29ubmVjdGlvbnMuXG4gKiBDbG91ZEZyb250IHNlcnZlcyB5b3VyIG9iamVjdHMgb25seSB0byBicm93c2VycyBvciBkZXZpY2VzIHRoYXQgc3VwcG9ydCBhdCBsZWFzdCB0aGUgU1NMIHZlcnNpb24gdGhhdCB5b3Ugc3BlY2lmeS5cbiAqL1xuZXhwb3J0IGVudW0gU2VjdXJpdHlQb2xpY3lQcm90b2NvbCB7XG4gIFNTTF9WMyA9ICdTU0x2MycsXG4gIFRMU19WMSA9ICdUTFN2MScsXG4gIFRMU19WMV8yMDE2ID0gJ1RMU3YxXzIwMTYnLFxuICBUTFNfVjFfMV8yMDE2ID0gJ1RMU3YxLjFfMjAxNicsXG4gIFRMU19WMV8yXzIwMTggPSAnVExTdjEuMl8yMDE4J1xufVxuXG4vKipcbiAqIExvZ2dpbmcgY29uZmlndXJhdGlvbiBmb3IgaW5jb21pbmcgcmVxdWVzdHNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMb2dnaW5nQ29uZmlndXJhdGlvbiB7XG4gIC8qKlxuICAgKiBCdWNrZXQgdG8gbG9nIHJlcXVlc3RzIHRvXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQSBsb2dnaW5nIGJ1Y2tldCBpcyBhdXRvbWF0aWNhbGx5IGNyZWF0ZWQuXG4gICAqL1xuICByZWFkb25seSBidWNrZXQ/OiBzMy5JQnVja2V0LFxuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRvIGluY2x1ZGUgdGhlIGNvb2tpZXMgaW4gdGhlIGxvZ3NcbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGluY2x1ZGVDb29raWVzPzogYm9vbGVhbixcblxuICAvKipcbiAgICogV2hlcmUgaW4gdGhlIGJ1Y2tldCB0byBzdG9yZSBsb2dzXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gcHJlZml4LlxuICAgKi9cbiAgcmVhZG9ubHkgcHJlZml4Pzogc3RyaW5nXG59XG5cbi8qKlxuICogQSBzb3VyY2UgY29uZmlndXJhdGlvbiBpcyBhIHdyYXBwZXIgZm9yIENsb3VkRnJvbnQgb3JpZ2lucyBhbmQgYmVoYXZpb3JzLlxuICogQW4gb3JpZ2luIGlzIHdoYXQgQ2xvdWRGcm9udCB3aWxsIFwiYmUgaW4gZnJvbnQgb2ZcIiAtIHRoYXQgaXMsIENsb3VkRnJvbnQgd2lsbCBwdWxsIGl0J3MgYXNzZXRzIGZyb20gYW4gb3JpZ2luLlxuICpcbiAqIElmIHlvdSdyZSB1c2luZyBzMyBhcyBhIHNvdXJjZSAtIHBhc3MgdGhlIGBzM09yaWdpbmAgcHJvcGVydHksIG90aGVyd2lzZSwgcGFzcyB0aGUgYGN1c3RvbU9yaWdpblNvdXJjZWAgcHJvcGVydHkuXG4gKlxuICogT25lIG9yIHRoZSBvdGhlciBtdXN0IGJlIHBhc3NlZCwgYW5kIGl0IGlzIGludmFsaWQgdG8gcGFzcyBib3RoIGluIHRoZSBzYW1lIFNvdXJjZUNvbmZpZ3VyYXRpb24uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU291cmNlQ29uZmlndXJhdGlvbiB7XG4gIC8qKlxuICAgKiBBbiBzMyBvcmlnaW4gc291cmNlIC0gaWYgeW91J3JlIHVzaW5nIHMzIGZvciB5b3VyIGFzc2V0c1xuICAgKi9cbiAgcmVhZG9ubHkgczNPcmlnaW5Tb3VyY2U/OiBTM09yaWdpbkNvbmZpZ1xuXG4gIC8qKlxuICAgKiBBIGN1c3RvbSBvcmlnaW4gc291cmNlIC0gZm9yIGFsbCBub24tczMgc291cmNlcy5cbiAgICovXG4gIHJlYWRvbmx5IGN1c3RvbU9yaWdpblNvdXJjZT86IEN1c3RvbU9yaWdpbkNvbmZpZyxcblxuICAvKipcbiAgICogVGhlIGJlaGF2aW9ycyBhc3NvY2lhdGVkIHdpdGggdGhpcyBzb3VyY2UuXG4gICAqIEF0IGxlYXN0IG9uZSAoZGVmYXVsdCkgYmVoYXZpb3IgbXVzdCBiZSBpbmNsdWRlZC5cbiAgICovXG4gIHJlYWRvbmx5IGJlaGF2aW9yczogQmVoYXZpb3JbXTtcblxuICAvKipcbiAgICogVGhlIHJlbGF0aXZlIHBhdGggdG8gdGhlIG9yaWdpbiByb290IHRvIHVzZSBmb3Igc291cmNlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgL1xuICAgKi9cbiAgcmVhZG9ubHkgb3JpZ2luUGF0aD86IHN0cmluZyxcblxuICAvKipcbiAgICogQW55IGFkZGl0aW9uYWwgaGVhZGVycyB0byBwYXNzIHRvIHRoZSBvcmlnaW5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBhZGRpdGlvbmFsIGhlYWRlcnMgYXJlIHBhc3NlZC5cbiAgICovXG4gIHJlYWRvbmx5IG9yaWdpbkhlYWRlcnM/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xufVxuXG4vKipcbiAqIEEgY3VzdG9tIG9yaWdpbiBjb25maWd1cmF0aW9uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ3VzdG9tT3JpZ2luQ29uZmlnIHtcbiAgLyoqXG4gICAqIFRoZSBkb21haW4gbmFtZSBvZiB0aGUgY3VzdG9tIG9yaWdpbi4gU2hvdWxkIG5vdCBpbmNsdWRlIHRoZSBwYXRoIC0gdGhhdCBzaG91bGQgYmUgaW4gdGhlIHBhcmVudCBTb3VyY2VDb25maWd1cmF0aW9uXG4gICAqL1xuICByZWFkb25seSBkb21haW5OYW1lOiBzdHJpbmcsXG5cbiAgLyoqXG4gICAqIFRoZSBvcmlnaW4gSFRUUCBwb3J0XG4gICAqXG4gICAqIEBkZWZhdWx0IDgwXG4gICAqL1xuICByZWFkb25seSBodHRwUG9ydD86IG51bWJlcixcblxuICAvKipcbiAgICogVGhlIG9yaWdpbiBIVFRQUyBwb3J0XG4gICAqXG4gICAqIEBkZWZhdWx0IDQ0M1xuICAgKi9cbiAgcmVhZG9ubHkgaHR0cHNQb3J0PzogbnVtYmVyLFxuXG4gIC8qKlxuICAgKiBUaGUga2VlcCBhbGl2ZSB0aW1lb3V0IHdoZW4gbWFraW5nIGNhbGxzIGluIHNlY29uZHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IER1cmF0aW9uLnNlY29uZHMoNSlcbiAgICovXG4gIHJlYWRvbmx5IG9yaWdpbktlZXBhbGl2ZVRpbWVvdXQ/OiBjZGsuRHVyYXRpb24sXG5cbiAgLyoqXG4gICAqIFRoZSBwcm90b2NvbCAoaHR0cCBvciBodHRwcykgcG9saWN5IHRvIHVzZSB3aGVuIGludGVyYWN0aW5nIHdpdGggdGhlIG9yaWdpbi5cbiAgICpcbiAgICogQGRlZmF1bHQgT3JpZ2luUHJvdG9jb2xQb2xpY3kuSHR0cHNPbmx5XG4gICAqL1xuICByZWFkb25seSBvcmlnaW5Qcm90b2NvbFBvbGljeT86IE9yaWdpblByb3RvY29sUG9saWN5LFxuXG4gIC8qKlxuICAgKiBUaGUgcmVhZCB0aW1lb3V0IHdoZW4gY2FsbGluZyB0aGUgb3JpZ2luIGluIHNlY29uZHNcbiAgICpcbiAgICogQGRlZmF1bHQgRHVyYXRpb24uc2Vjb25kcygzMClcbiAgICovXG4gIHJlYWRvbmx5IG9yaWdpblJlYWRUaW1lb3V0PzogY2RrLkR1cmF0aW9uXG5cbiAgLyoqXG4gICAqIFRoZSBTU0wgdmVyc2lvbnMgdG8gdXNlIHdoZW4gaW50ZXJhY3Rpbmcgd2l0aCB0aGUgb3JpZ2luLlxuICAgKlxuICAgKiBAZGVmYXVsdCBPcmlnaW5Tc2xQb2xpY3kuVExTdjFfMlxuICAgKi9cbiAgcmVhZG9ubHkgYWxsb3dlZE9yaWdpblNTTFZlcnNpb25zPzogT3JpZ2luU3NsUG9saWN5W107XG5cbn1cblxuZXhwb3J0IGVudW0gT3JpZ2luU3NsUG9saWN5IHtcbiAgU1NMX1YzID0gJ1NTTHYzJyxcbiAgVExTX1YxID0gJ1RMU3YxJyxcbiAgVExTX1YxXzEgPSAnVExTdjEuMScsXG4gIFRMU19WMV8yID0gJ1RMU3YxLjInLFxufVxuXG5leHBvcnQgZW51bSBPcmlnaW5Qcm90b2NvbFBvbGljeSB7XG4gIEhUVFBfT05MWSA9ICdodHRwLW9ubHknLFxuICBNQVRDSF9WSUVXRVIgPSAnbWF0Y2gtdmlld2VyJyxcbiAgSFRUUFNfT05MWSA9ICdodHRwcy1vbmx5Jyxcbn1cblxuLyoqXG4gKiBTMyBvcmlnaW4gY29uZmlndXJhdGlvbiBmb3IgQ2xvdWRGcm9udFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFMzT3JpZ2luQ29uZmlnIHtcbiAgLyoqXG4gICAqIFRoZSBzb3VyY2UgYnVja2V0IHRvIHNlcnZlIGNvbnRlbnQgZnJvbVxuICAgKi9cbiAgcmVhZG9ubHkgczNCdWNrZXRTb3VyY2U6IHMzLklCdWNrZXQ7XG5cbiAgLyoqXG4gICAqIFRoZSBvcHRpb25hbCBPcmlnaW4gQWNjZXNzIElkZW50aXR5IG9mIHRoZSBvcmlnaW4gaWRlbnRpdHkgY2xvdWRmcm9udCB3aWxsIHVzZSB3aGVuIGNhbGxpbmcgeW91ciBzMyBidWNrZXQuXG4gICAqXG4gICAqIEBkZWZhdWx0IE5vIE9yaWdpbiBBY2Nlc3MgSWRlbnRpdHkgd2hpY2ggcmVxdWlyZXMgdGhlIFMzIGJ1Y2tldCB0byBiZSBwdWJsaWMgYWNjZXNzaWJsZVxuICAgKi9cbiAgcmVhZG9ubHkgb3JpZ2luQWNjZXNzSWRlbnRpdHk/OiBJT3JpZ2luQWNjZXNzSWRlbnRpdHk7XG59XG5cbi8qKlxuICogQW4gZW51bSBmb3IgdGhlIHN1cHBvcnRlZCBtZXRob2RzIHRvIGEgQ2xvdWRGcm9udCBkaXN0cmlidXRpb24uXG4gKi9cbmV4cG9ydCBlbnVtIENsb3VkRnJvbnRBbGxvd2VkTWV0aG9kcyB7XG4gIEdFVF9IRUFEID0gJ0dIJyxcbiAgR0VUX0hFQURfT1BUSU9OUyA9ICdHSE8nLFxuICBBTEwgPSAnQUxMJ1xufVxuXG4vKipcbiAqIEVudW1zIGZvciB0aGUgbWV0aG9kcyBDbG91ZEZyb250IGNhbiBjYWNoZS5cbiAqL1xuZXhwb3J0IGVudW0gQ2xvdWRGcm9udEFsbG93ZWRDYWNoZWRNZXRob2RzIHtcbiAgR0VUX0hFQUQgPSAnR0gnLFxuICBHRVRfSEVBRF9PUFRJT05TID0gJ0dITycsXG59XG5cbi8qKlxuICogQSBDbG91ZEZyb250IGJlaGF2aW9yIHdyYXBwZXIuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQmVoYXZpb3Ige1xuXG4gIC8qKlxuICAgKiBJZiBDbG91ZEZyb250IHNob3VsZCBhdXRvbWF0aWNhbGx5IGNvbXByZXNzIHNvbWUgY29udGVudCB0eXBlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgY29tcHJlc3M/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBJZiB0aGlzIGJlaGF2aW9yIGlzIHRoZSBkZWZhdWx0IGJlaGF2aW9yIGZvciB0aGUgZGlzdHJpYnV0aW9uLlxuICAgKlxuICAgKiBZb3UgbXVzdCBzcGVjaWZ5IGV4YWN0bHkgb25lIGRlZmF1bHQgZGlzdHJpYnV0aW9uIHBlciBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbi5cbiAgICogVGhlIGRlZmF1bHQgYmVoYXZpb3IgaXMgYWxsb3dlZCB0byBvbWl0IHRoZSBcInBhdGhcIiBwcm9wZXJ0eS5cbiAgICovXG4gIHJlYWRvbmx5IGlzRGVmYXVsdEJlaGF2aW9yPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVHJ1c3RlZCBzaWduZXJzIGlzIGhvdyBDbG91ZEZyb250IGFsbG93cyB5b3UgdG8gc2VydmUgcHJpdmF0ZSBjb250ZW50LlxuICAgKiBUaGUgc2lnbmVycyBhcmUgdGhlIGFjY291bnQgSURzIHRoYXQgYXJlIGFsbG93ZWQgdG8gc2lnbiBjb29raWVzL3ByZXNpZ25lZCBVUkxzIGZvciB0aGlzIGRpc3RyaWJ1dGlvbi5cbiAgICpcbiAgICogSWYgeW91IHBhc3MgYSBub24gZW1wdHkgdmFsdWUsIGFsbCByZXF1ZXN0cyBmb3IgdGhpcyBiZWhhdmlvciBtdXN0IGJlIHNpZ25lZCAobm8gcHVibGljIGFjY2VzcyB3aWxsIGJlIGFsbG93ZWQpXG4gICAqL1xuICByZWFkb25seSB0cnVzdGVkU2lnbmVycz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKlxuICAgKiBUaGUgZGVmYXVsdCBhbW91bnQgb2YgdGltZSBDbG91ZEZyb250IHdpbGwgY2FjaGUgYW4gb2JqZWN0LlxuICAgKlxuICAgKiBUaGlzIHZhbHVlIGFwcGxpZXMgb25seSB3aGVuIHlvdXIgY3VzdG9tIG9yaWdpbiBkb2VzIG5vdCBhZGQgSFRUUCBoZWFkZXJzLFxuICAgKiBzdWNoIGFzIENhY2hlLUNvbnRyb2wgbWF4LWFnZSwgQ2FjaGUtQ29udHJvbCBzLW1heGFnZSwgYW5kIEV4cGlyZXMgdG8gb2JqZWN0cy5cbiAgICogQGRlZmF1bHQgODY0MDAgKDEgZGF5KVxuICAgKlxuICAgKi9cbiAgcmVhZG9ubHkgZGVmYXVsdFR0bD86IGNkay5EdXJhdGlvbjtcblxuICAvKipcbiAgICogVGhlIG1ldGhvZCB0aGlzIENsb3VkRnJvbnQgZGlzdHJpYnV0aW9uIHJlc3BvbmRzIGRvLlxuICAgKlxuICAgKiBAZGVmYXVsdCBHRVRfSEVBRFxuICAgKi9cbiAgcmVhZG9ubHkgYWxsb3dlZE1ldGhvZHM/OiBDbG91ZEZyb250QWxsb3dlZE1ldGhvZHM7XG5cbiAgLyoqXG4gICAqIFRoZSBwYXRoIHRoaXMgYmVoYXZpb3IgcmVzcG9uZHMgdG8uXG4gICAqIFJlcXVpcmVkIGZvciBhbGwgbm9uLWRlZmF1bHQgYmVoYXZpb3JzLiAoVGhlIGRlZmF1bHQgYmVoYXZpb3IgaW1wbGljaXRseSBoYXMgXCIqXCIgYXMgdGhlIHBhdGggcGF0dGVybi4gKVxuICAgKlxuICAgKi9cbiAgcmVhZG9ubHkgcGF0aFBhdHRlcm4/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFdoaWNoIG1ldGhvZHMgYXJlIGNhY2hlZCBieSBDbG91ZEZyb250IGJ5IGRlZmF1bHQuXG4gICAqXG4gICAqIEBkZWZhdWx0IEdFVF9IRUFEXG4gICAqL1xuICByZWFkb25seSBjYWNoZWRNZXRob2RzPzogQ2xvdWRGcm9udEFsbG93ZWRDYWNoZWRNZXRob2RzO1xuXG4gIC8qKlxuICAgKiBUaGUgdmFsdWVzIENsb3VkRnJvbnQgd2lsbCBmb3J3YXJkIHRvIHRoZSBvcmlnaW4gd2hlbiBtYWtpbmcgYSByZXF1ZXN0LlxuICAgKlxuICAgKiBAZGVmYXVsdCBub25lIChubyBjb29raWVzIC0gbm8gaGVhZGVycylcbiAgICpcbiAgICovXG4gIHJlYWRvbmx5IGZvcndhcmRlZFZhbHVlcz86IENmbkRpc3RyaWJ1dGlvbi5Gb3J3YXJkZWRWYWx1ZXNQcm9wZXJ0eTtcblxuICAvKipcbiAgICogVGhlIG1pbmltdW0gYW1vdW50IG9mIHRpbWUgdGhhdCB5b3Ugd2FudCBvYmplY3RzIHRvIHN0YXkgaW4gdGhlIGNhY2hlXG4gICAqIGJlZm9yZSBDbG91ZEZyb250IHF1ZXJpZXMgeW91ciBvcmlnaW4uXG4gICAqL1xuICByZWFkb25seSBtaW5UdGw/OiBjZGsuRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSBtYXggYW1vdW50IG9mIHRpbWUgeW91IHdhbnQgb2JqZWN0cyB0byBzdGF5IGluIHRoZSBjYWNoZVxuICAgKiBiZWZvcmUgQ2xvdWRGcm9udCBxdWVyaWVzIHlvdXIgb3JpZ2luLlxuICAgKlxuICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5zZWNvbmRzKDMxNTM2MDAwKSAob25lIHllYXIpXG4gICAqL1xuICByZWFkb25seSBtYXhUdGw/OiBjZGsuRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIERlY2xhcmVzIGFzc29jaWF0ZWQgbGFtYmRhQGVkZ2UgZnVuY3Rpb25zIGZvciB0aGlzIGRpc3RyaWJ1dGlvbiBiZWhhdmlvdXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IE5vIGxhbWJkYSBmdW5jdGlvbiBhc3NvY2lhdGVkXG4gICAqL1xuICByZWFkb25seSBsYW1iZGFGdW5jdGlvbkFzc29jaWF0aW9ucz86IExhbWJkYUZ1bmN0aW9uQXNzb2NpYXRpb25bXTtcblxufVxuXG5leHBvcnQgaW50ZXJmYWNlIExhbWJkYUZ1bmN0aW9uQXNzb2NpYXRpb24ge1xuXG4gIC8qKlxuICAgKiBUaGUgbGFtYmRhIGV2ZW50IHR5cGUgZGVmaW5lcyBhdCB3aGljaCBldmVudCB0aGUgbGFtYmRhXG4gICAqIGlzIGNhbGxlZCBkdXJpbmcgdGhlIHJlcXVlc3QgbGlmZWN5Y2xlXG4gICAqL1xuICByZWFkb25seSBldmVudFR5cGU6IExhbWJkYUVkZ2VFdmVudFR5cGU7XG5cbiAgLyoqXG4gICAqIEEgdmVyc2lvbiBvZiB0aGUgbGFtYmRhIHRvIGFzc29jaWF0ZVxuICAgKi9cbiAgcmVhZG9ubHkgbGFtYmRhRnVuY3Rpb246IGxhbWJkYS5JVmVyc2lvbjtcbn1cblxuZXhwb3J0IGVudW0gTGFtYmRhRWRnZUV2ZW50VHlwZSB7XG4gIC8qKlxuICAgKiBUaGUgb3JpZ2luLXJlcXVlc3Qgc3BlY2lmaWVzIHRoZSByZXF1ZXN0IHRvIHRoZVxuICAgKiBvcmlnaW4gbG9jYXRpb24gKGUuZy4gUzMpXG4gICAqL1xuICBPUklHSU5fUkVRVUVTVCA9ICdvcmlnaW4tcmVxdWVzdCcsXG4gIC8qKlxuICAgKiBUaGUgb3JpZ2luLXJlc3BvbnNlIHNwZWNpZmllcyB0aGUgcmVzcG9uc2UgZnJvbSB0aGVcbiAgICogb3JpZ2luIGxvY2F0aW9uIChlLmcuIFMzKVxuICAgKi9cbiAgT1JJR0lOX1JFU1BPTlNFID0gJ29yaWdpbi1yZXNwb25zZScsXG4gIC8qKlxuICAgKiBUaGUgdmlld2VyLXJlcXVlc3Qgc3BlY2lmaWVzIHRoZSBpbmNvbWluZyByZXF1ZXN0XG4gICAqL1xuICBWSUVXRVJfUkVRVUVTVCA9ICd2aWV3ZXItcmVxdWVzdCcsXG4gIC8qKlxuICAgKiBUaGUgdmlld2VyLXJlc3BvbnNlIHNwZWNpZmllcyB0aGUgb3V0Z29pbmcgcmVwb25zZVxuICAgKi9cbiAgVklFV0VSX1JFU1BPTlNFID0gJ3ZpZXdlci1yZXNwb25zZScsXG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgVmlld2VyQ2VydGlmaWNhdGVPcHRpb25zIHtcbiAgLyoqXG4gICAqIEhvdyBDbG91ZEZyb250IHNob3VsZCBzZXJ2ZSBIVFRQUyByZXF1ZXN0cy5cbiAgICpcbiAgICogU2VlIHRoZSBub3RlcyBvbiBTU0xNZXRob2QgaWYgeW91IHdpc2ggdG8gdXNlIG90aGVyIFNTTCB0ZXJtaW5hdGlvbiB0eXBlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgU1NMTWV0aG9kLlNOSVxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jbG91ZGZyb250L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1ZpZXdlckNlcnRpZmljYXRlLmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IHNzbE1ldGhvZD86IFNTTE1ldGhvZDtcblxuICAvKipcbiAgICogVGhlIG1pbmltdW0gdmVyc2lvbiBvZiB0aGUgU1NMIHByb3RvY29sIHRoYXQgeW91IHdhbnQgQ2xvdWRGcm9udCB0byB1c2UgZm9yIEhUVFBTIGNvbm5lY3Rpb25zLlxuICAgKlxuICAgKiBDbG91ZEZyb250IHNlcnZlcyB5b3VyIG9iamVjdHMgb25seSB0byBicm93c2VycyBvciBkZXZpY2VzIHRoYXQgc3VwcG9ydCBhdFxuICAgKiBsZWFzdCB0aGUgU1NMIHZlcnNpb24gdGhhdCB5b3Ugc3BlY2lmeS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBTU0x2MyBpZiBzc2xNZXRob2QgVklQLCBUTFN2MSBpZiBzc2xNZXRob2QgU05JXG4gICAqL1xuICByZWFkb25seSBzZWN1cml0eVBvbGljeT86IFNlY3VyaXR5UG9saWN5UHJvdG9jb2w7XG5cbiAgLyoqXG4gICAqIERvbWFpbiBuYW1lcyBvbiB0aGUgY2VydGlmaWNhdGUgKGJvdGggbWFpbiBkb21haW4gbmFtZSBhbmQgU3ViamVjdCBBbHRlcm5hdGl2ZSBuYW1lcylcbiAgICovXG4gIHJlYWRvbmx5IGFsaWFzZXM/OiBzdHJpbmdbXTtcbn1cblxuLyoqXG4gKiBWaWV3ZXIgY2VydGlmaWNhdGUgY29uZmlndXJhdGlvbiBjbGFzc1xuICovXG5leHBvcnQgY2xhc3MgVmlld2VyQ2VydGlmaWNhdGUge1xuICAvKipcbiAgICogR2VuZXJhdGUgYW4gQVdTIENlcnRpZmljYXRlIE1hbmFnZXIgKEFDTSkgdmlld2VyIGNlcnRpZmljYXRlIGNvbmZpZ3VyYXRpb25cbiAgICpcbiAgICogQHBhcmFtIGNlcnRpZmljYXRlIEFXUyBDZXJ0aWZpY2F0ZSBNYW5hZ2VyIChBQ00pIGNlcnRpZmljYXRlLlxuICAgKiAgICAgICAgICAgICAgICAgICAgWW91ciBjZXJ0aWZpY2F0ZSBtdXN0IGJlIGxvY2F0ZWQgaW4gdGhlIHVzLWVhc3QtMSAoVVMgRWFzdCAoTi4gVmlyZ2luaWEpKSByZWdpb24gdG8gYmUgYWNjZXNzZWQgYnkgQ2xvdWRGcm9udFxuICAgKiBAcGFyYW0gb3B0aW9ucyBjZXJ0aWZpY2F0ZSBjb25maWd1cmF0aW9uIG9wdGlvbnNcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUFjbUNlcnRpZmljYXRlKGNlcnRpZmljYXRlOiBjZXJ0aWZpY2F0ZW1hbmFnZXIuSUNlcnRpZmljYXRlLCBvcHRpb25zOiBWaWV3ZXJDZXJ0aWZpY2F0ZU9wdGlvbnMgPSB7fSkge1xuICAgIGNvbnN0IHtcbiAgICAgIHNzbE1ldGhvZDogc3NsU3VwcG9ydE1ldGhvZCA9IFNTTE1ldGhvZC5TTkksXG4gICAgICBzZWN1cml0eVBvbGljeTogbWluaW11bVByb3RvY29sVmVyc2lvbixcbiAgICAgIGFsaWFzZXMsXG4gICAgfSA9IG9wdGlvbnM7XG5cbiAgICByZXR1cm4gbmV3IFZpZXdlckNlcnRpZmljYXRlKHtcbiAgICAgIGFjbUNlcnRpZmljYXRlQXJuOiBjZXJ0aWZpY2F0ZS5jZXJ0aWZpY2F0ZUFybiwgc3NsU3VwcG9ydE1ldGhvZCwgbWluaW11bVByb3RvY29sVmVyc2lvbixcbiAgICB9LCBhbGlhc2VzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZSBhbiBJQU0gdmlld2VyIGNlcnRpZmljYXRlIGNvbmZpZ3VyYXRpb25cbiAgICpcbiAgICogQHBhcmFtIGlhbUNlcnRpZmljYXRlSWQgSWRlbnRpZmllciBvZiB0aGUgSUFNIGNlcnRpZmljYXRlXG4gICAqIEBwYXJhbSBvcHRpb25zIGNlcnRpZmljYXRlIGNvbmZpZ3VyYXRpb24gb3B0aW9uc1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tSWFtQ2VydGlmaWNhdGUoaWFtQ2VydGlmaWNhdGVJZDogc3RyaW5nLCBvcHRpb25zOiBWaWV3ZXJDZXJ0aWZpY2F0ZU9wdGlvbnMgPSB7fSkge1xuICAgIGNvbnN0IHtcbiAgICAgIHNzbE1ldGhvZDogc3NsU3VwcG9ydE1ldGhvZCA9IFNTTE1ldGhvZC5TTkksXG4gICAgICBzZWN1cml0eVBvbGljeTogbWluaW11bVByb3RvY29sVmVyc2lvbixcbiAgICAgIGFsaWFzZXMsXG4gICAgfSA9IG9wdGlvbnM7XG5cbiAgICByZXR1cm4gbmV3IFZpZXdlckNlcnRpZmljYXRlKHtcbiAgICAgIGlhbUNlcnRpZmljYXRlSWQsIHNzbFN1cHBvcnRNZXRob2QsIG1pbmltdW1Qcm90b2NvbFZlcnNpb24sXG4gICAgfSwgYWxpYXNlcyk7XG4gIH1cblxuICAvKipcbiAgICogR2VuZXJhdGUgYSB2aWV3ZXIgY2VydGlmY2F0ZSBjb25maWd1cmF0aW9uIHVzaW5nXG4gICAqIHRoZSBDbG91ZEZyb250IGRlZmF1bHQgY2VydGlmaWNhdGUgKGUuZy4gZDExMTExMWFiY2RlZjguY2xvdWRmcm9udC5uZXQpXG4gICAqIGFuZCBhIHtAbGluayBTZWN1cml0eVBvbGljeVByb3RvY29sLlRMU19WMX0gc2VjdXJpdHkgcG9saWN5LlxuICAgKlxuICAgKiBAcGFyYW0gYWxpYXNlcyBBbHRlcm5hdGl2ZSBDTkFNRSBhbGlhc2VzXG4gICAqICAgICAgICAgICAgICAgIFlvdSBhbHNvIG11c3QgY3JlYXRlIGEgQ05BTUUgcmVjb3JkIHdpdGggeW91ciBETlMgc2VydmljZSB0byByb3V0ZSBxdWVyaWVzXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21DbG91ZEZyb250RGVmYXVsdENlcnRpZmljYXRlKC4uLmFsaWFzZXM6IHN0cmluZ1tdKSB7XG4gICAgcmV0dXJuIG5ldyBWaWV3ZXJDZXJ0aWZpY2F0ZSh7IGNsb3VkRnJvbnREZWZhdWx0Q2VydGlmaWNhdGU6IHRydWUgfSwgYWxpYXNlcyk7XG4gIH1cblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKFxuICAgIHB1YmxpYyByZWFkb25seSBwcm9wczogQ2ZuRGlzdHJpYnV0aW9uLlZpZXdlckNlcnRpZmljYXRlUHJvcGVydHksXG4gICAgcHVibGljIHJlYWRvbmx5IGFsaWFzZXM6IHN0cmluZ1tdID0gW10pIHsgfVxufVxuXG5leHBvcnQgaW50ZXJmYWNlIENsb3VkRnJvbnRXZWJEaXN0cmlidXRpb25Qcm9wcyB7XG5cbiAgLyoqXG4gICAqIEFsaWFzQ29uZmlndXJhdGlvbiBpcyB1c2VkIHRvIGNvbmZpZ3VyZWQgQ2xvdWRGcm9udCB0byByZXNwb25kIHRvIHJlcXVlc3RzIG9uIGN1c3RvbSBkb21haW4gbmFtZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZS5cbiAgICogQGRlcHJlY2F0ZWQgc2VlIHtAbGluayBDbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uUHJvcHMjdmlld2VyQ2VydGlmaWNhdGV9IHdpdGgge0BsaW5rIFZpZXdlckNlcnRpZmljYXRlI2FjbUNlcnRpZmljYXRlfVxuICAgKi9cbiAgcmVhZG9ubHkgYWxpYXNDb25maWd1cmF0aW9uPzogQWxpYXNDb25maWd1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBBIGNvbW1lbnQgZm9yIHRoaXMgZGlzdHJpYnV0aW9uIGluIHRoZSBDbG91ZEZyb250IGNvbnNvbGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gY29tbWVudCBpcyBhZGRlZCB0byBkaXN0cmlidXRpb24uXG4gICAqL1xuICByZWFkb25seSBjb21tZW50Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgZGVmYXVsdCBvYmplY3QgdG8gc2VydmUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gXCJpbmRleC5odG1sXCIgaXMgc2VydmVkLlxuICAgKi9cbiAgcmVhZG9ubHkgZGVmYXVsdFJvb3RPYmplY3Q/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIElmIHlvdXIgZGlzdHJpYnV0aW9uIHNob3VsZCBoYXZlIElQdjYgZW5hYmxlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgZW5hYmxlSXBWNj86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBtYXggc3VwcG9ydGVkIEhUVFAgVmVyc2lvbnMuXG4gICAqXG4gICAqIEBkZWZhdWx0IEh0dHBWZXJzaW9uLkhUVFAyXG4gICAqL1xuICByZWFkb25seSBodHRwVmVyc2lvbj86IEh0dHBWZXJzaW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgcHJpY2UgY2xhc3MgZm9yIHRoZSBkaXN0cmlidXRpb24gKHRoaXMgaW1wYWN0cyBob3cgbWFueSBsb2NhdGlvbnMgQ2xvdWRGcm9udCB1c2VzIGZvciB5b3VyIGRpc3RyaWJ1dGlvbiwgYW5kIGJpbGxpbmcpXG4gICAqXG4gICAqIEBkZWZhdWx0IFByaWNlQ2xhc3MuUHJpY2VDbGFzczEwMCB0aGUgY2hlYXBlc3Qgb3B0aW9uIGZvciBDbG91ZEZyb250IGlzIHBpY2tlZCBieSBkZWZhdWx0LlxuICAgKi9cbiAgcmVhZG9ubHkgcHJpY2VDbGFzcz86IFByaWNlQ2xhc3M7XG5cbiAgLyoqXG4gICAqIFRoZSBkZWZhdWx0IHZpZXdlciBwb2xpY3kgZm9yIGluY29taW5nIGNsaWVudHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IFJlZGlyZWN0VG9IVFRQc1xuICAgKi9cbiAgcmVhZG9ubHkgdmlld2VyUHJvdG9jb2xQb2xpY3k/OiBWaWV3ZXJQcm90b2NvbFBvbGljeTtcblxuICAvKipcbiAgICogVGhlIG9yaWdpbiBjb25maWd1cmF0aW9ucyBmb3IgdGhpcyBkaXN0cmlidXRpb24uIEJlaGF2aW9ycyBhcmUgYSBwYXJ0IG9mIHRoZSBvcmlnaW4uXG4gICAqL1xuICByZWFkb25seSBvcmlnaW5Db25maWdzOiBTb3VyY2VDb25maWd1cmF0aW9uW107XG5cbiAgLyoqXG4gICAqIE9wdGlvbmFsIC0gaWYgd2Ugc2hvdWxkIGVuYWJsZSBsb2dnaW5nLlxuICAgKiBZb3UgY2FuIHBhc3MgYW4gZW1wdHkgb2JqZWN0ICh7fSkgdG8gaGF2ZSB1cyBhdXRvIGNyZWF0ZSBhIGJ1Y2tldCBmb3IgbG9nZ2luZy5cbiAgICogT21pc3Npb24gb2YgdGhpcyBwcm9wZXJ0eSBpbmRpY2F0ZXMgbm8gbG9nZ2luZyBpcyB0byBiZSBlbmFibGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGxvZ2dpbmcgaXMgZW5hYmxlZCBieSBkZWZhdWx0LlxuICAgKi9cbiAgcmVhZG9ubHkgbG9nZ2luZ0NvbmZpZz86IExvZ2dpbmdDb25maWd1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBIb3cgQ2xvdWRGcm9udCBzaG91bGQgaGFuZGxlIHJlcXVlc3RzIHRoYXQgYXJlIG5vdCBzdWNjZXNzZnVsIChlZyBQYWdlTm90Rm91bmQpXG4gICAqXG4gICAqIEJ5IGRlZmF1bHQsIENsb3VkRnJvbnQgZG9lcyBub3QgcmVwbGFjZSBIVFRQIHN0YXR1cyBjb2RlcyBpbiB0aGUgNHh4IGFuZCA1eHggcmFuZ2VcbiAgICogd2l0aCBjdXN0b20gZXJyb3IgbWVzc2FnZXMuIENsb3VkRnJvbnQgZG9lcyBub3QgY2FjaGUgSFRUUCBzdGF0dXMgY29kZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gY3VzdG9tIGVycm9yIGNvbmZpZ3VyYXRpb24uXG4gICAqL1xuICByZWFkb25seSBlcnJvckNvbmZpZ3VyYXRpb25zPzogQ2ZuRGlzdHJpYnV0aW9uLkN1c3RvbUVycm9yUmVzcG9uc2VQcm9wZXJ0eVtdO1xuXG4gIC8qKlxuICAgKiBVbmlxdWUgaWRlbnRpZmllciB0aGF0IHNwZWNpZmllcyB0aGUgQVdTIFdBRiB3ZWIgQUNMIHRvIGFzc29jaWF0ZSB3aXRoIHRoaXMgQ2xvdWRGcm9udCBkaXN0cmlidXRpb24uXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvd2hhdC1pcy1hd3Mtd2FmLmh0bWxcbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBBV1MgV2ViIEFwcGxpY2F0aW9uIEZpcmV3YWxsIHdlYiBhY2Nlc3MgY29udHJvbCBsaXN0ICh3ZWIgQUNMKS5cbiAgICovXG4gIHJlYWRvbmx5IHdlYkFDTElkPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgd2hldGhlciB5b3Ugd2FudCB2aWV3ZXJzIHRvIHVzZSBIVFRQIG9yIEhUVFBTIHRvIHJlcXVlc3QgeW91ciBvYmplY3RzLFxuICAgKiB3aGV0aGVyIHlvdSdyZSB1c2luZyBhbiBhbHRlcm5hdGUgZG9tYWluIG5hbWUgd2l0aCBIVFRQUywgYW5kIGlmIHNvLFxuICAgKiBpZiB5b3UncmUgdXNpbmcgQVdTIENlcnRpZmljYXRlIE1hbmFnZXIgKEFDTSkgb3IgYSB0aGlyZC1wYXJ0eSBjZXJ0aWZpY2F0ZSBhdXRob3JpdHkuXG4gICAqXG4gICAqIEBkZWZhdWx0IFZpZXdlckNlcnRpZmljYXRlLmZyb21DbG91ZEZyb250RGVmYXVsdENlcnRpZmljYXRlKClcbiAgICpcbiAgICogQHNlZSBodHRwczovL2F3cy5hbWF6b24uY29tL3ByZW1pdW1zdXBwb3J0L2tub3dsZWRnZS1jZW50ZXIvY3VzdG9tLXNzbC1jZXJ0aWZpY2F0ZS1jbG91ZGZyb250L1xuICAgKi9cbiAgcmVhZG9ubHkgdmlld2VyQ2VydGlmaWNhdGU/OiBWaWV3ZXJDZXJ0aWZpY2F0ZTtcbn1cblxuLyoqXG4gKiBJbnRlcm5hbCBvbmx5IC0ganVzdCBhZGRzIHRoZSBvcmlnaW5JZCBzdHJpbmcgdG8gdGhlIEJlaGF2aW9yXG4gKi9cbmludGVyZmFjZSBCZWhhdmlvcldpdGhPcmlnaW4gZXh0ZW5kcyBCZWhhdmlvciB7XG4gIHJlYWRvbmx5IHRhcmdldE9yaWdpbklkOiBzdHJpbmc7XG59XG5cbi8qKlxuICogQW1hem9uIENsb3VkRnJvbnQgaXMgYSBnbG9iYWwgY29udGVudCBkZWxpdmVyeSBuZXR3b3JrIChDRE4pIHNlcnZpY2UgdGhhdCBzZWN1cmVseSBkZWxpdmVycyBkYXRhLCB2aWRlb3MsXG4gKiBhcHBsaWNhdGlvbnMsIGFuZCBBUElzIHRvIHlvdXIgdmlld2VycyB3aXRoIGxvdyBsYXRlbmN5IGFuZCBoaWdoIHRyYW5zZmVyIHNwZWVkcy5cbiAqIENsb3VkRnJvbnQgZnJvbnRzIHVzZXIgcHJvdmlkZWQgY29udGVudCBhbmQgY2FjaGVzIGl0IGF0IGVkZ2UgbG9jYXRpb25zIGFjcm9zcyB0aGUgd29ybGQuXG4gKlxuICogSGVyZSdzIGhvdyB5b3UgY2FuIHVzZSB0aGlzIGNvbnN0cnVjdDpcbiAqXG4gKiBgYGB0c1xuICogaW1wb3J0IHsgQ2xvdWRGcm9udFdlYkRpc3RyaWJ1dGlvbiB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZGZyb250J1xuICpcbiAqIGNvbnN0IHNvdXJjZUJ1Y2tldCA9IG5ldyBCdWNrZXQodGhpcywgJ0J1Y2tldCcpO1xuICpcbiAqIGNvbnN0IGRpc3RyaWJ1dGlvbiA9IG5ldyBDbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uKHRoaXMsICdNeURpc3RyaWJ1dGlvbicsIHtcbiAqICBvcmlnaW5Db25maWdzOiBbXG4gKiAgICB7XG4gKiAgICAgIHMzT3JpZ2luU291cmNlOiB7XG4gKiAgICAgIHMzQnVja2V0U291cmNlOiBzb3VyY2VCdWNrZXRcbiAqICAgICAgfSxcbiAqICAgICAgYmVoYXZpb3JzIDogWyB7aXNEZWZhdWx0QmVoYXZpb3I6IHRydWV9XVxuICogICAgfVxuICogIF1cbiAqIH0pO1xuICogYGBgXG4gKlxuICogVGhpcyB3aWxsIGNyZWF0ZSBhIENsb3VkRnJvbnQgZGlzdHJpYnV0aW9uIHRoYXQgdXNlcyB5b3VyIFMzQnVja2V0IGFzIGl0J3Mgb3JpZ2luLlxuICpcbiAqIFlvdSBjYW4gY3VzdG9taXplIHRoZSBkaXN0cmlidXRpb24gdXNpbmcgYWRkaXRpb25hbCBwcm9wZXJ0aWVzIGZyb20gdGhlIENsb3VkRnJvbnRXZWJEaXN0cmlidXRpb25Qcm9wcyBpbnRlcmZhY2UuXG4gKlxuICpcbiAqL1xuZXhwb3J0IGNsYXNzIENsb3VkRnJvbnRXZWJEaXN0cmlidXRpb24gZXh0ZW5kcyBjZGsuQ29uc3RydWN0IGltcGxlbWVudHMgSURpc3RyaWJ1dGlvbiB7XG4gIC8qKlxuICAgKiBUaGUgbG9nZ2luZyBidWNrZXQgZm9yIHRoaXMgQ2xvdWRGcm9udCBkaXN0cmlidXRpb24uXG4gICAqIElmIGxvZ2dpbmcgaXMgbm90IGVuYWJsZWQgZm9yIHRoaXMgZGlzdHJpYnV0aW9uIC0gdGhpcyBwcm9wZXJ0eSB3aWxsIGJlIHVuZGVmaW5lZC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBsb2dnaW5nQnVja2V0PzogczMuSUJ1Y2tldDtcblxuICAvKipcbiAgICogVGhlIGRvbWFpbiBuYW1lIGNyZWF0ZWQgYnkgQ2xvdWRGcm9udCBmb3IgdGhpcyBkaXN0cmlidXRpb24uXG4gICAqIElmIHlvdSBhcmUgdXNpbmcgYWxpYXNlcyBmb3IgeW91ciBkaXN0cmlidXRpb24sIHRoaXMgaXMgdGhlIGRvbWFpbk5hbWUgeW91ciBETlMgcmVjb3JkcyBzaG91bGQgcG9pbnQgdG8uXG4gICAqIChJbiBSb3V0ZTUzLCB5b3UgY291bGQgY3JlYXRlIGFuIEFMSUFTIHJlY29yZCB0byB0aGlzIHZhbHVlLCBmb3IgZXhhbXBsZS4gKVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGRvbWFpbk5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGRpc3RyaWJ1dGlvbiBJRCBmb3IgdGhpcyBkaXN0cmlidXRpb24uXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZGlzdHJpYnV0aW9uSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogTWFwcyBvdXIgbWV0aG9kcyB0byB0aGUgc3RyaW5nIGFycmF5cyB0aGV5IGFyZVxuICAgKi9cbiAgcHJpdmF0ZSByZWFkb25seSBNRVRIT0RfTE9PS1VQX01BUCA9IHtcbiAgICBHSDogWydHRVQnLCAnSEVBRCddLFxuICAgIEdITzogWydHRVQnLCAnSEVBRCcsICdPUFRJT05TJ10sXG4gICAgQUxMOiBbJ0RFTEVURScsICdHRVQnLCAnSEVBRCcsICdPUFRJT05TJywgJ1BBVENIJywgJ1BPU1QnLCAnUFVUJ10sXG4gIH07XG5cbiAgLyoqXG4gICAqIE1hcHMgZm9yIHdoaWNoIFNlY3VyaXR5UG9saWN5UHJvdG9jb2wgYXJlIGF2YWlsYWJsZSB0byB3aGljaCBTU0xNZXRob2RzXG4gICAqL1xuICBwcml2YXRlIHJlYWRvbmx5IFZBTElEX1NTTF9QUk9UT0NPTFM6IHsgW21ldGhvZCBpbiBTU0xNZXRob2RdOiBzdHJpbmdbXSB9ID0ge1xuICAgIFtTU0xNZXRob2QuU05JXTogW1xuICAgICAgU2VjdXJpdHlQb2xpY3lQcm90b2NvbC5UTFNfVjEsIFNlY3VyaXR5UG9saWN5UHJvdG9jb2wuVExTX1YxXzFfMjAxNixcbiAgICAgIFNlY3VyaXR5UG9saWN5UHJvdG9jb2wuVExTX1YxXzIwMTYsIFNlY3VyaXR5UG9saWN5UHJvdG9jb2wuVExTX1YxXzJfMjAxOCxcbiAgICBdLFxuICAgIFtTU0xNZXRob2QuVklQXTogW1NlY3VyaXR5UG9saWN5UHJvdG9jb2wuU1NMX1YzLCBTZWN1cml0eVBvbGljeVByb3RvY29sLlRMU19WMV0sXG4gIH07XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgbGV0IGRpc3RyaWJ1dGlvbkNvbmZpZzogQ2ZuRGlzdHJpYnV0aW9uLkRpc3RyaWJ1dGlvbkNvbmZpZ1Byb3BlcnR5ID0ge1xuICAgICAgY29tbWVudDogcHJvcHMuY29tbWVudCxcbiAgICAgIGVuYWJsZWQ6IHRydWUsXG4gICAgICBkZWZhdWx0Um9vdE9iamVjdDogcHJvcHMuZGVmYXVsdFJvb3RPYmplY3QgIT09IHVuZGVmaW5lZCA/IHByb3BzLmRlZmF1bHRSb290T2JqZWN0IDogJ2luZGV4Lmh0bWwnLFxuICAgICAgaHR0cFZlcnNpb246IHByb3BzLmh0dHBWZXJzaW9uIHx8IEh0dHBWZXJzaW9uLkhUVFAyLFxuICAgICAgcHJpY2VDbGFzczogcHJvcHMucHJpY2VDbGFzcyB8fCBQcmljZUNsYXNzLlBSSUNFX0NMQVNTXzEwMCxcbiAgICAgIGlwdjZFbmFibGVkOiAocHJvcHMuZW5hYmxlSXBWNiAhPT0gdW5kZWZpbmVkKSA/IHByb3BzLmVuYWJsZUlwVjYgOiB0cnVlLFxuICAgICAgLy8gdHNsaW50OmRpc2FibGUtbmV4dC1saW5lOm1heC1saW5lLWxlbmd0aFxuICAgICAgY3VzdG9tRXJyb3JSZXNwb25zZXM6IHByb3BzLmVycm9yQ29uZmlndXJhdGlvbnMsIC8vIFRPRE86IHZhbGlkYXRpb24gOiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTQ2xvdWRGb3JtYXRpb24vbGF0ZXN0L1VzZXJHdWlkZS9hd3MtcHJvcGVydGllcy1jbG91ZGZyb250LWRpc3RyaWJ1dGlvbi1jdXN0b21lcnJvcnJlc3BvbnNlLmh0bWwjY2ZuLWNsb3VkZnJvbnQtZGlzdHJpYnV0aW9uLWN1c3RvbWVycm9ycmVzcG9uc2UtZXJyb3JjYWNoaW5nbWludHRsXG4gICAgICB3ZWJBY2xJZDogcHJvcHMud2ViQUNMSWQsXG4gICAgfTtcblxuICAgIGNvbnN0IGJlaGF2aW9yczogQmVoYXZpb3JXaXRoT3JpZ2luW10gPSBbXTtcblxuICAgIGNvbnN0IG9yaWdpbnM6IENmbkRpc3RyaWJ1dGlvbi5PcmlnaW5Qcm9wZXJ0eVtdID0gW107XG5cbiAgICBsZXQgb3JpZ2luSW5kZXggPSAxO1xuICAgIGZvciAoY29uc3Qgb3JpZ2luQ29uZmlnIG9mIHByb3BzLm9yaWdpbkNvbmZpZ3MpIHtcbiAgICAgIGNvbnN0IG9yaWdpbklkID0gYG9yaWdpbiR7b3JpZ2luSW5kZXh9YDtcbiAgICAgIGlmICghb3JpZ2luQ29uZmlnLnMzT3JpZ2luU291cmNlICYmICFvcmlnaW5Db25maWcuY3VzdG9tT3JpZ2luU291cmNlKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignVGhlcmUgbXVzdCBiZSBhdCBsZWFzdCBvbmUgb3JpZ2luIHNvdXJjZSAtIGVpdGhlciBhbiBzM09yaWdpblNvdXJjZSBvciBhIGN1c3RvbU9yaWdpblNvdXJjZScpO1xuICAgICAgfVxuICAgICAgaWYgKG9yaWdpbkNvbmZpZy5jdXN0b21PcmlnaW5Tb3VyY2UgJiYgb3JpZ2luQ29uZmlnLnMzT3JpZ2luU291cmNlKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignVGhlcmUgY2Fubm90IGJlIGJvdGggYW4gczNPcmlnaW5Tb3VyY2UgYW5kIGEgY3VzdG9tT3JpZ2luU291cmNlIGluIHRoZSBzYW1lIFNvdXJjZUNvbmZpZ3VyYXRpb24uJyk7XG4gICAgICB9XG5cbiAgICAgIGNvbnN0IG9yaWdpbkhlYWRlcnM6IENmbkRpc3RyaWJ1dGlvbi5PcmlnaW5DdXN0b21IZWFkZXJQcm9wZXJ0eVtdID0gW107XG4gICAgICBpZiAob3JpZ2luQ29uZmlnLm9yaWdpbkhlYWRlcnMpIHtcbiAgICAgICAgT2JqZWN0LmtleXMob3JpZ2luQ29uZmlnLm9yaWdpbkhlYWRlcnMpLmZvckVhY2goa2V5ID0+IHtcbiAgICAgICAgICBjb25zdCBvSGVhZGVyOiBDZm5EaXN0cmlidXRpb24uT3JpZ2luQ3VzdG9tSGVhZGVyUHJvcGVydHkgPSB7XG4gICAgICAgICAgICBoZWFkZXJOYW1lOiBrZXksXG4gICAgICAgICAgICBoZWFkZXJWYWx1ZTogb3JpZ2luQ29uZmlnLm9yaWdpbkhlYWRlcnMhW2tleV0sXG4gICAgICAgICAgfTtcbiAgICAgICAgICBvcmlnaW5IZWFkZXJzLnB1c2gob0hlYWRlcik7XG4gICAgICAgIH0pO1xuICAgICAgfVxuXG4gICAgICBsZXQgczNPcmlnaW5Db25maWc6IENmbkRpc3RyaWJ1dGlvbi5TM09yaWdpbkNvbmZpZ1Byb3BlcnR5IHwgdW5kZWZpbmVkO1xuICAgICAgaWYgKG9yaWdpbkNvbmZpZy5zM09yaWdpblNvdXJjZSkge1xuICAgICAgICAvLyBmaXJzdCBjYXNlIGZvciBiYWNrd2FyZHMgY29tcGF0aWJpbGl0eVxuICAgICAgICBpZiAob3JpZ2luQ29uZmlnLnMzT3JpZ2luU291cmNlLm9yaWdpbkFjY2Vzc0lkZW50aXR5KSB7XG4gICAgICAgICAgLy8gZ3JhbnQgQ2xvdWRGcm9udCBPcmlnaW5BY2Nlc3NJZGVudGl0eSByZWFkIGFjY2VzcyB0byBTMyBidWNrZXRcbiAgICAgICAgICBvcmlnaW5Db25maWcuczNPcmlnaW5Tb3VyY2UuczNCdWNrZXRTb3VyY2UuZ3JhbnRSZWFkKG9yaWdpbkNvbmZpZy5zM09yaWdpblNvdXJjZS5vcmlnaW5BY2Nlc3NJZGVudGl0eSk7XG5cbiAgICAgICAgICBzM09yaWdpbkNvbmZpZyA9IHtcbiAgICAgICAgICAgIG9yaWdpbkFjY2Vzc0lkZW50aXR5OlxuICAgICAgICAgICAgICBgb3JpZ2luLWFjY2Vzcy1pZGVudGl0eS9jbG91ZGZyb250LyR7XG4gICAgICAgICAgICAgICAgb3JpZ2luQ29uZmlnLnMzT3JpZ2luU291cmNlLm9yaWdpbkFjY2Vzc0lkZW50aXR5Lm9yaWdpbkFjY2Vzc0lkZW50aXR5TmFtZVxuICAgICAgICAgICAgICB9YCxcbiAgICAgICAgICB9O1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIHMzT3JpZ2luQ29uZmlnID0ge307XG4gICAgICAgIH1cbiAgICAgIH1cblxuICAgICAgY29uc3Qgb3JpZ2luUHJvcGVydHk6IENmbkRpc3RyaWJ1dGlvbi5PcmlnaW5Qcm9wZXJ0eSA9IHtcbiAgICAgICAgaWQ6IG9yaWdpbklkLFxuICAgICAgICBkb21haW5OYW1lOiBvcmlnaW5Db25maWcuczNPcmlnaW5Tb3VyY2VcbiAgICAgICAgICA/IG9yaWdpbkNvbmZpZy5zM09yaWdpblNvdXJjZS5zM0J1Y2tldFNvdXJjZS5idWNrZXRSZWdpb25hbERvbWFpbk5hbWVcbiAgICAgICAgICA6IG9yaWdpbkNvbmZpZy5jdXN0b21PcmlnaW5Tb3VyY2UhLmRvbWFpbk5hbWUsXG4gICAgICAgIG9yaWdpblBhdGg6IG9yaWdpbkNvbmZpZy5vcmlnaW5QYXRoLFxuICAgICAgICBvcmlnaW5DdXN0b21IZWFkZXJzOiBvcmlnaW5IZWFkZXJzLmxlbmd0aCA+IDAgPyBvcmlnaW5IZWFkZXJzIDogdW5kZWZpbmVkLFxuICAgICAgICBzM09yaWdpbkNvbmZpZyxcbiAgICAgICAgY3VzdG9tT3JpZ2luQ29uZmlnOiBvcmlnaW5Db25maWcuY3VzdG9tT3JpZ2luU291cmNlXG4gICAgICAgICAgPyB7XG4gICAgICAgICAgICBodHRwUG9ydDogb3JpZ2luQ29uZmlnLmN1c3RvbU9yaWdpblNvdXJjZS5odHRwUG9ydCB8fCA4MCxcbiAgICAgICAgICAgIGh0dHBzUG9ydDogb3JpZ2luQ29uZmlnLmN1c3RvbU9yaWdpblNvdXJjZS5odHRwc1BvcnQgfHwgNDQzLFxuICAgICAgICAgICAgb3JpZ2luS2VlcGFsaXZlVGltZW91dDogb3JpZ2luQ29uZmlnLmN1c3RvbU9yaWdpblNvdXJjZS5vcmlnaW5LZWVwYWxpdmVUaW1lb3V0XG4gICAgICAgICAgICAgICYmIG9yaWdpbkNvbmZpZy5jdXN0b21PcmlnaW5Tb3VyY2Uub3JpZ2luS2VlcGFsaXZlVGltZW91dC50b1NlY29uZHMoKSB8fCA1LFxuICAgICAgICAgICAgb3JpZ2luUmVhZFRpbWVvdXQ6IG9yaWdpbkNvbmZpZy5jdXN0b21PcmlnaW5Tb3VyY2Uub3JpZ2luUmVhZFRpbWVvdXRcbiAgICAgICAgICAgICAgJiYgb3JpZ2luQ29uZmlnLmN1c3RvbU9yaWdpblNvdXJjZS5vcmlnaW5SZWFkVGltZW91dC50b1NlY29uZHMoKSB8fCAzMCxcbiAgICAgICAgICAgIG9yaWdpblByb3RvY29sUG9saWN5OiBvcmlnaW5Db25maWcuY3VzdG9tT3JpZ2luU291cmNlLm9yaWdpblByb3RvY29sUG9saWN5IHx8IE9yaWdpblByb3RvY29sUG9saWN5LkhUVFBTX09OTFksXG4gICAgICAgICAgICBvcmlnaW5Tc2xQcm90b2NvbHM6IG9yaWdpbkNvbmZpZy5jdXN0b21PcmlnaW5Tb3VyY2UuYWxsb3dlZE9yaWdpblNTTFZlcnNpb25zIHx8IFtPcmlnaW5Tc2xQb2xpY3kuVExTX1YxXzJdLFxuICAgICAgICAgIH1cbiAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgIH07XG5cbiAgICAgIGZvciAoY29uc3QgYmVoYXZpb3Igb2Ygb3JpZ2luQ29uZmlnLmJlaGF2aW9ycykge1xuICAgICAgICBiZWhhdmlvcnMucHVzaCh7IC4uLmJlaGF2aW9yLCB0YXJnZXRPcmlnaW5JZDogb3JpZ2luSWQgfSk7XG4gICAgICB9XG5cbiAgICAgIG9yaWdpbnMucHVzaChvcmlnaW5Qcm9wZXJ0eSk7XG4gICAgICBvcmlnaW5JbmRleCsrO1xuICAgIH1cblxuICAgIG9yaWdpbnMuZm9yRWFjaChvcmlnaW4gPT4ge1xuICAgICAgaWYgKCFvcmlnaW4uczNPcmlnaW5Db25maWcgJiYgIW9yaWdpbi5jdXN0b21PcmlnaW5Db25maWcpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBPcmlnaW4gJHtvcmlnaW4uZG9tYWluTmFtZX0gaXMgbWlzc2luZyBlaXRoZXIgUzNPcmlnaW5Db25maWcgb3IgQ3VzdG9tT3JpZ2luQ29uZmlnLiBBdCBsZWFzdCAxIG11c3QgYmUgc3BlY2lmaWVkLmApO1xuICAgICAgfVxuICAgIH0pO1xuICAgIGRpc3RyaWJ1dGlvbkNvbmZpZyA9IHtcbiAgICAgIC4uLmRpc3RyaWJ1dGlvbkNvbmZpZyxcbiAgICAgIG9yaWdpbnMsXG4gICAgfTtcblxuICAgIGNvbnN0IGRlZmF1bHRCZWhhdmlvcnMgPSBiZWhhdmlvcnMuZmlsdGVyKGJlaGF2aW9yID0+IGJlaGF2aW9yLmlzRGVmYXVsdEJlaGF2aW9yKTtcbiAgICBpZiAoZGVmYXVsdEJlaGF2aW9ycy5sZW5ndGggIT09IDEpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignVGhlcmUgY2FuIG9ubHkgYmUgb25lIGRlZmF1bHQgYmVoYXZpb3IgYWNyb3NzIGFsbCBzb3VyY2VzLiBbIE9uZSBkZWZhdWx0IGJlaGF2aW9yIHBlciBkaXN0cmlidXRpb24gXS4nKTtcbiAgICB9XG5cbiAgICBkaXN0cmlidXRpb25Db25maWcgPSB7IC4uLmRpc3RyaWJ1dGlvbkNvbmZpZywgZGVmYXVsdENhY2hlQmVoYXZpb3I6IHRoaXMudG9CZWhhdmlvcihkZWZhdWx0QmVoYXZpb3JzWzBdLCBwcm9wcy52aWV3ZXJQcm90b2NvbFBvbGljeSkgfTtcblxuICAgIGNvbnN0IG90aGVyQmVoYXZpb3JzOiBDZm5EaXN0cmlidXRpb24uQ2FjaGVCZWhhdmlvclByb3BlcnR5W10gPSBbXTtcbiAgICBmb3IgKGNvbnN0IGJlaGF2aW9yIG9mIGJlaGF2aW9ycy5maWx0ZXIoYiA9PiAhYi5pc0RlZmF1bHRCZWhhdmlvcikpIHtcbiAgICAgIGlmICghYmVoYXZpb3IucGF0aFBhdHRlcm4pIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdwYXRoUGF0dGVybiBpcyByZXF1aXJlZCBmb3IgYWxsIG5vbi1kZWZhdWx0IGJlaGF2aW9ycycpO1xuICAgICAgfVxuICAgICAgb3RoZXJCZWhhdmlvcnMucHVzaCh0aGlzLnRvQmVoYXZpb3IoYmVoYXZpb3IsIHByb3BzLnZpZXdlclByb3RvY29sUG9saWN5KSBhcyBDZm5EaXN0cmlidXRpb24uQ2FjaGVCZWhhdmlvclByb3BlcnR5KTtcbiAgICB9XG5cbiAgICBkaXN0cmlidXRpb25Db25maWcgPSB7IC4uLmRpc3RyaWJ1dGlvbkNvbmZpZywgY2FjaGVCZWhhdmlvcnM6IG90aGVyQmVoYXZpb3JzLmxlbmd0aCA+IDAgPyBvdGhlckJlaGF2aW9ycyA6IHVuZGVmaW5lZCB9O1xuXG4gICAgaWYgKHByb3BzLmFsaWFzQ29uZmlndXJhdGlvbiAmJiBwcm9wcy52aWV3ZXJDZXJ0aWZpY2F0ZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFtcbiAgICAgICAgJ1lvdSBjYW5ub3Qgc2V0IGJvdGggYWxpYXNDb25maWd1cmF0aW9uIGFuZCB2aWV3ZXJDZXJ0aWZpY2F0ZSBwcm9wZXJ0aWVzLicsXG4gICAgICAgICdQbGVhc2Ugb25seSB1c2Ugdmlld2VyQ2VydGlmaWNhdGUsIGFzIGFsaWFzQ29uZmlndXJhdGlvbiBpcyBkZXByZWNhdGVkLicsXG4gICAgICBdLmpvaW4oJyAnKSk7XG4gICAgfVxuXG4gICAgbGV0IF92aWV3ZXJDZXJ0aWZpY2F0ZSA9IHByb3BzLnZpZXdlckNlcnRpZmljYXRlO1xuICAgIGlmIChwcm9wcy5hbGlhc0NvbmZpZ3VyYXRpb24pIHtcbiAgICAgIGNvbnN0IHthY21DZXJ0UmVmLCBzZWN1cml0eVBvbGljeSwgc3NsTWV0aG9kLCBuYW1lczogYWxpYXNlc30gPSBwcm9wcy5hbGlhc0NvbmZpZ3VyYXRpb247XG5cbiAgICAgIF92aWV3ZXJDZXJ0aWZpY2F0ZSA9IFZpZXdlckNlcnRpZmljYXRlLmZyb21BY21DZXJ0aWZpY2F0ZShcbiAgICAgICAgY2VydGlmaWNhdGVtYW5hZ2VyLkNlcnRpZmljYXRlLmZyb21DZXJ0aWZpY2F0ZUFybih0aGlzLCAnQWxpYXNDb25maWd1cmF0aW9uQ2VydCcsIGFjbUNlcnRSZWYpLFxuICAgICAgICB7IHNlY3VyaXR5UG9saWN5LCBzc2xNZXRob2QsIGFsaWFzZXMgfSxcbiAgICAgICk7XG4gICAgfVxuXG4gICAgaWYgKF92aWV3ZXJDZXJ0aWZpY2F0ZSkge1xuICAgICAgY29uc3Qge3Byb3BzOiB2aWV3ZXJDZXJ0aWZpY2F0ZSwgYWxpYXNlc30gPSBfdmlld2VyQ2VydGlmaWNhdGU7XG4gICAgICBPYmplY3QuYXNzaWduKGRpc3RyaWJ1dGlvbkNvbmZpZywge2FsaWFzZXMsIHZpZXdlckNlcnRpZmljYXRlfSk7XG5cbiAgICAgIGNvbnN0IHttaW5pbXVtUHJvdG9jb2xWZXJzaW9uLCBzc2xTdXBwb3J0TWV0aG9kfSA9IHZpZXdlckNlcnRpZmljYXRlO1xuXG4gICAgICBpZiAobWluaW11bVByb3RvY29sVmVyc2lvbiAhPSBudWxsICYmIHNzbFN1cHBvcnRNZXRob2QgIT0gbnVsbCkge1xuICAgICAgICBjb25zdCB2YWxpZFByb3RvY29scyA9IHRoaXMuVkFMSURfU1NMX1BST1RPQ09MU1tzc2xTdXBwb3J0TWV0aG9kIGFzIFNTTE1ldGhvZF07XG5cbiAgICAgICAgaWYgKHZhbGlkUHJvdG9jb2xzLmluZGV4T2YobWluaW11bVByb3RvY29sVmVyc2lvbi50b1N0cmluZygpKSA9PT0gLTEpIHtcbiAgICAgICAgICAvLyB0c2xpbnQ6ZGlzYWJsZS1uZXh0LWxpbmU6bWF4LWxpbmUtbGVuZ3RoXG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGAke21pbmltdW1Qcm90b2NvbFZlcnNpb259IGlzIG5vdCBjb21wYWJ0aWJsZSB3aXRoIHNzbE1ldGhvZCAke3NzbFN1cHBvcnRNZXRob2R9LlxcblxcdFZhbGlkIFByb3RvY29scyBhcmU6ICR7dmFsaWRQcm90b2NvbHMuam9pbignLCAnKX1gKTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0gZWxzZSB7XG4gICAgICBkaXN0cmlidXRpb25Db25maWcgPSB7IC4uLmRpc3RyaWJ1dGlvbkNvbmZpZyxcbiAgICAgICAgdmlld2VyQ2VydGlmaWNhdGU6IHsgY2xvdWRGcm9udERlZmF1bHRDZXJ0aWZpY2F0ZTogdHJ1ZSB9LFxuICAgICAgfTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMubG9nZ2luZ0NvbmZpZykge1xuICAgICAgdGhpcy5sb2dnaW5nQnVja2V0ID0gcHJvcHMubG9nZ2luZ0NvbmZpZy5idWNrZXQgfHwgbmV3IHMzLkJ1Y2tldCh0aGlzLCAnTG9nZ2luZ0J1Y2tldCcpO1xuICAgICAgZGlzdHJpYnV0aW9uQ29uZmlnID0ge1xuICAgICAgICAuLi5kaXN0cmlidXRpb25Db25maWcsXG4gICAgICAgIGxvZ2dpbmc6IHtcbiAgICAgICAgICBidWNrZXQ6IHRoaXMubG9nZ2luZ0J1Y2tldC5idWNrZXRSZWdpb25hbERvbWFpbk5hbWUsXG4gICAgICAgICAgaW5jbHVkZUNvb2tpZXM6IHByb3BzLmxvZ2dpbmdDb25maWcuaW5jbHVkZUNvb2tpZXMgfHwgZmFsc2UsXG4gICAgICAgICAgcHJlZml4OiBwcm9wcy5sb2dnaW5nQ29uZmlnLnByZWZpeCxcbiAgICAgICAgfSxcbiAgICAgIH07XG4gICAgfVxuXG4gICAgY29uc3QgZGlzdHJpYnV0aW9uID0gbmV3IENmbkRpc3RyaWJ1dGlvbih0aGlzLCAnQ0ZEaXN0cmlidXRpb24nLCB7IGRpc3RyaWJ1dGlvbkNvbmZpZyB9KTtcbiAgICB0aGlzLm5vZGUuZGVmYXVsdENoaWxkID0gZGlzdHJpYnV0aW9uO1xuICAgIHRoaXMuZG9tYWluTmFtZSA9IGRpc3RyaWJ1dGlvbi5hdHRyRG9tYWluTmFtZTtcbiAgICB0aGlzLmRpc3RyaWJ1dGlvbklkID0gZGlzdHJpYnV0aW9uLnJlZjtcbiAgfVxuXG4gIHByaXZhdGUgdG9CZWhhdmlvcihpbnB1dDogQmVoYXZpb3JXaXRoT3JpZ2luLCBwcm90b1BvbGljeT86IFZpZXdlclByb3RvY29sUG9saWN5KSB7XG4gICAgbGV0IHRvUmV0dXJuID0ge1xuICAgICAgYWxsb3dlZE1ldGhvZHM6IHRoaXMuTUVUSE9EX0xPT0tVUF9NQVBbaW5wdXQuYWxsb3dlZE1ldGhvZHMgfHwgQ2xvdWRGcm9udEFsbG93ZWRNZXRob2RzLkdFVF9IRUFEXSxcbiAgICAgIGNhY2hlZE1ldGhvZHM6IHRoaXMuTUVUSE9EX0xPT0tVUF9NQVBbaW5wdXQuY2FjaGVkTWV0aG9kcyB8fCBDbG91ZEZyb250QWxsb3dlZENhY2hlZE1ldGhvZHMuR0VUX0hFQURdLFxuICAgICAgY29tcHJlc3M6IGlucHV0LmNvbXByZXNzICE9PSBmYWxzZSxcbiAgICAgIGRlZmF1bHRUdGw6IGlucHV0LmRlZmF1bHRUdGwgJiYgaW5wdXQuZGVmYXVsdFR0bC50b1NlY29uZHMoKSxcbiAgICAgIGZvcndhcmRlZFZhbHVlczogaW5wdXQuZm9yd2FyZGVkVmFsdWVzIHx8IHsgcXVlcnlTdHJpbmc6IGZhbHNlLCBjb29raWVzOiB7IGZvcndhcmQ6ICdub25lJyB9IH0sXG4gICAgICBtYXhUdGw6IGlucHV0Lm1heFR0bCAmJiBpbnB1dC5tYXhUdGwudG9TZWNvbmRzKCksXG4gICAgICBtaW5UdGw6IGlucHV0Lm1pblR0bCAmJiBpbnB1dC5taW5UdGwudG9TZWNvbmRzKCksXG4gICAgICB0cnVzdGVkU2lnbmVyczogaW5wdXQudHJ1c3RlZFNpZ25lcnMsXG4gICAgICB0YXJnZXRPcmlnaW5JZDogaW5wdXQudGFyZ2V0T3JpZ2luSWQsXG4gICAgICB2aWV3ZXJQcm90b2NvbFBvbGljeTogcHJvdG9Qb2xpY3kgfHwgVmlld2VyUHJvdG9jb2xQb2xpY3kuUkVESVJFQ1RfVE9fSFRUUFMsXG4gICAgfTtcbiAgICBpZiAoIWlucHV0LmlzRGVmYXVsdEJlaGF2aW9yKSB7XG4gICAgICB0b1JldHVybiA9IE9iamVjdC5hc3NpZ24odG9SZXR1cm4sIHsgcGF0aFBhdHRlcm46IGlucHV0LnBhdGhQYXR0ZXJuIH0pO1xuICAgIH1cbiAgICBpZiAoaW5wdXQubGFtYmRhRnVuY3Rpb25Bc3NvY2lhdGlvbnMpIHtcbiAgICAgIHRvUmV0dXJuID0gT2JqZWN0LmFzc2lnbih0b1JldHVybiwge1xuICAgICAgICBsYW1iZGFGdW5jdGlvbkFzc29jaWF0aW9uczogaW5wdXQubGFtYmRhRnVuY3Rpb25Bc3NvY2lhdGlvbnNcbiAgICAgICAgICAubWFwKGZuYSA9PiAoe1xuICAgICAgICAgICAgZXZlbnRUeXBlOiBmbmEuZXZlbnRUeXBlLFxuICAgICAgICAgICAgbGFtYmRhRnVuY3Rpb25Bcm46IGZuYS5sYW1iZGFGdW5jdGlvbiAmJiBmbmEubGFtYmRhRnVuY3Rpb24uZnVuY3Rpb25Bcm4sXG4gICAgICAgICAgfSkpLFxuICAgICAgfSk7XG5cbiAgICAgIC8vIGFsbG93IGVkZ2VsYW1iZGEuYW1hem9uYXdzLmNvbSB0byBhc3N1bWUgdGhlIGZ1bmN0aW9ucycgZXhlY3V0aW9uIHJvbGUuXG4gICAgICBmb3IgKGNvbnN0IGEgb2YgaW5wdXQubGFtYmRhRnVuY3Rpb25Bc3NvY2lhdGlvbnMpIHtcbiAgICAgICAgaWYgKGEubGFtYmRhRnVuY3Rpb24ucm9sZSAmJiBhLmxhbWJkYUZ1bmN0aW9uLnJvbGUgaW5zdGFuY2VvZiBpYW0uUm9sZSAmJiBhLmxhbWJkYUZ1bmN0aW9uLnJvbGUuYXNzdW1lUm9sZVBvbGljeSkge1xuICAgICAgICAgIGEubGFtYmRhRnVuY3Rpb24ucm9sZS5hc3N1bWVSb2xlUG9saWN5LmFkZFN0YXRlbWVudHMobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgYWN0aW9uczogWyAnc3RzOkFzc3VtZVJvbGUnIF0sXG4gICAgICAgICAgICBwcmluY2lwYWxzOiBbIG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWRnZWxhbWJkYS5hbWF6b25hd3MuY29tJykgXSxcbiAgICAgICAgICB9KSk7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIHRvUmV0dXJuO1xuICB9XG59XG4iXX0=