import { Duration } from './duration';
/**
 * Represents a date of expiration.
 *
 * The amount can be specified either as a Date object, timestamp, Duration or string.
 *
 * @stability stable
 */
export declare class Expiration {
    /**
     * Expire at the specified date.
     *
     * @param d date to expire at.
     * @stability stable
     */
    static atDate(d: Date): Expiration;
    /**
     * Expire at the specified timestamp.
     *
     * @param t timestamp in unix milliseconds.
     * @stability stable
     */
    static atTimestamp(t: number): Expiration;
    /**
     * Expire once the specified duration has passed since deployment time.
     *
     * @param t the duration to wait before expiring.
     * @stability stable
     */
    static after(t: Duration): Expiration;
    /**
     * Expire at specified date, represented as a string.
     *
     * @param s the string that represents date to expire at.
     * @stability stable
     */
    static fromString(s: string): Expiration;
    /**
     * Expiration value as a Date object.
     *
     * @stability stable
     */
    readonly date: Date;
    private constructor();
    /**
     * Exipration Value in a formatted Unix Epoch Time in seconds.
     *
     * @stability stable
     */
    toEpoch(): number;
    /**
     * Check if Exipiration expires before input.
     *
     * @param t the duration to check against.
     * @stability stable
     */
    isBefore(t: Duration): boolean;
    /**
     * Check if Exipiration expires after input.
     *
     * @param t the duration to check against.
     * @stability stable
     */
    isAfter(t: Duration): boolean;
}
