# AUTOGENERATED! DO NOT EDIT! File to edit: 01_dataset.ipynb (unless otherwise specified).

__all__ = ['conn', 'get_user_eeg_info', 'get_eeg', 'get_task', 'get_events', 'ch_names', 'get_timestamps',
           'parse_channel', 'parse_eeg', 'create_montage', 'create_raw', 'sync_task', 'get_events', 'get_annotations',
           'get_session', 'get_nback', 'dpath', 'download_sessions', 'get_sessions']

# Cell
import numpy as np
import pandas as pd
import mne
from pathlib import Path
import psycopg2
import matplotlib.pyplot as plt
from functools import partial
import os
import pickle

# Cell
conn = psycopg2.connect(user='prisma',
                        password='cognition365*',
                        host='34.76.231.74',
                        port='5432',
                        database='mydb')

# Cell
def get_user_eeg_info(email):
    sql = f"""
    SELECT
        U.email,
        EEG.id as "eeg_id",
        COUNT(EPOCH.id) as "n_epochs"
    FROM "User" as U
    JOIN "EEG" as EEG on EEG."userId" = U.id
    JOIN "EEGEpoch" as EPOCH on EPOCH."eegId" = EEG.id
    WHERE U.email='{email}'
    GROUP BY "email", "eeg_id"
    ORDER BY n_epochs DESC
    """
    cursor = conn.cursor()
    cursor.execute(sql)
    data = cursor.fetchall()
    return pd.DataFrame(data, columns=['email', 'eeg_id', 'n_epochs'])

# Cell
def get_eeg(eeg_id):
    sql = f"""
    SELECT
        U.id as "user_id", U.email,
        EEG.id as "eeg_id",
        EPOCH.id as "epoch_id", EPOCH."samplingRate" as "sampling_rate", EPOCH."startTime" as "start_time",
        SAMPLE.id as "sample_id", SAMPLE.index as "sample_index", SAMPLE.data

    FROM "User" as U
    JOIN "EEG" as EEG on EEG."userId" = U.id
    JOIN "EEGEpoch" as EPOCH on EPOCH."eegId" = EEG.id
    JOIN "EEGSample" as SAMPLE on SAMPLE."epochId" = EPOCH.id
    WHERE EEG.id='{eeg_id}'
    ORDER BY start_time, sample_id, sample_index
    """
    cursor = conn.cursor()
    cursor.execute(sql)
    data = cursor.fetchall()
    return pd.DataFrame(data, columns=['user_id', 'email', 'eeg_id', 'epoch_id', 'sampling_rate', 'start_time', 'sample_id', 'sample_index', 'data'])

# Cell
def get_task(eeg_id):
    sql = f"""
    SELECT
        U.id as "user_id", U.email,
        TASK.id as "task_id", TASK.type, TASK.name, TASK.completed, TASK.comment, TASK."eegId",
        TRIAL.id as "trial_id", TRIAL.index, TRIAL.critical, TRIAL.stimulus, TRIAL.expected,
        RESP.id as "response_id", RESP."presentedAt", RESP."respondedAt", RESP.response, RESP.result

    FROM "User" as U
    JOIN "Task" as TASK on TASK."userId" = U.id
    JOIN "Trial" as TRIAL on TRIAL."taskId" = Task.id
    JOIN "TrialResponse" as RESP on RESP."trialId" = TRIAL.id

    WHERE TASK."eegId"='{eeg_id}'
    """
    cursor = conn.cursor()
    cursor.execute(sql)
    data = cursor.fetchall()
    return pd.DataFrame(data, columns=['user_id', 'email', 'task_id', 'type', 'name', 'completed', 'comments', 'eeg_id',
                                       'trial_id', 'index', 'critical', 'stimulus', 'expected',
                                       'response_id', 'presented_at', 'responded_at', 'response', 'result'])

# Cell
def get_events(eeg_id):
    sql = f"""
        SELECT
        U.id as "user_id", U.email,
        EEG.id as "eeg_id",
        EVENT.id as "event_id", EVENT.timestamp, EVENT.value

    FROM "User" as U
    JOIN "EEG" as EEG on EEG."userId" = U.id
    JOIN "EEGEvent" as EVENT on EVENT."eegId" = EEG.id
    WHERE EVENT."eegId"='{eeg_id}'
    ORDER BY timestamp, event_id
    """
    cursor = conn.cursor()
    cursor.execute(sql)
    data = cursor.fetchall()
    return pd.DataFrame(data, columns=['user_id', 'email', 'eeg_id', 'event_id', 'timestamp', 'value'])

# Cell
ch_names = ['TP9', 'AF7', 'AF8', 'TP10', 'AUX']

# Cell
def get_timestamps(ch):
    starts = pd.to_datetime(ch.start_time.astype('int'), unit='ms').values
    periods = np.array(ch.sampling_rate.values, dtype=np.int)
    # subtract 1ms from next start_time to avoid overlap, and add 1 sec to last trial
    ends = np.append(starts[1:] - np.timedelta64(1, 'ms'), starts[-1] + np.timedelta64(1, 's'))
    assert len(starts) == len(periods) == len(ends)
    return np.array([pd.date_range(start=s, end=e, periods=p) for s,e,p in zip(starts, ends, periods)]).flatten()

# Cell
def parse_channel(ch):
    timestamps = get_timestamps(ch)
    data = ch.data.explode().values
    assert len(timestamps) == len(data)
    ch_name = ch_names[np.unique(ch.sample_index)[0]]
    return pd.DataFrame(data.T, index=timestamps, columns=[ch_name])

# Cell
def parse_eeg(eeg):
    eeg_data = eeg[['sampling_rate', 'start_time', 'sample_id', 'sample_index', 'data']]
    sfreq = eeg.sampling_rate.unique()[0]
    channels = [eeg_data[eeg_data.sample_index == i] for i, ch in enumerate(ch_names)]
    return pd.concat([parse_channel(ch) for ch in channels], axis=1)

# Cell
def create_montage(ref_montage=Path('./standard_1005.tsv')):
    ch_labels = ch_names[:-1] # remove AUX
    ch_labels.append('Nz'); ch_labels

    df = pd.read_csv(ref_montage, sep='\t')
    ch_pos = df.set_index('label')
    ch_pos = ch_pos.loc[ch_labels, ['x', 'y', 'z']]

    ch_pos = {k:v for k,v in zip(ch_pos.T, ch_pos.values)}

    return mne.channels.make_dig_montage(nasion=ch_pos['Nz'],
                                            lpa=ch_pos['TP9'],
                                            rpa=ch_pos['TP10'],
                                            ch_pos=ch_pos,
                                            coord_frame='unknown',
                                            hsp=None,
                                            hpi=None)

# Cell
def create_raw(data, sfreq):
    info = mne.create_info(ch_names, sfreq, ch_types='eeg'); info
    raw = mne.io.RawArray(data.T, info)
    raw.drop_channels('AUX')
    montage = create_montage()
    raw.set_montage(montage)
    return raw

# Cell
def sync_task(data, task, event_dir):
    presented_at = pd.to_datetime(task.presented_at.astype('int'), unit='ms').values
    responded_at = pd.to_datetime(task.responded_at.astype('int'), unit='ms').values

    p_events = [[data.index.get_loc(t, method='nearest'), 0, event_dir['presented_at']] for t in presented_at]
    r_events = [[data.index.get_loc(t, method='nearest'), event_dir['presented_at'], event_dir['responded_at']] for t in responded_at]

    events = np.concatenate([p_events, r_events])
    events_df = pd.DataFrame(events, columns=['time_index','prior_event', 'event']).set_index('time_index')

    return events_df.sort_index()

# Cell
def get_events(data, task, event_dir):
    events = sync_task(data, task, event_dir)
    time_index = events.index
    prior, event = events.values.T

    return np.array([time_index, prior, event]).T

# Cell
def get_annotations(data, task, event_dir):
    events_df = sync_task(data, task, event_dir)
    idx2event = {v:k for k,v in event_dir.items()}

    p_events = events_df.loc[events_df['event'] == 1]
    r_events = events_df.loc[events_df['event'] == 2]

    # presented at
    p_len = len(p_events)

    p_dur = (r_events.index - p_events.index) / 100
    p_desc = np.full(p_len, idx2event[1])
    p_onset = p_events.index / 100
    p_annots = np.array([p_onset, p_dur, p_desc]).T


    # responded at
    r_len = len(r_events)

    r_dur = np.full(r_len, 0.5) # can be calculated with timed trial (eg. 2000ms)
    r_desc = np.full(r_len, idx2event[2])
    r_onset = r_events.index / 100
    r_annots = np.array([r_onset, r_dur, r_desc]).T

    annots = np.concatenate([p_annots, r_annots])

    annots_df = pd.DataFrame(annots, columns=['onset', 'duration', 'description']).set_index('onset').sort_index()

    onset = annots_df.index
    duration = annots_df['duration']
    description = annots_df['description']

    annots = mne.Annotations(onset, duration, description)
    return annots

# Cell
def get_session(eeg_id, event_dir=dict()):
        # TODO: check if id is correct
        eeg = get_eeg(eeg_id)
        task = get_task(eeg_id)
        data = parse_eeg(eeg)
        sfreq = eeg.sampling_rate.unique()[0]

        raw = create_raw(data, sfreq)
        events = get_events(data, task, event_dir)

        ## TODO: improve annotations
        if event_dir['presented_at'] == 1:
            print("Annotating")
            annots = get_annotations(data, task, event_dir)
            raw.set_annotations(annots)

        return (raw, events, event_dir)

# Cell
get_nback = partial(get_session,event_dir=dict(presented_at=1, responded_at=2))

# Cell
dpath = Path('./data/eeg-sessions.p')

# Cell
def download_sessions(eeg_ids, write=True, dpath=Path('./data/eeg-sessions.p')):
    sessions = []
    for eeg_id in eeg_ids:
        task = dataset.get_task(eeg_id)
        if len(task['type']):
            _type = task['type'][0]
            if _type == 'NBACK':
                print(f"Downloading: {_type}")
                sessions.append(('NBACK', get_nback(eeg_id)))
        else:
            print(f"Dowloading: FREE")
            sessions.append(('FREE', get_session(eeg_id, event_dir=dict())))

    if write: pickle.dump(sessions, open(dpath, 'wb'))
    return sessions

# Cell
def get_sessions(eeg_ids):
    ## TODO: if the folder contains files for specific id
    is_empty = len(os.listdir('./data')) < 1; is_empty
    sessions = download_sessions(eeg_ids) if is_empty else pickle.load(open(dpath, 'rb'))
    return sessions