import numpy
import scipy.interpolate


def compute_ang_avg_tck(size=100000000, seed=101):
    """Compute angle-averaged detector response function B-spline interpolant

    This should not ever need to be calculated manually, as the
    pre-computed output should have supplied in pickle form with the
    source and made available as ANG_AVG_TCK.

    This takes a while (>30 seconds) even on reasonable machines.

    """
    orig_state = numpy.random.get_state()
    numpy.random.seed(seed)

    cosi = numpy.random.uniform(-1.0, 1.0, size=size)
    costheta = numpy.random.uniform(-1.0, 1.0, size=size)
    phi = numpy.random.uniform(0.0, numpy.pi, size=size)
    psi = numpy.random.uniform(0.0, numpy.pi, size=size)
    cos2phi = numpy.cos(2.0 * phi)
    sin2phi = numpy.sin(2.0 * phi)
    cos2psi = numpy.cos(2.0 * psi)
    sin2psi = numpy.sin(2.0 * psi)
    fp = +0.5*(1.0 + costheta**2) * cos2phi * cos2psi - costheta * sin2phi * sin2psi
    fc = +0.5*(1.0 + costheta**2) * cos2phi * sin2psi + costheta * sin2phi * cos2psi
    defffac = ((0.5 * (1.0 + cosi**2) * fp)**2 + (cosi * fc)**2)**-0.5

    x = 1.0 / (1.0 - numpy.linspace(0.0, 1.0, 100, endpoint=False))**2.0
    defffac.sort()
    frac = numpy.searchsorted(defffac, x) / float(size)
    log1mfrac = numpy.log(1 - frac)
    tck = scipy.interpolate.splrep(numpy.log(x), log1mfrac)

    numpy.random.set_state(orig_state)

    return tck


ANG_AVG_TCK = (
    numpy.array(
        [0.        , 0.        , 0.        , 0.        , 0.04040541,
         0.06091841, 0.08164399, 0.10258659, 0.12375081, 0.14514139,
         0.16676322, 0.18862136, 0.21072103, 0.23306763, 0.25566674,
         0.27852413, 0.30164578, 0.32503786, 0.34870677, 0.37265916,
         0.39690188, 0.42144206, 0.4462871 , 0.47144467, 0.49692272,
         0.52272953, 0.54887369, 0.57536414, 0.60221019, 0.62942149,
         0.65700813, 0.68498062, 0.71334989, 0.74212736, 0.77132496,
         0.80095513, 0.83103089, 0.86156583, 0.89257421, 0.92407092,
         0.9560716 , 0.98859264, 1.02165125, 1.05526548, 1.08945435,
         1.12423784, 1.15963699, 1.195674  , 1.23237228, 1.26975654,
         1.30785293, 1.34668911, 1.38629436, 1.42669978, 1.46793835,
         1.51004517, 1.55305758, 1.59701539, 1.6419611 , 1.68794014,
         1.73500114, 1.78319624, 1.83258146, 1.88321708, 1.93516805,
         1.98850455, 2.0433025 , 2.09964425, 2.15761932, 2.21732525,
         2.27886857, 2.34236596, 2.40794561, 2.47574871, 2.54593135,
         2.61866664, 2.6941473 , 2.77258872, 2.85423271, 2.93935194,
         3.02825547, 3.1212955 , 3.21887582, 3.32146241, 3.42959686,
         3.54391368, 3.66516293, 3.79423997, 3.93222571, 4.08044166,
         4.24052707, 4.41454983, 4.60517019, 4.81589122, 5.05145729,
         5.31852007, 5.62682143, 5.99146455, 6.43775165, 7.01311579,
         9.21034037, 9.21034037, 9.21034037, 9.21034037]),
    numpy.array(
        [ 1.34446994e-20, -5.38168620e-06, -4.11071419e-04, -1.75358847e-03,
         -3.19118624e-03, -5.04252678e-03, -7.32043322e-03, -1.00358011e-02,
         -1.32139717e-02, -1.68210418e-02, -2.09155000e-02, -2.55164947e-02,
         -3.05903247e-02, -3.61847891e-02, -4.23219015e-02, -4.90056643e-02,
         -5.62464091e-02, -6.41063713e-02, -7.25830126e-02, -8.17432599e-02,
         -9.16081704e-02, -1.02213946e-01, -1.13631721e-01, -1.25827020e-01,
         -1.39025090e-01, -1.53174067e-01, -1.68490412e-01, -1.85053814e-01,
         -2.03021427e-01, -2.22654990e-01, -2.45025724e-01, -2.69734580e-01,
         -2.96273954e-01, -3.24501024e-01, -3.54482318e-01, -3.86039500e-01,
         -4.19254381e-01, -4.54202947e-01, -4.90883478e-01, -5.29220946e-01,
         -5.69381766e-01, -6.11522963e-01, -6.55516450e-01, -7.01703438e-01,
         -7.50041109e-01, -8.00681220e-01, -8.53843170e-01, -9.09734720e-01,
         -9.68357308e-01, -1.03026317e+00, -1.09576185e+00, -1.16397580e+00,
         -1.23313253e+00, -1.30403588e+00, -1.37650319e+00, -1.45064707e+00,
         -1.52663780e+00, -1.60437279e+00, -1.68414960e+00, -1.76570993e+00,
         -1.84940258e+00, -1.93556342e+00, -2.02414210e+00, -2.11499761e+00,
         -2.20902800e+00, -2.30525787e+00, -2.40470258e+00, -2.50748338e+00,
         -2.61348515e+00, -2.72286620e+00, -2.83695156e+00, -2.95452778e+00,
         -3.07662223e+00, -3.20341416e+00, -3.33529272e+00, -3.47281704e+00,
         -3.61646404e+00, -3.76636629e+00, -3.92301149e+00, -4.08936956e+00,
         -4.26126575e+00, -4.44378247e+00, -4.63745191e+00, -4.84154169e+00,
         -5.06157480e+00, -5.29272394e+00, -5.54299496e+00, -5.81095734e+00,
         -6.10349712e+00, -6.42191190e+00, -6.77236199e+00, -7.15599858e+00,
         -7.59186882e+00, -8.09538100e+00, -8.66749595e+00, -9.36396730e+00,
         -1.02471504e+01, -1.22240111e+01, -1.40270198e+01, -1.54762418e+01,
          0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00]),
    3,
)


def ang_avg(x):
    """Return angle-averaged detector response for a given SNR

    """
    x = numpy.log(x)
    if x < 0.0:
        f = 0.0
    elif x > max(ANG_AVG_TCK[0]):
        f = 1.0
    else:
        f = 1.0 - numpy.exp(scipy.interpolate.splev(x, ANG_AVG_TCK))
    return f

####################

if __name__ == '__main__':
    print(compute_ang_avg_tck())
