"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const sfn = require("../lib");
describe('Task base', () => {
    let stack;
    let task;
    beforeEach(() => {
        // GIVEN
        stack = new cdk.Stack();
        task = new FakeTask(stack, 'my-task', {
            metrics: {
                metricPrefixPlural: '',
                metricPrefixSingular: '',
            },
        });
    });
    test('instantiate a concrete implementation with properties', () => {
        // WHEN
        task = new FakeTask(stack, 'my-exciting-task', {
            comment: 'my exciting task',
            heartbeat: cdk.Duration.seconds(10),
            timeout: cdk.Duration.minutes(10),
        });
        // THEN
        expect(render(task)).toEqual({
            StartAt: 'my-exciting-task',
            States: {
                'my-exciting-task': {
                    End: true,
                    Type: 'Task',
                    Comment: 'my exciting task',
                    TimeoutSeconds: 600,
                    HeartbeatSeconds: 10,
                    Resource: 'my-resource',
                    Parameters: { MyParameter: 'myParameter' },
                },
            },
        });
    });
    test('add catch configuration', () => {
        // GIVEN
        const failure = new sfn.Fail(stack, 'failed', {
            error: 'DidNotWork',
            cause: 'We got stuck',
        });
        // WHEN
        task.addCatch(failure);
        // THEN
        expect(render(task)).toEqual({
            StartAt: 'my-task',
            States: {
                'my-task': {
                    End: true,
                    Catch: [{
                            ErrorEquals: ['States.ALL'],
                            Next: 'failed',
                        }],
                    Type: 'Task',
                    Resource: 'my-resource',
                    Parameters: { MyParameter: 'myParameter' },
                },
                'failed': {
                    Type: 'Fail',
                    Error: 'DidNotWork',
                    Cause: 'We got stuck',
                },
            },
        });
    });
    test('add retry configuration', () => {
        // WHEN
        task.addRetry({ errors: ['HTTPError'], maxAttempts: 2 })
            .addRetry(); // adds default retry
        // THEN
        expect(render(task)).toEqual({
            StartAt: 'my-task',
            States: {
                'my-task': {
                    End: true,
                    Retry: [
                        {
                            ErrorEquals: ['HTTPError'],
                            MaxAttempts: 2,
                        },
                        {
                            ErrorEquals: ['States.ALL'],
                        },
                    ],
                    Type: 'Task',
                    Resource: 'my-resource',
                    Parameters: { MyParameter: 'myParameter' },
                },
            },
        });
    });
    test('add a next state to the task in the chain', () => {
        // WHEN
        task.next(new sfn.Pass(stack, 'passState'));
        // THEN
        expect(render(task)).toEqual({
            StartAt: 'my-task',
            States: {
                'my-task': {
                    Next: 'passState',
                    Type: 'Task',
                    Resource: 'my-resource',
                    Parameters: { MyParameter: 'myParameter' },
                },
                'passState': { Type: 'Pass', End: true },
            },
        });
    });
    test('get named metric for this task', () => {
        // WHEN
        const metric = task.metric('my-metric');
        // THEN
        verifyMetric(metric, 'my-metric', 'Sum');
    });
    test('add metric for task state run time', () => {
        // WHEN
        const metric = task.metricRunTime();
        // THEN
        verifyMetric(metric, 'RunTime', 'Average');
    });
    test('add metric for task schedule time', () => {
        // WHEN
        const metric = task.metricScheduleTime();
        // THEN
        verifyMetric(metric, 'ScheduleTime', 'Average');
    });
    test('add metric for time between task being scheduled to closing', () => {
        // WHEN
        const metric = task.metricTime();
        // THEN
        verifyMetric(metric, 'Time', 'Average');
    });
    test('add metric for number of times the task is scheduled', () => {
        // WHEN
        const metric = task.metricScheduled();
        // THEN
        verifyMetric(metric, 'Scheduled', 'Sum');
    });
    test('add metric for number of times the task times out', () => {
        // WHEN
        const metric = task.metricTimedOut();
        // THEN
        verifyMetric(metric, 'TimedOut', 'Sum');
    });
    test('add metric for number of times the task was started', () => {
        // WHEN
        const metric = task.metricStarted();
        // THEN
        verifyMetric(metric, 'Started', 'Sum');
    });
    test('add metric for number of times the task succeeded', () => {
        // WHEN
        const metric = task.metricSucceeded();
        // THEN
        verifyMetric(metric, 'Succeeded', 'Sum');
    });
    test('add metric for number of times the task failed', () => {
        // WHEN
        const metric = task.metricFailed();
        // THEN
        verifyMetric(metric, 'Failed', 'Sum');
    });
    test('add metric for number of times the metrics heartbeat timed out', () => {
        // WHEN
        const metric = task.metricHeartbeatTimedOut();
        // THEN
        verifyMetric(metric, 'HeartbeatTimedOut', 'Sum');
    });
    test('metrics must be configured to use metric* APIs', () => {
        // GIVEN
        task = new FakeTask(stack, 'mytask', {});
        // THEN
        expect(() => {
            task.metricFailed();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricHeartbeatTimedOut();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricRunTime();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricScheduleTime();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricScheduled();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricStarted();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricSucceeded();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricTime();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricTimedOut();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
    });
});
function verifyMetric(metric, metricName, statistic) {
    expect(metric).toEqual({
        period: {
            amount: 5,
            unit: {
                label: 'minutes',
                inMillis: 60000,
            },
        },
        namespace: 'AWS/States',
        metricName,
        statistic,
    });
}
function render(sm) {
    return new cdk.Stack().resolve(new sfn.StateGraph(sm.startState, 'Test Graph').toGraphJson());
}
class FakeTask extends sfn.TaskStateBase {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.taskMetrics = props.metrics;
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: 'my-resource',
            Parameters: sfn.FieldUtils.renderObject({
                MyParameter: 'myParameter',
            }),
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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