"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const stepfunctions = require("../lib");
describe('States Language', () => {
    test('A single task is a State Machine', () => {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const chain = new stepfunctions.Pass(stack, 'Some State');
        // THEN
        expect(render(chain)).toStrictEqual({
            StartAt: 'Some State',
            States: {
                'Some State': { Type: 'Pass', End: true },
            },
        });
    }),
        test('A sequence of two tasks is a State Machine', () => {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const task1 = new stepfunctions.Pass(stack, 'State One');
            const task2 = new stepfunctions.Pass(stack, 'State Two');
            const chain = stepfunctions.Chain
                .start(task1)
                .next(task2);
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'State One',
                States: {
                    'State One': { Type: 'Pass', Next: 'State Two' },
                    'State Two': { Type: 'Pass', End: true },
                },
            });
        }),
        test('You dont need to hold on to the state to render the entire state machine correctly', () => {
            const stack = new cdk.Stack();
            // WHEN
            const task1 = new stepfunctions.Pass(stack, 'State One');
            const task2 = new stepfunctions.Pass(stack, 'State Two');
            task1.next(task2);
            // THEN
            expect(render(task1)).toStrictEqual({
                StartAt: 'State One',
                States: {
                    'State One': { Type: 'Pass', Next: 'State Two' },
                    'State Two': { Type: 'Pass', End: true },
                },
            });
        }),
        test('A chain can be appended to', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Pass(stack, 'State One');
            const task2 = new stepfunctions.Pass(stack, 'State Two');
            const task3 = new stepfunctions.Pass(stack, 'State Three');
            // WHEN
            const chain = stepfunctions.Chain
                .start(task1)
                .next(task2)
                .next(task3);
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'State One',
                States: {
                    'State One': { Type: 'Pass', Next: 'State Two' },
                    'State Two': { Type: 'Pass', Next: 'State Three' },
                    'State Three': { Type: 'Pass', End: true },
                },
            });
        }),
        test('A state machine can be appended to another state machine', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Pass(stack, 'State One');
            const task2 = new stepfunctions.Pass(stack, 'State Two');
            const task3 = new stepfunctions.Wait(stack, 'State Three', {
                time: stepfunctions.WaitTime.duration(cdk.Duration.seconds(10)),
            });
            // WHEN
            const chain = stepfunctions.Chain
                .start(task1)
                .next(stepfunctions.Chain.start(task2).next(task3));
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'State One',
                States: {
                    'State One': { Type: 'Pass', Next: 'State Two' },
                    'State Two': { Type: 'Pass', Next: 'State Three' },
                    'State Three': { Type: 'Wait', End: true, Seconds: 10 },
                },
            });
        }),
        test('A state machine definition can be instantiated and chained', () => {
            const stack = new cdk.Stack();
            const before = new stepfunctions.Pass(stack, 'Before');
            const after = new stepfunctions.Pass(stack, 'After');
            // WHEN
            const chain = before.next(new ReusableStateMachine(stack, 'Reusable')).next(after);
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'Before',
                States: {
                    'Before': { Type: 'Pass', Next: 'Choice' },
                    'Choice': {
                        Type: 'Choice',
                        Choices: [
                            { Variable: '$.branch', StringEquals: 'left', Next: 'Left Branch' },
                            { Variable: '$.branch', StringEquals: 'right', Next: 'Right Branch' },
                        ],
                    },
                    'Left Branch': { Type: 'Pass', Next: 'After' },
                    'Right Branch': { Type: 'Pass', Next: 'After' },
                    'After': { Type: 'Pass', End: true },
                },
            });
        }),
        test('A success state cannot be chained onto', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const succeed = new stepfunctions.Succeed(stack, 'Succeed');
            const pass = new stepfunctions.Pass(stack, 'Pass');
            // WHEN
            expect(() => pass.next(succeed).next(pass)).toThrow();
        }),
        test('A failure state cannot be chained onto', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const fail = new stepfunctions.Fail(stack, 'Fail', { error: 'X', cause: 'Y' });
            const pass = new stepfunctions.Pass(stack, 'Pass');
            // WHEN
            expect(() => pass.next(fail).next(pass)).toThrow();
        }),
        test('Parallels can contain direct states', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const one = new stepfunctions.Pass(stack, 'One');
            const two = new stepfunctions.Pass(stack, 'Two');
            const three = new stepfunctions.Pass(stack, 'Three');
            // WHEN
            const para = new stepfunctions.Parallel(stack, 'Parallel');
            para.branch(one.next(two));
            para.branch(three);
            // THEN
            expect(render(para)).toStrictEqual({
                StartAt: 'Parallel',
                States: {
                    Parallel: {
                        Type: 'Parallel',
                        End: true,
                        Branches: [
                            {
                                StartAt: 'One',
                                States: {
                                    One: { Type: 'Pass', Next: 'Two' },
                                    Two: { Type: 'Pass', End: true },
                                },
                            },
                            {
                                StartAt: 'Three',
                                States: {
                                    Three: { Type: 'Pass', End: true },
                                },
                            },
                        ],
                    },
                },
            });
        }),
        test('Parallels can contain instantiated reusable definitions', () => {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const para = new stepfunctions.Parallel(stack, 'Parallel');
            para.branch(new ReusableStateMachine(stack, 'Reusable1').prefixStates('Reusable1/'));
            para.branch(new ReusableStateMachine(stack, 'Reusable2').prefixStates('Reusable2/'));
            // THEN
            expect(render(para)).toStrictEqual({
                StartAt: 'Parallel',
                States: {
                    Parallel: {
                        Type: 'Parallel',
                        End: true,
                        Branches: [
                            {
                                StartAt: 'Reusable1/Choice',
                                States: {
                                    'Reusable1/Choice': {
                                        Type: 'Choice',
                                        Choices: [
                                            { Variable: '$.branch', StringEquals: 'left', Next: 'Reusable1/Left Branch' },
                                            { Variable: '$.branch', StringEquals: 'right', Next: 'Reusable1/Right Branch' },
                                        ],
                                    },
                                    'Reusable1/Left Branch': { Type: 'Pass', End: true },
                                    'Reusable1/Right Branch': { Type: 'Pass', End: true },
                                },
                            },
                            {
                                StartAt: 'Reusable2/Choice',
                                States: {
                                    'Reusable2/Choice': {
                                        Type: 'Choice',
                                        Choices: [
                                            { Variable: '$.branch', StringEquals: 'left', Next: 'Reusable2/Left Branch' },
                                            { Variable: '$.branch', StringEquals: 'right', Next: 'Reusable2/Right Branch' },
                                        ],
                                    },
                                    'Reusable2/Left Branch': { Type: 'Pass', End: true },
                                    'Reusable2/Right Branch': { Type: 'Pass', End: true },
                                },
                            },
                        ],
                    },
                },
            });
        }),
        test('State Machine Fragments can be wrapped in a single state', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const reusable = new SimpleChain(stack, 'Hello');
            const state = reusable.toSingleState();
            expect(render(state)).toStrictEqual({
                StartAt: 'Hello',
                States: {
                    Hello: {
                        Type: 'Parallel',
                        End: true,
                        Branches: [
                            {
                                StartAt: 'Hello: Task1',
                                States: {
                                    'Hello: Task1': { Type: 'Task', Next: 'Hello: Task2', Resource: 'resource' },
                                    'Hello: Task2': { Type: 'Task', End: true, Resource: 'resource' },
                                },
                            },
                        ],
                    },
                },
            });
        }),
        test('Chaining onto branched failure state ignores failure state', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const yes = new stepfunctions.Pass(stack, 'Yes');
            const no = new stepfunctions.Fail(stack, 'No', { error: 'Failure', cause: 'Wrong branch' });
            const enfin = new stepfunctions.Pass(stack, 'Finally');
            const choice = new stepfunctions.Choice(stack, 'Choice')
                .when(stepfunctions.Condition.stringEquals('$.foo', 'bar'), yes)
                .otherwise(no);
            // WHEN
            choice.afterwards().next(enfin);
            // THEN
            expect(render(choice)).toStrictEqual({
                StartAt: 'Choice',
                States: {
                    Choice: {
                        Type: 'Choice',
                        Choices: [
                            { Variable: '$.foo', StringEquals: 'bar', Next: 'Yes' },
                        ],
                        Default: 'No',
                    },
                    Yes: { Type: 'Pass', Next: 'Finally' },
                    No: { Type: 'Fail', Error: 'Failure', Cause: 'Wrong branch' },
                    Finally: { Type: 'Pass', End: true },
                },
            });
        }),
        test('Can include OTHERWISE transition for Choice in afterwards()', () => {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const chain = new stepfunctions.Choice(stack, 'Choice')
                .when(stepfunctions.Condition.stringEquals('$.foo', 'bar'), new stepfunctions.Pass(stack, 'Yes'))
                .afterwards({ includeOtherwise: true })
                .next(new stepfunctions.Pass(stack, 'Finally'));
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'Choice',
                States: {
                    Choice: {
                        Type: 'Choice',
                        Choices: [
                            { Variable: '$.foo', StringEquals: 'bar', Next: 'Yes' },
                        ],
                        Default: 'Finally',
                    },
                    Yes: { Type: 'Pass', Next: 'Finally' },
                    Finally: { Type: 'Pass', End: true },
                },
            });
        }),
        test('State machines can have unconstrainted gotos', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const one = new stepfunctions.Pass(stack, 'One');
            const two = new stepfunctions.Pass(stack, 'Two');
            // WHEN
            const chain = one.next(two).next(one);
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'One',
                States: {
                    One: { Type: 'Pass', Next: 'Two' },
                    Two: { Type: 'Pass', Next: 'One' },
                },
            });
        }),
        test('States can have error branches', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const failure = new stepfunctions.Fail(stack, 'Failed', { error: 'DidNotWork', cause: 'We got stuck' });
            // WHEN
            const chain = task1.addCatch(failure);
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'Task1',
                States: {
                    Task1: {
                        Type: 'Task',
                        Resource: 'resource',
                        End: true,
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'Failed' },
                        ],
                    },
                    Failed: {
                        Type: 'Fail',
                        Error: 'DidNotWork',
                        Cause: 'We got stuck',
                    },
                },
            });
        }),
        test('Retries and errors with a result path', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const failure = new stepfunctions.Fail(stack, 'Failed', { error: 'DidNotWork', cause: 'We got stuck' });
            // WHEN
            const chain = task1.addRetry({ errors: ['HTTPError'], maxAttempts: 2 }).addCatch(failure, { resultPath: '$.some_error' }).next(failure);
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'Task1',
                States: {
                    Task1: {
                        Type: 'Task',
                        Resource: 'resource',
                        Catch: [{ ErrorEquals: ['States.ALL'], Next: 'Failed', ResultPath: '$.some_error' }],
                        Retry: [{ ErrorEquals: ['HTTPError'], MaxAttempts: 2 }],
                        Next: 'Failed',
                    },
                    Failed: {
                        Type: 'Fail',
                        Error: 'DidNotWork',
                        Cause: 'We got stuck',
                    },
                },
            });
        }),
        test('Can wrap chain and attach error handler', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const errorHandler = new stepfunctions.Pass(stack, 'ErrorHandler');
            // WHEN
            const chain = task1.next(task2).toSingleState('Wrapped').addCatch(errorHandler);
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'Wrapped',
                States: {
                    Wrapped: {
                        Type: 'Parallel',
                        Branches: [
                            {
                                StartAt: 'Task1',
                                States: {
                                    Task1: {
                                        Type: 'Task',
                                        Resource: 'resource',
                                        Next: 'Task2',
                                    },
                                    Task2: {
                                        Type: 'Task',
                                        Resource: 'resource',
                                        End: true,
                                    },
                                },
                            },
                        ],
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'ErrorHandler' },
                        ],
                        End: true,
                    },
                    ErrorHandler: { Type: 'Pass', End: true },
                },
            });
        }),
        test('Chaining does not chain onto error handler state', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const errorHandler = new stepfunctions.Pass(stack, 'ErrorHandler');
            // WHEN
            const chain = task1.addCatch(errorHandler).next(task2);
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'Task1',
                States: {
                    Task1: {
                        Type: 'Task',
                        Resource: 'resource',
                        Next: 'Task2',
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'ErrorHandler' },
                        ],
                    },
                    Task2: { Type: 'Task', Resource: 'resource', End: true },
                    ErrorHandler: { Type: 'Pass', End: true },
                },
            });
        }),
        test('Chaining does not chain onto error handler, extended', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const task3 = new stepfunctions.Task(stack, 'Task3', { task: new FakeTask() });
            const errorHandler = new stepfunctions.Pass(stack, 'ErrorHandler');
            // WHEN
            const chain = task1.addCatch(errorHandler)
                .next(task2.addCatch(errorHandler))
                .next(task3.addCatch(errorHandler));
            // THEN
            const sharedTaskProps = { Type: 'Task', Resource: 'resource', Catch: [{ ErrorEquals: ['States.ALL'], Next: 'ErrorHandler' }] };
            expect(render(chain)).toStrictEqual({
                StartAt: 'Task1',
                States: {
                    Task1: { Next: 'Task2', ...sharedTaskProps },
                    Task2: { Next: 'Task3', ...sharedTaskProps },
                    Task3: { End: true, ...sharedTaskProps },
                    ErrorHandler: { Type: 'Pass', End: true },
                },
            });
        }),
        test('Error handler with a fragment', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const errorHandler = new stepfunctions.Pass(stack, 'ErrorHandler');
            // WHEN
            task1.addCatch(errorHandler)
                .next(new SimpleChain(stack, 'Chain').catch(errorHandler))
                .next(task2.addCatch(errorHandler));
        }),
        test('Can merge state machines with shared states', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const failure = new stepfunctions.Fail(stack, 'Failed', { error: 'DidNotWork', cause: 'We got stuck' });
            // WHEN
            task1.addCatch(failure);
            task2.addCatch(failure);
            task1.next(task2);
            // THEN
            expect(render(task1)).toStrictEqual({
                StartAt: 'Task1',
                States: {
                    Task1: {
                        Type: 'Task',
                        Resource: 'resource',
                        Next: 'Task2',
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'Failed' },
                        ],
                    },
                    Task2: {
                        Type: 'Task',
                        Resource: 'resource',
                        End: true,
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'Failed' },
                        ],
                    },
                    Failed: {
                        Type: 'Fail',
                        Error: 'DidNotWork',
                        Cause: 'We got stuck',
                    },
                },
            });
        }),
        test('No duplicate state IDs', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const intermediateParent = new cdk.Construct(stack, 'Parent');
            const state1 = new stepfunctions.Pass(stack, 'State');
            const state2 = new stepfunctions.Pass(intermediateParent, 'State');
            state1.next(state2);
            // WHEN
            expect(() => render(state1)).toThrow();
        }),
        test('No duplicate state IDs even across Parallel branches', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const intermediateParent = new cdk.Construct(stack, 'Parent');
            const state1 = new stepfunctions.Pass(stack, 'State');
            const state2 = new stepfunctions.Pass(intermediateParent, 'State');
            const parallel = new stepfunctions.Parallel(stack, 'Parallel')
                .branch(state1)
                .branch(state2);
            // WHEN
            expect(() => render(parallel)).toThrow();
        }),
        test('No cross-parallel jumps', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const state1 = new stepfunctions.Pass(stack, 'State1');
            const state2 = new stepfunctions.Pass(stack, 'State2');
            expect(() => new stepfunctions.Parallel(stack, 'Parallel')
                .branch(state1.next(state2))
                .branch(state2)).toThrow();
        }),
        describe('findReachableStates', () => {
            test('Can retrieve possible states from initial state', () => {
                // GIVEN
                const stack = new cdk.Stack();
                const state1 = new stepfunctions.Pass(stack, 'State1');
                const state2 = new stepfunctions.Pass(stack, 'State2');
                const state3 = new stepfunctions.Pass(stack, 'State3');
                const definition = state1
                    .next(state2)
                    .next(state3);
                // WHEN
                const states = stepfunctions.State.findReachableStates(definition.startState);
                // THEN
                expect(state1.id).toStrictEqual(states[0].id);
                expect(state2.id).toStrictEqual(states[1].id);
                expect(state3.id).toStrictEqual(states[2].id);
            });
            test('Does not retrieve unreachable states', () => {
                // GIVEN
                const stack = new cdk.Stack();
                const state1 = new stepfunctions.Pass(stack, 'State1');
                const state2 = new stepfunctions.Pass(stack, 'State2');
                const state3 = new stepfunctions.Pass(stack, 'State3');
                state1.next(state2).next(state3);
                // WHEN
                const states = stepfunctions.State.findReachableStates(state2);
                // THEN
                expect(state2.id).toStrictEqual(states[0].id);
                expect(state3.id).toStrictEqual(states[1].id);
                expect(states.length).toStrictEqual(2);
            });
            test('Works with Choice and Parallel states', () => {
                // GIVEN
                const stack = new cdk.Stack();
                const state1 = new stepfunctions.Choice(stack, 'MainChoice');
                const stateCA = new stepfunctions.Pass(stack, 'StateA');
                const stateCB = new stepfunctions.Pass(stack, 'StateB');
                const statePA = new stepfunctions.Pass(stack, 'ParallelA');
                const statePB = new stepfunctions.Pass(stack, 'ParallelB');
                const state2 = new stepfunctions.Parallel(stack, 'RunParallel');
                const state3 = new stepfunctions.Pass(stack, 'FinalState');
                state2.branch(statePA);
                state2.branch(statePB);
                state1.when(stepfunctions.Condition.stringEquals('$.myInput', 'A'), stateCA);
                state1.when(stepfunctions.Condition.stringEquals('$.myInput', 'B'), stateCB);
                stateCA.next(state2);
                state2.next(state3);
                const definition = state1.otherwise(stateCA);
                // WHEN
                const statesFromStateCB = stepfunctions.State.findReachableStates(stateCB);
                const statesFromState1 = stepfunctions.State.findReachableStates(definition);
                // THEN
                const expectedFromState1 = [state1, stateCA, stateCB, state2, state3];
                for (let i = 0; i < expectedFromState1.length; i++) {
                    expect(statesFromState1[i].id).toStrictEqual(expectedFromState1[i].id);
                }
                expect(statesFromStateCB[0].id).toStrictEqual(stateCB.id);
            });
        });
});
class ReusableStateMachine extends stepfunctions.StateMachineFragment {
    constructor(scope, id) {
        super(scope, id);
        const choice = new stepfunctions.Choice(this, 'Choice')
            .when(stepfunctions.Condition.stringEquals('$.branch', 'left'), new stepfunctions.Pass(this, 'Left Branch'))
            .when(stepfunctions.Condition.stringEquals('$.branch', 'right'), new stepfunctions.Pass(this, 'Right Branch'));
        this.startState = choice;
        this.endStates = choice.afterwards().endStates;
    }
}
class SimpleChain extends stepfunctions.StateMachineFragment {
    constructor(scope, id) {
        super(scope, id);
        const task1 = new stepfunctions.Task(this, 'Task1', { task: new FakeTask() });
        this.task2 = new stepfunctions.Task(this, 'Task2', { task: new FakeTask() });
        task1.next(this.task2);
        this.startState = task1;
        this.endStates = [this.task2];
    }
    catch(state, props) {
        this.task2.addCatch(state, props);
        return this;
    }
}
function render(sm) {
    return new cdk.Stack().resolve(new stepfunctions.StateGraph(sm.startState, 'Test Graph').toGraphJson());
}
class FakeTask {
    bind(_task) {
        return {
            resourceArn: 'resource',
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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