# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['xrpl',
 'xrpl.account',
 'xrpl.asyncio',
 'xrpl.asyncio.account',
 'xrpl.asyncio.clients',
 'xrpl.asyncio.ledger',
 'xrpl.asyncio.transaction',
 'xrpl.asyncio.wallet',
 'xrpl.clients',
 'xrpl.core',
 'xrpl.core.addresscodec',
 'xrpl.core.binarycodec',
 'xrpl.core.binarycodec.binary_wrappers',
 'xrpl.core.binarycodec.definitions',
 'xrpl.core.binarycodec.types',
 'xrpl.core.keypairs',
 'xrpl.ledger',
 'xrpl.models',
 'xrpl.models.amounts',
 'xrpl.models.currencies',
 'xrpl.models.requests',
 'xrpl.models.transactions',
 'xrpl.models.transactions.pseudo_transactions',
 'xrpl.models.transactions.types',
 'xrpl.transaction',
 'xrpl.utils',
 'xrpl.utils.txn_parser',
 'xrpl.utils.txn_parser.utils',
 'xrpl.wallet']

package_data = \
{'': ['*']}

install_requires = \
['Deprecated>=1.2.13,<2.0.0',
 'ECPy>=1.2.5,<2.0.0',
 'base58>=2.1.0,<3.0.0',
 'httpx>=0.18.1,<0.24.0',
 'pycryptodome>=3.16.0,<4.0.0',
 'types-Deprecated>=1.2.9,<2.0.0',
 'typing-extensions>=4.2.0,<5.0.0']

extras_require = \
{':python_version >= "3.10" and python_version < "4.0"': ['websockets>=10.0,<11.0'],
 ':python_version >= "3.7" and python_version < "3.10"': ['websockets>=9.0.1,<11.0']}

setup_kwargs = {
    'name': 'xrpl-py',
    'version': '1.8.0b1',
    'description': 'A complete Python library for interacting with the XRP ledger',
    'long_description': '[![Documentation Status](https://readthedocs.org/projects/xrpl-py/badge)](https://xrpl-py.readthedocs.io/)\n\n# xrpl-py\n\nA pure Python implementation for interacting with the XRP Ledger, the `xrpl-py` library simplifies the hardest parts of XRP Ledger interaction, like serialization and transaction signing, by providing native Python methods and models for [XRP Ledger transactions](https://xrpl.org/transaction-formats.html) and core server [API](https://xrpl.org/api-conventions.html) ([`rippled`](https://github.com/ripple/rippled)) objects.\n\n\n\n```py\n# create a network client\nfrom xrpl.clients import JsonRpcClient\nclient = JsonRpcClient("https://s.altnet.rippletest.net:51234")\n\n# create a wallet on the testnet\nfrom xrpl.wallet import generate_faucet_wallet\ntest_wallet = generate_faucet_wallet(client)\nprint(test_wallet)\npublic_key: ED3CC1BBD0952A60088E89FA502921895FC81FBD79CAE9109A8FE2D23659AD5D56\nprivate_key: -HIDDEN-\nclassic_address: rBtXmAdEYcno9LWRnAGfT9qBxCeDvuVRZo\n\n# look up account info\nfrom xrpl.models.requests.account_info import AccountInfo\nacct_info = AccountInfo(\n    account="rBtXmAdEYcno9LWRnAGfT9qBxCeDvuVRZo",\n    ledger_index="current",\n    queue=True,\n    strict=True,\n)\nresponse = client.request(acct_info)\nresult = response.result\nimport json\nprint(json.dumps(result["account_data"], indent=4, sort_keys=True))\n# {\n#     "Account": "rBtXmAdEYcno9LWRnAGfT9qBxCeDvuVRZo",\n#     "Balance": "1000000000",\n#     "Flags": 0,\n#     "LedgerEntryType": "AccountRoot",\n#     "OwnerCount": 0,\n#     "PreviousTxnID": "73CD4A37537A992270AAC8472F6681F44E400CBDE04EC8983C34B519F56AB107",\n#     "PreviousTxnLgrSeq": 16233962,\n#     "Sequence": 16233962,\n#     "index": "FD66EC588B52712DCE74831DCB08B24157DC3198C29A0116AA64D310A58512D7"\n# }\n```\n\n[![Downloads](https://pepy.tech/badge/xrpl-py/month)](https://pepy.tech/project/xrpl-py/month)\n[![Contributors](https://img.shields.io/github/contributors/xpring-eng/xrpl-py.svg)](https://github.com/xpring-eng/xrpl-py/graphs/contributors)\n\n## Installation and supported versions\n\nThe `xrpl-py` library is available on [PyPI](https://pypi.org/). Install with `pip`:\n\n\n```\npip3 install xrpl-py\n```\n\nThe library supports [Python 3.7](https://www.python.org/downloads/) and later.\n\n[![Supported Versions](https://img.shields.io/pypi/pyversions/xrpl-py.svg)](https://pypi.org/project/xrpl-py)\n\n\n## Features\n\nUse `xrpl-py` to build Python applications that leverage the [XRP Ledger](https://xrpl.org/). The library helps with all aspects of interacting with the XRP Ledger, including:\n\n* Key and wallet management\n* Serialization\n* Transaction Signing\n\n`xrpl-py` also provides:\n\n* A network client — See [`xrpl.clients`](https://xrpl-py.readthedocs.io/en/stable/source/xrpl.clients.html) for more information.\n* Methods for inspecting accounts — See [XRPL Account Methods](https://xrpl-py.readthedocs.io/en/stable/source/xrpl.account.html) for more information.\n* Codecs for encoding and decoding addresses and other objects — See [Core Codecs](https://xrpl-py.readthedocs.io/en/stable/source/xrpl.core.html) for more information.\n\n## [➡️ Reference Documentation](https://xrpl-py.readthedocs.io/en/stable/)\n\nSee the complete [`xrpl-py` reference documentation on Read the Docs](https://xrpl-py.readthedocs.io/en/stable/index.html).\n\n\n## Usage\n\nThe following sections describe some of the most commonly used modules in the `xrpl-py` library and provide sample code.\n\n### Network client\n\nUse the `xrpl.clients` library to create a network client for connecting to the XRP Ledger.\n\n```py\nfrom xrpl.clients import JsonRpcClient\nJSON_RPC_URL = "https://s.altnet.rippletest.net:51234"\nclient = JsonRpcClient(JSON_RPC_URL)\n```\n\n### Manage keys and wallets\n\n#### `xrpl.wallet`\n\nUse the [`xrpl.wallet`](https://xrpl-py.readthedocs.io/en/stable/source/xrpl.wallet.html) module to create a wallet from a given seed or or via a [Testnet faucet](https://xrpl.org/xrp-testnet-faucet.html).\n\nTo create a wallet from a seed (in this case, the value generated using [`xrpl.keypairs`](#xrpl-keypairs)):\n\n```py\nwallet_from_seed = xrpl.wallet.Wallet(seed, 0)\nprint(wallet_from_seed)\n# pub_key: ED46949E414A3D6D758D347BAEC9340DC78F7397FEE893132AAF5D56E4D7DE77B0\n# priv_key: -HIDDEN-\n# classic_address: rG5ZvYsK5BPi9f1Nb8mhFGDTNMJhEhufn6\n```\n\nTo create a wallet from a Testnet faucet:\n\n```py\ntest_wallet = generate_faucet_wallet(client)\ntest_account = test_wallet.classic_address\nprint("Classic address:", test_account)\n# Classic address: rEQB2hhp3rg7sHj6L8YyR4GG47Cb7pfcuw\n```\n\n#### `xrpl.core.keypairs`\n\nUse the [`xrpl.core.keypairs`](https://xrpl-py.readthedocs.io/en/stable/source/xrpl.core.keypairs.html#module-xrpl.core.keypairs) module to generate seeds and derive keypairs and addresses from those seed values.\n\nHere\'s an example of how to generate a `seed` value and derive an [XRP Ledger "classic" address](https://xrpl.org/cryptographic-keys.html#account-id-and-address) from that seed.\n\n\n```py\nfrom xrpl.core import keypairs\nseed = keypairs.generate_seed()\npublic, private = keypairs.derive_keypair(seed)\ntest_account = keypairs.derive_classic_address(public)\nprint("Here\'s the public key:")\nprint(public)\nprint("Here\'s the private key:")\nprint(private)\nprint("Store this in a secure place!")\n# Here\'s the public key:\n# ED3CC1BBD0952A60088E89FA502921895FC81FBD79CAE9109A8FE2D23659AD5D56\n# Here\'s the private key:\n# EDE65EE7882847EF5345A43BFB8E6F5EEC60F45461696C384639B99B26AAA7A5CD\n# Store this in a secure place!\n```\n\n**Note:** You can use `xrpl.core.keypairs.sign` to sign transactions but `xrpl-py` also provides explicit methods for safely signing and submitting transactions. See [Transaction Signing](#transaction-signing) and [XRPL Transaction Methods](https://xrpl-py.readthedocs.io/en/stable/source/xrpl.transaction.html#module-xrpl.transaction) for more information.\n\n\n### Serialize and sign transactions\n\nTo securely submit transactions to the XRP Ledger, you need to first serialize data from JSON and other formats into the [XRP Ledger\'s canonical format](https://xrpl.org/serialization.html), then to [authorize the transaction](https://xrpl.org/transaction-basics.html#authorizing-transactions) by digitally [signing it](https://xrpl-py.readthedocs.io/en/stable/source/xrpl.core.keypairs.html?highlight=sign#xrpl.core.keypairs.sign) with the account\'s private key. The `xrpl-py` library provides several methods to simplify this process.\n\n\nUse the [`xrpl.transaction`](https://xrpl-py.readthedocs.io/en/stable/source/xrpl.transaction.html) module to sign and submit transactions. The module offers three ways to do this:\n\n* [`sign_and_submit`](https://xrpl-py.readthedocs.io/en/stable/source/xrpl.transaction.html#xrpl.transaction.sign_and_submit) — Signs a transaction locally, then submits it to the XRP Ledger. This method does not implement [reliable transaction submission](https://xrpl.org/reliable-transaction-submission.html#reliable-transaction-submission) best practices, so only use it for development or testing purposes.\n\n* [`sign`](https://xrpl-py.readthedocs.io/en/stable/source/xrpl.transaction.html#xrpl.transaction.sign) — Signs a transaction locally. This method **does  not** submit the transaction to the XRP Ledger.\n\n* [`send_reliable_submission`](https://xrpl-py.readthedocs.io/en/stable/source/xrpl.transaction.html#xrpl.transaction.send_reliable_submission) — An implementation of the [reliable transaction submission guidelines](https://xrpl.org/reliable-transaction-submission.html#reliable-transaction-submission), this method submits a signed transaction to the XRP Ledger and then verifies that it has been included in a validated ledger (or has failed to do so). Use this method to submit transactions for production purposes.\n\n\n```py\nfrom xrpl.models.transactions import Payment\nfrom xrpl.transaction import sign, send_reliable_submission\nfrom xrpl.ledger import get_latest_validated_ledger_sequence\nfrom xrpl.account import get_next_valid_seq_number\n\ncurrent_validated_ledger = get_latest_validated_ledger_sequence(client)\ntest_wallet.sequence = get_next_valid_seq_number(test_wallet.classic_address, client)\n\n# prepare the transaction\n# the amount is expressed in drops, not XRP\n# see https://xrpl.org/basic-data-types.html#specifying-currency-amounts\nmy_tx_payment = Payment(\n    account=test_wallet.classic_address,\n    amount="2200000",\n    destination="rPT1Sjq2YGrBMTttX4GZHjKu9dyfzbpAYe",\n    last_ledger_sequence=current_validated_ledger + 20,\n    sequence=test_wallet.sequence,\n    fee="10",\n)\n# sign the transaction\nmy_tx_payment_signed = sign(my_tx_payment,test_wallet)\n\n# submit the transaction\ntx_response = send_reliable_submission(my_tx_payment_signed, client)\n```\n\n#### Get fee from the XRP Ledger\n\n\nIn most cases, you can specify the minimum [transaction cost](https://xrpl.org/transaction-cost.html#current-transaction-cost) of `"10"` for the `fee` field unless you have a strong reason not to. But if you want to get the [current load-balanced transaction cost](https://xrpl.org/transaction-cost.html#current-transaction-cost) from the network, you can use the `get_fee` function:\n\n```py\nfrom xrpl.ledger import get_fee\nfee = get_fee(client)\nprint(fee)\n# 10\n```\n\n#### Auto-filled fields\n\nThe `xrpl-py` library automatically populates the `fee`, `sequence` and `last_ledger_sequence` fields when you create transactions. In the example above, you could omit those fields and let the library fill them in for you.\n\n```py\nfrom xrpl.models.transactions import Payment\nfrom xrpl.transaction import send_reliable_submission, autofill_and_sign\n# prepare the transaction\n# the amount is expressed in drops, not XRP\n# see https://xrpl.org/basic-data-types.html#specifying-currency-amounts\nmy_tx_payment = Payment(\n    account=test_wallet.classic_address,\n    amount="2200000",\n    destination="rPT1Sjq2YGrBMTttX4GZHjKu9dyfzbpAYe"\n)\n\n# sign the transaction with the autofill method\n# (this will auto-populate the fee, sequence, and last_ledger_sequence)\nmy_tx_payment_signed = autofill_and_sign(my_tx_payment, test_wallet, client)\nprint(my_tx_payment_signed)\n# Payment(\n#     account=\'rMPUKmzmDWEX1tQhzQ8oGFNfAEhnWNFwz\',\n#     transaction_type=<TransactionType.PAYMENT: \'Payment\'>,\n#     fee=\'10\',\n#     sequence=16034065,\n#     account_txn_id=None,\n#     flags=0,\n#     last_ledger_sequence=10268600,\n#     memos=None,\n#     signers=None,\n#     source_tag=None,\n#     signing_pub_key=\'EDD9540FA398915F0BCBD6E65579C03BE5424836CB68B7EB1D6573F2382156B444\',\n#     txn_signature=\'938FB22AE7FE76CF26FD11F8F97668E175DFAABD2977BCA397233117E7E1C4A1E39681091CC4D6DF21403682803AB54CC21DC4FA2F6848811DEE10FFEF74D809\',\n#     amount=\'2200000\',\n#     destination=\'rPT1Sjq2YGrBMTttX4GZHjKu9dyfzbpAYe\',\n#     destination_tag=None,\n#     invoice_id=None,\n#     paths=None,\n#     send_max=None,\n#     deliver_min=None\n# )\n\n# submit the transaction\ntx_response = send_reliable_submission(my_tx_payment_signed, client)\n```\n\n\n### Subscribe to ledger updates\n\nYou can send `subscribe` and `unsubscribe` requests only using the WebSocket network client. These request methods allow you to be alerted of certain situations as they occur, such as when a new ledger is declared.\n\n```py\nfrom xrpl.clients import WebsocketClient\nurl = "wss://s.altnet.rippletest.net/"\nfrom xrpl.models.requests import Subscribe, StreamParameter\nreq = Subscribe(streams=[StreamParameter.LEDGER])\n# NOTE: this code will run forever without a timeout, until the process is killed\nwith WebsocketClient(url) as client:\n    client.send(req)\n    for message in client:\n        print(message)\n# {\'result\': {\'fee_base\': 10, \'fee_ref\': 10, \'ledger_hash\': \'7CD50477F23FF158B430772D8E82A961376A7B40E13C695AA849811EDF66C5C0\', \'ledger_index\': 18183504, \'ledger_time\': 676412962, \'reserve_base\': 20000000, \'reserve_inc\': 5000000, \'validated_ledgers\': \'17469391-18183504\'}, \'status\': \'success\', \'type\': \'response\'}\n# {\'fee_base\': 10, \'fee_ref\': 10, \'ledger_hash\': \'BAA743DABD168BD434804416C8087B7BDEF7E6D7EAD412B9102281DD83B10D00\', \'ledger_index\': 18183505, \'ledger_time\': 676412970, \'reserve_base\': 20000000, \'reserve_inc\': 5000000, \'txn_count\': 0, \'type\': \'ledgerClosed\', \'validated_ledgers\': \'17469391-18183505\'}\n# {\'fee_base\': 10, \'fee_ref\': 10, \'ledger_hash\': \'D8227DAF8F745AE3F907B251D40B4081E019D013ABC23B68C0B1431DBADA1A46\', \'ledger_index\': 18183506, \'ledger_time\': 676412971, \'reserve_base\': 20000000, \'reserve_inc\': 5000000, \'txn_count\': 0, \'type\': \'ledgerClosed\', \'validated_ledgers\': \'17469391-18183506\'}\n# {\'fee_base\': 10, \'fee_ref\': 10, \'ledger_hash\': \'CFC412B6DDB9A402662832A781C23F0F2E842EAE6CFC539FEEB287318092C0DE\', \'ledger_index\': 18183507, \'ledger_time\': 676412972, \'reserve_base\': 20000000, \'reserve_inc\': 5000000, \'txn_count\': 0, \'type\': \'ledgerClosed\', \'validated_ledgers\': \'17469391-18183507\'}\n```\n\n\n### Asynchronous Code\n\nThis library supports Python\'s [`asyncio`](https://docs.python.org/3/library/asyncio.html) package, which is used to run asynchronous code. All the async code is in [`xrpl.asyncio`](https://xrpl-py.readthedocs.io/en/stable/source/xrpl.asyncio.html) If you are writing asynchronous code, please note that you will not be able to use any synchronous sugar functions, due to how event loops are handled. However, every synchronous method has a corresponding asynchronous method that you can use.\n\nThis sample code is the asynchronous equivalent of the above section on submitting a transaction.\n\n```py\nimport asyncio\nfrom xrpl.models.transactions import Payment\nfrom xrpl.asyncio.transaction import sign, send_reliable_submission\nfrom xrpl.asyncio.ledger import get_latest_validated_ledger_sequence\nfrom xrpl.asyncio.account import get_next_valid_seq_number\nfrom xrpl.asyncio.clients import AsyncJsonRpcClient\n\nasync_client = AsyncJsonRpcClient(JSON_RPC_URL)\n\nasync def submit_sample_transaction():\n    current_validated_ledger = await get_latest_validated_ledger_sequence(async_client)\n    test_wallet.sequence = await get_next_valid_seq_number(test_wallet.classic_address, async_client)\n\n    # prepare the transaction\n    # the amount is expressed in drops, not XRP\n    # see https://xrpl.org/basic-data-types.html#specifying-currency-amounts\n    my_tx_payment = Payment(\n        account=test_wallet.classic_address,\n        amount="2200000",\n        destination="rPT1Sjq2YGrBMTttX4GZHjKu9dyfzbpAYe",\n        last_ledger_sequence=current_validated_ledger + 20,\n        sequence=test_wallet.sequence,\n        fee="10",\n    )\n    # sign the transaction\n    my_tx_payment_signed = await sign(my_tx_payment,test_wallet)\n\n    # submit the transaction\n    tx_response = await send_reliable_submission(my_tx_payment_signed, async_client)\n\nasyncio.run(submit_sample_transaction())\n```\n\n### Encode addresses\n\nUse [`xrpl.core.addresscodec`](https://xrpl-py.readthedocs.io/en/stable/source/xrpl.core.addresscodec.html) to encode and decode addresses into and from the ["classic" and X-address formats](https://xrpl.org/accounts.html#addresses).\n\n```py\n# convert classic address to x-address\nfrom xrpl.core import addresscodec\ntestnet_xaddress = (\n    addresscodec.classic_address_to_xaddress(\n        "rMPUKmzmDWEX1tQhzQ8oGFNfAEhnWNFwz",\n        tag=0,\n        is_test_network=True,\n    )\n)\nprint(testnet_xaddress)\n# T7QDemmxnuN7a52A62nx2fxGPWcRahLCf3qaswfrsNW9Lps\n```\n\n\n## Contributing\n\nIf you want to contribute to this project, see [CONTRIBUTING.md].\n\n### Mailing Lists\n\nWe have a low-traffic mailing list for announcements of new `xrpl-py` releases. (About 1 email per week)\n\n+ [Subscribe to xrpl-announce](https://groups.google.com/g/xrpl-announce)\n\nIf you\'re using the XRP Ledger in production, you should run a [rippled server](https://github.com/ripple/rippled) and subscribe to the ripple-server mailing list as well.\n\n+ [Subscribe to ripple-server](https://groups.google.com/g/ripple-server)\n\n### Code Samples\n- For samples of common use cases, see the [XRPL.org Code Samples](https://xrpl.org/code-samples.html) page.\n- You can also browse those samples [directly on GitHub](https://github.com/XRPLF/xrpl-dev-portal/tree/master/content/_code-samples).\n\n### Report an issue\n\nExperienced an issue? Report it [here](https://github.com/XRPLF/xrpl-py/issues/new).\n\n## License\n\nThe `xrpl-py` library is licensed under the ISC License. See [LICENSE] for more information.\n\n\n\n[CONTRIBUTING.md]: CONTRIBUTING.md\n[LICENSE]: LICENSE\n',
    'author': 'Mayukha Vadari',
    'author_email': 'mvadari@ripple.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/XRPLF/xrpl-py',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
