"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const have_resource_1 = require("@aws-cdk/assert/lib/assertions/have-resource");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('User Pool', () => {
    test('default setup', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool');
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPool', {
            AdminCreateUserConfig: {
                AllowAdminCreateUserOnly: true,
                InviteMessageTemplate: have_resource_1.ABSENT
            },
            EmailVerificationMessage: 'Hello {username}, Your verification code is {####}',
            EmailVerificationSubject: 'Verify your new account',
            SmsVerificationMessage: 'The verification code to your new account is {####}',
            VerificationMessageTemplate: {
                DefaultEmailOption: 'CONFIRM_WITH_CODE',
                EmailMessage: 'Hello {username}, Your verification code is {####}',
                EmailSubject: 'Verify your new account',
                SmsMessage: 'The verification code to your new account is {####}',
            },
            SmsConfiguration: {
                SnsCallerArn: {
                    'Fn::GetAtt': ['PoolsmsRoleC3352CE6', 'Arn'],
                },
                ExternalId: 'Pool'
            }
        });
        expect(stack).toHaveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: 'sts:AssumeRole',
                        Condition: {
                            StringEquals: {
                                'sts:ExternalId': 'Pool'
                            }
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: 'cognito-idp.amazonaws.com'
                        }
                    }
                ],
                Version: '2012-10-17'
            },
            Policies: [
                {
                    PolicyDocument: {
                        Statement: [
                            {
                                Action: 'sns:Publish',
                                Effect: 'Allow',
                                Resource: '*'
                            }
                        ],
                        Version: '2012-10-17'
                    },
                    PolicyName: 'sns-publish'
                }
            ]
        });
    });
    test('self sign up option is correctly configured', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            selfSignUpEnabled: true
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPool', {
            AdminCreateUserConfig: {
                AllowAdminCreateUserOnly: false
            }
        });
    });
    test('email verification via link is configured correctly', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            userVerification: {
                emailStyle: lib_1.VerificationEmailStyle.LINK
            }
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            EmailVerificationMessage: 'Hello {username}, Your verification code is {####}',
            EmailVerificationSubject: 'Verify your new account',
            VerificationMessageTemplate: {
                DefaultEmailOption: 'CONFIRM_WITH_LINK',
                EmailMessageByLink: 'Hello {username}, Your verification code is {####}',
                EmailSubjectByLink: 'Verify your new account',
            }
        });
    }),
        test('user invitation messages are configured correctly', () => {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.UserPool(stack, 'Pool', {
                userInvitation: {
                    emailBody: 'invitation email body',
                    emailSubject: 'invitation email subject',
                    smsMessage: 'invitation sms'
                }
            });
            // THEN
            expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
                AdminCreateUserConfig: {
                    InviteMessageTemplate: {
                        EmailMessage: 'invitation email body',
                        EmailSubject: 'invitation email subject',
                        SMSMessage: 'invitation sms'
                    }
                }
            });
        });
    test('smsRole property is recognized', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const role = aws_iam_1.Role.fromRoleArn(stack, 'smsRole', 'arn:aws:iam::664773442901:role/sms-role');
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            smsRole: role,
            smsRoleExternalId: 'test-external-id'
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            SmsConfiguration: {
                ExternalId: 'test-external-id',
                SnsCallerArn: role.roleArn
            }
        });
    });
    test('import using id', () => {
        // GIVEN
        const stack = new core_1.Stack(undefined, undefined, {
            env: { region: 'some-region-1', account: '0123456789012' }
        });
        const userPoolId = 'test-user-pool';
        // WHEN
        const pool = lib_1.UserPool.fromUserPoolId(stack, 'userpool', userPoolId);
        expect(pool.userPoolId).toEqual(userPoolId);
        expect(pool.userPoolArn).toMatch(/cognito-idp:some-region-1:0123456789012:userpool\/test-user-pool/);
    });
    test('import using arn', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const userPoolArn = 'arn:aws:cognito-idp:us-east-1:0123456789012:userpool/test-user-pool';
        // WHEN
        const pool = lib_1.UserPool.fromUserPoolArn(stack, 'userpool', userPoolArn);
        expect(pool.userPoolId).toEqual('test-user-pool');
        expect(pool.userPoolArn).toEqual(userPoolArn);
    });
    test('support tags', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const pool = new lib_1.UserPool(stack, 'Pool', {
            userPoolName: 'myPool',
        });
        core_1.Tag.add(pool, 'PoolTag', 'PoolParty');
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'myPool',
            UserPoolTags: {
                PoolTag: 'PoolParty',
            }
        });
    });
    test('lambda triggers are defined', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        const pool = new lib_1.UserPool(stack, 'Pool', {
            lambdaTriggers: {
                preSignUp: fn
            }
        });
        pool.addCustomMessageTrigger(fn);
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            LambdaConfig: {
                PreSignUp: stack.resolve(fn.functionArn),
                CustomMessage: stack.resolve(fn.functionArn)
            }
        });
    });
    test('on* API correctly appends triggers', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const createAuthChallengeLambdaFn = new lambda.Function(stack, 'createAuthChallengeLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const customMessageLambdaFn = new lambda.Function(stack, 'customMessageLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const defineAuthChallengeLambdaFn = new lambda.Function(stack, 'defineAuthChallengeLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const postAuthenticationLambdaFn = new lambda.Function(stack, 'postAuthenticationLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const postConfirmationLambdaFn = new lambda.Function(stack, 'postConfirmationLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const preAuthenticationLambdaFn = new lambda.Function(stack, 'preAuthenticationLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const preSignUpLambdaFn = new lambda.Function(stack, 'preSignUpLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const preTokenGenerationLambdaFn = new lambda.Function(stack, 'preTokenGenerationLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const userMigrationLambdaFn = new lambda.Function(stack, 'userMigrationLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const verifyAuthChallengeResponseLambdaFn = new lambda.Function(stack, 'verifyAuthChallengeResponseLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        const pool = new lib_1.UserPool(stack, 'Pool', {});
        pool.addCreateAuthChallengeTrigger(createAuthChallengeLambdaFn);
        pool.addCustomMessageTrigger(customMessageLambdaFn);
        pool.addDefineAuthChallengeTrigger(defineAuthChallengeLambdaFn);
        pool.addPostAuthenticationTrigger(postAuthenticationLambdaFn);
        pool.addPostConfirmationTrigger(postConfirmationLambdaFn);
        pool.addPreAuthenticationTrigger(preAuthenticationLambdaFn);
        pool.addPreSignUpTrigger(preSignUpLambdaFn);
        pool.addPreTokenGenerationTrigger(preTokenGenerationLambdaFn);
        pool.addUserMigrationTrigger(userMigrationLambdaFn);
        pool.addVerifyAuthChallengeResponseTrigger(verifyAuthChallengeResponseLambdaFn);
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            LambdaConfig: {
                CreateAuthChallenge: stack.resolve(createAuthChallengeLambdaFn.functionArn),
                CustomMessage: stack.resolve(customMessageLambdaFn.functionArn),
                DefineAuthChallenge: stack.resolve(defineAuthChallengeLambdaFn.functionArn),
                PostAuthentication: stack.resolve(postAuthenticationLambdaFn.functionArn),
                PostConfirmation: stack.resolve(postConfirmationLambdaFn.functionArn),
                PreAuthentication: stack.resolve(preAuthenticationLambdaFn.functionArn),
                PreSignUp: stack.resolve(preSignUpLambdaFn.functionArn),
                PreTokenGeneration: stack.resolve(preTokenGenerationLambdaFn.functionArn),
                UserMigration: stack.resolve(userMigrationLambdaFn.functionArn),
                VerifyAuthChallengeResponse: stack.resolve(verifyAuthChallengeResponseLambdaFn.functionArn)
            }
        });
    });
    test('lambdas are given cognito service grant', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            lambdaTriggers: {
                preSignUp: fn
            }
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Lambda::Permission', {
            FunctionName: stack.resolve(fn.functionArn),
            Principal: 'cognito-idp.amazonaws.com'
        });
    });
    test('no username aliases specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool');
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: have_resource_1.ABSENT,
            AliasAttributes: have_resource_1.ABSENT,
        });
    });
    test('fails when preferredUsername is used without username', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { preferredUsername: true }
        })).toThrow(/username/);
    });
    test('username and email are specified as the username aliases', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { username: true, email: true }
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: have_resource_1.ABSENT,
            AliasAttributes: ['email'],
        });
    });
    test('email and phone number are specified as the username aliases', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { email: true, phone: true }
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: ['email', 'phone_number'],
            AliasAttributes: have_resource_1.ABSENT,
        });
    });
    test('email and phone number are autoverified, by default, if they are specified as signIn', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
            signInAliases: { email: true }
        });
        new lib_1.UserPool(stack, 'Pool2', {
            userPoolName: 'Pool2',
            signInAliases: { email: true, phone: true }
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            AutoVerifiedAttributes: ['email'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool2',
            AutoVerifiedAttributes: ['email', 'phone_number'],
        });
    });
    test('explicit autoverify are correctly picked up', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { username: true },
            autoVerify: { email: true, phone: true },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            AutoVerifiedAttributes: ['email', 'phone_number'],
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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