"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const cognito_generated_1 = require("./cognito.generated");
/**
 * Standard attributes
 * Specified following the OpenID Connect spec
 * @see https://openid.net/specs/openid-connect-core-1_0.html#StandardClaims
 */
var UserPoolAttribute;
(function (UserPoolAttribute) {
    /**
     * End-User's preferred postal address.
     */
    UserPoolAttribute["ADDRESS"] = "address";
    /**
     * End-User's birthday, represented as an ISO 8601:2004 [ISO8601‑2004] YYYY-MM-DD format.
     * The year MAY be 0000, indicating that it is omitted.
     * To represent only the year, YYYY format is allowed.
     */
    UserPoolAttribute["BIRTHDATE"] = "birthdate";
    /**
     * End-User's preferred e-mail address.
     * Its value MUST conform to the RFC 5322 [RFC5322] addr-spec syntax.
     */
    UserPoolAttribute["EMAIL"] = "email";
    /**
     * Surname(s) or last name(s) of the End-User.
     * Note that in some cultures, people can have multiple family names or no family name;
     * all can be present, with the names being separated by space characters.
     */
    UserPoolAttribute["FAMILY_NAME"] = "family_name";
    /**
     * End-User's gender.
     */
    UserPoolAttribute["GENDER"] = "gender";
    /**
     * Given name(s) or first name(s) of the End-User.
     * Note that in some cultures, people can have multiple given names;
     * all can be present, with the names being separated by space characters.
     */
    UserPoolAttribute["GIVEN_NAME"] = "given_name";
    /**
     * End-User's locale, represented as a BCP47 [RFC5646] language tag.
     * This is typically an ISO 639-1 Alpha-2 [ISO639‑1] language code in lowercase
     * and an ISO 3166-1 Alpha-2 [ISO3166‑1] country code in uppercase, separated by a dash.
     * For example, en-US or fr-CA.
     */
    UserPoolAttribute["LOCALE"] = "locale";
    /**
     * Middle name(s) of the End-User.
     * Note that in some cultures, people can have multiple middle names;
     * all can be present, with the names being separated by space characters.
     * Also note that in some cultures, middle names are not used.
     */
    UserPoolAttribute["MIDDLE_NAME"] = "middle_name";
    /**
     * End-User's full name in displayable form including all name parts,
     * possibly including titles and suffixes, ordered according to the End-User's locale and preferences.
     */
    UserPoolAttribute["NAME"] = "name";
    /**
     * Casual name of the End-User that may or may not be the same as the given_name.
     * For instance, a nickname value of Mike might be returned alongside a given_name value of Michael.
     */
    UserPoolAttribute["NICKNAME"] = "nickname";
    /**
     * End-User's preferred telephone number.
     * E.164 [E.164] is RECOMMENDED as the format of this Claim, for example, +1 (425) 555-1212 or +56 (2) 687 2400.
     * If the phone number contains an extension, it is RECOMMENDED that the extension be represented using the
     * RFC 3966 [RFC3966] extension syntax, for example, +1 (604) 555-1234;ext=5678.
     */
    UserPoolAttribute["PHONE_NUMBER"] = "phone_number";
    /**
     * URL of the End-User's profile picture.
     * This URL MUST refer to an image file (for example, a PNG, JPEG, or GIF image file),
     * rather than to a Web page containing an image.
     * Note that this URL SHOULD specifically reference a profile photo of the End-User
     * suitable for displaying when describing the End-User, rather than an arbitrary photo taken by the End-User
     */
    UserPoolAttribute["PICTURE"] = "picture";
    /**
     * Shorthand name by which the End-User wishes to be referred to.
     */
    UserPoolAttribute["PREFERRED_USERNAME"] = "preferred_username";
    /**
     * URL of the End-User's profile page. The contents of this Web page SHOULD be about the End-User.
     */
    UserPoolAttribute["PROFILE"] = "profile";
    /**
     * The End-User's time zone
     */
    UserPoolAttribute["TIMEZONE"] = "zoneinfo";
    /**
     * Time the End-User's information was last updated.
     * Its value is a JSON number representing the number of seconds from 1970-01-01T0:0:0Z
     * as measured in UTC until the date/time.
     */
    UserPoolAttribute["UPDATED_AT"] = "updated_at";
    /**
     * URL of the End-User's Web page or blog.
     * This Web page SHOULD contain information published by the End-User or an organization that the End-User is affiliated with.
     */
    UserPoolAttribute["WEBSITE"] = "website";
})(UserPoolAttribute = exports.UserPoolAttribute || (exports.UserPoolAttribute = {}));
/**
 * The email verification style
 */
var VerificationEmailStyle;
(function (VerificationEmailStyle) {
    /** Verify email via code */
    VerificationEmailStyle["CODE"] = "CONFIRM_WITH_CODE";
    /** Verify email via link */
    VerificationEmailStyle["LINK"] = "CONFIRM_WITH_LINK";
})(VerificationEmailStyle = exports.VerificationEmailStyle || (exports.VerificationEmailStyle = {}));
/**
 * Define a Cognito User Pool
 */
class UserPool extends core_1.Resource {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l;
        super(scope, id);
        this.triggers = {};
        const signIn = this.signInConfiguration(props);
        if (props.lambdaTriggers) {
            for (const t of Object.keys(props.lambdaTriggers)) {
                const trigger = props.lambdaTriggers[t];
                if (trigger !== undefined) {
                    this.addLambdaPermission(trigger, t);
                    this.triggers[t] = trigger.functionArn;
                }
            }
        }
        const emailVerificationSubject = (_b = (_a = props.userVerification) === null || _a === void 0 ? void 0 : _a.emailSubject, (_b !== null && _b !== void 0 ? _b : 'Verify your new account'));
        const emailVerificationMessage = (_d = (_c = props.userVerification) === null || _c === void 0 ? void 0 : _c.emailBody, (_d !== null && _d !== void 0 ? _d : 'Hello {username}, Your verification code is {####}'));
        const smsVerificationMessage = (_f = (_e = props.userVerification) === null || _e === void 0 ? void 0 : _e.smsMessage, (_f !== null && _f !== void 0 ? _f : 'The verification code to your new account is {####}'));
        const defaultEmailOption = (_h = (_g = props.userVerification) === null || _g === void 0 ? void 0 : _g.emailStyle, (_h !== null && _h !== void 0 ? _h : VerificationEmailStyle.CODE));
        const verificationMessageTemplate = (defaultEmailOption === VerificationEmailStyle.CODE) ? {
            defaultEmailOption,
            emailMessage: emailVerificationMessage,
            emailSubject: emailVerificationSubject,
            smsMessage: smsVerificationMessage,
        } : {
            defaultEmailOption,
            emailMessageByLink: emailVerificationMessage,
            emailSubjectByLink: emailVerificationSubject,
            smsMessage: smsVerificationMessage
        };
        const inviteMessageTemplate = {
            emailMessage: (_j = props.userInvitation) === null || _j === void 0 ? void 0 : _j.emailBody,
            emailSubject: (_k = props.userInvitation) === null || _k === void 0 ? void 0 : _k.emailSubject,
            smsMessage: (_l = props.userInvitation) === null || _l === void 0 ? void 0 : _l.smsMessage,
        };
        const selfSignUpEnabled = props.selfSignUpEnabled !== undefined ? props.selfSignUpEnabled : false;
        const adminCreateUserConfig = {
            allowAdminCreateUserOnly: !selfSignUpEnabled,
            inviteMessageTemplate: props.userInvitation !== undefined ? inviteMessageTemplate : undefined,
        };
        const userPool = new cognito_generated_1.CfnUserPool(this, 'Resource', {
            userPoolName: props.userPoolName,
            usernameAttributes: signIn.usernameAttrs,
            aliasAttributes: signIn.aliasAttrs,
            autoVerifiedAttributes: signIn.autoVerifyAttrs,
            lambdaConfig: core_1.Lazy.anyValue({ produce: () => this.triggers }),
            smsConfiguration: this.smsConfiguration(props),
            adminCreateUserConfig,
            emailVerificationMessage,
            emailVerificationSubject,
            smsVerificationMessage,
            verificationMessageTemplate,
        });
        this.userPoolId = userPool.ref;
        this.userPoolArn = userPool.attrArn;
        this.userPoolProviderName = userPool.attrProviderName;
        this.userPoolProviderUrl = userPool.attrProviderUrl;
    }
    /**
     * Import an existing user pool based on its id.
     */
    static fromUserPoolId(scope, id, userPoolId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolId = userPoolId;
                this.userPoolArn = core_1.Stack.of(this).formatArn({
                    service: 'cognito-idp',
                    resource: 'userpool',
                    resourceName: userPoolId,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing user pool based on its ARN.
     */
    static fromUserPoolArn(scope, id, userPoolArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolArn = userPoolArn;
                this.userPoolId = core_1.Stack.of(this).parseArn(userPoolArn).resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Attach 'Create Auth Challenge' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-create-auth-challenge.html
     * @param fn the lambda function to attach
     */
    addCreateAuthChallengeTrigger(fn) {
        this.addLambdaPermission(fn, 'CreateAuthChallenge');
        this.triggers = { ...this.triggers, createAuthChallenge: fn.functionArn };
    }
    /**
     * Attach 'Custom Message' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-message.html
     * @param fn the lambda function to attach
     */
    addCustomMessageTrigger(fn) {
        this.addLambdaPermission(fn, 'CustomMessage');
        this.triggers = { ...this.triggers, customMessage: fn.functionArn };
    }
    /**
     * Attach 'Define Auth Challenge' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-define-auth-challenge.html
     * @param fn the lambda function to attach
     */
    addDefineAuthChallengeTrigger(fn) {
        this.addLambdaPermission(fn, 'DefineAuthChallenge');
        this.triggers = { ...this.triggers, defineAuthChallenge: fn.functionArn };
    }
    /**
     * Attach 'Post Authentication' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html
     * @param fn the lambda function to attach
     */
    addPostAuthenticationTrigger(fn) {
        this.addLambdaPermission(fn, 'PostAuthentication');
        this.triggers = { ...this.triggers, postAuthentication: fn.functionArn };
    }
    /**
     * Attach 'Post Confirmation' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html
     * @param fn the lambda function to attach
     */
    addPostConfirmationTrigger(fn) {
        this.addLambdaPermission(fn, 'PostConfirmation');
        this.triggers = { ...this.triggers, postConfirmation: fn.functionArn };
    }
    /**
     * Attach 'Pre Authentication' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html
     * @param fn the lambda function to attach
     */
    addPreAuthenticationTrigger(fn) {
        this.addLambdaPermission(fn, 'PreAuthentication');
        this.triggers = { ...this.triggers, preAuthentication: fn.functionArn };
    }
    /**
     * Attach 'Pre Sign Up' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html
     * @param fn the lambda function to attach
     */
    addPreSignUpTrigger(fn) {
        this.addLambdaPermission(fn, 'PreSignUp');
        this.triggers = { ...this.triggers, preSignUp: fn.functionArn };
    }
    /**
     * Attach 'Pre Token Generation' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html
     * @param fn the lambda function to attach
     */
    addPreTokenGenerationTrigger(fn) {
        this.addLambdaPermission(fn, 'PreTokenGeneration');
        this.triggers = { ...this.triggers, preTokenGeneration: fn.functionArn };
    }
    /**
     * Attach 'User Migration' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html
     * @param fn the lambda function to attach
     */
    addUserMigrationTrigger(fn) {
        this.addLambdaPermission(fn, 'UserMigration');
        this.triggers = { ...this.triggers, userMigration: fn.functionArn };
    }
    /**
     * Attach 'Verify Auth Challenge Response' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-verify-auth-challenge-response.html
     * @param fn the lambda function to attach
     */
    addVerifyAuthChallengeResponseTrigger(fn) {
        this.addLambdaPermission(fn, 'VerifyAuthChallengeResponse');
        this.triggers = { ...this.triggers, verifyAuthChallengeResponse: fn.functionArn };
    }
    addLambdaPermission(fn, name) {
        const normalize = name.charAt(0).toUpperCase() + name.slice(1);
        fn.addPermission(`${normalize}Cognito`, {
            principal: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com'),
            sourceArn: this.userPoolArn
        });
    }
    signInConfiguration(props) {
        var _a;
        let aliasAttrs;
        let usernameAttrs;
        let autoVerifyAttrs;
        const signIn = (_a = props.signInAliases, (_a !== null && _a !== void 0 ? _a : { username: true }));
        if (signIn.preferredUsername && !signIn.username) {
            throw new Error('username signIn must be enabled if preferredUsername is enabled');
        }
        if (signIn.username) {
            aliasAttrs = [];
            if (signIn.email) {
                aliasAttrs.push(UserPoolAttribute.EMAIL);
            }
            if (signIn.phone) {
                aliasAttrs.push(UserPoolAttribute.PHONE_NUMBER);
            }
            if (signIn.preferredUsername) {
                aliasAttrs.push(UserPoolAttribute.PREFERRED_USERNAME);
            }
            if (aliasAttrs.length === 0) {
                aliasAttrs = undefined;
            }
        }
        else {
            usernameAttrs = [];
            if (signIn.email) {
                usernameAttrs.push(UserPoolAttribute.EMAIL);
            }
            if (signIn.phone) {
                usernameAttrs.push(UserPoolAttribute.PHONE_NUMBER);
            }
        }
        if (props.autoVerify) {
            autoVerifyAttrs = [];
            if (props.autoVerify.email) {
                autoVerifyAttrs.push(UserPoolAttribute.EMAIL);
            }
            if (props.autoVerify.phone) {
                autoVerifyAttrs.push(UserPoolAttribute.PHONE_NUMBER);
            }
        }
        else if (signIn.email || signIn.phone) {
            autoVerifyAttrs = [];
            if (signIn.email) {
                autoVerifyAttrs.push(UserPoolAttribute.EMAIL);
            }
            if (signIn.phone) {
                autoVerifyAttrs.push(UserPoolAttribute.PHONE_NUMBER);
            }
        }
        return { usernameAttrs, aliasAttrs, autoVerifyAttrs };
    }
    smsConfiguration(props) {
        var _a;
        if (props.smsRole) {
            return {
                snsCallerArn: props.smsRole.roleArn,
                externalId: props.smsRoleExternalId
            };
        }
        else {
            const smsRoleExternalId = this.node.uniqueId.substr(0, 1223); // sts:ExternalId max length of 1224
            const smsRole = (_a = props.smsRole, (_a !== null && _a !== void 0 ? _a : new aws_iam_1.Role(this, 'smsRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com', {
                    conditions: {
                        StringEquals: { 'sts:ExternalId': smsRoleExternalId }
                    }
                }),
                inlinePolicies: {
                    /*
                     * The UserPool is very particular that it must contain an 'sns:Publish' action as an inline policy.
                     * Ideally, a conditional that restricts this action to 'sms' protocol needs to be attached, but the UserPool deployment fails validation.
                     * Seems like a case of being excessively strict.
                     */
                    'sns-publish': new aws_iam_1.PolicyDocument({
                        statements: [
                            new aws_iam_1.PolicyStatement({
                                actions: ['sns:Publish'],
                                resources: ['*'],
                            })
                        ]
                    })
                }
            })));
            return {
                externalId: smsRoleExternalId,
                snsCallerArn: smsRole.roleArn
            };
        }
    }
}
exports.UserPool = UserPool;
//# sourceMappingURL=data:application/json;base64,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