# -*- coding: utf-8 -*-
"""
Testing class for audit trail endpoints of the Castor EDC API Wrapper.
Link: https://data.castoredc.com/api#/country

@author: R.C.A. van Linschoten
https://orcid.org/0000-0003-3052-596X
"""
import pytest
from httpx import HTTPStatusError


class TestAuditTrail:
    def test_audit_trail_basic(self, client):
        """Test basic audit trail functionality"""
        audit_trail = client.audit_trail(date_from="2021-10-12", date_to="2021-10-12")
        assert audit_trail == self.test_audit_trail_model

    def test_audit_trail_fail(self, client):
        """Test audit trail fails correctly"""
        with pytest.raises(ValueError) as e:
            client.audit_trail(date_from="12-10-2021", date_to="12-10-2021")
        assert "does not match format" in str(e.value)

    def test_audit_trail_user(self, client):
        """Test audit trail functionality while filtering on user"""
        audit_trail = client.audit_trail(
            date_from="2021-10-12",
            date_to="2021-10-12",
            user_id="06A8C79E-F76F-4824-AB1A-93F0457C5A61",
        )
        assert audit_trail == self.test_audit_trail_user_model

    test_audit_trail_model = [
        {
            "datetime": {
                "date": "2021-10-12 09:08:29.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "143",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:08:33.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Diastolic blood pressure in mmHg"',
                "Field Value": "71",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:08:33.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Laboratory", Field: "Hematocrit value"',
                "Field Value": "23",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:08:33.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "207",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:08:55.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:08:55+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000218",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:08:55+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:08:56.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record randomized",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Progress": 0,
                "Randomized By": "castoredcapi.github@gmail.com",
                "Randomized Group Name": "Control",
                "Randomized Group Number": 1,
                "Randomized Id": "007",
                "Randomized On": "12-10-2021 09:08:56",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:08:56.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Randomization", Field: "Randomization allocation"',
                "Field Value": 1,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:08:59.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:08:59+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000219",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:08:59+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:09:09.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000003",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "8745EB23-CF9D-495C-B081-0EB24C68BD8F",
                "Report Name Custom": "89041199",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:09:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "6ED6EC83-218C-4610-8387-E82FADBA6D7B",
                "Report Name Custom": "68217351",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:09:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "D88F938D-BCC3-407B-835F-AD4C2DB92F53",
                "Report Name Custom": "17893202",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:09:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "5358BFA2-0742-4625-88C9-C50E74D12FAD",
                "Report Name Custom": "73314828",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:09:30.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Sub-survey of package invitation created",
            "user_id": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "user_name": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "user_email": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "event_details": {
                "Archived": 0,
                "Auto Lock On Finish": 0,
                "Createdby": "PythonAPI Github",
                "Createdon": "2021-10-12 09:09:30",
                "Finished On": None,
                "First Opened On": None,
                "Last Updated By": None,
                "Last Updated On": None,
                "Locked": 0,
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 0,
                "Record Id": "000001",
                "Sent On": None,
                "Survey Auto Send": 0,
                "Survey Id": "D70C1273-B5D8-45CD-BFE8-A0BA75C44B7E",
                "Survey Instance Id": "F42A35BE-4F7B-4C30-97A7-0D2B00CC21E7",
                "Survey Invitation": None,
                "Survey Invitation Subject": None,
                "Survey Package Instance Id": "BBAD255C-807C-4D43-B2A8-CAA732ED4BC6",
                "Survey Url String": None,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:09:32.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Survey package invitation unlocked",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Archived": False,
                "Auto Lock On Finish": False,
                "Auto Send": False,
                "Completion Time": "2020-08-12T16:53:00+0200",
                "Created By": "Reinier van Linschoten (R.linschoten@franciscus.nl)",
                "Created On": "2019-11-04T14:59:22+0100",
                "First Opened On": None,
                "Locked": False,
                "Package Invitation": 'Dear participant,\n\nYou are participating in the study "Example Study" and we would like to ask you to fill in a survey.\n\nPlease click the link below to complete our survey.\n\n{url}\n\n{logo}',
                "Package Invitation Subject": "Please fill in this survey for Example Study",
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 100,
                "Received Time": "2020-08-12T16:53:00+0200",
                "Record Id": "000002",
                "Sent On": None,
                "Start Time": None,
                "Status": "Created",
                "Survey Package Id": "71C01598-4682-4A4C-90E6-69C0BD38EA47",
                "Survey Package Instance Id": "23B4FD48-BA41-4C9B-BAEF-D5C3DD5F8E5C",
                "Survey Url String": "XWNSC9W5",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:09:32+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:36:26.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "128",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:36:29.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Diastolic blood pressure in mmHg"',
                "Field Value": "32",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:36:29.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Laboratory", Field: "Hematocrit value"',
                "Field Value": "27",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:36:29.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "79",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:36:51.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:36:51+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000220",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:36:51+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:36:51.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record randomized",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Progress": 0,
                "Randomized By": "castoredcapi.github@gmail.com",
                "Randomized Group Name": "Control",
                "Randomized Group Number": 1,
                "Randomized Id": "008",
                "Randomized On": "12-10-2021 09:36:51",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:36:51.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Randomization", Field: "Randomization allocation"',
                "Field Value": 1,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:36:57.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:36:57+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000221",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:36:57+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:37:08.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000003",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "7B61051A-A7C3-4976-B2CF-13FF43CE61E6",
                "Report Name Custom": "39626898",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:37:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "523C81AA-8CDB-42B9-9860-11F1283DDDCC",
                "Report Name Custom": "22329694",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:37:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "53C8FE42-E790-4876-856D-B5A6714E702D",
                "Report Name Custom": "23616042",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:37:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "6600CEFD-9441-4C55-877B-7CC95A9F0CEA",
                "Report Name Custom": "44612112",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:37:30.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Sub-survey of package invitation created",
            "user_id": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "user_name": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "user_email": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "event_details": {
                "Archived": 0,
                "Auto Lock On Finish": 0,
                "Createdby": "PythonAPI Github",
                "Createdon": "2021-10-12 09:37:30",
                "Finished On": None,
                "First Opened On": None,
                "Last Updated By": None,
                "Last Updated On": None,
                "Locked": 0,
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 0,
                "Record Id": "000001",
                "Sent On": None,
                "Survey Auto Send": 0,
                "Survey Id": "D70C1273-B5D8-45CD-BFE8-A0BA75C44B7E",
                "Survey Instance Id": "A634DE1E-FB4F-4549-9A4B-C66560B13442",
                "Survey Invitation": None,
                "Survey Invitation Subject": None,
                "Survey Package Instance Id": "6DD6AC03-BF1C-489C-A660-6CA55B4EDFA2",
                "Survey Url String": None,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:37:33.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Survey package invitation locked",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Archived": False,
                "Auto Lock On Finish": False,
                "Auto Send": False,
                "Completion Time": "2020-08-12T16:53:00+0200",
                "Created By": "Reinier van Linschoten (R.linschoten@franciscus.nl)",
                "Created On": "2019-11-04T14:59:22+0100",
                "First Opened On": None,
                "Locked": True,
                "Package Invitation": 'Dear participant,\n\nYou are participating in the study "Example Study" and we would like to ask you to fill in a survey.\n\nPlease click the link below to complete our survey.\n\n{url}\n\n{logo}',
                "Package Invitation Subject": "Please fill in this survey for Example Study",
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 100,
                "Received Time": "2020-08-12T16:53:00+0200",
                "Record Id": "000002",
                "Sent On": None,
                "Start Time": None,
                "Status": "Locked",
                "Survey Package Id": "71C01598-4682-4A4C-90E6-69C0BD38EA47",
                "Survey Package Instance Id": "23B4FD48-BA41-4C9B-BAEF-D5C3DD5F8E5C",
                "Survey Url String": "XWNSC9W5",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:37:33+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:43:13.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Sub-survey of package invitation created",
            "user_id": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "user_name": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "user_email": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "event_details": {
                "Archived": 0,
                "Auto Lock On Finish": 0,
                "Createdby": "PythonAPI Github",
                "Createdon": "2021-10-12 09:43:13",
                "Finished On": None,
                "First Opened On": None,
                "Last Updated By": None,
                "Last Updated On": None,
                "Locked": 0,
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 0,
                "Record Id": "000001",
                "Sent On": None,
                "Survey Auto Send": 0,
                "Survey Id": "D70C1273-B5D8-45CD-BFE8-A0BA75C44B7E",
                "Survey Instance Id": "4E369673-5C65-4508-B9A4-898A87FAEC89",
                "Survey Invitation": None,
                "Survey Invitation Subject": None,
                "Survey Package Instance Id": "AABB381C-3A69-47E6-9AA3-03E5BD4CA1D4",
                "Survey Url String": None,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:43:15.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Survey package invitation unlocked",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Archived": False,
                "Auto Lock On Finish": False,
                "Auto Send": False,
                "Completion Time": "2020-08-12T16:53:00+0200",
                "Created By": "Reinier van Linschoten (R.linschoten@franciscus.nl)",
                "Created On": "2019-11-04T14:59:22+0100",
                "First Opened On": None,
                "Locked": False,
                "Package Invitation": 'Dear participant,\n\nYou are participating in the study "Example Study" and we would like to ask you to fill in a survey.\n\nPlease click the link below to complete our survey.\n\n{url}\n\n{logo}',
                "Package Invitation Subject": "Please fill in this survey for Example Study",
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 100,
                "Received Time": "2020-08-12T16:53:00+0200",
                "Record Id": "000002",
                "Sent On": None,
                "Start Time": None,
                "Status": "Created",
                "Survey Package Id": "71C01598-4682-4A4C-90E6-69C0BD38EA47",
                "Survey Package Instance Id": "23B4FD48-BA41-4C9B-BAEF-D5C3DD5F8E5C",
                "Survey Url String": "XWNSC9W5",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:43:15+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:44:45.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Sub-survey of package invitation created",
            "user_id": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "user_name": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "user_email": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "event_details": {
                "Archived": 0,
                "Auto Lock On Finish": 0,
                "Createdby": "PythonAPI Github",
                "Createdon": "2021-10-12 09:44:45",
                "Finished On": None,
                "First Opened On": None,
                "Last Updated By": None,
                "Last Updated On": None,
                "Locked": 0,
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 0,
                "Record Id": "000001",
                "Sent On": None,
                "Survey Auto Send": 0,
                "Survey Id": "D70C1273-B5D8-45CD-BFE8-A0BA75C44B7E",
                "Survey Instance Id": "52015DCC-5A4D-4A7B-B17C-E3BEFC5E9C3D",
                "Survey Invitation": None,
                "Survey Invitation Subject": None,
                "Survey Package Instance Id": "548E5961-6EF3-4285-8F0C-779060D9F4E5",
                "Survey Url String": None,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:44:47.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Survey package invitation locked",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Archived": False,
                "Auto Lock On Finish": False,
                "Auto Send": False,
                "Completion Time": "2020-08-12T16:53:00+0200",
                "Created By": "Reinier van Linschoten (R.linschoten@franciscus.nl)",
                "Created On": "2019-11-04T14:59:22+0100",
                "First Opened On": None,
                "Locked": True,
                "Package Invitation": 'Dear participant,\n\nYou are participating in the study "Example Study" and we would like to ask you to fill in a survey.\n\nPlease click the link below to complete our survey.\n\n{url}\n\n{logo}',
                "Package Invitation Subject": "Please fill in this survey for Example Study",
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 100,
                "Received Time": "2020-08-12T16:53:00+0200",
                "Record Id": "000002",
                "Sent On": None,
                "Start Time": None,
                "Status": "Locked",
                "Survey Package Id": "71C01598-4682-4A4C-90E6-69C0BD38EA47",
                "Survey Package Instance Id": "23B4FD48-BA41-4C9B-BAEF-D5C3DD5F8E5C",
                "Survey Url String": "XWNSC9W5",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:44:47+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:45:31.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "166",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:45:35.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Diastolic blood pressure in mmHg"',
                "Field Value": "142",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:45:35.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Laboratory", Field: "Hematocrit value"',
                "Field Value": "33",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:45:35.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "198",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:46:00.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:46:00+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000222",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:46:00+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:46:01.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record randomized",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Progress": 0,
                "Randomized By": "castoredcapi.github@gmail.com",
                "Randomized Group Name": "Control",
                "Randomized Group Number": 1,
                "Randomized Id": "009",
                "Randomized On": "12-10-2021 09:46:01",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:46:01.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Randomization", Field: "Randomization allocation"',
                "Field Value": 1,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:46:05.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:46:05+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000223",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:46:05+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:46:16.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000003",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "88198C3C-5A3B-4562-8BE5-C1E9F9FF9C7E",
                "Report Name Custom": "12493419",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:46:18.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "1E899D5F-DFCE-4BC4-8591-FF58E0173EF3",
                "Report Name Custom": "38923775",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:46:18.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "2D7B5771-8CDD-40B4-9521-E8AE904DB944",
                "Report Name Custom": "40373747",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:46:18.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "6A166FF3-046C-4A00-B8EB-66D6C375D74A",
                "Report Name Custom": "44159104",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:46:41.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Sub-survey of package invitation created",
            "user_id": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "user_name": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "user_email": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "event_details": {
                "Archived": 0,
                "Auto Lock On Finish": 0,
                "Createdby": "PythonAPI Github",
                "Createdon": "2021-10-12 09:46:41",
                "Finished On": None,
                "First Opened On": None,
                "Last Updated By": None,
                "Last Updated On": None,
                "Locked": 0,
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 0,
                "Record Id": "000001",
                "Sent On": None,
                "Survey Auto Send": 0,
                "Survey Id": "D70C1273-B5D8-45CD-BFE8-A0BA75C44B7E",
                "Survey Instance Id": "D7F08BE0-499C-4AEC-B96A-34B2F1549C6B",
                "Survey Invitation": None,
                "Survey Invitation Subject": None,
                "Survey Package Instance Id": "538FF291-1639-4FC6-812C-6C96B906D9D5",
                "Survey Url String": None,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:46:44.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Survey package invitation unlocked",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Archived": False,
                "Auto Lock On Finish": False,
                "Auto Send": False,
                "Completion Time": "2020-08-12T16:53:00+0200",
                "Created By": "Reinier van Linschoten (R.linschoten@franciscus.nl)",
                "Created On": "2019-11-04T14:59:22+0100",
                "First Opened On": None,
                "Locked": False,
                "Package Invitation": 'Dear participant,\n\nYou are participating in the study "Example Study" and we would like to ask you to fill in a survey.\n\nPlease click the link below to complete our survey.\n\n{url}\n\n{logo}',
                "Package Invitation Subject": "Please fill in this survey for Example Study",
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 100,
                "Received Time": "2020-08-12T16:53:00+0200",
                "Record Id": "000002",
                "Sent On": None,
                "Start Time": None,
                "Status": "Created",
                "Survey Package Id": "71C01598-4682-4A4C-90E6-69C0BD38EA47",
                "Survey Package Instance Id": "23B4FD48-BA41-4C9B-BAEF-D5C3DD5F8E5C",
                "Survey Url String": "XWNSC9W5",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:46:44+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:50:06.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:50:06+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000224",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:50:06+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:50:55.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:50:55+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000225",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:50:55+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:51:27.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "152",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:51:32.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Diastolic blood pressure in mmHg"',
                "Field Value": "161",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:51:32.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Laboratory", Field: "Hematocrit value"',
                "Field Value": "26",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:51:32.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "166",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:51:55.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:51:55+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000226",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:51:55+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:51:55.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record randomized",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Progress": 0,
                "Randomized By": "castoredcapi.github@gmail.com",
                "Randomized Group Name": "Intervention",
                "Randomized Group Number": 2,
                "Randomized Id": "009",
                "Randomized On": "12-10-2021 09:51:55",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:51:55.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Randomization", Field: "Randomization allocation"',
                "Field Value": 2,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:51:59.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:51:59+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000227",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:51:59+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:52:09.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000003",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "18B1F9B0-D6A5-4733-9B64-664BE2B2876C",
                "Report Name Custom": "70331968",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:52:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "46587E64-769C-42B7-9D22-E3A83F6FC7FD",
                "Report Name Custom": "91424793",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:52:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "D393292F-78CC-4E23-9230-0EFABBB3E74D",
                "Report Name Custom": "11994679",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:52:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "1E174647-FE91-46C0-93C2-45D0285DFBFE",
                "Report Name Custom": "59538294",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:52:34.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Sub-survey of package invitation created",
            "user_id": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "user_name": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "user_email": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "event_details": {
                "Archived": 0,
                "Auto Lock On Finish": 0,
                "Createdby": "PythonAPI Github",
                "Createdon": "2021-10-12 09:52:34",
                "Finished On": None,
                "First Opened On": None,
                "Last Updated By": None,
                "Last Updated On": None,
                "Locked": 0,
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 0,
                "Record Id": "000001",
                "Sent On": None,
                "Survey Auto Send": 0,
                "Survey Id": "D70C1273-B5D8-45CD-BFE8-A0BA75C44B7E",
                "Survey Instance Id": "9A486F7F-DAEA-4929-AA9B-FCD68DFF0627",
                "Survey Invitation": None,
                "Survey Invitation Subject": None,
                "Survey Package Instance Id": "CE29B846-6B91-49C8-AC55-B4B17B706546",
                "Survey Url String": None,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:52:36.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Survey package invitation locked",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Archived": False,
                "Auto Lock On Finish": False,
                "Auto Send": False,
                "Completion Time": "2020-08-12T16:53:00+0200",
                "Created By": "Reinier van Linschoten (R.linschoten@franciscus.nl)",
                "Created On": "2019-11-04T14:59:22+0100",
                "First Opened On": None,
                "Locked": True,
                "Package Invitation": 'Dear participant,\n\nYou are participating in the study "Example Study" and we would like to ask you to fill in a survey.\n\nPlease click the link below to complete our survey.\n\n{url}\n\n{logo}',
                "Package Invitation Subject": "Please fill in this survey for Example Study",
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 100,
                "Received Time": "2020-08-12T16:53:00+0200",
                "Record Id": "000002",
                "Sent On": None,
                "Start Time": None,
                "Status": "Locked",
                "Survey Package Id": "71C01598-4682-4A4C-90E6-69C0BD38EA47",
                "Survey Package Instance Id": "23B4FD48-BA41-4C9B-BAEF-D5C3DD5F8E5C",
                "Survey Url String": "XWNSC9W5",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:52:36+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:15.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "68",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:20.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Diastolic blood pressure in mmHg"',
                "Field Value": "127",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:20.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Laboratory", Field: "Hematocrit value"',
                "Field Value": "54",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:20.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "139",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:43.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:58:43+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000228",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:58:43+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:43.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record randomized",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Progress": 0,
                "Randomized By": "castoredcapi.github@gmail.com",
                "Randomized Group Name": "Control",
                "Randomized Group Number": 1,
                "Randomized Id": "010",
                "Randomized On": "12-10-2021 09:58:43",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:43.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Randomization", Field: "Randomization allocation"',
                "Field Value": 1,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:47.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:58:47+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000229",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:58:47+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:57.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000003",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "0AD51C79-0BC9-4B90-BE9E-5FB8DA20D135",
                "Report Name Custom": "47954216",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:59.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "5B3E84EF-0CFA-40B7-8160-0D1FF2F93830",
                "Report Name Custom": "42821786",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:59.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "3DADFCFB-4348-4798-8194-BBF00E5EAC30",
                "Report Name Custom": "79292451",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:59.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "E2855588-6248-40AD-A5F6-E010AD2231F5",
                "Report Name Custom": "36306512",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:59:18.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Sub-survey of package invitation created",
            "user_id": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "user_name": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "user_email": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "event_details": {
                "Archived": 0,
                "Auto Lock On Finish": 0,
                "Createdby": "PythonAPI Github",
                "Createdon": "2021-10-12 09:59:18",
                "Finished On": None,
                "First Opened On": None,
                "Last Updated By": None,
                "Last Updated On": None,
                "Locked": 0,
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 0,
                "Record Id": "000001",
                "Sent On": None,
                "Survey Auto Send": 0,
                "Survey Id": "D70C1273-B5D8-45CD-BFE8-A0BA75C44B7E",
                "Survey Instance Id": "772C8D10-80DF-4CF7-95FE-431C9EC4BB43",
                "Survey Invitation": None,
                "Survey Invitation Subject": None,
                "Survey Package Instance Id": "5F187B5A-8BE0-4640-B201-ACC1F0A0336A",
                "Survey Url String": None,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:59:20.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Survey package invitation unlocked",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Archived": False,
                "Auto Lock On Finish": False,
                "Auto Send": False,
                "Completion Time": "2020-08-12T16:53:00+0200",
                "Created By": "Reinier van Linschoten (R.linschoten@franciscus.nl)",
                "Created On": "2019-11-04T14:59:22+0100",
                "First Opened On": None,
                "Locked": False,
                "Package Invitation": 'Dear participant,\n\nYou are participating in the study "Example Study" and we would like to ask you to fill in a survey.\n\nPlease click the link below to complete our survey.\n\n{url}\n\n{logo}',
                "Package Invitation Subject": "Please fill in this survey for Example Study",
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 100,
                "Received Time": "2020-08-12T16:53:00+0200",
                "Record Id": "000002",
                "Sent On": None,
                "Start Time": None,
                "Status": "Created",
                "Survey Package Id": "71C01598-4682-4A4C-90E6-69C0BD38EA47",
                "Survey Package Instance Id": "23B4FD48-BA41-4C9B-BAEF-D5C3DD5F8E5C",
                "Survey Url String": "XWNSC9W5",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:59:20+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:00:13.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "135",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:00:17.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Diastolic blood pressure in mmHg"',
                "Field Value": "168",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:00:17.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Laboratory", Field: "Hematocrit value"',
                "Field Value": "86",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:00:17.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "55",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:00:50.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T10:00:50+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000230",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T10:00:50+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:00:51.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record randomized",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Progress": 0,
                "Randomized By": "castoredcapi.github@gmail.com",
                "Randomized Group Name": "Control",
                "Randomized Group Number": 1,
                "Randomized Id": "011",
                "Randomized On": "12-10-2021 10:00:51",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:00:51.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Randomization", Field: "Randomization allocation"',
                "Field Value": 1,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:00:55.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T10:00:55+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000231",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T10:00:55+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:01:07.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000003",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "4F020D84-B28C-4E35-94E6-4B16F5871917",
                "Report Name Custom": "60993389",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:01:08.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "04C9DCFC-122F-4895-8C09-F580A81F7E9C",
                "Report Name Custom": "99561698",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:01:08.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "3A8A1C68-9679-4340-BC5B-BCA987B043A8",
                "Report Name Custom": "93624573",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:01:08.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "D7938C45-E751-4D88-9C26-BB9CF8D965EE",
                "Report Name Custom": "84116011",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:01:40.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Sub-survey of package invitation created",
            "user_id": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "user_name": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "user_email": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "event_details": {
                "Archived": 0,
                "Auto Lock On Finish": 0,
                "Createdby": "PythonAPI Github",
                "Createdon": "2021-10-12 10:01:40",
                "Finished On": None,
                "First Opened On": None,
                "Last Updated By": None,
                "Last Updated On": None,
                "Locked": 0,
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 0,
                "Record Id": "000001",
                "Sent On": None,
                "Survey Auto Send": 0,
                "Survey Id": "D70C1273-B5D8-45CD-BFE8-A0BA75C44B7E",
                "Survey Instance Id": "5AFF838C-A1BE-4869-A8D2-443B498FB97F",
                "Survey Invitation": None,
                "Survey Invitation Subject": None,
                "Survey Package Instance Id": "B0D01A2E-B24D-4F5F-8196-0A7B6DECAB17",
                "Survey Url String": None,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:01:43.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Survey package invitation locked",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Archived": False,
                "Auto Lock On Finish": False,
                "Auto Send": False,
                "Completion Time": "2020-08-12T16:53:00+0200",
                "Created By": "Reinier van Linschoten (R.linschoten@franciscus.nl)",
                "Created On": "2019-11-04T14:59:22+0100",
                "First Opened On": None,
                "Locked": True,
                "Package Invitation": 'Dear participant,\n\nYou are participating in the study "Example Study" and we would like to ask you to fill in a survey.\n\nPlease click the link below to complete our survey.\n\n{url}\n\n{logo}',
                "Package Invitation Subject": "Please fill in this survey for Example Study",
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 100,
                "Received Time": "2020-08-12T16:53:00+0200",
                "Record Id": "000002",
                "Sent On": None,
                "Start Time": None,
                "Status": "Locked",
                "Survey Package Id": "71C01598-4682-4A4C-90E6-69C0BD38EA47",
                "Survey Package Instance Id": "23B4FD48-BA41-4C9B-BAEF-D5C3DD5F8E5C",
                "Survey Url String": "XWNSC9W5",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T10:01:43+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:04:48.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "50",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:04:51.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Diastolic blood pressure in mmHg"',
                "Field Value": "92",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:04:51.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Laboratory", Field: "Hematocrit value"',
                "Field Value": "48",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:04:51.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "74",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:05:16.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T10:05:16+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000232",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T10:05:16+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:05:17.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record randomized",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Progress": 0,
                "Randomized By": "castoredcapi.github@gmail.com",
                "Randomized Group Name": "Intervention",
                "Randomized Group Number": 2,
                "Randomized Id": "010",
                "Randomized On": "12-10-2021 10:05:17",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:05:17.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Randomization", Field: "Randomization allocation"',
                "Field Value": 2,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:05:21.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T10:05:21+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000233",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T10:05:21+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:05:33.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000003",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "B8012A7A-FB84-4EB6-B5BB-4A641C60E43A",
                "Report Name Custom": "64163572",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:05:35.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "E1A62955-B488-4778-8F1C-56E2B1F9CB08",
                "Report Name Custom": "35997075",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:05:35.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "165F889B-A383-4AC9-A3AD-E0E015BEE913",
                "Report Name Custom": "50005335",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:05:35.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "87221943-839C-4176-9C44-031785BEAA05",
                "Report Name Custom": "80909364",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:07:38.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "250",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:07:42.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Diastolic blood pressure in mmHg"',
                "Field Value": "125",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:07:42.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Laboratory", Field: "Hematocrit value"',
                "Field Value": "45",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:07:42.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "87",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:08:09.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T10:08:09+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000234",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T10:08:09+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:08:09.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record randomized",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Progress": 0,
                "Randomized By": "castoredcapi.github@gmail.com",
                "Randomized Group Name": "Intervention",
                "Randomized Group Number": 2,
                "Randomized Id": "011",
                "Randomized On": "12-10-2021 10:08:09",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:08:09.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Randomization", Field: "Randomization allocation"',
                "Field Value": 2,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:08:13.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T10:08:13+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000235",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T10:08:13+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:08:24.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000003",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "13F937B1-EA60-48EB-8219-49D749A9FA9E",
                "Report Name Custom": "40068813",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:08:25.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "34C7BC11-FA74-4928-AADC-365FE8FEB59D",
                "Report Name Custom": "47262143",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:08:25.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "539BDFDF-F6BE-4106-955B-108E8728BF65",
                "Report Name Custom": "15125046",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:08:25.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "5A6F035E-E4AA-4354-AFFE-6A9BD5036CE8",
                "Report Name Custom": "50244726",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:08:46.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Sub-survey of package invitation created",
            "user_id": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "user_name": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "user_email": "AUTOMATION_ENGINE_OR_UNKNOWN",
            "event_details": {
                "Archived": 0,
                "Auto Lock On Finish": 0,
                "Createdby": "PythonAPI Github",
                "Createdon": "2021-10-12 10:08:46",
                "Finished On": None,
                "First Opened On": None,
                "Last Updated By": None,
                "Last Updated On": None,
                "Locked": 0,
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 0,
                "Record Id": "000001",
                "Sent On": None,
                "Survey Auto Send": 0,
                "Survey Id": "D70C1273-B5D8-45CD-BFE8-A0BA75C44B7E",
                "Survey Instance Id": "222169D1-BE7F-4A19-9BA1-D132FCD016B5",
                "Survey Invitation": None,
                "Survey Invitation Subject": None,
                "Survey Package Instance Id": "49C32EF2-C466-4638-ADCD-78116292BE31",
                "Survey Url String": None,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:08:50.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Survey package invitation unlocked",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Archived": False,
                "Auto Lock On Finish": False,
                "Auto Send": False,
                "Completion Time": "2020-08-12T16:53:00+0200",
                "Created By": "Reinier van Linschoten (R.linschoten@franciscus.nl)",
                "Created On": "2019-11-04T14:59:22+0100",
                "First Opened On": None,
                "Locked": False,
                "Package Invitation": 'Dear participant,\n\nYou are participating in the study "Example Study" and we would like to ask you to fill in a survey.\n\nPlease click the link below to complete our survey.\n\n{url}\n\n{logo}',
                "Package Invitation Subject": "Please fill in this survey for Example Study",
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 100,
                "Received Time": "2020-08-12T16:53:00+0200",
                "Record Id": "000002",
                "Sent On": None,
                "Start Time": None,
                "Status": "Created",
                "Survey Package Id": "71C01598-4682-4A4C-90E6-69C0BD38EA47",
                "Survey Package Instance Id": "23B4FD48-BA41-4C9B-BAEF-D5C3DD5F8E5C",
                "Survey Url String": "XWNSC9W5",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T10:08:50+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 15:27:29.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Permissions - User invited to study",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Account Created": "2021-10-12 15:27:29",
                "Invited User Email Address": "reinier.v.linschoten@gmail.com",
                "Invited User ID": "5376D4F9-8525-4E87-8CF4-7DA7EDBA6168",
                "Inviting User ID": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
                "Inviting User Name": "PythonAPI Github",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 16:31:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Role created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": [],
        },
    ]

    test_audit_trail_user_model = [
        {
            "datetime": {
                "date": "2021-10-12 09:08:29.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "143",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:08:33.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Diastolic blood pressure in mmHg"',
                "Field Value": "71",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:08:33.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Laboratory", Field: "Hematocrit value"',
                "Field Value": "23",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:08:33.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "207",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:08:55.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:08:55+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000218",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:08:55+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:08:56.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record randomized",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Progress": 0,
                "Randomized By": "castoredcapi.github@gmail.com",
                "Randomized Group Name": "Control",
                "Randomized Group Number": 1,
                "Randomized Id": "007",
                "Randomized On": "12-10-2021 09:08:56",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:08:56.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Randomization", Field: "Randomization allocation"',
                "Field Value": 1,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:08:59.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:08:59+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000219",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:08:59+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:09:09.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000003",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "8745EB23-CF9D-495C-B081-0EB24C68BD8F",
                "Report Name Custom": "89041199",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:09:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "6ED6EC83-218C-4610-8387-E82FADBA6D7B",
                "Report Name Custom": "68217351",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:09:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "D88F938D-BCC3-407B-835F-AD4C2DB92F53",
                "Report Name Custom": "17893202",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:09:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "5358BFA2-0742-4625-88C9-C50E74D12FAD",
                "Report Name Custom": "73314828",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:09:32.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Survey package invitation unlocked",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Archived": False,
                "Auto Lock On Finish": False,
                "Auto Send": False,
                "Completion Time": "2020-08-12T16:53:00+0200",
                "Created By": "Reinier van Linschoten (R.linschoten@franciscus.nl)",
                "Created On": "2019-11-04T14:59:22+0100",
                "First Opened On": None,
                "Locked": False,
                "Package Invitation": 'Dear participant,\n\nYou are participating in the study "Example Study" and we would like to ask you to fill in a survey.\n\nPlease click the link below to complete our survey.\n\n{url}\n\n{logo}',
                "Package Invitation Subject": "Please fill in this survey for Example Study",
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 100,
                "Received Time": "2020-08-12T16:53:00+0200",
                "Record Id": "000002",
                "Sent On": None,
                "Start Time": None,
                "Status": "Created",
                "Survey Package Id": "71C01598-4682-4A4C-90E6-69C0BD38EA47",
                "Survey Package Instance Id": "23B4FD48-BA41-4C9B-BAEF-D5C3DD5F8E5C",
                "Survey Url String": "XWNSC9W5",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:09:32+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:36:26.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "128",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:36:29.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Diastolic blood pressure in mmHg"',
                "Field Value": "32",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:36:29.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Laboratory", Field: "Hematocrit value"',
                "Field Value": "27",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:36:29.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "79",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:36:51.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:36:51+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000220",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:36:51+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:36:51.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record randomized",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Progress": 0,
                "Randomized By": "castoredcapi.github@gmail.com",
                "Randomized Group Name": "Control",
                "Randomized Group Number": 1,
                "Randomized Id": "008",
                "Randomized On": "12-10-2021 09:36:51",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:36:51.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Randomization", Field: "Randomization allocation"',
                "Field Value": 1,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:36:57.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:36:57+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000221",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:36:57+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:37:08.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000003",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "7B61051A-A7C3-4976-B2CF-13FF43CE61E6",
                "Report Name Custom": "39626898",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:37:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "523C81AA-8CDB-42B9-9860-11F1283DDDCC",
                "Report Name Custom": "22329694",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:37:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "53C8FE42-E790-4876-856D-B5A6714E702D",
                "Report Name Custom": "23616042",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:37:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "6600CEFD-9441-4C55-877B-7CC95A9F0CEA",
                "Report Name Custom": "44612112",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:37:33.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Survey package invitation locked",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Archived": False,
                "Auto Lock On Finish": False,
                "Auto Send": False,
                "Completion Time": "2020-08-12T16:53:00+0200",
                "Created By": "Reinier van Linschoten (R.linschoten@franciscus.nl)",
                "Created On": "2019-11-04T14:59:22+0100",
                "First Opened On": None,
                "Locked": True,
                "Package Invitation": 'Dear participant,\n\nYou are participating in the study "Example Study" and we would like to ask you to fill in a survey.\n\nPlease click the link below to complete our survey.\n\n{url}\n\n{logo}',
                "Package Invitation Subject": "Please fill in this survey for Example Study",
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 100,
                "Received Time": "2020-08-12T16:53:00+0200",
                "Record Id": "000002",
                "Sent On": None,
                "Start Time": None,
                "Status": "Locked",
                "Survey Package Id": "71C01598-4682-4A4C-90E6-69C0BD38EA47",
                "Survey Package Instance Id": "23B4FD48-BA41-4C9B-BAEF-D5C3DD5F8E5C",
                "Survey Url String": "XWNSC9W5",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:37:33+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:43:15.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Survey package invitation unlocked",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Archived": False,
                "Auto Lock On Finish": False,
                "Auto Send": False,
                "Completion Time": "2020-08-12T16:53:00+0200",
                "Created By": "Reinier van Linschoten (R.linschoten@franciscus.nl)",
                "Created On": "2019-11-04T14:59:22+0100",
                "First Opened On": None,
                "Locked": False,
                "Package Invitation": 'Dear participant,\n\nYou are participating in the study "Example Study" and we would like to ask you to fill in a survey.\n\nPlease click the link below to complete our survey.\n\n{url}\n\n{logo}',
                "Package Invitation Subject": "Please fill in this survey for Example Study",
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 100,
                "Received Time": "2020-08-12T16:53:00+0200",
                "Record Id": "000002",
                "Sent On": None,
                "Start Time": None,
                "Status": "Created",
                "Survey Package Id": "71C01598-4682-4A4C-90E6-69C0BD38EA47",
                "Survey Package Instance Id": "23B4FD48-BA41-4C9B-BAEF-D5C3DD5F8E5C",
                "Survey Url String": "XWNSC9W5",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:43:15+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:44:47.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Survey package invitation locked",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Archived": False,
                "Auto Lock On Finish": False,
                "Auto Send": False,
                "Completion Time": "2020-08-12T16:53:00+0200",
                "Created By": "Reinier van Linschoten (R.linschoten@franciscus.nl)",
                "Created On": "2019-11-04T14:59:22+0100",
                "First Opened On": None,
                "Locked": True,
                "Package Invitation": 'Dear participant,\n\nYou are participating in the study "Example Study" and we would like to ask you to fill in a survey.\n\nPlease click the link below to complete our survey.\n\n{url}\n\n{logo}',
                "Package Invitation Subject": "Please fill in this survey for Example Study",
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 100,
                "Received Time": "2020-08-12T16:53:00+0200",
                "Record Id": "000002",
                "Sent On": None,
                "Start Time": None,
                "Status": "Locked",
                "Survey Package Id": "71C01598-4682-4A4C-90E6-69C0BD38EA47",
                "Survey Package Instance Id": "23B4FD48-BA41-4C9B-BAEF-D5C3DD5F8E5C",
                "Survey Url String": "XWNSC9W5",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:44:47+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:45:31.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "166",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:45:35.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Diastolic blood pressure in mmHg"',
                "Field Value": "142",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:45:35.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Laboratory", Field: "Hematocrit value"',
                "Field Value": "33",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:45:35.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "198",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:46:00.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:46:00+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000222",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:46:00+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:46:01.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record randomized",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Progress": 0,
                "Randomized By": "castoredcapi.github@gmail.com",
                "Randomized Group Name": "Control",
                "Randomized Group Number": 1,
                "Randomized Id": "009",
                "Randomized On": "12-10-2021 09:46:01",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:46:01.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Randomization", Field: "Randomization allocation"',
                "Field Value": 1,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:46:05.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:46:05+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000223",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:46:05+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:46:16.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000003",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "88198C3C-5A3B-4562-8BE5-C1E9F9FF9C7E",
                "Report Name Custom": "12493419",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:46:18.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "1E899D5F-DFCE-4BC4-8591-FF58E0173EF3",
                "Report Name Custom": "38923775",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:46:18.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "2D7B5771-8CDD-40B4-9521-E8AE904DB944",
                "Report Name Custom": "40373747",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:46:18.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "6A166FF3-046C-4A00-B8EB-66D6C375D74A",
                "Report Name Custom": "44159104",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:46:44.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Survey package invitation unlocked",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Archived": False,
                "Auto Lock On Finish": False,
                "Auto Send": False,
                "Completion Time": "2020-08-12T16:53:00+0200",
                "Created By": "Reinier van Linschoten (R.linschoten@franciscus.nl)",
                "Created On": "2019-11-04T14:59:22+0100",
                "First Opened On": None,
                "Locked": False,
                "Package Invitation": 'Dear participant,\n\nYou are participating in the study "Example Study" and we would like to ask you to fill in a survey.\n\nPlease click the link below to complete our survey.\n\n{url}\n\n{logo}',
                "Package Invitation Subject": "Please fill in this survey for Example Study",
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 100,
                "Received Time": "2020-08-12T16:53:00+0200",
                "Record Id": "000002",
                "Sent On": None,
                "Start Time": None,
                "Status": "Created",
                "Survey Package Id": "71C01598-4682-4A4C-90E6-69C0BD38EA47",
                "Survey Package Instance Id": "23B4FD48-BA41-4C9B-BAEF-D5C3DD5F8E5C",
                "Survey Url String": "XWNSC9W5",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:46:44+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:50:06.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:50:06+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000224",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:50:06+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:50:55.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:50:55+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000225",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:50:55+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:51:27.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "152",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:51:32.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Diastolic blood pressure in mmHg"',
                "Field Value": "161",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:51:32.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Laboratory", Field: "Hematocrit value"',
                "Field Value": "26",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:51:32.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "166",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:51:55.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:51:55+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000226",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:51:55+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:51:55.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record randomized",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Progress": 0,
                "Randomized By": "castoredcapi.github@gmail.com",
                "Randomized Group Name": "Intervention",
                "Randomized Group Number": 2,
                "Randomized Id": "009",
                "Randomized On": "12-10-2021 09:51:55",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:51:55.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Randomization", Field: "Randomization allocation"',
                "Field Value": 2,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:51:59.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:51:59+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000227",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:51:59+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:52:09.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000003",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "18B1F9B0-D6A5-4733-9B64-664BE2B2876C",
                "Report Name Custom": "70331968",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:52:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "46587E64-769C-42B7-9D22-E3A83F6FC7FD",
                "Report Name Custom": "91424793",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:52:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "D393292F-78CC-4E23-9230-0EFABBB3E74D",
                "Report Name Custom": "11994679",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:52:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "1E174647-FE91-46C0-93C2-45D0285DFBFE",
                "Report Name Custom": "59538294",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:52:36.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Survey package invitation locked",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Archived": False,
                "Auto Lock On Finish": False,
                "Auto Send": False,
                "Completion Time": "2020-08-12T16:53:00+0200",
                "Created By": "Reinier van Linschoten (R.linschoten@franciscus.nl)",
                "Created On": "2019-11-04T14:59:22+0100",
                "First Opened On": None,
                "Locked": True,
                "Package Invitation": 'Dear participant,\n\nYou are participating in the study "Example Study" and we would like to ask you to fill in a survey.\n\nPlease click the link below to complete our survey.\n\n{url}\n\n{logo}',
                "Package Invitation Subject": "Please fill in this survey for Example Study",
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 100,
                "Received Time": "2020-08-12T16:53:00+0200",
                "Record Id": "000002",
                "Sent On": None,
                "Start Time": None,
                "Status": "Locked",
                "Survey Package Id": "71C01598-4682-4A4C-90E6-69C0BD38EA47",
                "Survey Package Instance Id": "23B4FD48-BA41-4C9B-BAEF-D5C3DD5F8E5C",
                "Survey Url String": "XWNSC9W5",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:52:36+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:15.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "68",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:20.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Diastolic blood pressure in mmHg"',
                "Field Value": "127",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:20.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Laboratory", Field: "Hematocrit value"',
                "Field Value": "54",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:20.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "139",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:43.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:58:43+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000228",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:58:43+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:43.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record randomized",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Progress": 0,
                "Randomized By": "castoredcapi.github@gmail.com",
                "Randomized Group Name": "Control",
                "Randomized Group Number": 1,
                "Randomized Id": "010",
                "Randomized On": "12-10-2021 09:58:43",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:43.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Randomization", Field: "Randomization allocation"',
                "Field Value": 1,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:47.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T09:58:47+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000229",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:58:47+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:57.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000003",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "0AD51C79-0BC9-4B90-BE9E-5FB8DA20D135",
                "Report Name Custom": "47954216",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:59.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "5B3E84EF-0CFA-40B7-8160-0D1FF2F93830",
                "Report Name Custom": "42821786",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:59.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "3DADFCFB-4348-4798-8194-BBF00E5EAC30",
                "Report Name Custom": "79292451",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:58:59.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "E2855588-6248-40AD-A5F6-E010AD2231F5",
                "Report Name Custom": "36306512",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 09:59:20.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Survey package invitation unlocked",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Archived": False,
                "Auto Lock On Finish": False,
                "Auto Send": False,
                "Completion Time": "2020-08-12T16:53:00+0200",
                "Created By": "Reinier van Linschoten (R.linschoten@franciscus.nl)",
                "Created On": "2019-11-04T14:59:22+0100",
                "First Opened On": None,
                "Locked": False,
                "Package Invitation": 'Dear participant,\n\nYou are participating in the study "Example Study" and we would like to ask you to fill in a survey.\n\nPlease click the link below to complete our survey.\n\n{url}\n\n{logo}',
                "Package Invitation Subject": "Please fill in this survey for Example Study",
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 100,
                "Received Time": "2020-08-12T16:53:00+0200",
                "Record Id": "000002",
                "Sent On": None,
                "Start Time": None,
                "Status": "Created",
                "Survey Package Id": "71C01598-4682-4A4C-90E6-69C0BD38EA47",
                "Survey Package Instance Id": "23B4FD48-BA41-4C9B-BAEF-D5C3DD5F8E5C",
                "Survey Url String": "XWNSC9W5",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T09:59:20+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:00:13.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "135",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:00:17.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Diastolic blood pressure in mmHg"',
                "Field Value": "168",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:00:17.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Laboratory", Field: "Hematocrit value"',
                "Field Value": "86",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:00:17.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "55",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:00:50.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T10:00:50+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000230",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T10:00:50+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:00:51.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record randomized",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Progress": 0,
                "Randomized By": "castoredcapi.github@gmail.com",
                "Randomized Group Name": "Control",
                "Randomized Group Number": 1,
                "Randomized Id": "011",
                "Randomized On": "12-10-2021 10:00:51",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:00:51.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Randomization", Field: "Randomization allocation"',
                "Field Value": 1,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:00:55.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T10:00:55+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000231",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T10:00:55+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:01:07.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000003",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "4F020D84-B28C-4E35-94E6-4B16F5871917",
                "Report Name Custom": "60993389",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:01:08.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "04C9DCFC-122F-4895-8C09-F580A81F7E9C",
                "Report Name Custom": "99561698",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:01:08.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "3A8A1C68-9679-4340-BC5B-BCA987B043A8",
                "Report Name Custom": "93624573",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:01:08.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "D7938C45-E751-4D88-9C26-BB9CF8D965EE",
                "Report Name Custom": "84116011",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:01:43.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Survey package invitation locked",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Archived": False,
                "Auto Lock On Finish": False,
                "Auto Send": False,
                "Completion Time": "2020-08-12T16:53:00+0200",
                "Created By": "Reinier van Linschoten (R.linschoten@franciscus.nl)",
                "Created On": "2019-11-04T14:59:22+0100",
                "First Opened On": None,
                "Locked": True,
                "Package Invitation": 'Dear participant,\n\nYou are participating in the study "Example Study" and we would like to ask you to fill in a survey.\n\nPlease click the link below to complete our survey.\n\n{url}\n\n{logo}',
                "Package Invitation Subject": "Please fill in this survey for Example Study",
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 100,
                "Received Time": "2020-08-12T16:53:00+0200",
                "Record Id": "000002",
                "Sent On": None,
                "Start Time": None,
                "Status": "Locked",
                "Survey Package Id": "71C01598-4682-4A4C-90E6-69C0BD38EA47",
                "Survey Package Instance Id": "23B4FD48-BA41-4C9B-BAEF-D5C3DD5F8E5C",
                "Survey Url String": "XWNSC9W5",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T10:01:43+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:04:48.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "50",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:04:51.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Diastolic blood pressure in mmHg"',
                "Field Value": "92",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:04:51.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Laboratory", Field: "Hematocrit value"',
                "Field Value": "48",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:04:51.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "74",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:05:16.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T10:05:16+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000232",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T10:05:16+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:05:17.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record randomized",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Progress": 0,
                "Randomized By": "castoredcapi.github@gmail.com",
                "Randomized Group Name": "Intervention",
                "Randomized Group Number": 2,
                "Randomized Id": "010",
                "Randomized On": "12-10-2021 10:05:17",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:05:17.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Randomization", Field: "Randomization allocation"',
                "Field Value": 2,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:05:21.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T10:05:21+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000233",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T10:05:21+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:05:33.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000003",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "B8012A7A-FB84-4EB6-B5BB-4A641C60E43A",
                "Report Name Custom": "64163572",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:05:35.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "E1A62955-B488-4778-8F1C-56E2B1F9CB08",
                "Report Name Custom": "35997075",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:05:35.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "165F889B-A383-4AC9-A3AD-E0E015BEE913",
                "Report Name Custom": "50005335",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:05:35.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "87221943-839C-4176-9C44-031785BEAA05",
                "Report Name Custom": "80909364",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:07:38.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "250",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:07:42.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Diastolic blood pressure in mmHg"',
                "Field Value": "125",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:07:42.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Laboratory", Field: "Hematocrit value"',
                "Field Value": "45",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:07:42.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Report "Unscheduled visit", Step: "Physical exam", Field: "Systolic blood pressure in mmHg"',
                "Field Value": "87",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:08:09.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T10:08:09+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000234",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T10:08:09+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:08:09.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record randomized",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Progress": 0,
                "Randomized By": "castoredcapi.github@gmail.com",
                "Randomized Group Name": "Intervention",
                "Randomized Group Number": 2,
                "Randomized Id": "011",
                "Randomized On": "12-10-2021 10:08:09",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:08:09.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record result",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Current Location": 'Phase: "Baseline", Step: "Randomization", Field: "Randomization allocation"',
                "Field Value": 2,
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:08:13.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Record created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Created By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Created On": "2021-10-12T10:08:13+0200",
                "Institute Id": "Test Institute",
                "Record Id": "000235",
                "Status": "open",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T10:08:13+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:08:24.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000003",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "13F937B1-EA60-48EB-8219-49D749A9FA9E",
                "Report Name Custom": "40068813",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:08:25.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "34C7BC11-FA74-4928-AADC-365FE8FEB59D",
                "Report Name Custom": "47262143",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:08:25.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "539BDFDF-F6BE-4106-955B-108E8728BF65",
                "Report Name Custom": "15125046",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:08:25.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Report instance created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Record Id": "000005",
                "Report Id": "770DB401-6100-4CF5-A95F-3402B55EAC48",
                "Report Instance Id": "5A6F035E-E4AA-4354-AFFE-6A9BD5036CE8",
                "Report Name Custom": "50244726",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 10:08:50.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Survey package invitation unlocked",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Archived": False,
                "Auto Lock On Finish": False,
                "Auto Send": False,
                "Completion Time": "2020-08-12T16:53:00+0200",
                "Created By": "Reinier van Linschoten (R.linschoten@franciscus.nl)",
                "Created On": "2019-11-04T14:59:22+0100",
                "First Opened On": None,
                "Locked": False,
                "Package Invitation": 'Dear participant,\n\nYou are participating in the study "Example Study" and we would like to ask you to fill in a survey.\n\nPlease click the link below to complete our survey.\n\n{url}\n\n{logo}',
                "Package Invitation Subject": "Please fill in this survey for Example Study",
                "Parent Id": None,
                "Parent Type": None,
                "Planned Send Date": None,
                "Progress": 100,
                "Received Time": "2020-08-12T16:53:00+0200",
                "Record Id": "000002",
                "Sent On": None,
                "Start Time": None,
                "Status": "Created",
                "Survey Package Id": "71C01598-4682-4A4C-90E6-69C0BD38EA47",
                "Survey Package Instance Id": "23B4FD48-BA41-4C9B-BAEF-D5C3DD5F8E5C",
                "Survey Url String": "XWNSC9W5",
                "Updated By": "PythonAPI Github (castoredcapi.github@gmail.com)",
                "Updated On": "2021-10-12T10:08:50+0200",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 15:27:29.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Permissions - User invited to study",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": {
                "Account Created": "2021-10-12 15:27:29",
                "Invited User Email Address": "reinier.v.linschoten@gmail.com",
                "Invited User ID": "5376D4F9-8525-4E87-8CF4-7DA7EDBA6168",
                "Inviting User ID": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
                "Inviting User Name": "PythonAPI Github",
            },
        },
        {
            "datetime": {
                "date": "2021-10-12 16:31:10.000000",
                "timezone_type": 3,
                "timezone": "Europe/Amsterdam",
            },
            "event_type": "Role created",
            "user_id": "06A8C79E-F76F-4824-AB1A-93F0457C5A61",
            "user_name": "PythonAPI Github",
            "user_email": "castoredcapi.github@gmail.com",
            "event_details": [],
        },
    ]
