import os
import pathlib
from typing import FrozenSet, List, Optional, Set

BLACKLIST = frozenset(
    [
        "id.ac",
        "2011.cqut.edu.cn",
        "2012.cqut.edu.cn",
        "2013.cqut.edu.cn",
        "2014.cqut.edu.cn",
        "2015.cqut.edu.cn",
        "2016.cqut.edu.cn",
        "2017.cqut.edu.cn",
        "2018.cqut.edu.cn",
        "aa.seu.edu.cn",
        "ahlg.ac.cn",
        "ahzyygz.edu.cn",
        "aiit.edu.cn",
        "alu.cdut.edu.cn",
        "alu.hdu.edu.cn",
        "alu.uestc.edu.cn",
        "alumni.hust.edu.cn",
        "alumni.pku.edu.cn",
        "alumni.sjtu.edu.cn",
        "alumni.stu.edu.cn",
        "alumni.zstu.edu.cn",
        "aufe.edu.cn",
        "axhu.edu.cn",
        "ayit.edu.cn",
        "bbc.edu.cn",
        "bese.ac.cn",
        "bfsu.edu.cn",
        "bit.edu.cn",
        "bitzh.edu.cn",
        "bjdx.ac.cn",
        "bjlg.ac.cn",
        "bjmu.edu.cn",
        "buaa.edu.cn",
        "bucea.edu.cn",
        "bupt.cn",
        "caa.edu.cn",
        "cau.edu.cn",
        "ccnu.edu.cn",
        "cczu.edu.cn",
        "cdmc.edu.cn",
        "chd.edu.cn",
        "chu.edu.cn",
        "cjlu.edu.cn",
        "codesra.cn",
        "cque.edu.cn",
        "ctgu.edu.cn",
        "cug.edu.cn",
        "cumtb.edu.cn",
        "dhu.edu.cn",
        "dlut.edu.cn",
        "dxss.ac.cn",
        "dzxy.ntu.edu.cn",
        "e.gzhu.edu.cn",
        "ecit.edu.cn",
        "ecupl.edu.cn",
        "ecut.edu.cn",
        "educn.ac.cn",
        "email.cugb.edu.cn",
        "email.ncu.edu.cn",
        "email.szu.edu.cn",
        "email.zzuli.edu.cn",
        "emlmuak.cn",
        "fafu.edu.cn",
        "fjys.edu.cn",
        "gduf.edu.cn",
        "gnnu.edu.cn",
        "gydx.ac.cn",
        "gzhtcm.edu.cn",
        "gzhu.edu.cn",
        "hainu.edu.cn",
        "hait.edu.cn",
        "haust.edu.cn",
        "hbdx.ac.cn",
        "hbtcm.edu.cn",
        "hbu.edu.cn",
        "hbucm.edu.cn",
        "hbut.edu.cn",
        "henu.edu.cn",
        "hhu.edu.cn",
        "hndx.ac.cn",
        "hnit.ed.cn",
        "hqu.edu.cn",
        "hrbeu.edu.cn",
        "hsc.pku.edu.cn",
        "huanghuai.edu.cn",
        "huas.edu.cn",
        "hubu.edu.cn",
        "hufe.edu.cn",
        "hunnu.edu.cn",
        "huse.edu.cn",
        "hust.edu.cn",
        "hznu.edu.cn",
        "ict.ac.cn",
        "idea.edu.cn",
        "imnu.edu.cn",
        "ipdbuu.com.cn",
        "itu.ac.tzbjtu.edu.cn",
        "jiangnan.edu.cn",
        "jlu.edu.cn",
        "jmu.edu.cn",
        "jnu.edu.cn",
        "jou.edu.cn",
        "jsahvc.edu.cn",
        "jsust.edu.cn",
        "jsut.edu.cn",
        "kfu.edu.cn",
        "kjdx.ac.cn",
        "kmust.edu.cn",
        "kust.edu.cn",
        "lcu.edu.cn",
        "lfnu.edu.cn",
        "lhmc.edu.cn",
        "lixin.edu.cn",
        "lzjtu.edu.cn",
        "lzpcc.edu.cn",
        "lzu.edu.cn",
        "m.fafu.edu.cn",
        "m.ldu.edu.cn",
        "mail.bnuz.edu.cn",
        "mail.ccmu.edu.cn",
        "mail.chzu.edu.cn",
        "mail.dhu.edu.cn",
        "mail.fmftsk.cn",
        "mail.hbut.edu.cn",
        "mail.imu.edu.cn",
        "mail.nwpu.edu.cn",
        "mail.scuec.edu.cn",
        "mail.sustc.edu.cn",
        "mail.uic.edu.cn",
        "mail.xjtu.edu.cn",
        "mail2.gdut.edu.cn",
        "mails.cust.edu.cn",
        "mails.hufe.edu.cn",
        "mails.imnu.edu.cn",
        "mails.jlu.edu.cn",
        "mails.mmnnsf.cn",
        "mails.qust.edu.cn",
        "mails.zstu.edu.cn",
        "masu.edu.cn",
        "muc.edu.cn",
        "nankai.edu.cn",
        "nbu.edu.cn",
        "ncepu.cn",
        "ncepu.edu.cn",
        "ncist.edu.cn",
        "ncwu.edu.cn",
        "nenu.edu.cn",
        "nit.edu.cn",
        "njau.edu.cn",
        "njit.edu.cn",
        "njnu.edu.cn",
        "njtc.edu.cn",
        "njtech.edu.cn",
        "nkdx.ac.cn",
        "nnu.edu.cn",
        "northfacing.ac.cn",
        "nuc.edu.cn",
        "nwafu.edu.cn",
        "nwnu.edu.cn",
        "nxmu.edu.cn",
        "nyist.edu.cn",
        "ouc.edu.cn",
        "pdsu.edu.cn",
        "pycharm.ac.cn",
        "qhnu.edu.cn",
        "qust.edu.cn",
        "qvtu.edu.cn",
        "qzit.edu.cn",
        "ruc.edu.cn",
        "s.hlju.edu.cn",
        "s.slcupc.edu.cn",
        "s.upc.edu.cn",
        "scnu.edu.cn",
        "scu.edu.cn",
        "scuec.edu.cn",
        "scun.edu.cn",
        "sdju.edu.cn",
        "sdnu.edu.cn",
        "sdufe.edu.cn",
        "sdust.edu.cn",
        "sdut.edu.cn",
        "sem.tsinghua.edu.cn",
        "sfu.edu.cn",
        "shafc.edu.cn",
        "shisu.edu.cn",
        "shmtu.edu.cn",
        "sicau.edu.cn",
        "siit.edu.cn",
        "sit.edu.cn",
        "smail.cczu.edu.cn",
        "smail.xtu.edu.cn",
        "snnu.edu.cn",
        "sqxy.edu.cn",
        "st.btbu.edu.cn",
        "st.gxu.edu.cn",
        "st.nuc.edu.cn",
        "st.sandau.edu.cn",
        "st.usst.edu.cn",
        "std.uestc.edu.cn",
        "stu.ahu.edu.cn",
        "stu.bzuu.edu.cn",
        "stu.ccit.edu.cn",
        "stu.ccsu.edu.cn",
        "stu.cdut.edu.cn",
        "stu.cpu.edu.cn",
        "stu.csust.edu.cn",
        "stu.cuz.edu.cn",
        "stu.fafu.edu.cn",
        "stu.fjmu.edu.cn",
        "stu.fynu.edu.cn",
        "stu.gzucm.edu.cn",
        "stu.hafu.edu.cn",
        "stu.haust.edu.cn",
        "stu.haut.edu.cn",
        "stu.hcnu.edu.cn",
        "stu.hebmu.edu.cn",
        "stu.hnie.edu.cn",
        "stu.hnucm.edu.cn",
        "stu.hrbust.edu.cn",
        "stu.hubu.edu.cn",
        "stu.huel.edu.cn",
        "stu.jhun.edu.cn",
        "stu.jiangnan.edu.cn",
        "stu.just.edu.cn",
        "stu.kust.edu.cn",
        "stu.mdit.edu.cn",
        "stu.mju.edu.cn",
        "stu.nchu.edu.cn",
        "stu.ncwu.edu.cn",
        "stu.nmu.edu.cn",
        "stu.nun.edu.cn",
        "stu.nwupl.edu.cn",
        "stu.peihua.edu.cn",
        "stu.qlu.edu.cn",
        "stu.scu.edu.cn",
        "stu.sdnu.edu.cn",
        "stu.shzu.edu.cn",
        "stu.sicnu.edu.cn",
        "stu.sqmc.edu.cn",
        "stu.sqxy.edu.cn",
        "stu.sumhs.edu.cn",
        "stu.suse.edu.cn",
        "stu.swun.edu.cn",
        "stu.sxau.edu.cn",
        "stu.sxit.edu.cn",
        "stu.tjcu.edu.cn",
        "stu.ujn.edu.cn",
        "stu.usc.edu.cn",
        "stu.wxic.edu.cn",
        "stu.xhsysu.edu.cn",
        "stu.xhu.edu.cn",
        "stu.xidian.edu.cn",
        "stu.xjtu.edu.cn",
        "stu.yznu.cn",
        "stu.zjhu.edu.cn",
        "stu.zuel.edu.cn",
        "stud.acdt.edu.cn",
        "student.cumtb.edu.cn",
        "student.cup.edu.cn",
        "stumail.hbu.edu.cn",
        "stumail.nwu.edu.cn",
        "stumail.sdut.edu.cn",
        "suda.edu.cn",
        "sues.edu.cn",
        "sufe.edu.cn",
        "sust.edu.cn",
        "swpu.edu.cn",
        "swu.edu.cn",
        "sxlg.ac.cn",
        "sziit.edu.cn",
        "t.odmail.cn",
        "tiangong.edu.cn",
        "tijmu.edu.cn",
        "tjdx.ac.cn",
        "tju.edu.cn",
        "tmu.edu.cn",
        "tongji.edu.cn",
        "tralu.hdu.edu.cn",
        "tzc.edu.cn",
        "uibe.edu.cn",
        "ujs.edu.cn",
        "usst.edu.cn",
        "ustb.edu.cn",
        "ustc.edu.cn",
        "vip.hnist.edu.cn",
        "vjrq.cn",
        "wfl.sh.edu.cn",
        "whit.edu.cn",
        "whu.edu.cn",
        "wust.edu.cn",
        "wxic.edu.cn",
        "wxit.edu.cn",
        "wzu.edu.cn",
        "wzut.edu.cn",
        "x.pycharm.ac.cn",
        "xauat.edu.cn",
        "xbdx.ac.cn",
        "xinfu.ac.cn",
        "xiyou.edu.cn",
        "xjdx.ac.cn",
        "xmu.edu.cn",
        "xs.hnit.edu.cn",
        "xupt.edu.cn",
        "xy.dlpu.edu.cn",
        "xy.hbuas.edu.cn",
        "xzyz.edu.cn",
        "yangtzeu.edu.cn",
        "yznu.edu.cn",
        "yzpc.edu.cn",
        "zcmu.edu.cn",
        "zjff.edu.cn",
        "zjgsu.edu.cn",
        "zjnu.edu.cn",
        "zjsru.edu.cn",
        "zju.edu.cn",
        "zknu.edu.cn",
        "zucc.edu.cn",
        "zuel.edu.cn",
        "zut.edu.cn",
        "alum.utoronto.ca",
        "alumni.albany.edu",
        "alumni.anu.edu.au",
        "alumni.bham.ac.uk",
        "alumni.cmu.edu",
        "alum.cs.umass.edu",
        "alumni.conncoll.edu",
        "alumni.edu.vn",
        "alumni.gsb.stanford.edu",
        "alumni.fh-campuswien.ac.at",
        "alumni.habib.edu.pk",
        "alumni.imperial.ac.uk",
        "alumni.iu.edu",
        "alumni.manchester.ac.uk",
        "alumni.montclair.edu",
        "alumni.msoe.edu",
        "alumni.ncyu.edu.tw",
        "alumni.nottingham.ac.uk",
        "alumni.npust.edu.tw",
        "alumni.sjtu.edu.cn",
        "alumni.ssst.edu.ba",
        "alumni.stanford.edu",
        "alumni.stu.edu.cn",
        "alumni.tku.edu.tw",
        "alumni.ubc.ca",
        "alumni.unc.edu",
        "alumni.unca.edu",
        "alumni.uniba.it",
        "alumni.upenn.edu",
        "alumni.vcu.edu",
        "alumni.virginia.edu",
        "alumni365.dmu.ac.uk",
        "alumni.undip.ac.id",
        "office.undip.ac.id",
        "mdu.edu.rs",
        "nfs.edu.rs",
        "aaedu.edu.pl",
        "k.tut.edu.pl",
        "lxiv.secondary.edu.pl",
        "wstkt.edu.pl",
        "mfk.app",
        "aof.anadolu.edu.tr",
        "art.edu.lv",
        "m.uno.edu.gr",
        "univ-barackobama.com",
        "hnie.edu.cn",
        "student.miyazaki-c.ed.jp",
        "alumni.berklee.edu",
        "alumni.cuhk.net",
        "uagraria.edu.ec",
        "connect.hku.hk",
        "graduate.hku.hk",
        "ljb.edu.hk",
        "student.isf.edu.hk",
        "365.ntcu.edu.tw",
        "ad1.nutc.edu.tw",
        "alum.ccu.edu.tw",
        "apps.ntpc.edu.tw",
        "cc.ntut.edu.tw",
        "ccu.edu.tw",
        "chc.edu.tw",
        "cloud.isu.edu.tw",
        "go.edu.tw",
        "hc.edu.tw",
        "hcc.edu.tw",
        "just.edu.tw",
        "live.kmu.edu.tw",
        "mail.aeust.edu.tw",
        "mail.edu.tw",
        "mail.ncyu.edu.tw",
        "mail.ntust.edu.tw",
        "mail.ypu.edu.tw",
        "mail4u.tnu.edu.tw",
        "ms.edu.tw",
        "ms1.mcu.edu.tw",
        "nccu.edu.tw",
        "ncku.edu.tw",
        "ntnu.edu.tw",
        "o365.cyut.edu.tw",
        "o365st.cycu.edu.tw",
        "off365.ncku.edu.tw",
        "office.stust.edu.tw",
        "office365.ncu.edu.tw",
        "office365.nkust.edu.tw",
        "profundity.com.tw",
        "tp.edu.tw",
        "tut.edu.tw",
        "ulive.pccu.edu.tw",
        "ydu.edu.tw",
        "tusur.ru",
        "k.kkk.ac.nz",
        "unswalumni.com",
        "ayushkids.edu.in",
        "usx.edu.cn",
        "meduca.edu.pa",
        "social.helwan.edu.eg",
        "alumni.uci.edu",
        "mail.rmutt.ac.th",
        "lsts.edu.vn",
        "ucass.edu.cn",
        "m.stc.edu.hk",
        "est.domingocomin.edu.ec",
        "sxdtdx.edu.cn",
        "alu.gxu.edu.cn",
        "xhu.edu.cn",
        "stu.xaau.edu.cn",
        "sol-du.ac.in",
        "sesgs.sckr.si",
        "agr.dmu.edu.eg",
        "ahyz.edu.cn",
        "lvu.edu.vn",
        "npcmstsfc.com",
        "alumni.stcl.edu",
        "csa.concordia.ab.ca",
        "metodista.br",
        "ga",
        "gagpk.ru",
        "app",
        "365.stucom.com",
        "shct.edu.om",
        "mapscollege.edu.mv",
        "tkc.edu",
        "itsqmet.edu.ec",
        "st.tvu.edu.vn",
        "jpu.edu.jo",
        "365.cute.edu.tw",
        "stu.bayburt.edu.tr",
        "alumni.rice.edu",
        "cet.edu",
        "innopolis.university",
        "shu.edu.tw",
        "mkhb.moe.gov.sa",
        "horizoncc.co.uk",
        "p4ed.com",
        "cantho.edu.vn",
        "caugiay.edu.vn",
        "umak.edu.ph",
        "normaljujuy.edu.ar",
        "posgrado.unp.edu.pe",
        "presbiteriano.g12.br",
        "psu.palawan.edu.ph",
        "kellerisd.com",
        "office.knu.ac.kr",
        "azeri.edu.az",
        "mdis.uz",
        "ms.srru.ac.th",
        "icprjc.edu",
        "student.guhsdaz.org",
        "o365.prz.edu.pl",
        "ind.suezuni.edu.eg",
        "busan.ac.kr",
        "bnu.kr",
        "mtu.kr",
        "student.cgcc.edu",
        "atk.com.kg",
        "hyejeon.ac.kr",
        "gcu.ac.kr",
        "ube.edu.ec",
        "swmich.edu",
        "student.ssc.edu",
        "wcstudents.wc.edu",
        "my.lsco.edu",
        "student.alvincollege.edu",
        "stu.howardcollege.edu",
        "students.panola.edu",
        "stu.siva.edu.cn",
        "myregent.ac.za",
        "stu.cqupt.edu.cn",
        "mails.cqjtu.edu.cn",
        "alumni.cuhk.net",
        "m365.nycu.edu.tw",
        "stu.cqupt.edu.cn",
        "stu.ahbvc.edu.cn",
        "o365.nuu.edu.tw",
        "o365.ntou.edu.tw",
        "chntheatre.edu.cn",
        "email.cufe.edu.cn",
        "alumni.bentley.edu",
        "altoona.k12.wi.us",
        "wauwatosa.k12.wi.us",
        "as.edu.sa",
        "lanesville.k12.in.us",
        "student.umc.edu.dz",
        "edu.menofia.edu.eg",
        "mail.dusit.ac.th",
        "stu.scott.kyschools.us",
        "cqwu.edu.cn",
        "cumt.edu.cn",
        "hncj.edu.cn",
        "huuc.edu.cn",
        "mastc.edu.cn",
        "njts.edu.cn",
        "sspu.edu.cn",
        "stu.hunau.edu.cn",
        "stu.nsu.edu.cn",
        "stu.sdp.edu.cn",
        "stumail.sztu.edu.cn",
        "szjm.edu.cn",
        "cc.ncu.edu.tw",
        "live.fy.edu.tw",
        "mail.ntou.edu.tw",
        "scu.edu.tw",
        "learner.hkuspace.hku.hk",
        "student.haywood.k12.nc.us",
        "hendrix.edu",
        "cantho.edu.vn",
        "2trom.com",
        "bannsai.com",
        "office.ptu.ac.kr",
        "sgs.edu.co",
        "uettaxila.edu.pk",
        "dent.tanta.edu.eg",
        "huttoisd.org",
        "skhhcps.edu.hk",
        "ae1odemira.edu.pt",
        "uno-r.edu.ph",
        "sonoranschools.org",
        "hanma.kr",
        "alumni.ied.edu",
        "office.yonam.ac.kr",
        "alu.xpu.edu.cn",
        "zjut.edu.cn",
        "o365.msu.ac.th",
        "g.seoultech.ac.kr",
        "sisu.edu.cn",
        "gatewaycollege.lk",
        "kalama.k12.wa.us",
        "stu.nthurston.k12.wa.us",
        "ipe.buet.ac.bd",
        "caa.columbia.edu",
        "itc.edu.kh",
        "o365.lhu.edu.tw",
        "stu.cute.edu.tw",
        "edu.kfs.edu.eg",
        "365.ghu.ac.kr",
        "m365.nycu.edu.tw",
        "languagesystems.edu",
        "sagrado.edu",
        "office.kaya.ac.kr",
        "bh.knu.ac.kr",
        "smpblbekasi.sch.id",
        "aroy.edu.pl",
        "ztbu.edu.cn",
        "gs.anyang.ac.kr",
        "uvmx.onmicrosoft.com",
        "mails.jlau.edu.cn",
        "aip.edu.pa",
        "monaco.edu",
        "ds.ac.kr",
        "riceindia.org",
        "tarbia.tn",
        "alum.mit.edu",
        "phystech.edu",
        "mail.sbu.ac.ir",
        "sbu.ac.ir",
        "sharif.ir",
        "kkk.ac.nz",
        "kkk.school.nz",
        "mhzayt.online",
        "twzhhq.online",
        "otp39.ru",
        "sstu-edu.ru",
        "udsu.ru",
        "kbtut.tj",
        "mail.0du.win",
        "90888.cc",
        "alum.urmc.rochester.edu",
        "fast.edu.pl",
        "hcdsol.tech",
        "aik.edu.kg",
        "tcdsb.ca",
        "emds.edu.pt",
        "australia.edu",
        "america.edu",
        "cpu.edu.kg",
        "neu.edu.kg",
        "tss.edu.kg",
        "micschools.com",
        "id.lv",
        "cmtc.ac.th",
        "aaks.dk",
        "kana.edu.kg",
        "h.edu.kg",
        "live",
        "oshtu.me",
        "oshsu.email",
        "kuosat.tech",
        "acu.edu.kg",
        "ink",
        "upscmd.tech",
        "poly.edu.rs",
        "kun.edu.kg",
        "my.csun.edu",
        "ust.edu.kg",
        "kousat.org",
        "kousat.tech",
        "uest.edu.kg",
        "aynu.me",
        "top",
        "oshtu.edu.kg",
        "oshsu.me",
        "osdri.me",
        "senacminas.edu.br",
        "uphsl.edu.ph",
        "batmu.edu.kg",
        "knau.edu.kg",
        "tss.edu.kg",
        "aa.edu.kg",
        "pccinternational.org",
        "c5.hk",
        "alumni.ncu.edu.tw",
        "smkmuhkarangmojo.com",
        "aimu.edu.kg",
        "poornima.ac.in",
        "mail.centroeducativoinvironda.edu.do",
        "theflightsims.edu.pl",
        "stu.biticrak.ae",
        "hut.edu.pa",
        "jasukg.me",
        "kbtu.me",
        "osdri.me",
        "cedara.ac.za",
        "campus-numerique-in-the-alps.com",
        "escoladotrabalhador40.com.br",
        "chosun-g.gen.hs.kr",
        "aryacs.edu.in",
        "hs.taquangbuu-bk.edu.vn",
        "aue.ae",
        "cug.ac.in",
        "live.hcu.ac.th",
        "outlook.up.edu.ph",
        "dongeui.ms.kr",
        "duksung.kr",
        "stu.owosso.k12.mi.us",
        "infochat.edu.pl",
        "skei.edu.in",
        "fzs.unsa.ba",
        "scpc.edu.in",
        "due.edu.vn",
        "aschool.edu.vn",
        "uva.br",
        "alumno.cch.unam.mx",
        "edisonschools.edu.vn",
        "nuc.edu.kg",
        "fzs.unsa.ba",
        "outlook.rmutr.ac.th",
        "kr.ac",
        "in.ac",
        "ali.edu.pl",
        "mu.ac.kr",
        "alazhar.edu.ps",
        "mf.unsa.ba",
        "apps.nmims.edu.in",
        "universityofmanagement.sn",
        "sbjacksonville.com",
        "foundationuniversity.com",
        "students.sdcsc.k12.in.us",
        "xzc.edu.pl",
        "ea.ufpa.br",
        "iced.ufpa.br",
        "stu.sxpi.edu.cn",
        "mus.edu.rs",
        "ipp.cl",
        "bkc.edu.vn",
        "hungyen.itrithuc.vn",
        "ntpc.ufpa.br",
        "ics.ufpa.br",
        "chiba-c.ed.jp",
        "estudiante.usat.edu.pe",
        "mc-class.gr",
        "mtmcollege.in",
        "rsat.org.uk",
        "altinbas.edu.tr",
        "mu.ac.kr",
        "stumail.zcs.k12.in.us",
        "dusit.ac.th",
        "nyfbox.hu",
        "suezuni.edu.eg",
        "o365.rmutp.ac.th",
        "wmoenglishacademy.co.in",
        "actonlascruces.com",
        "csiriicb.in",
        "essat-gabes-elearning.com",
        "gmiv.onl",
        "hallgato.uni-szie.hu",
        "hiete.hu",
        "intech-so.fr",
        "intech-sud.fr",
        "itcalumnos.cl",
        "kvsroguwahati.in",
        "spbgut.com",
        "southern-bd.info",
        "supco-montpellier.fr",
        "vsavm.com",
        "bdu.edu.et",
        "alu.hit.edu.cn",
        "ilford-school.co.uk",
        "edu.ac.kr",
        "thc.tu.edu.np",
        "man10jakartabarat.sch.id",
        "fpz.hr",
        "vicaria.edu.ar",
        "uaconcagua.cl",
        "kojimachi.ed.jp",
        "chiba-c.ed.jp",
        "student.miyazaki-c.ed.jp",
        "huiu.edu.vn",
    ]
)

TLDS = frozenset(
    [
        "ac.be",
        "ac.gg",
        "ac.gn",
        "ac.il",
        "ac.in",
        "ac.jp",
        "ac.me",
        "ac.ni",
        "ac.om",
        "ac.pg",
        "ac.pr",
        "ac.rw",
        "ac.sz",
        "ac.yu",
        "ac.za",
        "ac.zm",
        "ed.ao",
        "ed.cr",
        "ed.jp",
        "edu.al",
        "edu.az",
        "edu.bb",
        "edu.bh",
        "edu.bs",
        "edu.bz",
        "edu.ck",
        "edu.cn",
        "edu.dz",
        "edu.ee",
        "edu.er",
        "edu.hk",
        "edu.in",
        "edu.kz",
        "edu.lr",
        "edu.ly",
        "edu.mg",
        "edu.ml",
        "edu.mo",
        "edu.mw",
        "edu.ni",
        "edu.pa",
        "edu.ph",
        "edu.pr",
        "edu.pt",
        "edu.pw",
        "edu.qa",
        "edu.sc",
        "edu.sd",
        "edu.sh",
        "edu.sl",
        "edu.sy",
        "edu.ws",
        "edu.ye",
        "edu.zm",
        "es.kr",
        "g12.br",
        "hs.kr",
        "ms.kr",
        "sc.kr",
        "sch.ae",
        "sch.gg",
        "sch.je",
        "sch.jo",
        "sch.lk",
        "sch.ly",
        "sch.my",
        "sch.om",
        "sch.ps",
        "sch.sa",
        "school.za",
        "ort.org.il",
        "k12.al.us",
        "k12.ak.us",
        "k12.az.us",
        "k12.ar.us",
        "k12.ct.us",
        "k12.de.us",
        "k12.fl.us",
        "k12.ga.us",
        "k12.hi.us",
        "k12.il.us",
        "k12.in.us",
        "k12.ia.us",
        "k12.ky.us",
        "k12.la.us",
        "k12.me.us",
        "k12.md.us",
        "k12.ma.us",
        "k12.mi.us",
        "k12.mn.us",
        "k12.ms.us",
        "k12.mo.us",
        "k12.mt.us",
        "k12.nh.us",
        "k12.nj.us",
        "k12.ny.us",
        "k12.nc.us",
        "k12.oh.us",
        "k12.ok.us",
        "k12.or.us",
        "k12.pa.us",
        "k12.sc.us",
        "k12.sd.us",
        "k12.tn.us",
        "k12.ut.us",
        "k12.vt.us",
        "k12.va.us",
        "k12.wv.us",
        "k12.wi.us",
        "k12.wy.us",
    ]
)


class SWOT:
    tlds = None
    stoplist = None

    @staticmethod
    def read_list(resource: str) -> Optional[Set[str]]:
        base_path = f'{pathlib.Path(os.path.abspath(os.path.dirname(__file__)))}/domains'
        file_path = f'{base_path}/{resource}'
        if os.path.exists(file_path):
            with open(file_path, "r") as f:
                return set(line.strip() for line in f)
        return None

def is_blacklisted(parts: List[str]) -> bool:
    return check_parts(BLACKLIST, parts)

def is_under_tld(parts: List[str]) -> bool:
    return check_parts(TLDS, parts)

def is_academic(email: str) -> bool:
    parts = domain_parts(email)
    return not is_blacklisted(parts) and (is_under_tld(parts) or bool(find_school_names(email)))


def find_school_names(email_or_domain: str) -> List[str]:
    return find_school_names_from_parts(domain_parts(email_or_domain))

def find_school_names_from_parts(parts: List[str]) -> List[str]:
    resource_path = ""
    for part in parts:
        resource_path = f"{resource_path}/{part}"
        school = SWOT.read_list(f'{resource_path.lstrip("/")}.txt')
        if school is not None:
            return list(school)
    return []


def domain_parts(email_or_domain: str) -> List[str]:
    cleaned = email_or_domain.strip().lower()
    after_at = cleaned.split("@")[-1]
    after_protocol = after_at.split("://")[-1]
    before_port = after_protocol.split(":")[0]
    return before_port.split(".")[::-1]  # reversed



def check_parts(search: FrozenSet[str], parts: List[str]) -> bool:
    """Check if domain information is held in any specific set."""
    slow_join = ""
    for part in parts:
        slow_join = part + slow_join
        if slow_join in search:
            return True
        slow_join = "." + slow_join

    return False
