"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Manifest = exports.VERSION_MISMATCH = void 0;
const fs = require("fs");
const jsonschema = require("jsonschema");
const semver = require("semver");
const assembly = require("./cloud-assembly");
// this prefix is used by the CLI to identify this specific error.
// in which case we want to instruct the user to upgrade his CLI.
// see exec.ts#createAssembly
exports.VERSION_MISMATCH = "Cloud assembly schema version mismatch";
// tslint:disable: no-var-requires
// eslint-disable-next-line @typescript-eslint/no-require-imports
const ASSEMBLY_SCHEMA = require("../schema/cloud-assembly.schema.json");
/**
 * Version is shared for both manifests
 */
// eslint-disable-next-line @typescript-eslint/no-require-imports
const SCHEMA_VERSION = require("../schema/cloud-assembly.version.json").version;
// tslint:enable: no-var-requires
/**
 * Protocol utility class.
 */
class Manifest {
    /**
     * Validates and saves the cloud assembly manifest to file.
     *
     * @param manifest - manifest.
     * @param filePath - output file path.
     */
    static saveAssemblyManifest(manifest, filePath) {
        Manifest.saveManifest(manifest, filePath, ASSEMBLY_SCHEMA);
    }
    /**
     * Load and validates the cloud assembly manifest from file.
     *
     * @param filePath - path to the manifest file.
     */
    static loadAssemblyManifest(filePath) {
        return Manifest.loadManifest(filePath, ASSEMBLY_SCHEMA, (obj) => Manifest.patchStackTags(obj));
    }
    /**
     * Fetch the current schema version number.
     */
    static version() {
        return SCHEMA_VERSION;
    }
    /**
     * Deprecated
     * @deprecated use `saveAssemblyManifest()`
     */
    static save(manifest, filePath) {
        return this.saveAssemblyManifest(manifest, filePath);
    }
    /**
     * Deprecated
     * @deprecated use `loadAssemblyManifest()`
     */
    static load(filePath) {
        return this.loadAssemblyManifest(filePath);
    }
    static validate(manifest, schema) {
        function parseVersion(version) {
            const ver = semver.valid(version);
            if (!ver) {
                throw new Error(`Invalid semver string: "${version}"`);
            }
            return ver;
        }
        const maxSupported = parseVersion(Manifest.version());
        const actual = parseVersion(manifest.version);
        // first validate the version should be accepted.
        if (semver.gt(actual, maxSupported)) {
            // we use a well known error prefix so that the CLI can identify this specific error
            // and print some more context to the user.
            throw new Error(`${exports.VERSION_MISMATCH}: Maximum schema version supported is ${maxSupported}, but found ${actual}`);
        }
        // now validate the format is good.
        const validator = new jsonschema.Validator();
        const result = validator.validate(manifest, schema, {
            // does exist but is not in the TypeScript definitions
            nestedErrors: true,
            allowUnknownAttributes: false,
        });
        if (!result.valid) {
            throw new Error(`Invalid assembly manifest:\n${result}`);
        }
    }
    static saveManifest(manifest, filePath, schema) {
        const withVersion = { ...manifest, version: Manifest.version() };
        Manifest.validate(withVersion, schema);
        fs.writeFileSync(filePath, JSON.stringify(withVersion, undefined, 2));
    }
    static loadManifest(filePath, schema, preprocess) {
        let obj = JSON.parse(fs.readFileSync(filePath, { encoding: "utf-8" }));
        if (preprocess) {
            obj = preprocess(obj);
        }
        Manifest.validate(obj, schema);
        return obj;
    }
    /**
     * This requires some explaining...
     *
     * We previously used `{ Key, Value }` for the object that represents a stack tag. (Notice the casing)
     *
     * When that object moved to this package, it had to be JSII compliant, which meant the property
     * names must be `camelCased`, and not `PascalCased`. This meant it no longer matches the structure in the `manifest.json` file.
     * In order to support current manifest files, we have to translate the `PascalCased` representation to the new `camelCased` one.
     *
     * Note that the serialization itself still writes `PascalCased` because it relates to how ROS expects it.
     *
     * Ideally, we would start writing the `camelCased` and translate to how ROS expects it when needed. But this requires nasty
     * backwards-compatibility code and it just doesn't seem to be worth the effort.
     */
    static patchStackTags(manifest) {
        for (const artifact of Object.values(manifest.artifacts || [])) {
            if (artifact.type === assembly.ArtifactType.ALIYUN_ROS_STACK) {
                for (const metadataEntries of Object.values(artifact.metadata || [])) {
                    for (const metadataEntry of metadataEntries) {
                        if (metadataEntry.type ===
                            assembly.ArtifactMetadataEntryType.STACK_TAGS &&
                            metadataEntry.data) {
                            const metadataAny = metadataEntry;
                            metadataAny.data = metadataAny.data.map((t) => ({
                                key: t.Key,
                                value: t.Value,
                            }));
                        }
                    }
                }
            }
        }
        return manifest;
    }
    constructor() { }
}
exports.Manifest = Manifest;
//# sourceMappingURL=data:application/json;base64,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