"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const stepfunctions = require("../lib");
describe('Map State', () => {
    test('State Machine With Map State', () => {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const map = new stepfunctions.Map(stack, 'Map State', {
            maxConcurrency: 1,
            itemsPath: stepfunctions.JsonPath.stringAt('$.inputForMap'),
            parameters: {
                foo: 'foo',
                bar: stepfunctions.JsonPath.stringAt('$.bar'),
            },
        });
        map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
        // THEN
        expect(render(map)).toStrictEqual({
            StartAt: 'Map State',
            States: {
                'Map State': {
                    Type: 'Map',
                    End: true,
                    Parameters: {
                        'foo': 'foo',
                        'bar.$': '$.bar',
                    },
                    Iterator: {
                        StartAt: 'Pass State',
                        States: {
                            'Pass State': {
                                Type: 'Pass',
                                End: true,
                            },
                        },
                    },
                    ItemsPath: '$.inputForMap',
                    MaxConcurrency: 1,
                },
            },
        });
    }),
        test('synth is successful', () => {
            const app = createAppWithMap((stack) => {
                const map = new stepfunctions.Map(stack, 'Map State', {
                    maxConcurrency: 1,
                    itemsPath: stepfunctions.JsonPath.stringAt('$.inputForMap'),
                });
                map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
                return map;
            });
            app.synth();
        }),
        test('fails in synthesis if iterator is missing', () => {
            const app = createAppWithMap((stack) => {
                const map = new stepfunctions.Map(stack, 'Map State', {
                    maxConcurrency: 1,
                    itemsPath: stepfunctions.JsonPath.stringAt('$.inputForMap'),
                });
                return map;
            });
            expect(() => app.synth()).toThrow(/Map state must have a non-empty iterator/);
        }),
        test('fails in synthesis when maxConcurrency is a float', () => {
            const app = createAppWithMap((stack) => {
                const map = new stepfunctions.Map(stack, 'Map State', {
                    maxConcurrency: 1.2,
                    itemsPath: stepfunctions.JsonPath.stringAt('$.inputForMap'),
                });
                map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
                return map;
            });
            expect(() => app.synth()).toThrow(/maxConcurrency has to be a positive integer/);
        }),
        test('fails in synthesis when maxConcurrency is a negative integer', () => {
            const app = createAppWithMap((stack) => {
                const map = new stepfunctions.Map(stack, 'Map State', {
                    maxConcurrency: -1,
                    itemsPath: stepfunctions.JsonPath.stringAt('$.inputForMap'),
                });
                map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
                return map;
            });
            expect(() => app.synth()).toThrow(/maxConcurrency has to be a positive integer/);
        }),
        test('fails in synthesis when maxConcurrency is too big to be an integer', () => {
            const app = createAppWithMap((stack) => {
                const map = new stepfunctions.Map(stack, 'Map State', {
                    maxConcurrency: Number.MAX_VALUE,
                    itemsPath: stepfunctions.JsonPath.stringAt('$.inputForMap'),
                });
                map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
                return map;
            });
            expect(() => app.synth()).toThrow(/maxConcurrency has to be a positive integer/);
        }),
        test('isPositiveInteger is false with negative number', () => {
            expect(stepfunctions.isPositiveInteger(-1)).toEqual(false);
        }),
        test('isPositiveInteger is false with decimal number', () => {
            expect(stepfunctions.isPositiveInteger(1.2)).toEqual(false);
        }),
        test('isPositiveInteger is false with a value greater than safe integer', () => {
            const valueToTest = Number.MAX_SAFE_INTEGER + 1;
            expect(stepfunctions.isPositiveInteger(valueToTest)).toEqual(false);
        }),
        test('isPositiveInteger is true with 0', () => {
            expect(stepfunctions.isPositiveInteger(0)).toEqual(true);
        }),
        test('isPositiveInteger is true with 10', () => {
            expect(stepfunctions.isPositiveInteger(10)).toEqual(true);
        }),
        test('isPositiveInteger is true with max integer value', () => {
            expect(stepfunctions.isPositiveInteger(Number.MAX_SAFE_INTEGER)).toEqual(true);
        });
});
function render(sm) {
    return new cdk.Stack().resolve(new stepfunctions.StateGraph(sm.startState, 'Test Graph').toGraphJson());
}
function createAppWithMap(mapFactory) {
    const app = new cdk.App();
    const stack = new cdk.Stack(app, 'my-stack');
    const map = mapFactory(stack);
    new stepfunctions.StateGraph(map, 'Test Graph');
    return app;
}
//# sourceMappingURL=data:application/json;base64,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