"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const lib_1 = require("../lib");
describe('Fields', () => {
    const jsonPathValidationErrorMsg = /exactly '\$', '\$\$', start with '\$.', start with '\$\$.' or start with '\$\['/;
    test('deep replace correctly handles fields in arrays', () => {
        expect(lib_1.FieldUtils.renderObject({
            unknown: undefined,
            bool: true,
            literal: 'literal',
            field: lib_1.JsonPath.stringAt('$.stringField'),
            listField: lib_1.JsonPath.listAt('$.listField'),
            deep: [
                'literal',
                {
                    deepField: lib_1.JsonPath.numberAt('$.numField'),
                },
            ],
        })).toStrictEqual({
            'bool': true,
            'literal': 'literal',
            'field.$': '$.stringField',
            'listField.$': '$.listField',
            'deep': [
                'literal',
                {
                    'deepField.$': '$.numField',
                },
            ],
        });
    }),
        test('exercise contextpaths', () => {
            expect(lib_1.FieldUtils.renderObject({
                str: lib_1.JsonPath.stringAt('$$.Execution.StartTime'),
                count: lib_1.JsonPath.numberAt('$$.State.RetryCount'),
                token: lib_1.JsonPath.taskToken,
                entire: lib_1.JsonPath.entireContext,
            })).toStrictEqual({
                'str.$': '$$.Execution.StartTime',
                'count.$': '$$.State.RetryCount',
                'token.$': '$$.Task.Token',
                'entire.$': '$$',
            });
        }),
        test('find all referenced paths', () => {
            expect(lib_1.FieldUtils.findReferencedPaths({
                bool: false,
                literal: 'literal',
                field: lib_1.JsonPath.stringAt('$.stringField'),
                listField: lib_1.JsonPath.listAt('$.listField'),
                deep: [
                    'literal',
                    {
                        field: lib_1.JsonPath.stringAt('$.stringField'),
                        deepField: lib_1.JsonPath.numberAt('$.numField'),
                    },
                ],
            })).toStrictEqual(['$.listField', '$.numField', '$.stringField']);
        }),
        test('cannot have JsonPath fields in arrays', () => {
            expect(() => lib_1.FieldUtils.renderObject({
                deep: [lib_1.JsonPath.stringAt('$.hello')],
            })).toThrowError(/Cannot use JsonPath fields in an array/);
        }),
        test('datafield path must be correct', () => {
            expect(lib_1.JsonPath.stringAt('$')).toBeDefined();
            expect(() => lib_1.JsonPath.stringAt('$hello')).toThrowError(jsonPathValidationErrorMsg);
            expect(() => lib_1.JsonPath.stringAt('hello')).toThrowError(jsonPathValidationErrorMsg);
        }),
        test('context path must be correct', () => {
            expect(lib_1.JsonPath.stringAt('$$')).toBeDefined();
            expect(() => lib_1.JsonPath.stringAt('$$hello')).toThrowError(jsonPathValidationErrorMsg);
            expect(() => lib_1.JsonPath.stringAt('hello')).toThrowError(jsonPathValidationErrorMsg);
        }),
        test('datafield path with array must be correct', () => {
            expect(lib_1.JsonPath.stringAt('$[0]')).toBeDefined();
            expect(lib_1.JsonPath.stringAt("$['abc']")).toBeDefined();
        }),
        test('test contains task token', () => {
            expect(true).toEqual(lib_1.FieldUtils.containsTaskToken({
                field: lib_1.JsonPath.taskToken,
            }));
            expect(true).toEqual(lib_1.FieldUtils.containsTaskToken({
                field: lib_1.JsonPath.stringAt('$$.Task'),
            }));
            expect(true).toEqual(lib_1.FieldUtils.containsTaskToken({
                field: lib_1.JsonPath.entireContext,
            }));
            expect(false).toEqual(lib_1.FieldUtils.containsTaskToken({
                oops: 'not here',
            }));
            expect(false).toEqual(lib_1.FieldUtils.containsTaskToken({
                oops: lib_1.JsonPath.stringAt('$$.Execution.StartTime'),
            }));
        }),
        test('arbitrary JSONPath fields are not replaced', () => {
            expect(lib_1.FieldUtils.renderObject({
                field: '$.content',
            })).toStrictEqual({
                field: '$.content',
            });
        }),
        test('fields cannot be used somewhere in a string interpolation', () => {
            expect(() => lib_1.FieldUtils.renderObject({
                field: `contains ${lib_1.JsonPath.stringAt('$.hello')}`,
            })).toThrowError(/Field references must be the entire string/);
        });
});
//# sourceMappingURL=data:application/json;base64,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