# -*- coding: utf-8 -*-
"""
Created on Fri Sep  3 13:40:56 2021

@author: ormondt
"""

import os
from pyproj import CRS
from pyproj import Transformer
import pandas as pd
import xarray as xr
import numpy as np
import glob
import datetime 

def nest1(overall, detail, option=None):
    
    # Returns a list with observation point objects
    
    if overall.type.lower() == "delft3dfm":
        if detail.type.lower() == "delft3dfm":
            nest1_delft3dfm_in_delft3dfm(overall, detail)
        elif detail.type.lower() == "sfincs":
            nest1_sfincs_in_delft3dfm(overall, detail)
        elif detail.type.lower() == "beware":
            nest1_beware_in_delft3dfm(overall, detail)
            
    elif overall.type.lower() == "sfincs":
        if detail.type.lower() == "sfincs":
            nest1_sfincs_in_sfincs(overall, detail)
        elif detail.type.lower() == "xbeach":
            nest1_xbeach_in_sfincs(overall, detail)
        elif detail.type.lower() == "beware":
            nest1_beware_in_sfincs(overall, detail)

    elif overall.type.lower() == "hurrywave":
        if detail.type.lower() == "hurrywave":
            nest1_hurrywave_in_hurrywave(overall, detail)
        elif detail.type.lower() == "xbeach":    
            if not option:
                option = "timeseries"
            nest1_xbeach_in_hurrywave(overall, detail, option=option)
        elif detail.type.lower() == "sfincs":    
            nest1_sfincs_in_hurrywave(overall, detail)
        elif detail.type.lower() == "beware":
            nest1_beware_in_hurrywave(overall, detail)

    elif overall.type.lower() == "beware":
        if detail.type.lower() == "sfincs":
            # No need to do anything here. BEWARE output points are fixed
            pass

#        elif detail.type == "delft3dfm":
#            obs = nest1_delft3dfm_in_sfincs(overall, detail)

#    return obs    

def nest2(overall,
          detail,
          boundary_water_level_correction=None,
          output_path=None,
          output_file=None,
	      option=None,
          return_maximum=False,
          bc_file=None):


    if not boundary_water_level_correction:
        # Path of the overall output time series
        boundary_water_level_correction = 0.0
    
    if overall.type.lower() == "delft3dfm":

        if detail.type.lower() == "delft3dfm":
            nest2_delft3dfm_in_delft3dfm(overall,
                                         detail,
                                         output_path,
                                         output_file,
                                         boundary_water_level_correction)

        elif detail.type.lower() == "sfincs":
            nest2_sfincs_in_delft3dfm(overall,
                                      detail,
                                      output_path,
                                      output_file,
                                      boundary_water_level_correction)

        elif detail.type.lower() == "beware":
            nest2_beware_in_delft3dfm(overall,
                                      detail,
                                      output_path,
                                      output_file,
                                      boundary_water_level_correction,
                                      option)
            
    elif overall.type.lower() == "sfincs":

        if detail.type.lower() == "sfincs":
            zs = nest2_sfincs_in_sfincs(overall,
                                        detail,
                                        output_path,
                                        output_file,
                                        boundary_water_level_correction,
                                        return_maximum=return_maximum)
            return zs
        elif detail.type.lower() == "xbeach":
            bc = nest2_xbeach_in_sfincs(overall,
                                        detail,
                                        output_path,
                                        output_file,
                                        boundary_water_level_correction,
                                        return_maximum=return_maximum)
            return bc
        elif detail.type.lower() == "beware":
            nest2_beware_in_sfincs(overall,
                                   detail,
                                   output_path,
                                   output_file,
                                   boundary_water_level_correction,
                                   option)

    elif overall.type.lower() == "hurrywave":

        if detail.type.lower() == "hurrywave":
            nest2_hurrywave_in_hurrywave(overall,
                                   detail,
                                   output_path,
                                   output_file,
                                   bc_file)
            pass
        elif detail.type.lower() == "xbeach":
            bc = nest2_xbeach_in_hurrywave(overall,
                                           detail,
                                           output_path,
                                           output_file,
                                           option,
                                           return_maximum=return_maximum)
            return bc
        elif detail.type.lower() == "sfincs":
            nest2_sfincs_in_hurrywave(overall,
                                      detail,
                                      output_path,
                                      output_file)
        elif detail.type.lower() == "beware":
            nest2_beware_in_hurrywave(overall,
                                      detail,
                                      output_path,
                                      output_file)

    elif overall.type.lower() == "beware":

        if detail.type.lower() == "sfincs":
            nest2_sfincs_in_beware(overall,
                                   detail,
                                   output_path,
                                   output_file,
                                   boundary_water_level_correction,
                                   option)

def nest1_delft3dfm_in_delft3dfm(overall, detail):
    
#    from delft3dfm import ObservationPoint as obspoint
    
    transformer = Transformer.from_crs(detail.crs,
                                       overall.crs,
                                       always_xy=True)
    
    for ind, bnd in enumerate(detail.boundary):        
        for ip, point in enumerate(bnd.point):
            x, y = transformer.transform(point.geometry.x,
                                         point.geometry.y)
            overall.add_observation_point(x, y, detail.name + "_" + point.name)
    
def nest1_sfincs_in_delft3dfm(overall, detail):
    
#    from delft3dfm import ObservationPoint as obspoint
    
    transformer = Transformer.from_crs(detail.crs,
                                       overall.crs,
                                       always_xy=True)
    
    for ind, point in enumerate(detail.flow_boundary_point):

        name = detail.name + "_" + str(ind + 1).zfill(4)
        x, y = transformer.transform(point.geometry.x,
                                     point.geometry.y)
#        obs_list.append(obspoint(x, y, name, crs=overall.crs))
        overall.add_observation_point(x, y, name)

def nest1_beware_in_delft3dfm(overall, detail):
        
    transformer = Transformer.from_crs(detail.crs,
                                       overall.crs,
                                       always_xy=True)
    
    for ind, point in enumerate(detail.flow_boundary_point):

        name = detail.name + "_" + point.name
        x, y = transformer.transform(point.geometry.x,
                                     point.geometry.y)
#        obs_list.append(obspoint(x, y, name, crs=overall.crs))
        overall.add_observation_point(x, y, name)
    
def nest1_sfincs_in_sfincs(overall, detail):
    
#    from sfincs import ObservationPoint as obspoint

    transformer = Transformer.from_crs(detail.crs,
                                       overall.crs,
                                       always_xy=True)
    
    for ind, point in enumerate(detail.flow_boundary_point):

        name = detail.name + "_" + str(ind + 1).zfill(4)
        x, y = transformer.transform(point.geometry.x,
                                     point.geometry.y)
#        obs_list.append(obspoint(x, y, name, crs=overall.crs))
        overall.add_observation_point(x, y, name)

def nest1_xbeach_in_sfincs(overall, detail):
    
    transformer = Transformer.from_crs(detail.crs,
                                       overall.crs,
                                       always_xy=True)
    
    for ind, point in enumerate(detail.flow_boundary_point):

        name = detail.name + "_" + str(ind + 1).zfill(4)
        x, y = transformer.transform(point.geometry.x,
                                     point.geometry.y)
#        obs_list.append(obspoint(x, y, name, crs=overall.crs))
        overall.add_observation_point(x, y, name)

def nest1_beware_in_sfincs(overall, detail):
        
    transformer = Transformer.from_crs(detail.crs,
                                       overall.crs,
                                       always_xy=True)
    
    for ind, point in enumerate(detail.flow_boundary_point):

        name = detail.name + "_" + point.name
        x, y = transformer.transform(point.geometry.x,
                                     point.geometry.y)
#        obs_list.append(obspoint(x, y, name, crs=overall.crs))
        overall.add_observation_point(x, y, name)

def nest1_hurrywave_in_hurrywave(overall, detail):
    
    transformer = Transformer.from_crs(detail.crs,
                                       overall.crs,
                                       always_xy=True)

    for ind, row in detail.boundary_conditions.gdf.iterrows():
        name = detail.name + "_" + str(ind + 1).zfill(4)
        x = row["geometry"].coords[0][0]
        y = row["geometry"].coords[0][1]
        x, y = transformer.transform(x, y)
        overall.observation_points_sp2.add_point(x, y, name)

def nest1_xbeach_in_hurrywave(overall, detail, option="sp2"):
    
    transformer = Transformer.from_crs(detail.crs,
                                       overall.crs,
                                       always_xy=True)
    
    for ind, point in enumerate(detail.wave_boundary_point):

        name = detail.name + "_" + str(ind + 1).zfill(4)
        x, y = transformer.transform(point.geometry.x,
                                     point.geometry.y)
#        obs_list.append(obspoint(x, y, name, crs=overall.crs))
#        if option=="sp2":
        overall.observation_points_sp2.add_point(x, y, name)
#        else:
        overall.observation_points_regular.add_point(x, y, name)

def nest1_sfincs_in_hurrywave(overall, detail):
    
    transformer = Transformer.from_crs(detail.crs,
                                       overall.crs,
                                       always_xy=True)

    for ind, point in enumerate(detail.wave_boundary_point):

        name = detail.name + "_" + str(ind + 1).zfill(4)
        x, y = transformer.transform(point.geometry.x,
                                     point.geometry.y)
        overall.observation_points_regular.add_point(x, y, name)

def nest1_beware_in_hurrywave(overall, detail):
    
    transformer = Transformer.from_crs(detail.crs,
                                       overall.crs,
                                       always_xy=True)

    for ind, point in enumerate(detail.wave_boundary_point):

        
        name = detail.name + "_" + point.name
        x, y = transformer.transform(point.geometry.x,
                                     point.geometry.y)
        overall.observation_points_regular.add_point(x, y, name)

            
def nest2_delft3dfm_in_delft3dfm(overall,
                                 detail,
                                 output_path,
                                 output_file,
                                 boundary_water_level_correction):

    if not output_file:
        # Path of the overall output time series
        output_file = overall.runid + "_his.nc"

#    output_file = os.path.join(output_path, output_file)

    for ind, bnd in enumerate(detail.boundary):        
        point_names = []
        for ip, point in enumerate(bnd.point):
            point_names.append(detail.name + "_" + point.name)
        # Return DataFrame bzs
        bzs = overall.read_timeseries_output(name_list=point_names,
                                             path=output_path,
                                             file_name=output_file)
        ts  = bzs.index
        for ip, point in enumerate(bnd.point):
            point.data = pd.Series(bzs.iloc[:,ip].values, index=ts) + boundary_water_level_correction    
    
def nest2_sfincs_in_delft3dfm(overall,
                              detail,
                              output_path,
                              output_file,
                              boundary_water_level_correction):

    if not output_file:
        # Path of the overall output time series
        output_file = overall.runid + "_his.nc"
    
    point_names = []
    for point in detail.flow_boundary_point:
        point_names.append(detail.name + "_" + point.name)                    
    output_file = os.path.join(output_path, output_file)

    # Return DataFrame bzs
    bzs = overall.read_timeseries_output(name_list=point_names,
                                         path=output_path,
                                         file_name=output_file)

    ts  = bzs.index
    for icol, point in enumerate(detail.flow_boundary_point):
        point.data = pd.Series(bzs.iloc[:,icol].values, index=ts) + boundary_water_level_correction    

def nest2_beware_in_delft3dfm(overall,
                              detail,
                              output_path,
                              output_file,
                              boundary_water_level_correction,
                              option):

    if option == 'flow':
        if not output_file:
            # Path of the overall output time series
            output_file = overall.runid + "_his.nc"
        
        point_names = []
        for point in detail.flow_boundary_point:
            point_names.append(detail.name + "_" + point.name) #reopen                   
            # point_names.append(point.name)                    
        # output_file = os.path.join(output_path, output_file)

        # Return DataFrame bzs
        bzs = overall.read_timeseries_output(name_list=point_names,
                                             path=output_path,
                                             file_name=output_file)
        ts  = bzs.index
        for icol, point in enumerate(detail.flow_boundary_point):
            point.data = pd.Series(bzs.iloc[:,icol].values, index=ts) + boundary_water_level_correction    
            
    if option == 'wave':


        if not output_path:
            # Path of the overall output time series
            output_path = overall.path
            
        if not output_file:
            file_name = glob.glob(os.path.join(output_path, "wavh*"))[1]
        else:
            file_name = os.path.join(output_path, output_file)
        
        # Open netcdf file
        file_name_dfm = glob.glob(os.path.join(output_path, "*_his.nc"))[0]
        ddd = xr.open_dataset(file_name_dfm)
        stations=ddd.station_name.values
        all_stations = []
        for ist, st in enumerate(stations):
            st=str(st.strip())[2:-1] #reopen
            # st=str(st.strip())[2:6]
            all_stations.append(st)

        point_names = []
        for point in detail.wave_boundary_point:
            point_names.append(detail.name + "_" + point.name) #reopen                   
            # point_names.append(point.name)  
        # ireq = []    
        # for ip, point in enumerate(point_names):
        #     for ist,st in enumerate(all_stations):
        #         if point.lower() == st.lower():
        #             ireq.append(ist)            
        #             break

        ddd = xr.open_dataset(file_name)
        times   = ddd.Hsig.time.values

        for ip, point in enumerate(detail.wave_boundary_point):
            for ist,st in enumerate(all_stations):
                ireq = -1
                if point_names[ip] == st.lower():
                    ireq=ist
                    break

            if ireq>-1:
                hm0     = ddd.Hsig.values[:,ireq]
                tp      = ddd.Tm01.values[:,ireq]
                wavdir  = ddd.Dir.values[:,ireq]
                dirspr  = ddd.Dspr.values[:,ireq]
    
                df = pd.DataFrame(index=times)
                df.insert(0,"hm0",hm0)
                df.insert(1,"tp",tp)
                df.insert(2,"wavdir",wavdir)
                df.insert(3,"dirspr",dirspr)
    
                point.data = df



def nest2_sfincs_in_sfincs(overall,
                           detail,
                           output_path,
                           output_file,
                           boundary_water_level_correction,
                           return_maximum=False):

    if not output_path:
        # Path of the overall output time series
        output_path = overall.path
    
        
    if overall.input.outputformat[0:3] == "bin":
        # ascii output        
        if not output_file:
            output_file = "zst.txt"
    else:
        # netcdf        
        if not output_file:
            output_file = "sfincs_his.nc"
    
    point_names = []
    for point in detail.flow_boundary_point:
        point_names.append(detail.name + "_" + point.name)                    
    zstfile = os.path.join(output_path, output_file)

    # Return DataFrame bzs
    bzs = overall.read_timeseries_output(name_list=point_names,
                                         file_name=zstfile)

    ts  = bzs.index
    for icol, point in enumerate(detail.flow_boundary_point):
        point.data = pd.Series(bzs.iloc[:,icol].values, index=ts) + boundary_water_level_correction

    if return_maximum:
        zmax = -999.0
        for icol, point in enumerate(detail.flow_boundary_point):
            zx = point.data.max()
            if zx>zmax:
                zs = point.data
                zmax = zx
        return zs        
                    
    else:    
        return detail.flow_boundary_point



def nest2_xbeach_in_sfincs(overall,
                           detail,
                           output_path,
                           output_file,
                           boundary_water_level_correction,
                           return_maximum=False):

    if not output_path:
        # Path of the overall output time series
        output_path = overall.path
        
    if overall.input.outputformat[0:3] == "bin":
        # ascii output        
        if not output_file:
            output_file = "zst.txt"
    else:
        # netcdf        
        if not output_file:
            output_file = "sfincs_his.nc"
    
    point_names = []
    for point in detail.flow_boundary_point:
        point_names.append(detail.name + "_" + point.name)                    
    zstfile = os.path.join(output_path, output_file)

    # Return DataFrame bzs
    bzs = overall.read_timeseries_output(name_list=point_names,
                                         file_name=zstfile)

    
    # Interpolate on desired format for XBeach forcing
    bzs_resampled = bzs.resample('10min').mean()
    bzs_interpolated = bzs_resampled.interpolate(method='linear')
    bzs_filtered = bzs_interpolated[detail.tref:detail.tstop]
    

    ts  = bzs_filtered.index
    for icol, point in enumerate(detail.flow_boundary_point):
        point.data = pd.Series(bzs_filtered.iloc[:,icol].values, index=ts) + boundary_water_level_correction

    if return_maximum:
        zmax = -999.0
        for icol, point in enumerate(detail.flow_boundary_point):
            zx = point.data.max()
            if zx>zmax:
                zs = point.data
                zmax = zx
        return zs        
                    
    else:    
        return detail.flow_boundary_point

def nest2_beware_in_sfincs(overall,
                           detail,
                           output_path,
                           output_file,
                           boundary_water_level_correction,
                           option):

    if option == 'flow':
        if not output_file:
            # Path of the overall output time series
            output_file = "sfincs_his.nc"
        
        point_names = []
        for point in detail.flow_boundary_point:
            point_names.append(detail.name + "_" + point.name) #reopen                   
            # point_names.append(point.name)                    
        # output_file = os.path.join(output_path, output_file)

        # Return DataFrame bzs
        bzs = overall.read_timeseries_output(name_list=point_names,
                                             file_name=os.path.join(output_path,
                                                                    output_file))

        # Replace -999.0 with zeros. This should not happen, but easy fix for now.
        bzs = bzs.replace(-999.0,0.0)
        for icol, point in enumerate(detail.flow_boundary_point):
            point.data = pd.Series(bzs.iloc[:,icol].values, index=bzs.index) + boundary_water_level_correction    
            

    
def nest2_hurrywave_in_hurrywave(overall,
                                 detail,
                                 output_path,
                                 output_file,
                                 bc_file):
    if not output_path:
        # Path of the overall output time series
        output_path = overall.path
        
    if not output_file:
        output_file = "hurrywave_sp2.nc"

    file_name = os.path.join(output_path, output_file)
    
    detail.boundary_conditions.forcing = "spectra"

    # Open netcdf file
    ddd = xr.open_dataset(file_name)
    stations=ddd.station_name.values
    all_stations = []
    for ist, st in enumerate(stations):
        st=str(st.strip())[2:-1]
        all_stations.append(st)

    point_names = []
    if len(detail.boundary_conditions.gdf)>0:
        for ind, row in detail.boundary_conditions.gdf.iterrows():
        # Find required boundary points        
            point_names.append(detail.name + "_" + row["name"])                    
        
    else:
        point_names = all_stations.copy()
        
    times   = ddd.point_spectrum2d.coords["time"].values
    sigma   = ddd.point_spectrum2d.coords["sigma"].values
    theta   = ddd.point_spectrum2d.coords["theta"].values

    ireq = []    
    for ip, point in enumerate(point_names):
        for ist,st in enumerate(all_stations):
            if point.lower() == st.lower():
                ireq.append(ist)            
                break

    for ind, row in detail.boundary_conditions.gdf.iterrows():

        sp2 = ddd.point_spectrum2d.values[:,ireq[ind],:,:]

        ds = xr.Dataset(
                data_vars = dict(point_spectrum2d=(["time", "theta", "sigma"], sp2)),
                coords    = dict(time=times,
                                 theta=theta,
                                 sigma=sigma)
                )
        detail.boundary_conditions.gdf.loc[ind, "spectra"] = ds.to_array()

    if bc_file is not None:
        detail.boundary_conditions.write_boundary_conditions_spectra(file_name=bc_file)
     
def nest2_xbeach_in_hurrywave(overall,
                              detail,
                              output_path,
                              output_file,
                              option,
                              return_maximum=False):
    if not output_path:
        # Path of the overall output time series
        output_path = overall.path

    if option == "sp2":    
        if not output_file:
            output_file = "hurrywave_sp2.nc"
    
        file_name = os.path.join(output_path, output_file)
    
        # Open netcdf file
        ddd = xr.open_dataset(file_name)
        stations=ddd.station_name.values
        all_stations = []
        for ist, st in enumerate(stations):
            st=str(st.strip())[2:-1]
            all_stations.append(st)
    
        point_names = []    
        if detail.wave_boundary_point:
            # Find required boundary points        
            for point in detail.wave_boundary_point:
                point_names.append(detail.name + "_" + point.name)                    
            
        else:
            point_names = all_stations.copy()
            
        times   = ddd.point_spectrum2d.coords["time"].values
        sigma   = ddd.point_spectrum2d.coords["sigma"].values
        theta   = ddd.point_spectrum2d.coords["theta"].values
    
        ireq = []    
        for ip, point in enumerate(point_names):
            for ist,st in enumerate(all_stations):
                if point.lower() == st.lower():
                    ireq.append(ist)            
                    break
    
        for ip, point in enumerate(detail.wave_boundary_point):
    
            sp2 = ddd.point_spectrum2d.values[:,ireq[ip],:,:]
    
            ds = xr.Dataset(
                    data_vars = dict(point_spectrum2d=(["time", "theta", "sigma"], sp2)),
                    coords    = dict(time=times,
                                     theta=theta,
                                     sigma=sigma)
                    )
            
            point.data = ds

    elif option == "timeseries":

        if not output_file:
            output_file = "hurrywave_his.nc"

        file_name = os.path.join(output_path, output_file)
    
        # Open netcdf file
        ddd = xr.open_dataset(file_name)
        stations=ddd.station_name.values
        all_stations = []
        for ist, st in enumerate(stations):
            st=str(st.strip())[2:-1]
            all_stations.append(st)
    
        point_names = []    
        if detail.wave_boundary_point:
            # Find required boundary points        
            for point in detail.wave_boundary_point:
                point_names.append(detail.name + "_" + point.name)                    
            
        else:
            point_names = all_stations.copy()
            
        times   = ddd.point_hm0.coords["time"].values

        ireq = []    
        for ip, point in enumerate(point_names):
            for ist,st in enumerate(all_stations):
                if point.lower() == st.lower():
                    ireq.append(ist)            
                    break
    
        for ip, point in enumerate(detail.wave_boundary_point):
    
            hm0     = ddd.point_hm0.values[:,ireq[ip]]
            tp      = ddd.point_tp.values[:,ireq[ip]]
            
            #set wavedir such that waves are forced perpendicular to coast instead of real direction
            wavdir =  np.mean([detail.params["thetamin"], detail.params["thetamax"]])
            #wavdir  = ddd.point_wavdir.values[:,ireq[ip]]
            
            #convert directional spread in degrees to xbeach spreading parameter
            dirspr  = ddd.point_dirspr.values[:,ireq[ip]]
            s = 2/(dirspr*np.pi/180)**2 - 1
            
    
            df = pd.DataFrame(index=times)
            df.insert(0,"hm0",hm0)
            df.insert(1,"tp",tp)
            df.insert(2,"wavdir",wavdir)
            df.insert(3,'gammajsp', 3.3)
            df.insert(4,"s",s)
    
            #resample to half-hourly data
            df_resampled = df.resample('30min').max()
            df_interpolated = df_resampled.interpolate(method='linear')
            mask = (df_interpolated.index >= detail.tref) & (df_interpolated.index <= detail.tstop)
            df_filtered = df_interpolated[mask]
    
            df_filtered.insert(5,'duration', 1800)
            df_filtered.insert(6,"dtbc",1)
            
            point.data = df_filtered

    if return_maximum:
        hmax = -999.0
        for icol, point in enumerate(detail.wave_boundary_point):
            hx = point.data["hm0"].max()
            if hx>hmax:
                hmx = point.data["hm0"]
                hmax = hx
        return hmx        
                    
    else:    
        return detail.flow_boundary_point


def nest2_sfincs_in_hurrywave(overall,
                              detail,
                              output_path,
                              output_file):
    if not output_path:
        # Path of the overall output time series
        output_path = overall.path
        
    if not output_file:
        output_file = "hurrywave_his.nc"


    file_name = os.path.join(output_path, output_file)

    # Open netcdf file
    ddd = xr.open_dataset(file_name)
    stations=ddd.station_name.values
    all_stations = []
    for ist, st in enumerate(stations):
        st=str(st.strip())[2:-1]
        all_stations.append(st)

    point_names = []    
    if detail.wave_boundary_point:
        # Find required boundary points        
        for point in detail.wave_boundary_point:
            point_names.append(detail.name + "_" + point.name)                    
        
    else:
        point_names = all_stations.copy()
        
    times   = ddd.point_hm0.coords["time"].values

    ireq = []    
    for ip, point in enumerate(point_names):
        for ist,st in enumerate(all_stations):
            if point.lower() == st.lower():
                ireq.append(ist)            
                break

    for ip, point in enumerate(detail.wave_boundary_point):

        hm0     = ddd.point_hm0.values[:,ireq[ip]]
        tp      = ddd.point_tp.values[:,ireq[ip]]
        wavdir  = ddd.point_wavdir.values[:,ireq[ip]]
        dirspr  = ddd.point_dirspr.values[:,ireq[ip]]

        df = pd.DataFrame(index=times)
        df.insert(0,"hm0",hm0)
        df.insert(1,"tp",tp)
        df.insert(2,"wavdir",wavdir)
        df.insert(3,"dirspr",dirspr)

        point.data = df

def nest2_beware_in_hurrywave(overall,
                              detail,
                              output_path,
                              output_file):

    if not output_path:
        # Path of the overall output time series
        output_path = overall.path
        
    if not output_file:
        output_file = "hurrywave_his.nc"

    file_name = os.path.join(output_path, output_file)

    # Open netcdf file
    ddd = xr.open_dataset(file_name)
    stations=ddd.station_name.values
    all_stations = []
    for ist, st in enumerate(stations):
        st=str(st.strip())[2:-1]
        all_stations.append(st)

    point_names = []    
    if detail.wave_boundary_point:
        # Find required boundary points        
        for point in detail.wave_boundary_point:
            point_names.append(detail.name + "_" + point.name)                    
        
    else:
        point_names = all_stations.copy()
        
    times   = ddd.point_hm0.coords["time"].values

    ireq = []    
    for ip, point in enumerate(point_names):
        for ist,st in enumerate(all_stations):
            if point.lower() == st.lower():
                ireq.append(ist)            
                break

    for ip, point in enumerate(detail.wave_boundary_point):

        hm0     = ddd.point_hm0.values[:,ireq[ip]]
        tp      = ddd.point_tp.values[:,ireq[ip]]
        wavdir  = ddd.point_wavdir.values[:,ireq[ip]]
        dirspr  = ddd.point_dirspr.values[:,ireq[ip]]

        df = pd.DataFrame(index=times)
        df.insert(0,"hm0",hm0)
        df.insert(1,"tp",tp)
        df.insert(2,"wavdir",wavdir)
        df.insert(3,"dirspr",dirspr)

        point.data = df


def nest2_sfincs_in_beware(overall,
                           detail,
                           output_path,
                           output_file,
                           boundary_water_level_correction,
                           option):

    from cht.sfincs.sfincs import FlowBoundaryPoint
    from cht.sfincs.sfincs import WaveMakerForcingPoint

    # Get bounding box for sfincs model
    # Convert bbox to beware crs
    x_range, y_range = detail.bounding_box(crs=overall.crs)
    dx = (x_range[1] - x_range[0])/10
    dy = (y_range[1] - y_range[0])/10
    x_range[0] = x_range[0] - dx
    x_range[1] = x_range[1] + dx
    y_range[0] = y_range[0] - dy
    y_range[1] = y_range[1] + dy
    
    # Read BEWARE offshore locations
    if not output_path:
        # Path of the overall output time series
        output_path = overall.path
        
    if not output_file:
        output_file = "beware_his.nc"

    file_name = os.path.join(output_path, output_file)

    # Open netcdf file
    ddd = xr.open_dataset(file_name)
    
    if option == "flow":
    
        xb = ddd.x_off.values
        yb = ddd.y_off.values
        
        # Find beware locations in bounding box
        inear = np.where((xb>x_range[0]) & (xb<x_range[1]) & (yb>y_range[0]) & (yb<y_range[1]))    
        xb=xb[inear]
        yb=yb[inear]
        nb = xb.size
        
        # Clear existing flow boundary points
        detail.flow_boundary_point = []
    
        # Convert to coordinate system of detail model
        transformer = Transformer.from_crs(overall.crs,
                                           detail.crs,
                                           always_xy=True)
        
        for ip in range(nb):
            name = str(ip + 1).zfill(4)        
            x, y = transformer.transform(xb[ip], yb[ip])
            point = FlowBoundaryPoint(x,
                                      y,
                                      name=name)
            detail.flow_boundary_point.append(point)
        
        # Extract data and set water level boundary conditions
        tref = datetime.datetime(1970,1,1)
        tsec = ddd.time.values # array of int64
        times = tref + tsec*datetime.timedelta(seconds=1)
        
        for ip, point in enumerate(detail.flow_boundary_point):
            point.data = pd.Series(ddd.WL.values[inear[0][ip],:], index=times) + boundary_water_level_correction

    elif option == "wave":

        xb = ddd.x_coast.values
        yb = ddd.y_coast.values
        
        # Find beware locations in bounding box
        inear = np.where((xb>x_range[0]) & (xb<x_range[1]) & (yb>y_range[0]) & (yb<y_range[1]))    
        xb=xb[inear]
        yb=yb[inear]
        nb = xb.size
        
        # Clear existing flow boundary points
        detail.wavemaker_forcing_point = []
    
        # Convert to coordinate system of detail model
        transformer = Transformer.from_crs(overall.crs,
                                           detail.crs,
                                           always_xy=True)
        
        for ip in range(nb):
            name = str(ip + 1).zfill(4)        
            x, y = transformer.transform(xb[ip], yb[ip])
            point = WaveMakerForcingPoint(x,
                                          y,
                                          name=name)
            detail.wavemaker_forcing_point.append(point)
        
        # Extract data and set water level boundary conditions
        tref = datetime.datetime(1970,1,1)
        tsec = ddd.time.values # array of int64
        times = tref + tsec*datetime.timedelta(seconds=1)
        
        for ip, point in enumerate(detail.wavemaker_forcing_point):

            df = pd.DataFrame()
            df["hm0_ig"] = ddd.obs_hm0_ig.values[inear[0][ip],:]
            df["tp_ig"]  = ddd.obs_tpig.values[inear[0][ip],:]
            df["setup"]  = ddd.obs_setup.values[inear[0][ip],:]
            df["time"]   = times
            df = df.set_index("time")
            
            df["hm0_ig"]=df["hm0_ig"].replace(np.nan, 0.1)
            df["tp_ig"]=df["tp_ig"].replace(np.nan, 60.0)
            df["setup"]=df["setup"].replace(np.nan, 0.0)
            
            point.data = df

    ddd.close()
    