# -*- coding: utf-8 -*-
"""
Created on Sat Jun 18 09:03:08 2022
@author: ormondt
"""

import os
from pyproj import CRS

from cht.model_builder.model_builder import ModelBuilder
from cht.hurrywave.hurrywave import HurryWave
from cht.hurrywave.hurrywave_domain import Bathymetry
from cht.hurrywave.hurrywave_domain import Mask
from cht.misc.geometry import RegularGrid
import cht.misc.fileops as fo

class HurryWaveBuilder(ModelBuilder):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
    
    def build(self,
              mskfile = "hurrywave.msk",
              depfile = "hurrywave.dep",
              make_mask=True,
              get_bathymetry=True,
              make_tiles=True,
              quiet=False):

        crs = CRS(self.setup_config["coordinates"]["crs"])
        
        ### Mask
        inpfile = os.path.join(self.model_path, "hurrywave.inp")
        
        hw = HurryWave()
        
        hw.crs            = crs
        hw.input.x0       = self.setup_config["coordinates"]["x0"]
        hw.input.y0       = self.setup_config["coordinates"]["y0"]
        hw.input.dx       = self.setup_config["coordinates"]["dx"]
        hw.input.dy       = self.setup_config["coordinates"]["dy"]
        hw.input.mmax     = self.setup_config["coordinates"]["mmax"]
        hw.input.nmax     = self.setup_config["coordinates"]["nmax"]
        hw.input.rotation = self.setup_config["coordinates"]["rotation"]
        hw.input.crs_name = crs.name
        
        if crs.is_geographic:
            hw.input.crs_type = "geographic"
            hw.input.crsgeo   = 1  
        else:    
            hw.input.crs_type = "projected"
            hw.input.crsgeo   = 0
            if "utm" in crs.name.lower():
               hw.input.crs_utmzone = crs.name[-3:]
        hw.input.crs_epsg = crs.to_epsg()
                   
        for key in self.setup_config["input"]:
            setattr(hw.input, key, self.setup_config["input"][key])

        # Copy hurrywave.bnd to model folder    
        if os.path.exists(os.path.join(self.data_path, "hurrywave.bnd")):
            fo.copy_file(os.path.join(self.data_path, "hurrywave.bnd"),
                        self.model_path)
            hw.input.bndfile = "hurrywave.bnd"

        hw.input.mskfile = mskfile
        hw.input.depfile = depfile

        hw.write_input_file(input_file=inpfile)

        ### Grid                   

        grid = RegularGrid(self.setup_config["coordinates"]["x0"],
                           self.setup_config["coordinates"]["y0"],
                           self.setup_config["coordinates"]["dx"],
                           self.setup_config["coordinates"]["dy"],
                           self.setup_config["coordinates"]["mmax"],
                           self.setup_config["coordinates"]["nmax"],
                           self.setup_config["coordinates"]["rotation"],
                           crs=crs)
        
        ### Bathymetry

        if get_bathymetry:
            # Get bathy/topo
            bathymetry = Bathymetry()
            bathymetry.get_bathymetry(grid, self.bathymetry_list, quiet=False)
            # Save bathymetry
            bathymetry.save(os.path.join(self.model_path, depfile))
        
        ### Mask

        if make_mask:
            mask = Mask(grid, bathymetry.z,
                        zmin=self.setup_config["mask"]["zmin"],
                        zmax=self.setup_config["mask"]["zmax"],
                        include_polygons=self.include_polygons,
                        exclude_polygons=self.exclude_polygons,
                        open_boundary_polygons=self.open_boundary_polygons)
            
            hw.mask = mask

            # Save
            mask.save(os.path.join(self.model_path, mskfile))

        ### Tiles        
        if make_tiles:
            dem_names = []
            z_range   = []
            zoom_range = []
            if self.setup_config["tiling"]["zmin"]>-99990.0 or self.setup_config["tiling"]["zmax"]<99990.0:
                z_range = [self.setup_config["tiling"]["zmin"],
                           self.setup_config["tiling"]["zmax"]]
                zoom_range = [self.setup_config["tiling"]["zoom_range_min"],
                              self.setup_config["tiling"]["zoom_range_max"]]
                for dem in self.bathymetry_list:
                    dem_names.append(dem.name)        
            hw.make_index_tiles(os.path.join(self.tile_path, "indices"),
                                zoom_range=zoom_range,
                                z_range=z_range,
                                dem_names=dem_names)
