# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['rhubarb', 'rhubarb.backends']

package_data = \
{'': ['*']}

install_requires = \
['anyio>=3.4.0,<4.0.0']

extras_require = \
{':extra == "kafka"': ['aiokafka>=0.7.2,<0.8.0'],
 ':extra == "postgres"': ['asyncpg>=0.24,<0.26'],
 ':extra == "rabbitmq"': ['aio-pika>=6.8.0,<7.0.0'],
 ':extra == "redis"': ['aioredis>=2.0.0,<3.0.0']}

setup_kwargs = {
    'name': 'rhubarb-py',
    'version': '1.6.0',
    'description': 'Rhubarb is a library that simplifies realtime streaming for a number of backends into a single API',
    'long_description': '# Rhubarb\n\n<div align="center">\n\n[![Build status](https://github.com/mopeyjellyfish/rhubarb/workflows/build/badge.svg?branch=main&event=push)](https://github.com/mopeyjellyfish/rhubarb/actions?query=workflow%3Abuild)\n[![Python Version](https://img.shields.io/pypi/pyversions/rhubarb-py.svg)](https://pypi.org/project/rhubarb-py)\n[![Dependencies Status](https://img.shields.io/badge/dependencies-up%20to%20date-brightgreen.svg)](https://github.com/mopeyjellyfish/rhubarb/pulls?utf8=%E2%9C%93&q=is%3Apr%20author%3Aapp%2Fdependabot)\n[![codecov](https://codecov.io/gh/mopeyjellyfish/rhubarb/branch/main/graph/badge.svg?token=E8F5LMKDBK)](https://codecov.io/gh/mopeyjellyfish/rhubarb)\n[![Documentation Status](https://readthedocs.org/projects/rhubarb-py/badge/?version=latest)](https://rhubarb-py.readthedocs.io/en/latest/?badge=latest)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![Security: bandit](https://img.shields.io/badge/security-bandit-green.svg)](https://github.com/PyCQA/bandit)\n[![Pre-commit](https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white)](https://github.com/mopeyjellyfish/rhubarb/blob/master/.pre-commit-config.yaml)\n[![Semantic Versions](https://img.shields.io/badge/%20%20%F0%9F%93%A6%F0%9F%9A%80-semantic--versions-e10079.svg)](https://github.com/mopeyjellyfish/rhubarb/releases)\n[![License](https://img.shields.io/github/license/mopeyjellyfish/rhubarb)](https://github.com/mopeyjellyfish/rhubarb/blob/master/LICENSE)\n\nRhubarb is a library that simplifies realtime streaming of events for a number of backends in to a single API. Currently supports [`Postgres`](https://github.com/MagicStack/asyncpg), [`kafka`](https://github.com/aio-libs/aiokafka), [`RabbitMQ`](https://github.com/mosquito/aio-pika), [`redis`](https://github.com/aio-libs/aioredis-py) as well as an internal memory backend useful for testing.\n\n</div>\n\n## Installation\n\nThere are a number of backends that can be used with Rhubarb:\n\n| Kafka | Postgres | Redis | RabbitMQ |\n| --------------------------------------------------------------------- | --------------------------------------------------------------------------------- | --------------------------------------------------------------------------------- |--------------------------------------------------------------------------------- |\n| <p align="center"><img src="./README_assets/kafka.png" width="60" height="100"/></p>    | <p align="center"><img src="./README_assets/postgres.png" width="100" height="100" /></p> | <p align="center"><img src="./README_assets/redis.png" width="100" height="80"/></p> | <p align="center"><img src="./README_assets/rabbitmq.jpg" width="100" height="100" /></p> |\n| `pip install rhubarb-py[kafka]` | `pip install rhubarb-py[postgres]` | `pip install rhubarb-py[redis]` | `pip install rhubarb-py[rabbitmq]` |\n\n## Backends\n\n- `Rhubarb("redis://localhost:6379/0")`\n- `Rhubarb("kafka://localhost:9092")`\n- `Rhubarb("postgres://postgres:postgres@localhost:5432/rhubarb")`\n- `Rhubarb("amqp://guest:guest@localhost/")`\n- `Rhubarb("memory://")`\n\n## Quick start\n\n### Simple event consumer\n\n```python\nasync with Rhubarb("redis://localhost:6379/0") as events:\n    async with events.subscribe(channel="CHATROOM") as subscriber:\n        async for event in subscriber:\n            await websocket.send_text(event.message)\n```\n\n### Simple event producer\n\n```python\nasync with Rhubarb("redis://localhost:6379/0") as events:\n    await events.publish("test message")\n```\n\n### History retrieval\n\n```python\nasync with Rhubarb("redis://localhost:6379/0") as events: \n    async with events.subscribe(channel="CHATROOM", history=10) as subscriber: # read the last 10 events published to the channel\n        async for event in subscriber:\n            await websocket.send_text(event.message)\n```\n\n### Custom serializer & deserializer\n\n```python\nasync with Rhubarb("redis://localhost:6379/0", serializer=json.dumps, deserializer=json.loads) as events:\n    async with events.subscribe(channel="CHATROOM", history=10) as subscriber: # read the last 10 events published to the channel\n        async for event in subscriber:\n            await websocket.send_text(event.message)\n```\n\n### Group subscribing (at-most-once processing)\n\n```python\nasync with Rhubarb("redis://localhost:6379/0", serializer=json.dumps, deserializer=json.loads) as events:\n    async with events.subscribe(\n        "TEST-GROUP-CHANNEL", group_name="TEST_GROUP", consumer_name="sub_1"\n    ) as subscriber_1:\n        async for event in subscriber:\n            await process_job(event)\n```\n\n## Example\n\nA minimal working example can be found in [example](https://github.com/mopeyjellyfish/rhubarb/blob/main/example/app.py) directory.\n\n```python\nimport os\n\nfrom starlette.applications import Starlette\nfrom starlette.concurrency import run_until_first_complete\nfrom starlette.responses import HTMLResponse\nfrom starlette.routing import Route, WebSocketRoute\n\nfrom rhubarb import Rhubarb\n\nURL = os.environ.get("URL", "redis://localhost:6379/0")\n\nevents = Rhubarb(URL)\n\nhtml = """\n<!DOCTYPE html>\n<html>\n    <head>\n        <title>Chat</title>\n    </head>\n    <body>\n        <h1>WebSocket Chat</h1>\n        <form action="" onsubmit="sendMessage(event)">\n            <input type="text" id="messageText" autocomplete="off"/>\n            <button>Send</button>\n        </form>\n        <ul id=\'messages\'>\n        </ul>\n        <script>\n            var ws = new WebSocket("ws://localhost:8000/ws");\n            ws.onmessage = function(event) {\n                var messages = document.getElementById(\'messages\')\n                var message = document.createElement(\'li\')\n                var content = document.createTextNode(event.data)\n                message.appendChild(content)\n                messages.appendChild(message)\n            };\n            function sendMessage(event) {\n                var input = document.getElementById("messageText")\n                ws.send(input.value)\n                input.value = \'\'\n                event.preventDefault()\n            }\n        </script>\n    </body>\n</html>\n"""\n\n\nasync def homepage(_):\n    return HTMLResponse(html)\n\n\nasync def room_consumer(websocket):\n    async for message in websocket.iter_text():\n        await events.publish(channel="chatroom", message=message)\n\n\nasync def room_producer(websocket):\n    async with events.subscribe(channel="chatroom") as subscriber:\n        async for event in subscriber:\n            await websocket.send_text(event.message)\n\n\nasync def ws(websocket):\n    await websocket.accept()\n    await run_until_first_complete(\n        (room_consumer, {"websocket": websocket}),\n        (room_producer, {"websocket": websocket}),\n    )\n\n\nroutes = [\n    Route("/", homepage),\n    WebSocketRoute("/ws", ws, name="chatroom_ws"),\n]\n\n\napp = Starlette(\n    routes=routes,\n    on_startup=[events.connect],\n    on_shutdown=[events.disconnect],\n)\n```\n\n## 🛡 License\n\n[![License](https://img.shields.io/github/license/mopeyjellyfish/rhubarb)](https://github.com/mopeyjellyfish/rhubarb/blob/master/LICENSE)\n\nThis project is licensed under the terms of the `MIT` license. See [LICENSE](https://github.com/mopeyjellyfish/rhubarb/blob/master/LICENSE) for more details.\n',
    'author': 'mopeyjellyfish',
    'author_email': 'dev@davidhall.tech',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/mopeyjellyfish/rhubarb',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
