"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SonarCodeScanner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const sonarqube_commands_1 = require("./sonarqube-commands");
const unpackSourceAndArtifacts = (includeGlobsForScan) => [
    'export BUILT_ARTIFACT_URI=`aws codebuild batch-get-builds --ids $SYNTH_BUILD_ID | jq -r \'.builds[0].secondaryArtifacts[] | select(.artifactIdentifier == "Synth__") | .location\' | awk \'{sub("arn:aws:s3:::","s3://")}1\' $1`',
    "export SYNTH_SOURCE_URI=`aws codebuild batch-get-builds --ids $SYNTH_BUILD_ID | jq -r '.builds[0].sourceVersion' | awk '{sub(\"arn:aws:s3:::\",\"s3://\")}1' $1`",
    "aws s3 cp $SYNTH_SOURCE_URI source.zip",
    "aws s3 cp $BUILT_ARTIFACT_URI built.zip",
    "unzip source.zip -d src",
    "unzip built.zip -d built",
    "rm source.zip built.zip",
    `rsync -a built/* src --include="*/" ${includeGlobsForScan
        ? includeGlobsForScan.map((g) => `--include ${g}`).join(" ")
        : ""} --include="**/coverage/**" --include="**/cdk.out/**" --exclude="**/node_modules/**/*" --exclude="**/.env/**" --exclude="*" --prune-empty-dirs`,
];
const owaspScan = () => `npx owasp-dependency-check --format HTML --out src/reports --exclude '**/.git/**/*' --scan src --enableExperimental --bin /tmp/dep-check --disableRetireJS`;
const cfnNagScan = (cdkOutDir, cfnNagIgnorePath) => cdkOutDir
    ? `cfn_nag ${cfnNagIgnorePath ? `--deny-list-path=${cfnNagIgnorePath}` : ""} built/${cdkOutDir}/**/*.template.json --output-format=json > src/reports/cfn-nag-report.json`
    : 'echo "skipping cfn_nag as no cdkOutDir was specified.';
class SonarCodeScanner extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const sonarQubeToken = new aws_secretsmanager_1.Secret(this, "SonarQubeToken");
        const synthBuildProject = aws_codebuild_1.Project.fromProjectArn(this, "SynthBuildProject", props.synthBuildArn);
        const validationProject = new aws_codebuild_1.Project(this, "ValidationProject", {
            environment: {
                buildImage: aws_codebuild_1.LinuxBuildImage.STANDARD_5_0,
            },
            environmentVariables: {
                SONARQUBE_TOKEN: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.SECRETS_MANAGER,
                    value: sonarQubeToken.secretArn,
                },
                SONARQUBE_ENDPOINT: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PLAINTEXT,
                    value: props.sonarqubeEndpoint,
                },
                PROJECT_NAME: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PLAINTEXT,
                    value: props.sonarqubeProjectName,
                },
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: "0.2",
                env: {
                    shell: "bash",
                },
                phases: {
                    install: {
                        commands: ["npm install -g aws-cdk", "gem install cfn-nag"],
                    },
                    build: {
                        commands: [
                            "export RESOLVED_SOURCE_VERSION=`aws codebuild batch-get-builds --ids $SYNTH_BUILD_ID | jq -r '.builds[0].resolvedSourceVersion'`",
                            ...unpackSourceAndArtifacts(props.includeGlobsForScan),
                            ...sonarqube_commands_1.createSonarqubeProject(props),
                            "mkdir -p src/reports",
                            owaspScan(),
                            cfnNagScan(props.cdkOutDir, props.cfnNagIgnorePath),
                            "cd src",
                            sonarqube_commands_1.sonarqubeScanner(props.excludeGlobsForScan),
                            ...sonarqube_commands_1.generateSonarqubeReports(),
                            ...(props.preArchiveCommands || []),
                        ],
                    },
                },
            }),
        });
        validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["codebuild:BatchGetBuilds"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [synthBuildProject.projectArn],
        }));
        validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["s3:GetObject*"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [props.artifactBucketArn, `${props.artifactBucketArn}/**`],
        }));
        props.artifactBucketKeyArn &&
            validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
                actions: ["kms:Decrypt", "kms:DescribeKey"],
                effect: aws_iam_1.Effect.ALLOW,
                resources: [props.artifactBucketKeyArn],
            }));
        synthBuildProject.onBuildSucceeded("OnSynthSuccess", {
            target: new aws_events_targets_1.CodeBuildProject(validationProject, {
                event: aws_events_1.RuleTargetInput.fromObject({
                    environmentVariablesOverride: [
                        {
                            name: "SYNTH_BUILD_ID",
                            type: "PLAINTEXT",
                            value: aws_events_1.EventField.fromPath("$.detail.build-id"),
                        },
                    ],
                }),
            }),
        });
        new aws_cdk_lib_1.CfnOutput(this, "SonarqubeSecretArn", {
            exportName: "SonarqubeSecretArn",
            value: sonarQubeToken.secretArn,
        });
        [
            "AwsSolutions-SMG4",
            "AwsPrototyping-SecretsManagerRotationEnabled",
        ].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(sonarQubeToken, [
                {
                    id: RuleId,
                    reason: "Key rotation is not possible as a user token needs to be generated from Sonarqube",
                },
            ]);
        });
        const stack = aws_cdk_lib_1.Stack.of(this);
        ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(validationProject.role, [
                {
                    id: RuleId,
                    reason: "Validation CodeBuild project requires access to the ArtifactsBucket and ability to create logs.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/codebuild/<.*SonarCodeScannerValidationProject.*>:\\*$/g`,
                        },
                        {
                            regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:codebuild:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:report-group/<.*SonarCodeScannerValidationProject.*>-\\*$/g`,
                        },
                        {
                            regex: `/^Action::s3:GetObject\\*$/g`,
                        },
                        {
                            regex: "/^Resource::<ArtifactsBucket.*.Arn>/\\*\\*$/g",
                        },
                    ],
                },
            ], true);
        });
    }
}
exports.SonarCodeScanner = SonarCodeScanner;
_a = JSII_RTTI_SYMBOL_1;
SonarCodeScanner[_a] = { fqn: "@aws-prototyping-sdk/pipeline.SonarCodeScanner", version: "0.13.5" };
//# sourceMappingURL=data:application/json;base64,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