"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmiPipelineLib = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const codepipeline_actions = require("@aws-cdk/aws-codepipeline-actions");
const ec2 = require("@aws-cdk/aws-ec2");
const events = require("@aws-cdk/aws-events");
const event_targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const imagebuilder = require("@aws-cdk/aws-imagebuilder");
const kms = require("@aws-cdk/aws-kms");
const sns = require("@aws-cdk/aws-sns");
const cdk = require("@aws-cdk/core");
const component_builder_1 = require("./component_builder");
const image_recipe_version_bump_1 = require("./image_recipe_version_bump");
const slack_notification_1 = require("./slack_notification");
const source_action_builder_1 = require("./source_action_builder");
const ssmupdate = require("./ssm_update");
const utils = require("./utils");
/**
 * Construct for creating a Codepipeline, EC2 Image builder pipeline from 1 pipeline configuration.
 *
 * @stability stable
 */
class AmiPipelineLib extends cdk.Construct {
    /**
     * Constructor.
     *
     * @stability stable
     */
    constructor(scope, id, pipelineConfig, componentDepsConfig, slackConfig) {
        super(scope, id);
        this.id = id;
        this.pipelineConfig = pipelineConfig;
        this.pipelineConfig.name = this.id;
        this.componentDepsConfig = componentDepsConfig;
        this.slackConfig = slackConfig;
        this.componentBuilder = new component_builder_1.ComponentBuilder(this, componentDepsConfig, this.id, this.pipelineConfig.parent_image);
        this.sourceActionBuilder = new source_action_builder_1.SourceActionBuilder(this, pipelineConfig.sources, this.id);
        this.createImagebuilderPipeline();
        this.createCodepipelineProject();
        this.createScheduledTask();
        if (this.topic) {
            new ssmupdate.SsmUpdateConstruct(this, 'SSMUpdate', this.topic, this.pipelineConfig);
            new slack_notification_1.SlackNotification(this, 'SlackNotification', this.topic, slackConfig, `${this.id}Recipe`);
        }
    }
    /**
     * @stability stable
     */
    createScheduledTask() {
        if (this.codepipeline && this.pipelineConfig.schedule) {
            const pipelineTarget = new event_targets.CodePipeline(this.codepipeline);
            new events.Rule(this, 'ScheduleRule', {
                schedule: events.Schedule.expression(this.pipelineConfig.schedule),
                targets: [pipelineTarget],
            });
        }
    }
    /**
     * @stability stable
     */
    createImagebuilderPipeline() {
        let compMap = this.componentBuilder.createComponentDependenciesMap();
        let image_id;
        let parentImage;
        if ('disk_size' in this.pipelineConfig && this.pipelineConfig.disk_size) {
            this.diskSize = this.pipelineConfig.disk_size;
        }
        if ('image_id' in this.pipelineConfig && this.pipelineConfig.image_id) {
            image_id = this.pipelineConfig.image_id;
        }
        else {
            switch (this.pipelineConfig.parent_image) {
                case 'AmazonLinux2':
                    parentImage = ec2.MachineImage.latestAmazonLinux({
                        generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                        edition: ec2.AmazonLinuxEdition.STANDARD,
                        virtualization: ec2.AmazonLinuxVirt.HVM,
                        storage: ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
                        cpuType: ec2.AmazonLinuxCpuType.X86_64,
                    });
                    break;
                default:
                    parentImage = ec2.MachineImage.lookup(this.getLookupCriteria(this.pipelineConfig.parent_image));
                    break;
            }
        }
        // Use a custom block device mapping if encryption is required.
        let blockDeviceMappings = this.createBlockDeviceMapping();
        this.recipe = new imagebuilder.CfnImageRecipe(this, 'ImageRecipe', {
            name: `${this.id}Recipe`,
            parentImage: image_id ? image_id : parentImage ? parentImage.getImage(this).imageId : null,
            version: this.getNextRecipeVersion(`${this.id}Recipe`),
            components: this.pipelineConfig.recipe.components.map((c) => ({
                componentArn: compMap[c.name].ref,
            })),
            blockDeviceMappings: blockDeviceMappings,
        });
        const builderRole = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilder'),
            ],
            inlinePolicies: this.getInlinePolicies(this.pipelineConfig),
        });
        const instanceProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [builderRole.roleName],
        });
        this.topic = new sns.Topic(this, 'AmiPipelineTopic');
        this.infrastructure = new imagebuilder.CfnInfrastructureConfiguration(this, 'Infrastructure', {
            name: `${this.id} - EC2 IB Infrastructure`,
            instanceProfileName: instanceProfile.ref,
            snsTopicArn: this.topic.topicArn,
            instanceTypes: this.pipelineConfig.instance_type ? [this.pipelineConfig.instance_type] : undefined,
            terminateInstanceOnFailure: !('terminate_on_failure' in this.pipelineConfig) || this.pipelineConfig.terminate_on_failure,
            subnetId: this.pipelineConfig.subnet_id,
            securityGroupIds: this.pipelineConfig.security_group_ids ? this.pipelineConfig.security_group_ids.split(',') : undefined,
        });
        this.distributionConfig = this.pipelineConfig.shared_with ? new imagebuilder.CfnDistributionConfiguration(this, 'DistributionConfig', {
            name: `${this.id} - Distribution`,
            distributions: this.pipelineConfig.shared_with.map((s) => ({
                region: s.region,
                amiDistributionConfiguration: {
                    Name: `${this.id} - {{ imagebuilder:buildDate }}`,
                    Description: `${this.id} Shared image`,
                    AmiTags: {
                        Name: this.id,
                    },
                    TargetAccountIds: s.account_ids,
                },
            })),
        }) : undefined;
        this.imagePipeline = new imagebuilder.CfnImagePipeline(this, 'AmiPipeline', {
            name: `${this.id} - AMI Pipeline`,
            imageRecipeArn: this.recipe.ref,
            infrastructureConfigurationArn: this.infrastructure.ref,
            distributionConfigurationArn: this.distributionConfig ? this.distributionConfig.ref : undefined,
        });
    }
    getInlinePolicies(pipelineConfig) {
        let result = {};
        const statements = pipelineConfig.policy_statements;
        if (statements) {
            for (let statement in statements) {
                result[statement] = new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            sid: statement,
                            actions: statements[statement].actions,
                            effect: iam.Effect.ALLOW,
                            resources: statements[statement].resources,
                        }),
                    ],
                });
            }
        }
        return result;
    }
    createBlockDeviceMapping() {
        const key = this.pipelineConfig.encrypted ? new kms.Key(this, 'PipelineEncryption', {
            description: `Image encryption for ${this.pipelineConfig.name}`,
            alias: `${utils.pascalize(this.pipelineConfig.name)}_key`,
        }) : null;
        let blockDeviceMappings = [
            {
                deviceName: this.pipelineConfig.parent_image.includes('AmazonLinux') ? '/dev/xvda' : '/dev/sda1',
                ebs: {
                    encrypted: key ? true : false,
                    volumeSize: this.pipelineConfig.disk_size ? this.pipelineConfig.disk_size : 8,
                    volumeType: 'gp2',
                    kmsKeyId: key === null || key === void 0 ? void 0 : key.keyArn,
                },
            },
        ];
        return blockDeviceMappings;
    }
    /**
     * @stability stable
     */
    getNextRecipeVersion(recipeName) {
        const recipeBumpCustomResource = new image_recipe_version_bump_1.ImageRecipeVersionBump(this, 'ImageRecipeVersionBump', recipeName);
        return recipeBumpCustomResource.nextVersion;
    }
    /**
     * @stability stable
     */
    createCodepipelineProject() {
        var _b;
        if (this.imagePipeline) {
            // Create CodePipeline project
            let imagebuilderBuild = new codebuild.PipelineProject(this, 'AmiPipelineBuild', {
                buildSpec: codebuild.BuildSpec.fromObject({
                    version: '0.2',
                    phases: {
                        install: {
                            commands: [
                                'apt-get update',
                                'apt-get install awscli -y',
                            ],
                        },
                        pre_build: {
                            commands: [
                                'curl -X POST -H \'Content-type: application/json\' --data "{\\"text\\":\\"$PIPELINE_NAME build has started.\\"}" $SLACK_WEBHOOK_URL',
                            ],
                        },
                        build: {
                            commands: 'aws imagebuilder start-image-pipeline-execution --image-pipeline-arn $IMAGE_PIPELINE_ARN',
                        },
                    },
                }),
                environment: {
                    buildImage: codebuild.LinuxBuildImage.STANDARD_4_0,
                    environmentVariables: {
                        IMAGE_PIPELINE_ARN: { value: this.imagePipeline.ref },
                        SLACK_WEBHOOK_URL: { value: this.slackConfig.slackWebhookUrl },
                        SLACK_CHANNEL: { value: this.slackConfig.channel },
                        SLACK_USERNAME: { value: this.slackConfig.username },
                        PIPELINE_NAME: { value: this.imagePipeline.name },
                    },
                },
            });
            imagebuilderBuild.addToRolePolicy(iam.PolicyStatement.fromJson({
                Sid: 'imagebuilderstart',
                Effect: 'Allow',
                Action: ['imagebuilder:StartImagePipelineExecution'],
                Resource: '*',
            }));
            const imageBuilderOutput = new codepipeline.Artifact('CdkBuildOutput');
            let sourceActions = this.sourceActionBuilder.createPipelineSources();
            const buildAction = new codepipeline_actions.CodeBuildAction({
                actionName: 'StartImageBuilder',
                project: imagebuilderBuild,
                input: (_b = sourceActions.find(a => a && a.sourceOutput)) === null || _b === void 0 ? void 0 : _b.sourceOutput,
                extraInputs: sourceActions.length > 1 ? sourceActions.slice(1).map(a => a.sourceOutput) : [],
                outputs: [imageBuilderOutput],
            });
            // Complete Pipeline Project
            this.codepipeline = new codepipeline.Pipeline(this, 'Pipeline', {
                restartExecutionOnUpdate: true,
                stages: [
                    {
                        stageName: 'Source',
                        actions: sourceActions.map(c => c.action).filter((c) => c != null),
                    },
                    {
                        stageName: 'Build',
                        actions: [buildAction],
                    },
                ],
            });
        }
    }
    /**
     * @stability stable
     */
    getLookupCriteria(parentImage) {
        switch (parentImage) {
            case 'Ubuntu1804':
                return {
                    name: 'ubuntu/images/hvm-ssd/ubuntu-bionic-18.04-amd64*',
                    owners: ['099720109477'],
                };
            case 'Ubuntu2004':
                return {
                    name: 'ubuntu/images/hvm-ssd/ubuntu-focal-20.04-amd64*',
                    owners: ['099720109477'],
                };
            case 'CentOS7':
                return {
                    name: '*',
                    owners: ['aws-marketplace'],
                    filters: {
                        'product-code': ['cvugziknvmxgqna9noibqnnsy'],
                    },
                };
            case 'CentOS8':
                return {
                    name: '*',
                    owners: ['aws-marketplace'],
                    filters: {
                        'product-code': ['47k9ia2igxpcce2bzo8u3kj03'],
                    },
                };
            default:
                return {
                    name: 'amzn-linux',
                };
        }
    }
}
exports.AmiPipelineLib = AmiPipelineLib;
_a = JSII_RTTI_SYMBOL_1;
AmiPipelineLib[_a] = { fqn: "halloumi-ami-pipelines.AmiPipelineLib", version: "0.0.14" };
//# sourceMappingURL=data:application/json;base64,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