# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['poetry_version_plugin']

package_data = \
{'': ['*']}

install_requires = \
['poetry>=1.2.0a1,<2.0.0']

entry_points = \
{'poetry.plugin': ['poetry-version-plugin = '
                   'poetry_version_plugin.plugin:VersionPlugin']}

setup_kwargs = {
    'name': 'poetry-version-plugin-dontuse',
    'version': '0.1.4',
    'description': 'Poetry plugin for dynamically extracting the package version from a __version__ variable or a Git tag.',
    'long_description': '# Poetry Version Plugin\n\n<a href="https://github.com/tiangolo/poetry-version-plugin/actions?query=workflow%3ATest" target="_blank">\n    <img src="https://github.com/tiangolo/poetry-version-plugin/workflows/Test/badge.svg" alt="Test">\n</a>\n<a href="https://github.com/tiangolo/poetry-version-plugin/actions?query=workflow%3APublish" target="_blank">\n    <img src="https://github.com/tiangolo/poetry-version-plugin/workflows/Publish/badge.svg" alt="Publish">\n</a>\n<a href="https://codecov.io/gh/tiangolo/poetry-version-plugin" target="_blank">\n    <img src="https://img.shields.io/codecov/c/github/tiangolo/poetry-version-plugin?color=%2334D058" alt="Coverage">\n</a>\n<a href="https://pypi.org/project/poetry-version-plugin" target="_blank">\n    <img src="https://img.shields.io/pypi/v/poetry-version-plugin?color=%2334D058&label=pypi%20package" alt="Package version">\n</a>\n\nA [**Poetry**](https://python-poetry.org/) plugin for dynamically extracting the package **version**.\n\nIt can read the version from a file `__init__.py` with:\n\n```python\n# __init__.py\n\n__version__ = "0.1.0"\n```\n\nAlternatively, it can read it from a **git tag**, set with a GitHub release or with:\n\n```console\n$ git tag 0.1.0\n```\n\n🚨 Consider this in the alpha stage. Read the warning below.\n\n## When to use\n\nThis is useful mainly if you are building a package library for others to use and you want to set the version in a place different than `pyproject.toml`, but you still want to keep a [single source of truth](https://en.wikipedia.org/wiki/Single_source_of_truth).\n\nIt won\'t be helpful in other use cases like managing local app environments with Poetry.\n\n## How to use\n\nMake sure you have Poetry version `1.2.0a1` or above. Read below for instructions to install it if you haven\'t.\n\n### Install Poetry Version Plugin\n\nInstall this plugin to your Poetry:\n\n```console\n$ poetry plugin add poetry-version-plugin\n\n--> 100%\n```\n\n### Set version in init file\n\nSet your package version in your file `__init__.py`, for example:\n\n```python\nfrom .main import do_awesome_stuff, AwesomeClass\n\n__version__ = "0.2.3"\n```\n\nAnd then edit your `pyproject.toml` with a section containing:\n\n```toml\n[tool.poetry-version-plugin]\nsource = "init"\n```\n\nNext, build your project. It will show an output like:\n\n```console\n$ poetry build\nUsing __init__.py file at my_awesome_package/__init__.py for dynamic version\nSetting package dynamic version to __version__ variable from __init__.py: 0.1.9\nBuilding my-awesome-package (0.1.9)\n  - Building sdist\n  - Built my-awesome-package-0.1.9.tar.gz\n  - Building wheel\n  - Built my-awesome-package-0.1.9-py3-none-any.whl\n```\n\n### Set the version in a Git tag\n\nAlternatively, to extract the version to use from a Git tag, add a section:\n\n```toml\n[tool.poetry-version-plugin]\nsource = "git-tag"\n```\n\nThen create a git tag, for example:\n\n```console\n$ git tag 0.1.3\n```\n\nIn this case, when building your project, it will show an output like:\n\n```console\n$ poetry build\nGit tag found, setting dynamic version to: 0.1.3\nBuilding my-awesome-package (0.1.3)\n  - Building sdist\n  - Built my-awesome-package-0.1.3.tar.gz\n  - Building wheel\n  - Built my-awesome-package-0.1.3-py3-none-any.whl\n```\n\n## Version in `pyproject.toml`\n\nCurrently (2021-05-24) Poetry requires a `version` configuration in the `pyproject.toml`, even if you use this plugin.\n\nWhen using this plugin, that `version` config won\'t be used, but Poetry still requires it to be present in the `pyproject.toml`.\n\nTo make it more evident that you are not using that `version` you can set it to `0`.\n\n```toml\n[tool.poetry]\nname = "my-awesome-package"\nversion = "0"\n```\n\nThat way, you will more easily notice if the plugin is not installed, as it will show that you are building a package with version `0` instead of the dynamic version set.\n\n## An example `pyproject.toml`\n\nA short, minimal example `pyproject.toml` could look like:\n\n```toml\n[tool.poetry]\nname = "my-awesome-package"\nversion = "0"\ndescription = ""\nauthors = ["Rick Sanchez <rick@rick-citadel.com>"]\nreadme = "README.md"\n\n[tool.poetry.dependencies]\npython = "^3.6"\n\n[build-system]\nrequires = ["poetry-core"]\nbuild-backend = "poetry.core.masonry.api"\n\n[tool.poetry-version-plugin]\nsource = "init"\n```\n\n## Why\n\nBy default, Poetry expects you to set your package version in `pyproject.toml`. And that would work in most cases.\n\nBut imagine you want to expose the version of your package in a `__version__` variable so that your users can do things like:\n\n```python\nimport my_awesome_package\nprint(my_awesome_package.__version__)\n```\n\nYou could manually write the `__version__` variable and handle the synchronization between it and the `pyproject.toml` yourself, which is very **error-prone**.\n\nThe current [official way of doing it without duplicating the value](https://github.com/python-poetry/poetry/pull/2366#issuecomment-652418094) is with `importlib.metadata`.\n\nBut that module is only available in Python 3.8 and above. So, for Python 3.7 and 3.6 you have to install a backport as a dependency of your package:\n\n```toml\n[tool.poetry.dependencies]\nimportlib-metadata = {version = "^1.0", python = "<3.8"}\n```\n\nBut then, when they release each new version of the backport (currently `4.0.1`), you have to update it (or not). And your users would have to manually handle conflicts with any other packages that also depend on `importlib-metadata` which could be multiple, as many packages could be doing the same trick (I\'ve dealt with that).\n\nThe other option is not to pin any version range of your `importlib-metadata` in your `pyproject.toml` and hope for the best.\n\nAnd then your `__init__.py` would have to include code using it, like:\n\n```python\n# I don\'t want this extra complexity 😔\n# And it doesn\'t work in Docker 🐋\ntry:\n    import importlib.metadata as importlib_metadata\nexcept ModuleNotFoundError:\n    import importlib_metadata\n\n__version__ = importlib_metadata.version(__name__)\n```\n\nBut that code is extra complexity and logic needed in your code, in each of your packages.\n\n🚨 Additionally, this only works when your package is installed in a Python environment. It won\'t work if, for example, you simply put your code in a **container**, which is common for web apps and distributed systems.\n\n### How this plugin solves it\n\nWith this plugin, your package doesn\'t depend on `importlib-metadata`, so your users won\'t need to handle conflicts or extra dependencies.\n\nInstead, your build system (Poetry) is what needs to have this plugin installed.\n\nThat avoids the extra code complexity on your side, dependency conflicts for your users, and support for other use cases like code copied directly inside a container.\n\n### Version from Git tag\n\nAlternatively, this plugin can also extract the version from a Git tag.\n\nSo, you could only create each version in a Git tag (for example, a GitHub release) instead of writing it in code.\n\nAnd then build the package on Continuous Integration (e.g. GitHub Actions). And this plugin would get the version of the package from that Git tag.\n\n## Install Poetry `1.2.0a1`\n\nFor this plugin to work, you need Poetry version `1.2.0a1` or above.\n\n[Poetry `1.2.0a1` was released recently](https://python-poetry.org/blog/announcing-poetry-1-2-0a1.html).\n\nThere\'s a high chance you already have installed Poetry `1.1.x`.\n\nThe first step is to uninstall it:\n\n```console\n$ curl -sSL https://raw.githubusercontent.com/python-poetry/poetry/master/get-poetry.py -O\n--> 100%\n\n$ python get-poetry.py --uninstall\n--> 100%\n```\n\nAnd then install the new Poetry with the new installer:\n\n```console\n$ curl -sSL https://raw.githubusercontent.com/python-poetry/poetry/master/install-poetry.py -O\n--> 100%\n\n$ python install-poetry.py --preview\n--> 100%\n```\n\n🔍 Notice that the new installer file is named `install-poetry.py` instead of `get-poetry.py`. Also, notice that, currently, you need to set `--preview` for it to install the alpha version `1.2.0a1`.\n\nYou can check that it worked with:\n\n```console\n$ poetry --version\nPoetry (version 1.2.0a1)\n```\n\n## Support for version in init file\n\nWhen using a `__version__` variable in your `__init__.py` you can have more logic in that file, import modules, and do more things above and below the declaration of that variable.\n\nBut the value has to be a literal string, like:\n\n```python\n___version___ = "0.2.0"\n```\n\n...instead of calling a function or something similar.\n\nAnd the variable has to be at the top-level, so it can\'t be inside an `if` statement or similar.\n\nThis is all fine and supported in your `__init__.py`:\n\n```python\n# __init__.py\n\n# This is all valid 👍✅\n\nfrom .main import do_awesome_stuff, AwesomeClass\n\nawesome = AwesomeClass()\n\n# Some comment explaining why this is commented out\n# __version__ = "1.0.0"\n\n__version__ = "0.2.3"\n\nif __name__ == "__main__":\n    awesome.run()\n```\n\nThis example is all valid and supported, and it includes:\n\n* Imports\n* Other objects and variables\n* Comments\n* The same string `__version__` inside a comment\n* If blocks around\n\n---\n\nBut this is not supported:\n\n```python\n# 🚨 Not supported\n\nif 2 == 2:\n    __version__ = "0.1.0\n```\n\nAnd this is not supported:\n\n```python\n# 🚨 Not supported\n\ndef get_version():\n    return "0.2.0"\n\n__version__ = get_version()\n```\n\n## How the plugin works\n\nPoetry runs the plugin when building a package, and it sets the version right before creating the "package distributable" (e.g., the wheel).\n\n### How the version variable works\n\nIf you have a package (a single package) declared in the `packages` config in your `pyproject.toml`, the plugin will use that package\'s `__init__.py` to find the `__version__` variable.\n\nIf you don\'t have any `packages` config, the plugin will assume that you have a single package named as your project, but in the module version (changing `-` for `_`). So, if your package is `my-awesome-project`, the plugin will use the file at `my_awesome_project/__init__.py` to find the `__version__` variable.\n\nThis file structure is the default if you create a new project with the command `poetry new`, so it should work as expected. ✨\n\nThe way the plugin works internally is by parsing the `__init__.py` file. Reading the Python\'s "Abstract Syntax Tree" using the `ast` standard module and extracting the literal value of the string. So, it doesn\'t execute the code in `__init__.py`, it only reads it as Python code.\n\nThe plugin doesn\'t try to import and execute that `__init__.py` file because that could require extra computation, external dependencies, etc. And it doesn\'t try to extract the `__version__` with regular expressions, as that would be prone to errors if, for example, there was some other `__version__` somewhere in the code, in a comment or inside a string.\n\n## Warning\n\n🚨 Consider this in the alpha stage. Poetry `1.2.0a1` with support for plugins was released on 2021-05-21. I started writing this plugin 3 days later, on 2021-05-24.\n\nThings might break in Poetry or in this plugin. So, please try it and test it very carefully before fully adopting it for delicate systems.\n\nThe way it works might change, and the specific configuration might change.\n\nAlso, if you don\'t find the following sections intuitive:\n\n```toml\n[tool.poetry-version-plugin]\nsource = "init"\n```\n\nand\n\n```toml\n[tool.poetry-version-plugin]\nsource = "git-tag"\n```\n\nlet me know what alternative configuration would make more sense and be more intuitive to you.\n\n👍 The good news is, assuming you are building packages to then upload them to PyPI for your users to download and use them, the **worst that could happen** if something broke is that you wouldn\'t be able to build a new version until something is fixed or changed. But your users shouldn\'t be affected in any way.\n\n## Release Notes\n\n### Latest Changes\n\n* ✏️ Fix typos and rewording in README.md. PR [#8](https://github.com/tiangolo/poetry-version-plugin/pull/8) by [@Gl0deanR](https://github.com/Gl0deanR).\n### 0.1.3\n\n* ✨ Improve logs, prefix with plugin name. PR [#6](https://github.com/tiangolo/poetry-version-plugin/pull/6) by [@tiangolo](https://github.com/tiangolo).\n* 🔧 Update pyproject metadata. PR [#5](https://github.com/tiangolo/poetry-version-plugin/pull/5) by [@tiangolo](https://github.com/tiangolo).\n* ✅ Fix coverage. PR [#4](https://github.com/tiangolo/poetry-version-plugin/pull/4) by [@tiangolo](https://github.com/tiangolo).\n* 📝 Improve docs. PR [#3](https://github.com/tiangolo/poetry-version-plugin/pull/3) by [@tiangolo](https://github.com/tiangolo).\n* 🐛 Fix tests for CI. PR [#1](https://github.com/tiangolo/poetry-version-plugin/pull/1) by [@tiangolo](https://github.com/tiangolo).\n* 👷 Fix Latest Changes action, set branch to main. PR [#2](https://github.com/tiangolo/poetry-version-plugin/pull/2) by [@tiangolo](https://github.com/tiangolo).\n\n## License\n\nThis project is licensed under the terms of the MIT license.\n',
    'author': 'Sebastián Ramírez',
    'author_email': 'tiangolo@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/tiangolo/poetry-version-plugin',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
