# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/03_notifyr.ipynb.

# %% auto 0
__all__ = ['send_nostr_message', 'set_private_key', 'get_private_key', 'delete_private_key', 'convert_to_hex',
           'get_notifyr_privkey', 'notifyr']

# %% ../nbs/03_notifyr.ipynb 6
from .client import Client
import time

# %% ../nbs/03_notifyr.ipynb 8
def send_nostr_message(notifyr_client: Client, message: str, recipient_pubkey_hex: str) -> None:
    """a simple function that takes a client and a message and
    sends the message to the client pubkey from the client pubkey
 
    Parameters
    ----------
    notifyr_client : Client
       A client class that will send an encrypted message for us
    message : str
       A message that will be encrypted and sent
    """
    with notifyr_client:
        event = notifyr_client.event_encrypted_message(recipient_hex=recipient_pubkey_hex,
                                                       message=message)
        notifyr_client.publish_event(event)
    pass


# %% ../nbs/03_notifyr.ipynb 10
import keyring
from keyring.errors import NoKeyringError
from .nostr import PrivateKey, PublicKey

# %% ../nbs/03_notifyr.ipynb 12
def set_private_key(notifyr_privkey_hex: str) -> None:
   """Set the private key in the computer keyring

   Parameters
   ----------
   notifyr_privkey_hex : str
       nostr hex private key
   """
   return keyring.set_password(service_name='nostr',
                                username='notifyr',
                                password=notifyr_privkey_hex)

def get_private_key() -> str:
    """get the nostr hex private key from the computer key ring

    Returns
    -------
    str
        nostr hex private key
    """
    return keyring.get_password(service_name='nostr',
                                username='notifyr')

def delete_private_key() -> None:
    """delete the nostr hex private key from the computer key ring.
    This is not used in the decorator function, but may be used
    if need for testing
    """
    return keyring.delete_password(service_name='nostr',
                                   username='notifyr')


# %% ../nbs/03_notifyr.ipynb 21
def convert_to_hex(pubkey: str) -> str:
    """make sure the pubkey is hex

    Parameters
    ----------
    pubkey : str
        hex or npub (bech32) pubkey

    Returns
    -------
    str
        hex pubkey
    """
    if pubkey.startswith('npub'):
         pubkey = \
            PublicKey.from_npub(pubkey).hex()
    return pubkey

# %% ../nbs/03_notifyr.ipynb 22
def get_notifyr_privkey() -> str:
    """returns a private key from keychain and
    sets a new one if one doesn't exist

    Returns
    -------
    str
        private key in hex format
    """
    privkey_hex = get_private_key()
    if privkey_hex is None:
        privkey_hex = PrivateKey().hex()
    set_private_key(privkey_hex)
    assert get_private_key() == privkey_hex
    return privkey_hex

# %% ../nbs/03_notifyr.ipynb 23
import functools

# %% ../nbs/03_notifyr.ipynb 24
def notifyr(func=None, recipient_pubkey: str = None, relay_urls: list[str] = None):
   """A decorator that will set a nostr private key to `func.notifyr_privkey_hex
   and use that key to send an encrypted message to it's own public key on the start
   and termination of the decorated function. The output will send whether the function
   runs completely or ends in an error with an informative message.

   Parameters
   ----------
   func : function
       the function to be decorated

   Returns
   -------
   function
       the decorated function

   Raises
   ------
   e
       if the function fails, else returns the function result
   """
   notifyr_privkey_hex = get_private_key()
   if notifyr_privkey_hex is None:
      notifyr_privkey_hex = PrivateKey().hex()
   set_private_key(notifyr_privkey_hex)
   assert get_private_key() == notifyr_privkey_hex
   if relay_urls is None:
      relay_urls = ['wss://relay.damus.io',
                    'wss://brb.io']
   if recipient_pubkey is None:
      recipient_pubkey_hex = \
         PrivateKey.from_hex(notifyr_privkey_hex).public_key.hex()
   else:
      if recipient_pubkey.startswith('npub'):
         recipient_pubkey_hex = \
            PublicKey.from_npub(recipient_pubkey).hex()
      else:
         recipient_pubkey_hex = recipient_pubkey

   if func is None:
        return lambda func: notifyr(func=func,
                                    recipient_pubkey=recipient_pubkey,
                                    relay_urls=relay_urls)

   @functools.wraps(func)
   def notifier(*args,**kwargs):
      notifyr_client = Client(private_key_hex=notifyr_privkey_hex,
                              relay_urls=relay_urls)
      notifyr_pubkey_hex = notifyr_client.public_key.hex()
      function_name = func.__name__
      message = f'**process name**: {function_name} started!'
      send_nostr_message(recipient_pubkey_hex=recipient_pubkey_hex,
                         notifyr_client=notifyr_client,
                         message=message)
      try:
         result = func(*args,**kwargs)
         message = f'**process name**: {function_name}\n' \
                   f'**finished** - preview of result:\n' \
                   f'-----------------------------\n\n'\
                   f'{str(result)[:100]}'
      except Exception as e:
         result = e
         message = f'**process name**: {function_name}\n' \
                   f'**failed** with error:\n\t{type(e).__name__}: {e}'
      send_nostr_message(recipient_pubkey_hex=recipient_pubkey_hex,
                         notifyr_client=notifyr_client,
                         message=message)
      if issubclass(type(result), Exception):
         raise result
      else:
         return result
   notifier.notifyr_private_key = notifyr_privkey_hex
   return notifier

# %% ../nbs/03_notifyr.ipynb 25
def notifyr(func=None, recipient_pubkey: str = None, relay_urls: list[str] = None):
   """A decorator that will set a nostr private key to `func.notifyr_privkey_hex
   and use that key to send an encrypted message to it's own public key on the start
   and termination of the decorated function. The output will send whether the function
   runs completely or ends in an error with an informative message.

   Parameters
   ----------
   func : function
       the function to be decorated

   Returns
   -------
   function
       the decorated function

   Raises
   ------
   e
       if the function fails, else returns the function result
   """
   notifyr_privkey_hex = get_notifyr_privkey()
   if relay_urls is None:
      relay_urls = ['wss://relay.damus.io',
                    'wss://brb.io']
   if recipient_pubkey is None:
      recipient_pubkey_hex = \
         PrivateKey.from_hex(notifyr_privkey_hex).public_key.hex()
   else:
      recipient_pubkey_hex = convert_to_hex(recipient_pubkey)
   notifyr_client = Client(private_key_hex=notifyr_privkey_hex,
                           relay_urls=relay_urls)
   if func is None:
        return lambda func: notifyr(func=func,
                                    recipient_pubkey=recipient_pubkey,
                                    relay_urls=relay_urls)
   @functools.wraps(func)
   def notifier(*args,**kwargs):
      function_name = func.__name__
      message = f'**process name**: {function_name} started!'
      send_nostr_message(recipient_pubkey_hex=recipient_pubkey_hex,
                         notifyr_client=notifyr_client,
                         message=message)
      try:
         result = func(*args,**kwargs)
         message = f'**process name**: {function_name}\n' \
                   f'**finished** - preview of result:\n' \
                   f'-----------------------------\n\n'\
                   f'{str(result)[:100]}'
      except Exception as e:
         result = e
         message = f'**process name**: {function_name}\n' \
                   f'**failed** with error:\n\t{type(e).__name__}: {e}'
      send_nostr_message(recipient_pubkey_hex=recipient_pubkey_hex,
                         notifyr_client=notifyr_client,
                         message=message)
      if issubclass(type(result), Exception):
         raise result
      else:
         return result
   notifier.notifyr_private_key = notifyr_privkey_hex
   return notifier
