from __future__ import annotations

import string
from abc import ABC
from typing import Any, Dict, Generic, List, Tuple, Type, TypeVar

from . import _google_terms as terms
from .connection import SheetsConnection
from .dtypes import (
    GOOGLE_DTYPES,
    TYPE_MAP,
    EffectiveFmt,
    EffectiveVal,
    Formula,
    String,
    UserEnteredVal,
)
from .interfaces import AuthConfig, TwoDRange
from ._core import GoogleDtype, GoogleValueType

T = TypeVar("T", bound="GSheetView")
FC = TypeVar("FC", bound="CellFormatting")
FG = TypeVar("FG", bound="GridFormatting")
FT = TypeVar("FT", bound="TextFormatting")


class NoConnectionError(Exception):
    """
    Error thrown when attempting to connect via a SheetsConnection that does not
    exist.
    """

    def __init__(self, vtype: Type[GSheetView], *args: object) -> None:
        msg = f"No SheetsConnection has been established for this {vtype}."
        super().__init__(msg, *args)


class Formatting:
    """
    Formatting objects contain methods for adding various update format requests
    to their parent Component.
    """

    def __init__(self, parent: Component[Any, Any, Any]):
        self._parent = parent

    def add_request(self, request: Dict[str, Any]) -> None:
        """
        Adds the passed request to the Formatting object's parent component.

        :param request: An api-ready request.
        :type request: Dict[str, Any]
        """
        self._parent.requests.append(request)

    def ensure_2d_range(self, rng: TwoDRange | None = None) -> TwoDRange:
        """
        Convenience method for ensuring that all requests generated by this
        Formatting object have a TwoDRange attached to them, if one isn't manually
        supplied.

        :param rng: A manually generated TwoDRange, defaults to None
        :type rng: TwoDRange, optional
        :return: The passed TwoDRange, or a range generated from the Formatting
            object's parent Component.
        :rtype: TwoDRange
        """
        return rng if rng else self._parent.range


class CellFormatting(Formatting):
    """
    Contains methods for generating format requests that update cell properties.
    """

    ...


class GridFormatting(Formatting):
    """
    Contains methods for generating format requests that update grid properties.
    """

    ...


class TextFormatting(Formatting):
    """
    Contains methods for generating format requests that update the properties of
    text in one or more cells.
    """

    ...


class GSheetView(ABC):
    """
    Abstract base class for the different ways of viewing a Google Sheet.
    """

    def __init__(
        self,
        gsheet_id: str,
        *,
        auth_config: AuthConfig | None = None,
        sheets_conn: SheetsConnection | None = None,
        autoconnect: bool = True,
    ) -> None:
        super().__init__()
        if not sheets_conn and autoconnect:
            sheets_conn = SheetsConnection(auth_config=auth_config)
        self._conn = sheets_conn
        self._auth = auth_config
        self._gsheet_id = gsheet_id
        self._requests: List[Dict[str, Any]] = []

    @property
    def requests(self) -> List[Dict[str, Any]]:
        """
        :return: The list of current (uncommitted) requests.
        :rtype: List[Dict[str, Any]]
        """
        return self._requests

    @property
    def conn(self) -> SheetsConnection:
        """
        :raises NoConnectionError: If the view's connection is null.
        :return: The view's SheetsConnection.
        :rtype: SheetsConnection
        """
        if not self._conn:
            raise NoConnectionError(type(self))
        return self._conn

    @property
    def auth(self) -> AuthConfig:
        """
        :raises NoConnectionError: If the view's auth config is null.
        :return: The view's AuthConfig.
        :rtype: AuthConfig
        """
        if not self._auth:
            raise NoConnectionError(type(self))
        return self._auth

    @property
    def gsheet_id(self) -> str:
        """
        :return: The id of the Google Sheet this view is connected to.
        :rtype: str
        """
        return self._gsheet_id

    def commit(self) -> Dict[str, Any]:
        """
        Commits the amassed requests on this view, sending them to the Sheets api
        as a batch update request.

        :raises NoConnectionError: If the view's SheetsConnection is null.
        :return: The response from the api.
        :rtype: Dict[str, Any]
        """
        if not self._conn:
            raise NoConnectionError(type(self))
        results = self._conn.execute_requests(self._gsheet_id, self._requests)
        self._requests = []
        return results

    @classmethod
    def _parse_row_data(
        cls,
        row_data: List[Dict[str, List[Dict[str, Any]]]],
        value_type: GoogleValueType = EffectiveVal,
    ) -> Tuple[List[List[Any]], List[List[Dict[str, Any]]]]:
        """
        Parses the dictionary returned by SheetsConnection.get_data and extracts
        only the relevant data.

        :param row_data: The raw data to parse.
        :type row_data: List[Dict[str, List[Dict[str, Any]]]]
        :param value_type: The value representation to extract from the raw data,
            defaults to EffectiveVal
        :type value_type: GoogleValueType, optional
        :return: A tuple containing a list of extracted data and another list of
            extracted formatting information.
        :rtype: Tuple[List[List[Any]], List[List[Dict[str, Any]]]]
        """
        values: List[List[Any]] = []
        formats: List[List[Dict[str, Any]]] = []
        for row in row_data:
            value_list: List[Any] = []
            fmt_list: List[Dict[str, Any]] = []
            for cell in row.get(terms.VALUES, []):
                raw_value = cell.get(str(value_type))
                fmt = cell.get(str(EffectiveFmt), {})
                value = raw_value
                if value_type in (UserEnteredVal, EffectiveVal):
                    if raw_value:
                        for dtype in GOOGLE_DTYPES:
                            value = raw_value.get(str(dtype))
                            if value:
                                if value_type == UserEnteredVal:  # type: ignore
                                    value = dtype.parse(value)
                                break
                value_list.append(value)
                fmt_list.append(fmt)
            values.append(value_list)
            formats.append(fmt_list)
        return values, formats

    def _get_data(
        self,
        gsheet_id: str,
        rng_str: str,
        value_type: GoogleValueType = EffectiveVal,
    ) -> Tuple[List[List[Any]], List[List[Dict[str, Any]]]]:
        """
        Fetches data from the view's SheetsConnection for the specified Google
        Sheet and parses it.

        :param gsheet_id: The Google Sheet to fetch data from.
        :type gsheet_id: str
        :param rng_str: The range within the Google Sheet to fetch data from.
        :type rng_str: str
        :param value_type: The value representation to extract from the raw data,
            defaults to EffectiveVal
        :type value_type: GoogleValueType, optional
        :return: A tuple containing a list of data values and another list of
            formatting information.
        :rtype: Tuple[List[List[Any]], List[List[Dict[str, Any]]]]
        """
        raw = self.conn.get_data(gsheet_id, [rng_str])
        row_data = raw[terms.TABS_PROP][0][terms.DATA][0][terms.ROWDATA]
        return self._parse_row_data(row_data, value_type=value_type)

    def _write_values(self: T, data: List[List[Any]], rng_dict: Dict[str, int]) -> T:
        """
        Generates an update cells request for writing values to the target range.

        :param data: The data to write.
        :type data: List[List[Any]]
        :param rng_dict: The range properties to write to.
        :type rng_dict: Dict[str, int]
        :return: This Formatting object.
        :rtype: T
        """
        write_values = [
            [self._gen_cell_write_value(val) for val in row] for row in data
        ]
        request = {
            "updateCells": {
                terms.FIELDS: "*",
                terms.ROWS: [{terms.VALUES: values} for values in write_values],
                terms.RNG: rng_dict,
            }
        }
        self._requests.append(request)
        return self

    @staticmethod
    def _gen_cell_write_value(python_val: Any) -> Dict[str, Any]:
        """
        Converts a python value to its corresponding Google Dtype and wraps it in
        the key-value payload expected by the Sheets api.

        :param python_val: Any value. Only numeric values, booleans, and "formulas"
            (strings that start with =) will be converted. All other python data
            will be converted to its string representation.
        :type python_val: Any
        :return: A dictionary containing the details the Sheet's api expects when
            writing data.
        :rtype: Dict[str, Any]
        """
        dtype: GoogleDtype
        type_ = type(python_val)
        if (
            isinstance(python_val, str)
            and len(python_val) >= 2
            and python_val[0] == "="
        ):
            dtype = Formula
        else:
            dtype = TYPE_MAP.get(type_, String)
        return {UserEnteredVal.value_key: {dtype.type_key: python_val}}

    # TODO: Delete this?
    @staticmethod
    def gen_alpha_keys(num: int) -> List[str]:
        """
        Generates a list of characters from the Latin alphabet a la gsheet/excel
        headers.

        :param num: The desired length of the list.
        :type num: int
        :return: A list containing as many letters and letter combos as desired.
            Can be used to generate sets up to 676 in length.
        :rtype: List[str]
        """
        a = string.ascii_uppercase
        result: List[str] = list()
        x = num // 26
        for i in range(x + 1):
            root = a[i - 1] if i > 0 else ""
            keys = [root + a[j] for j in range(26)]
            for k in keys:
                result.append(k) if len(result) < num else None
        return result


class Component(GSheetView, Generic[FC, FG, FT]):
    """
    Base class for Tab and Range.
    """

    def __init__(
        self,
        *,
        gsheet_range: TwoDRange,
        gsheet_id: str,
        grid_formatting: Type[FG],
        text_formatting: Type[FT],
        cell_formatting: Type[FC],
        auth_config: AuthConfig | None = None,
        sheets_conn: SheetsConnection | None = None,
        autoconnect: bool = True,
    ) -> None:
        super().__init__(
            gsheet_id,
            auth_config=auth_config,
            sheets_conn=sheets_conn,
            autoconnect=autoconnect,
        )
        self._rng = gsheet_range
        self._values: List[List[Any]] = []
        self._formats: List[List[Dict[str, Any]]] = []
        self._format_grid = grid_formatting(self)
        self._format_text = text_formatting(self)
        self._format_cell = cell_formatting(self)

    @property
    def tab_id(self) -> int:
        """
        :return: The id of the linked tab.
        :rtype: int
        """
        return self._rng.tab_id

    @property
    def range_str(self) -> str:
        """
        :return: The string representation of the range specified by this Component.
        :rtype: str
        """
        return str(self._rng)

    @property
    def range(self) -> TwoDRange:
        """
        :return: The TwoDRange representation of the range specified by this
            Component.
        :rtype: TwoDRange
        """
        return self._rng

    @property
    def format_grid(self) -> FG:
        """
        :return: The GridFormatting subclass associated with this Component type.
        :rtype: FG
        """
        return self._format_grid

    @property
    def format_text(self) -> FT:
        """
        :return: The TextFormatting subclass associated with this Component type.
        :rtype: FT
        """
        return self._format_text

    @property
    def format_cell(self) -> FC:
        """
        :return: The CellFormatting subclass associated with this Component type.
        :rtype: FC
        """
        return self._format_cell

    @property
    def values(self) -> List[List[Any]]:
        """
        :return: The fetched data values in this Component's cells.
        :rtype: List[List[Any]]
        """
        return self._values

    @values.setter
    def values(self, new_values: List[List[Any]]) -> None:
        """
        :param new_values: The values to overwrite this Component's current values
            with.
        :type new_values: List[List[Any]]
        """
        self._values = new_values

    @property
    def formats(self) -> List[List[Dict[str, Any]]]:
        """
        :return: The fetched formatting properties of this Component's cells.
        :rtype: List[List[Dict[str, Any]]]
        """
        return self._formats

    @formats.setter
    def formats(self, new_formats: List[List[Dict[str, Any]]]) -> None:
        """
        :param new_formats: The formats to overwrite this Component's current formats
            with.
        :type new_formats: List[List[Dict[str, Any]]]
        """
        self._formats = new_formats

    @property
    def data_shape(self) -> Tuple[int, int]:
        """
        :return: The row length and column width of this Component's data.
        :rtype: Tuple[int, int]
        """
        width = len(self._values[0]) if self._values else 0
        return len(self._values), width
