# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/01_enum.ipynb.

# %% auto 0
__all__ = ['Direction', 'Scope', 'ResolutionOrder', 'BoundFormat', 'DecoratedFormat']

# %% ../nbs/01_enum.ipynb 6
from inspect import (BoundArguments, )
from itertools import permutations
from enum import StrEnum, auto

# %% ../nbs/01_enum.ipynb 8
from typing import (Callable, Type, Self)

# %% ../nbs/01_enum.ipynb 10
#| export


# %% ../nbs/01_enum.ipynb 12
#| export


# %% ../nbs/01_enum.ipynb 14
from .type import T, P, BoundReturn, DecoReturn

# %% ../nbs/01_enum.ipynb 16
@classmethod
def safe(cls: Type[T], member: Type[T] | str | None = None)  -> T:
    '''Retrieve a safe enum member based on the given value. If the member is not a valid member,
    the default member is returned instead.

    Parameters
    ----------
    cls : Type[T]
        The enum class to operate on.
        
    member : Type[T] | str | None, optional
        The member to retrieve. Can be an enum member or its name as a string. Default is None.

    Returns
    -------
    T
        The retrieved enum member.
    '''
    return member if member in cls.__members__.values() else cls.default

# %% ../nbs/01_enum.ipynb 18
class Direction(StrEnum):
    '''
    Enumeration for direction, with a safe method for retrieval.

    Attributes
    ----------
    lr : auto
        Left to Right direction.
    rl : auto
        Right to Left direction.
    default : lr
        The default direction.
    
    Methods
    -------
    safe(cls: Type[T], member: Type[T] | str | None = None) -> T
        Method to safely retrieve enumeration members.
    '''
    lr = auto()
    rl = auto()
    default = lr
    safe = safe

# %% ../nbs/01_enum.ipynb 20
class Scope(StrEnum):
    '''Enumeration for scope, with methods for permutations and safe retrieval.

    Attributes
    ----------
    outer : auto
        Represents an outer scope.
    inner : auto
        Represents an inner scope.
    extra : auto
        Represents an extra scope, beyond inner and outer.
    default : inner
        The default scope.
        
    Methods
    -------
    safe(cls: Type[T], member: Type[T] | str | None = None) -> T
        Method to safely retrieve enumeration members.
        
    permutations(cls: Type[Self]) -> tuple[Type[Self], ...]:
        Generate all possible permutations of scope values.
        
    permutation_key(perm: tuple[Type[Self], ...]) -> str:
        Generate a key string from a scope permutation.
        
    permutation_keys(perms: tuple[Type[Self], ...] | None = None) -> tuple[str, ...]:
        Generate key strings for all or specified scope permutations.
        
    perm_dict(cls: Type[Self]) -> dict[str, Type[Self]]:
        Create a dictionary mapping permutation keys to permutations.
    '''
    outer = auto()
    inner = auto()
    extra = auto()
    default = inner
    safe = safe
    
    @classmethod
    def permutations(cls: Type[Self]) -> tuple[Type[Self], ...]:
        return list(permutations(cls, len(tuple(cls))))
    
    @classmethod
    def permutation_key(cls: Type[Self], perm: tuple[Type[Self], ...]) -> str:
        return ' '.join(perm).title().replace(' ', '')
    
    @classmethod
    def permutation_keys(cls: Type[Self], perms: tuple[Type[Self], ...] | None = None) -> tuple[str, ...]:
        perms = perms or cls.permutations()
        return tuple(map(cls.permutation_key, perms))
    
    @classmethod
    def perm_dict(cls: Type[Self]) -> dict[str, Type[Self]]:
        perms = cls.permutations()
        return dict(zip(cls.permutation_keys(perms), perms))

# %% ../nbs/01_enum.ipynb 22
class ResolutionOrder(StrEnum):
    '''Defines the order in which arguments are resolved.

    Attributes
    ----------
    OuterInnerExtra : auto
        Order where outer arguments are resolved before inner, followed by extra.
    OuterExtraInner : auto
        Order where outer arguments are resolved before extra, followed by inner.
    InnerOuterExtra : auto
        Order where inner arguments are resolved before outer, followed by extra.
    InnerExtraOuter : auto
        Order where inner arguments are resolved before extra, followed by outer.
    ExtraOuterInner : auto
        Order where extra arguments are resolved before outer, followed by inner.
    ExtraInnerOuter : auto
        Order where extra arguments are resolved before inner, followed by outer.
    
    Methods
    -------
    safe(cls: Type[T], member: Type[T] | str | None = None) -> T
        Method to safely retrieve enumeration members.
    '''
    OuterInnerExtra = auto()
    OuterExtraInner = auto()
    InnerOuterExtra = auto()
    InnerExtraOuter = auto()
    ExtraOuterInner = auto()
    ExtraInnerOuter = auto()
    
    OuterInner = OuterInnerExtra
    OuterExtra = OuterExtraInner
    InnerOuter = InnerOuterExtra
    InnerExtra = InnerExtraOuter
    ExtraOuter = ExtraOuterInner
    ExtraInner = ExtraInnerOuter
    
    
    Inner = InnerOuterExtra
    Extra = ExtraInnerOuter
    Outer = OuterInnerExtra
    
    default = InnerOuterExtra
    
    safe = safe

# %% ../nbs/01_enum.ipynb 24
class BoundFormat(StrEnum): # ReturnFormat
    '''Defines formats for returning bound arguments.

    Attributes
    ----------
    none : auto
        No return.
    nones : auto
        Return None in both positions of a tuple.
    bound : auto
        Return BoundArguments.
    boundnone : auto
        Return BoundArguments and None.
    original : auto
        Return the original *args and **kwargs.
    arguments : auto
        Return an OrderedDict of arguments.
    args : auto
        Return only *args.
    kwargs : auto
        Return only **kwargs.
    argskwargs : auto
        Return both *args and **kwargs in a tuple.
    
    Methods
    -------
    format(cls: Type[Self], bound: BoundArguments, *args, **kwargs) -> BoundReturn
        Formats the bound arguments according to the specified option.
        
    safe(cls: Type[T], member: Type[T] | str | None = None) -> T
        Method to safely retrieve enumeration members.
    '''
    none = auto()
    nones = auto()
    bound = auto()
    boundnone = auto()
    original = auto()
    arguments = auto()
    args = auto()
    kwargs = auto()
    argskwargs = auto()
    default = argskwargs
    
    safe = safe
    
    @classmethod
    def format(
        cls: Type[Self], 
        bound: BoundArguments, 
        *args: P.args,
        __format: Type[Self] = 'default',
        __locals_if_empty: bool = False,
        **kwargs: P.kwargs,
    ) -> BoundReturn:
        '''Formats the bound arguments according to the specified option.
        
        Parameters
        ----------
        bound : BoundArguments    
            The bound arguments to format.
        
        *args : P.args
            The positional arguments to format / maybe return.
            
        __format : BoundFormat, optional
            The format to use. Default is 'default'.
            
        __locals_if_empty : bool, optional
            If True, return the local arguments (*args /**kwargs) if the bound arguments are empty. 
            Default is False.
            
        **kwargs : P.kwargs
            The keyword arguments to format / maybe return.
            
        Returns
        -------
        BoundReturn
            The formatted bound arguments.
        '''
        __format = kwargs.get('__format', __format)
        
        __missing = kwargs.get('__locals_if_empty', __locals_if_empty)
        format = cls.safe(__format)
        
        # if the bound args and kwargs are empty, return the original args and kwargs
        
        marg, mkws = bound.args or args, bound.kwargs or kwargs
        match format:
            case cls.none: return None
            case cls.nones: return None, None
            case cls.bound: return bound
            case cls.boundnone: return bound, None
            case cls.original: return args, kwargs
            case cls.arguments: return bound.arguments
            case cls.args: return marg if __missing else bound.args
            case cls.kwargs: return mkws if __missing else bound.kwargs
            case cls.argskwargs: return(marg, mkws) if __missing else bound.args, bound.kwargs
            case _: return bound.args, bound.kwargs

# %% ../nbs/01_enum.ipynb 25
class DecoratedFormat(StrEnum): # DecoratedReturn
    '''Defines formats for returning the result of a decoration process.

    Attributes
    ----------
    decorator : auto
        Return the decorator itself.
    decorated : auto
        Return the decorated function.
    called : auto
        Apply the decorator and call the resulting function.
    error : auto
        Raise a TypeError if the function is not callable.
    none : auto
        No return.
    
    Methods
    -------
    format(cls: Type[Self], dec: Callable, fn: Callable | None,  /, *args, **kwargs) -> DecoReturn
        Formats the decoration result according to the specified option.
    
    safe(cls: Type[T], member: Type[T] | str | None = None) -> T
        Method to safely retrieve enumeration members.
    '''
    decorator = auto()
    decorated = auto()
    called = auto()
    error = auto()
    none = auto()
    default = none
    
    safe = safe
    
    @classmethod
    def format(
        cls: Type[Self], dec: Callable, fn: Callable | None = None,  /, *args: P.args, 
        __format: Type[Self] = 'decorated', **kwargs: P.kwargs
    ) -> DecoReturn:
        '''Formats the decoration result according to the specified option.
        
        Parameters
        ----------
        dec : Callable
            The decorator to format.
            
        fn : Callable | None, optional
            The function to decorate. Default is None.
            
        *args : P.args
            The positional arguments to pass to the decorator.
            
        __format : DecoratedFormat, optional
            The format to use. Default is 'decorated'.
            
        **kwargs : P.kwargs
            The keyword arguments to pass to the decorator.
        
        Returns
        -------
        DecoReturn
            The formatted decoration result.
        '''
        format = cls.safe(kwargs.get('__format', __format))
        match format:
            case cls.decorator: return dec
            case cls.decorated: return dec(fn)
            case cls.called: return dec(fn)(*args, **kwargs)
            case cls.error: raise TypeError(f'Expected callable, got {type(fn)}')
            case _: return None
            
    @classmethod
    def infer(cls: Type[Self], fn: Callable | None = None) -> Type[Self]:
        '''Infers the format to use based on whether or not the function is callable.'''
        if fn is None: return DecoratedFormat.decorator
        elif callable(fn): return DecoratedFormat.called
        else: return DecoratedFormat.error
