"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
/**
 * Minimum number of healthy hosts for a server deployment.
 */
class MinimumHealthyHosts {
    constructor(json) {
        this.json = json;
    }
    /**
     * The minimum healhty hosts threshold expressed as an absolute number.
     */
    static count(value) {
        return new MinimumHealthyHosts({
            type: 'HOST_COUNT',
            value
        });
    }
    /**
     * The minmum healhty hosts threshold expressed as a percentage of the fleet.
     */
    static percentage(value) {
        return new MinimumHealthyHosts({
            type: 'FLEET_PERCENT',
            value
        });
    }
    /**
     * @internal
     */
    get _json() {
        return this.json;
    }
}
exports.MinimumHealthyHosts = MinimumHealthyHosts;
/**
 * A custom Deployment Configuration for an EC2/on-premise Deployment Group.
 *
 * @resource AWS::CodeDeploy::DeploymentConfig
 */
class ServerDeploymentConfig extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.deploymentConfigName,
        });
        const resource = new codedeploy_generated_1.CfnDeploymentConfig(this, 'Resource', {
            deploymentConfigName: this.physicalName,
            minimumHealthyHosts: props.minimumHealthyHosts._json,
        });
        this.deploymentConfigName = resource.ref;
        this.deploymentConfigArn = utils_1.arnForDeploymentConfig(this.deploymentConfigName);
    }
    /**
     * Import a custom Deployment Configuration for an EC2/on-premise Deployment Group defined either outside the CDK app,
     * or in a different region.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param serverDeploymentConfigName the properties of the referenced custom Deployment Configuration
     * @returns a Construct representing a reference to an existing custom Deployment Configuration
     */
    static fromServerDeploymentConfigName(scope, id, serverDeploymentConfigName) {
        ignore(scope);
        ignore(id);
        return deploymentConfig(serverDeploymentConfigName);
    }
}
exports.ServerDeploymentConfig = ServerDeploymentConfig;
ServerDeploymentConfig.ONE_AT_A_TIME = deploymentConfig('CodeDeployDefault.OneAtATime');
ServerDeploymentConfig.HALF_AT_A_TIME = deploymentConfig('CodeDeployDefault.HalfAtATime');
ServerDeploymentConfig.ALL_AT_ONCE = deploymentConfig('CodeDeployDefault.AllAtOnce');
function deploymentConfig(name) {
    return {
        deploymentConfigName: name,
        deploymentConfigArn: utils_1.arnForDeploymentConfig(name),
    };
}
function ignore(_x) { return; }
//# sourceMappingURL=data:application/json;base64,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