"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn = require("@aws-cdk/aws-cloudformation");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const yaml = require("js-yaml");
const path = require("path");
const resourceType = 'Custom::SSM-Document';
const ID = `CFN::Resource::${resourceType}`;
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
class Document extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const stack = cdk.Stack.of(this).stackName;
        const fn = this.ensureLambda();
        const name = this.fixDocumentName(props.name);
        if (name.length < 3 || name.length > 128) {
            this.node.addError(`SSM Document name ${name} is invalid. The name must be between 3 and 128 characters.`);
            return;
        }
        let content = props.content;
        if (typeof content === 'string') {
            content = yaml.safeLoad(content.toString());
        }
        const tags = props.tags || {};
        tags.CreatedBy = ID;
        new cfn.CustomResource(this, `SSM-Document-${name}`, {
            provider: cfn.CustomResourceProvider.fromLambda(fn),
            resourceType: resourceType,
            properties: {
                updateDefaultVersion: props.updateDefaultVersion || true,
                name: name,
                content: content,
                documentType: props.documentType || 'Command',
                targetType: props.targetType || '/',
                StackName: stack,
                tags: tags,
            }
        });
    }
    ensureLambda() {
        const stack = cdk.Stack.of(this);
        const constructName = 'SSM-Document-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const policy = new iam.ManagedPolicy(stack, 'SSM-Document-Manager-Policy', {
            managedPolicyName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing SSM documents`,
            statements: [
                new iam.PolicyStatement({
                    actions: [
                        'ssm:ListDocuments',
                        'ssm:ListTagsForResource',
                    ],
                    resources: ['*'],
                }),
                new iam.PolicyStatement({
                    actions: [
                        'ssm:CreateDocument',
                        'ssm:AddTagsToResource',
                    ],
                    resources: ['*'],
                    conditions: {
                        StringEquals: {
                            'aws:RequestTag/CreatedBy': ID,
                        }
                    },
                }),
                new iam.PolicyStatement({
                    actions: [
                        'ssm:DeleteDocument',
                        'ssm:DescribeDocument',
                        'ssm:GetDocument',
                        'ssm:ListDocumentVersions',
                        'ssm:ModifyDocumentPermission',
                        'ssm:UpdateDocument',
                        'ssm:UpdateDocumentDefaultVersion',
                        'ssm:AddTagsToResource',
                        'ssm:RemoveTagsFromResource',
                    ],
                    resources: ['*'],
                    conditions: {
                        StringEquals: {
                            'aws:ResourceTag/CreatedBy': ID,
                        }
                    },
                }),
            ],
        });
        const role = new iam.Role(stack, 'SSM-Document-Manager-Role', {
            roleName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing SSM documents`,
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ]
        });
        const fn = new lambda.Function(stack, constructName, {
            functionName: `${stack.stackName}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage SSM Documents',
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: cdk.Duration.minutes(lambdaTimeout)
        });
        return fn;
    }
    fixDocumentName(name) {
        return name.replace(/[^a-zA-Z0-9_.-]+/g, '-');
    }
}
exports.Document = Document;
//# sourceMappingURL=data:application/json;base64,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