"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateCA = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
class PrivateCA extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.route53_zone = new aws_cdk_lib_1.aws_route53.PrivateHostedZone(this, 'CpInternalHostedZone', {
            vpc: props.cluster.vpc,
            zoneName: props.hostedZoneName,
        });
        //TODO : Customize this
        const cfnCertificateAuthority = new aws_cdk_lib_1.aws_acmpca.CfnCertificateAuthority(this, 'CA', {
            type: 'ROOT',
            keyAlgorithm: 'RSA_2048',
            signingAlgorithm: 'SHA256WITHRSA',
            subject: {
                country: 'US',
                organization: 'Amazon',
                organizationalUnit: 'AWS',
                state: 'WA',
                commonName: props.hostedZoneName,
                locality: 'Seattle',
            },
        });
        this.private_ca_arn = cfnCertificateAuthority.attrArn;
        const internalRootCaCert = new aws_cdk_lib_1.aws_acmpca.CfnCertificate(this, 'PrivCaCert', {
            certificateAuthorityArn: this.private_ca_arn,
            certificateSigningRequest: cfnCertificateAuthority.attrCertificateSigningRequest,
            templateArn: 'arn:aws:acm-pca:::template/RootCACertificate/V1',
            signingAlgorithm: 'SHA256WITHRSA',
            validity: {
                value: 10,
                type: 'YEARS',
            },
        });
        const internalRootCaActivation = new aws_cdk_lib_1.aws_acmpca.CfnCertificateAuthorityActivation(this, 'PrivCaActivation', {
            certificateAuthorityArn: this.private_ca_arn,
            certificate: internalRootCaCert.attrCertificate,
        });
        const wild_card_cert = new aws_cdk_lib_1.aws_certificatemanager.CfnCertificate(this, 'CpCert', {
            domainName: `*.${props.hostedZoneName}`,
            certificateAuthorityArn: this.private_ca_arn,
        });
        wild_card_cert.node.addDependency(internalRootCaActivation);
        const cluster_issuer_namespace_manifest = {
            apiVersion: 'v1',
            kind: 'Namespace',
            metadata: { name: 'aws-pca-issuer' },
        };
        const cert_issuer_namespace = props.cluster.addManifest('ClusterIssuerNamespaceManifest', cluster_issuer_namespace_manifest);
        const service_account = props.cluster.addServiceAccount('PrivateCASA');
        service_account.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: [this.private_ca_arn],
            actions: [
                'acm-pca:Get*',
                'acm-pca:Issue*',
                'acm-pca:Describe*',
            ],
        }));
        const service_account_external_dns = props.cluster.addServiceAccount('ExternalDns');
        service_account_external_dns.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'route53:*',
            ],
        }));
        // props.nodegroup.role.addToPrincipalPolicy(new PolicyStatement({
        //   resources: ['*'],
        //   actions: [
        //     'route53:*',
        //   ],
        // }));
        props.cluster.addHelmChart('ExternalDnsHelm', {
            repository: 'https://kubernetes-sigs.github.io/external-dns/',
            chart: 'external-dns',
            // namespace: 'external-dns',
            release: 'external-dns',
            wait: true,
            values: {
                provider: 'aws',
                policy: 'sync',
                aws: {
                    region: aws_cdk_lib_1.Stack.of(this).region,
                    zoneType: 'private',
                },
                // domainFilters: [
                //   hosted_zone.zoneName,
                // ],
                serviceAccount: {
                    create: false,
                    name: service_account_external_dns.serviceAccountName,
                },
            },
        });
        // TODO : Dont know why, but pod SA still wants node to lookup Describe* calls on PCA
        props.nodegroup.role.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: [this.private_ca_arn],
            actions: [
                'acm-pca:Get*',
                'acm-pca:Issue*',
                'acm-pca:Describe*',
            ],
        }));
        service_account.node.addDependency(cert_issuer_namespace);
        const deploy_cert_manager = props.cluster.addHelmChart('CertManager', {
            repository: 'https://charts.jetstack.io',
            chart: 'cert-manager',
            namespace: 'cert-manager',
            release: 'cert-manager',
            wait: true,
            version: 'v1.5.0',
            values: {
                installCRDs: true,
                webhook: {
                    timeoutSeconds: 30,
                },
            },
        });
        const priv_ca_issuer_helm = props.cluster.addHelmChart('PrivateCaIssuer', {
            repository: 'https://cert-manager.github.io/aws-privateca-issuer',
            chart: 'aws-pca-issuer',
            namespace: 'aws-pca-issuer',
            release: 'aws-pca-issuer',
            wait: true,
            values: {
                serviceAccount: {
                    create: 'false',
                    name: service_account.serviceAccountName,
                },
            },
        });
        priv_ca_issuer_helm.node.addDependency(internalRootCaActivation, deploy_cert_manager);
        const deploy_waiter = props.cluster.addHelmChart('DummyWaiterPostCertManager', {
            repository: 'https://anshrma.github.io/helm-charts',
            chart: 'helm-waiter',
            // namespace: 'default',
            release: 'helm-waiter',
            wait: true,
        });
        deploy_waiter.node.addDependency(priv_ca_issuer_helm);
        const cluster_issuer_manifest = {
            apiVersion: 'awspca.cert-manager.io/v1beta1',
            kind: 'AWSPCAClusterIssuer',
            metadata: { name: props.clusterIssuerName },
            spec: {
                arn: this.private_ca_arn,
                // arn: cfnCertificateAuthority.attrArn,
                region: aws_cdk_lib_1.Stack.of(this).region,
            },
        };
        const cluster_issuer = props.cluster.addManifest('ClusterIssuerManifest', cluster_issuer_manifest);
        cluster_issuer.node.addDependency(deploy_waiter);
        const namespace_manifest = {
            apiVersion: 'v1',
            kind: 'Namespace',
            metadata: { name: props.namespace },
        };
        const ns = props.cluster.addManifest('NSManifests', namespace_manifest);
        const cert_manifest = {
            apiVersion: 'cert-manager.io/v1',
            kind: 'Certificate',
            metadata: {
                name: props.cacertname,
                namespace: props.namespace,
            },
            spec: {
                commonName: props.hostedZoneName,
                dnsNames: props.dnsNames,
                duration: '2160h0m0s',
                issuerRef: {
                    group: 'awspca.cert-manager.io',
                    kind: 'AWSPCAClusterIssuer',
                    name: props.clusterIssuerName,
                },
                renewBefore: '360h0m0s',
                secretName: `${props.clusterIssuerName}-secret`,
                usages: [
                    'server auth',
                    'client auth',
                ],
                privateKey: {
                    algorithm: 'RSA',
                    size: 2048,
                },
            },
        };
        const cert = props.cluster.addManifest('CertManifests', cert_manifest);
        cert.node.addDependency(ns, cluster_issuer);
        // new CfnOutput(this, 'ExtDnsRoleArn', { value: service_account_external_dns.role.roleArn });
        // new CfnOutput(this, 'CertIssuerRoleArnOutput', { value: service_account.serviceAccountName });
    }
}
exports.PrivateCA = PrivateCA;
//# sourceMappingURL=data:application/json;base64,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