"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobDefinition = exports.PlatformCapabilities = exports.LogDriver = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_batch_1 = require("aws-cdk-lib/aws-batch");
const job_definition_image_config_1 = require("./job-definition-image-config");
/**
 * The log driver to use for the container.
 */
var LogDriver;
(function (LogDriver) {
    /**
     * Specifies the Amazon CloudWatch Logs logging driver.
     */
    LogDriver["AWSLOGS"] = "awslogs";
    /**
     * Specifies the Fluentd logging driver.
     */
    LogDriver["FLUENTD"] = "fluentd";
    /**
     * Specifies the Graylog Extended Format (GELF) logging driver.
     */
    LogDriver["GELF"] = "gelf";
    /**
     * Specifies the journald logging driver.
     */
    LogDriver["JOURNALD"] = "journald";
    /**
     * Specifies the logentries logging driver.
     */
    LogDriver["LOGENTRIES"] = "logentries";
    /**
     * Specifies the JSON file logging driver.
     */
    LogDriver["JSON_FILE"] = "json-file";
    /**
     * Specifies the Splunk logging driver.
     */
    LogDriver["SPLUNK"] = "splunk";
    /**
     * Specifies the syslog logging driver.
     */
    LogDriver["SYSLOG"] = "syslog";
})(LogDriver = exports.LogDriver || (exports.LogDriver = {}));
/**
 * Platform capabilities
 */
var PlatformCapabilities;
(function (PlatformCapabilities) {
    /**
     * Specifies EC2 environment.
     */
    PlatformCapabilities["EC2"] = "EC2";
    /**
     * Specifies Fargate environment.
     */
    PlatformCapabilities["FARGATE"] = "FARGATE";
})(PlatformCapabilities = exports.PlatformCapabilities || (exports.PlatformCapabilities = {}));
/**
 * Batch Job Definition.
 *
 * Defines a batch job definition to execute a specific batch job.
 */
class JobDefinition extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.jobDefinitionName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_batch_alpha_JobDefinitionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, JobDefinition);
            }
            throw error;
        }
        this.validateProps(props);
        this.imageConfig = new job_definition_image_config_1.JobDefinitionImageConfig(this, props.container);
        const isFargate = !!props.platformCapabilities?.includes(PlatformCapabilities.FARGATE);
        const jobDef = new aws_batch_1.CfnJobDefinition(this, 'Resource', {
            jobDefinitionName: props.jobDefinitionName,
            containerProperties: this.buildJobContainer(props.container, isFargate),
            type: 'container',
            nodeProperties: props.nodeProps
                ? {
                    mainNode: props.nodeProps.mainNode,
                    nodeRangeProperties: this.buildNodeRangeProps(props.nodeProps, isFargate),
                    numNodes: props.nodeProps.count,
                }
                : undefined,
            parameters: props.parameters,
            retryStrategy: {
                attempts: props.retryAttempts || 1,
            },
            timeout: {
                attemptDurationSeconds: props.timeout ? props.timeout.toSeconds() : undefined,
            },
            platformCapabilities: props.platformCapabilities ?? [PlatformCapabilities.EC2],
            propagateTags: props.propagateTags,
        });
        // add read secrets permission to execution role
        if (props.container.secrets && props.container.executionRole) {
            const executionRole = props.container.executionRole;
            Object.values(props.container.secrets).forEach((secret) => {
                secret.grantRead(executionRole);
            });
        }
        this.jobDefinitionArn = this.getResourceArnAttribute(jobDef.ref, {
            service: 'batch',
            resource: 'job-definition',
            resourceName: this.physicalName,
        });
        this.jobDefinitionName = this.getResourceNameAttribute(jobDef.ref);
    }
    /**
     * Imports an existing batch job definition by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param jobDefinitionArn
     */
    static fromJobDefinitionArn(scope, id, jobDefinitionArn) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const jobDefName = stack.splitArn(jobDefinitionArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefinitionArn;
                this.jobDefinitionName = jobDefName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports an existing batch job definition by its name.
     * If name is specified without a revision then the latest active revision is used.
     *
     * @param scope
     * @param id
     * @param jobDefinitionName
     */
    static fromJobDefinitionName(scope, id, jobDefinitionName) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const jobDefArn = stack.formatArn({
            service: 'batch',
            resource: 'job-definition',
            arnFormat: aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME,
            resourceName: jobDefinitionName,
        });
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefArn;
                this.jobDefinitionName = jobDefinitionName;
            }
        }
        return new Import(scope, id);
    }
    deserializeEnvVariables(env) {
        const vars = new Array();
        if (env === undefined) {
            return vars;
        }
        Object.keys(env).map((name) => {
            vars.push({ name, value: env[name] });
        });
        return vars;
    }
    /**
     * Validates the properties provided for a new job definition.
     */
    validateProps(props) {
        if (props === undefined) {
            return;
        }
        if (props.platformCapabilities !== undefined && props.platformCapabilities.includes(PlatformCapabilities.FARGATE)
            && props.container.executionRole === undefined) {
            throw new Error('Fargate job must have executionRole set');
        }
        if (props.platformCapabilities !== undefined && props.platformCapabilities.includes(PlatformCapabilities.FARGATE)
            && props.container.gpuCount !== undefined) {
            throw new Error('Fargate job must not have gpuCount set');
        }
        if ((props.platformCapabilities === undefined || props.platformCapabilities.includes(PlatformCapabilities.EC2))
            && props.container.assignPublicIp !== undefined) {
            throw new Error('EC2 job must not have assignPublicIp set');
        }
    }
    buildJobContainer(container, isFargate) {
        if (container === undefined) {
            return undefined;
        }
        // If the AWS_*** environment variables are not explicitly set to the container, infer them from the current environment.
        // This makes the usage of tools like AWS SDK inside the container frictionless
        const environment = this.deserializeEnvVariables(container.environment);
        if (!environment.find((x) => x.name === 'AWS_REGION')) {
            environment.push({
                name: 'AWS_REGION',
                value: aws_cdk_lib_1.Stack.of(this).region,
            });
        }
        if (!environment.find((x) => x.name === 'AWS_ACCOUNT')) {
            environment.push({
                name: 'AWS_ACCOUNT',
                value: aws_cdk_lib_1.Stack.of(this).account,
            });
        }
        return {
            command: container.command,
            environment,
            secrets: container.secrets
                ? Object.entries(container.secrets).map(([key, value]) => {
                    return {
                        name: key,
                        valueFrom: value.arn,
                    };
                })
                : undefined,
            image: this.imageConfig.imageName,
            instanceType: container.instanceType && container.instanceType.toString(),
            jobRoleArn: container.jobRole && container.jobRole.roleArn,
            executionRoleArn: container.executionRole && container.executionRole.roleArn,
            linuxParameters: container.linuxParams
                ? { devices: container.linuxParams.renderLinuxParameters().devices }
                : undefined,
            logConfiguration: container.logConfiguration ? {
                logDriver: container.logConfiguration.logDriver,
                options: container.logConfiguration.options,
                secretOptions: container.logConfiguration.secretOptions
                    ? this.buildLogConfigurationSecretOptions(container.logConfiguration.secretOptions)
                    : undefined,
            } : undefined,
            mountPoints: container.mountPoints,
            privileged: container.privileged || false,
            networkConfiguration: container.assignPublicIp ? {
                assignPublicIp: container.assignPublicIp ? 'ENABLED' : 'DISABLED',
            } : undefined,
            readonlyRootFilesystem: container.readOnly || false,
            ulimits: container.ulimits,
            user: container.user,
            volumes: container.volumes,
            fargatePlatformConfiguration: container.platformVersion ? {
                platformVersion: container.platformVersion,
            } : undefined,
            resourceRequirements: [
                { type: 'VCPU', value: String(container.vcpus || (isFargate ? 0.25 : 1)) },
                { type: 'MEMORY', value: String(container.memoryLimitMiB || (isFargate ? 512 : 4)) },
            ].concat(container.gpuCount ? [{ type: 'GPU', value: String(container.gpuCount) }] : []),
        };
    }
    buildNodeRangeProps(multiNodeProps, isFargate) {
        const rangeProps = new Array();
        for (const prop of multiNodeProps.rangeProps) {
            rangeProps.push({
                container: this.buildJobContainer(prop.container, isFargate),
                targetNodes: `${prop.fromNodeIndex || 0}:${prop.toNodeIndex || multiNodeProps.count}`,
            });
        }
        return rangeProps;
    }
    buildLogConfigurationSecretOptions(secretOptions) {
        return secretOptions.map(secretOption => {
            return {
                name: secretOption.optionName,
                valueFrom: secretOption.secretArn,
            };
        });
    }
}
exports.JobDefinition = JobDefinition;
_a = JSII_RTTI_SYMBOL_1;
JobDefinition[_a] = { fqn: "@aws-cdk/aws-batch-alpha.JobDefinition", version: "2.54.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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