# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/retraitement_erfs-fpr/modules/copules_01_algo_monte_carlo.ipynb (unless otherwise specified).

__all__ = [
    "config",
    "tc",
    "monte_carlo_simulation",
    "eval_error",
    "apply_Monte_Carlo",
    "force_mean_with_factor",
]

# Cell
# Import qui servent à la librairie

import random
import unittest
from time import time

import pandas as pd
from leximpact_socio_fisca_simu_etat.config import Configuration
from sklearn.metrics import mean_absolute_percentage_error
from tqdm.notebook import tqdm

config = Configuration(project_folder="leximpact-prepare-data")
tc = unittest.TestCase()

# Cell
def monte_carlo_simulation(bucket, debug=False):
    """
    Implementation d'une fonction probabilistique qui associe une valeur estimée pour une variable VAR
    dans l'ERFS-FPR selon
    la valeur d'une variable de référence VAR_REF, en utilisant des copules calculés à partir des
    distributions de VAR1 et VAR_REF dans POTE.
    Par exemple, VAR1 = Rk et VAR_REF = RFR

    ::bucket:: Dictionnaire contenant les caractéristiques du bucket et ses sous-buckets.
    """

    # Note du 19/08/2021 depuis que l'on a ajouté le bucket à 0 dans les buckets,
    # plus besoin de regarder spécifiquement le cas du 0
    randinnbgens = random.random() * sum(bucket["nb_foyer"].values())

    # Générer un rk à partir de cette distrib (Monte-Carlo probablement)
    sommepass = 0  # nombre de foyer dans les buckets déjà dépassés. Quand on a passé le nombre de gens qu'on veut, on a fini de générer !
    for bucket_var in bucket["buckets"]:
        sommepass += bucket_var["nombre_ff_tranche"]
        if sommepass > randinnbgens:
            if debug:
                print(
                    f'{sommepass} > {randinnbgens} => return {bucket_var["mean_tranche_var"]}'
                )
            return bucket_var["mean_tranche_var"]
    print("ERROR returned nothing !")


# Cell


def eval_error(sub_buckets, predict: pd.DataFrame, debug=False) -> float:
    """
    Fonction d'évaluation de l'écart entre le résultat obtenu et le résultat attendu.
    On utilise les données de chaque sous bucket pour calculer une erreur au niveau du bucket.
    ::dataset:: Le DataFrame de VAR_REF que l'on souhaite analyser
    ::predict:: Le DataFrame des données générées
    """
    # On commence par générer les sous-buckets des données
    # Cela nous donnera des chiffres à comparer
    # On ne peut pas utiliser get_calib car on ne peut pas toujours trouver le même nombre de bucket
    # Idéalement il faudrait propager un paramètre jusqu'à get_frontiere pour forcer les mêmes frontières
    # mc_subbuckets = get_calib(pandas_to_vaex(predict), predict.columns[0], nb_bucket_asked)['buckets']
    values = sorted(predict[predict.columns[0]].to_list())
    mc_subbuckets = []
    y_test_mean, y_pred_mean = [], []
    tc.assertTrue(isinstance(sub_buckets, list))
    sum_mean_pred = 0
    sum_mean_test = 0
    for bucket in sub_buckets:
        tc.assertIsNotNone(bucket.get("nombre_ff_tranche"))
        # fin = debut + bucket['nombre_ff_tranche']
        # On prend les foyers dans la tranche de var
        extrait = [
            v for v in values if bucket["seuil_var_inf"] < v <= bucket["seuil_var_supp"]
        ]
        buck = {
            "nombre_ff_tranche": len(extrait),  # N'a pas d'intérêt
            "sum_tranche_var": sum(extrait),
            "mean_tranche_var": 0 if len(extrait) == 0 else sum(extrait) / len(extrait),
        }

        mc_subbuckets.append(buck)
        # tc.assertEqual(len(extrait), bucket['nombre_ff_tranche'])
        #         tc.assertEqual(sum(extrait), bucket['sum_tranche_var'])
        #         tc.assertEqual(0 if len(extrait) == 0 else sum(extrait)/ len(extrait), bucket['mean_tranche_var'])

        # On donnera ensuite ces valeurs à Scikit-Learn pour qu'il calcul l'erreur
        sum_mean_pred += buck["mean_tranche_var"]
        sum_mean_test += bucket["mean_tranche_var"]
        y_test_mean.append(bucket["mean_tranche_var"])
        y_pred_mean.append(buck["mean_tranche_var"])
        # La somme n'a pas de sens car POTE et ERFS n'ont pas la même taille
    #         y_test_sum.append(bucket['sum_tranche_var'])
    #         y_pred_sum.append(buck['sum_tranche_var'])
    tc.assertEqual(len(sub_buckets), len(mc_subbuckets))
    mean_pred = sum_mean_pred / len(sub_buckets)
    mean_test = sum_mean_test / len(sub_buckets)
    # On utilise la métrique mean_absolute_percentage_error qui est indépendante de la dimension des variables (une erreur entre 1 et 2 est la même qu'entre 1000 et 2000)
    # mean_squared_log_error : this metric penalizes an under-predicted estimate greater than an over-predicted estimate.
    # mean_squared_error : Pas satisfaisant car dépend de l'empleur des valeurs
    err_mean = mean_absolute_percentage_error(y_test_mean, y_pred_mean)
    err_all_sub_bucket_mean = mean_absolute_percentage_error(
        [mean_test, mean_test], [mean_pred, mean_pred]
    )
    if debug:
        print(f"{mean_test} {mean_pred} => error : {err_all_sub_bucket_mean}")
    # r2_sum = r2_score(y_test_sum,y_pred_sum)
    errors = [err_mean, err_all_sub_bucket_mean]
    error = sum(errors) / len(errors)

    if debug:
        print(f"{err_mean} {err_all_sub_bucket_mean} => error : {error}")
    # On soustrait l'erreur à 1 pour avoir un comportement "lower is better"
    # A noter que l'erreur R2 peut être négative
    return error


# Cell
def apply_Monte_Carlo(
    df: pd.DataFrame,
    copules: dict,
    bucket_level_col_name="revkire",
    out_col="monte_carlo_fake_var",
    nb_tirage=20,
    seed=None,
    use_force_mean_with_factor=False,
    debug=False,
) -> pd.DataFrame:
    """
    Applique la méthode de Monte-Carlo a un jeux de données
    ::df:: Le dataframe auquel injecter les données
    ::copules:: Le dictionnaire de copule complet
    ::bucket_level_col_name:: Le nom de la colonne utilisée pour calculer le bucket principal.
    ::out_col:: Le nom de la colonne en sortie pour le résultat de la méthode de Monte-Carlo
    ::nb_tirage:: Nombre de tirages aléatoires pour chaque bucket
    """
    final_df = pd.DataFrame()
    df_best_result = None
    # Set the seed to have reproductible results
    seed = int(time()) if seed is None else seed
    random.seed(seed)
    errors = []
    print(f"Seed used : {seed}")
    # On vérifie que l'on est au bon niveau
    if not isinstance(copules, list):
        if copules.get("copules"):
            copules = copules["copules"]
    tc.assertIsNotNone(copules[0].get("buckets"))
    for bucket in tqdm(copules):
        min_error = 1e99
        seuil_inf = bucket["lower_bound"]  # noqa: F841
        seuil_supp = bucket["upper_bound"]  # noqa: F841
        df_bucket = df.query(
            "@seuil_inf <= " + bucket_level_col_name + " < @seuil_supp"
        ).copy()
        # if debug:
        #    print(f"{seuil_inf} {seuil_supp} => {len(df_bucket)}")
        # On fait nb_tirage simulation de Monte-Carlo
        for i in range(nb_tirage):
            serie = df_bucket[bucket_level_col_name].apply(
                lambda rfr: monte_carlo_simulation(bucket)
            )
            df_tmp = pd.DataFrame({"monte_carlo_fake_var_" + str(i): serie})
            error = eval_error(bucket["buckets"], df_tmp)
            if error < min_error:
                min_error = error
                df_bucket["monte_carlo_fake_var"] = df_tmp[
                    "monte_carlo_fake_var_" + str(i)
                ]
                df_best_result = df_bucket
        errors.append(min_error)
        # On conserve le meilleur résultat
        if df_best_result is None:
            print("WARNING : apply_Monte_Carlo did not work")
            df_bucket["monte_carlo_fake_var"] = 0
            df_best_result = df_bucket
        if use_force_mean_with_factor:
            df_best_result = force_mean_with_factor(
                bucket["buckets"], df_best_result, col_name="monte_carlo_fake_var"
            )
            error = eval_error(bucket["buckets"], df_best_result)
            if debug:
                print(f"error after {error}")
        final_df = final_df.append(df_best_result)
        df_best_result = None
    tc.assertEqual(len(final_df), len(df))
    df[out_col] = final_df["monte_carlo_fake_var"]
    return df, errors


# Cell
def force_mean_with_factor(
    sub_buckets, df_target, col_name="monte_carlo_fake_var", debug=False
):
    """
    On calcul la moyenne du dataframe, on calcul ensuite le ration qui va permettre
    de corriger les valeurs pour obtenir la moyenne du bucket.
    ::bucket:: Le bucket dont on veut s'approcher de la moyenne
    ::df_target:: Le DataFrame contenant les données
    ::col_name:: Le nom de la colonne à corriger
    ::return:: Le datframe d'entrée modifié
    """
    df_result = df_target.copy()
    for bucket in sub_buckets:
        inf = bucket["seuil_var_inf"]  # noqa: F841
        supp = bucket["seuil_var_supp"]  # noqa: F841
        buck_mean = bucket["mean_tranche_var"]
        df_bucket = df_target.query("@inf <= " + col_name + " < @supp").copy()
        df_mean = df_bucket[col_name].mean()
        factor = buck_mean / df_mean
        df_bucket[col_name] = df_bucket[col_name] * factor
        df_result.update(df_bucket[[col_name]])
        if debug:
            print(
                f"Mean before {df_mean}, mean after : {df_bucket[col_name].mean()} {buck_mean} {factor}"
            )
    return df_result
