# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/retraitement_erfs-fpr/modules/calib_and_copule.ipynb (unless otherwise specified).

__all__ = [
    "tc",
    "DatasetNotSorted",
    "get_frontieres",
    "prepare_tranche_rfr_vaex",
    "list_revenus_each_bucket",
    "frontieres_valeurs",
    "compute_copule_vaex",
    "DistribDeVarVaex",
    "get_fake_data",
    "pandas_to_vaex",
    "get_calib",
    "bucket_merge_with_above",
    "reduce_bucket_number",
    "get_copules",
    "compute_pop_copules",
    "copules_to_df",
    "calib_to_df",
    "get_ecart_frontiere",
]

# Cell
import copy
import random
import statistics
import unittest
from copy import deepcopy
from time import time
from typing import Dict, List

import pandas as pd
import vaex

tc = unittest.TestCase()

# Cell


class DatasetNotSorted(Exception):
    pass


# Cell


def get_frontieres(
    nb_lignes: int,
    nb_bucket: int,
    ajout_ratio_supp=[0.1, 1e-2, 1e-3, 1e-4, 1e-5, 1e-6],
    nb_respect_secret_statistique=12,
    debug=False,
) -> List:
    # Si nous n'avons pas assez de personnes au dessus de zéro, on arrête là.
    if nb_lignes < nb_respect_secret_statistique:
        print(
            f"WARNING get_frontieres, moins de {nb_respect_secret_statistique} éléments => On retourne une liste vide. !!!!!!!!!!"
        )
        return []

    # Si nous savons déjà qu'on n'a pas assez de foyer pour remplir les bucket demandés, on fait moins de bucket
    nb_bucket = min(nb_lignes // nb_respect_secret_statistique, nb_bucket)
    # S'il ne reste qu'un bucket, on le retourne
    if nb_bucket == 1:
        return [nb_lignes]

    # On génère les frontières de base
    frontieres = [int(nb_lignes * i / nb_bucket) for i in range(1, nb_bucket)]
    if debug:
        print("get_frontieres frontieres de base", frontieres)
    # On ajoute des frontières supérieure
    for r in ajout_ratio_supp:
        nouvelle_frontiere = int(nb_lignes * r)
        if (
            nouvelle_frontiere < (nb_lignes - frontieres[-1])
            and nb_lignes - nouvelle_frontiere < nb_lignes
        ):
            frontieres += [nb_lignes - nouvelle_frontiere]

    # On ajoute la frontière supérieure
    frontieres += [nb_lignes]
    if debug:
        print("get_frontieres frontieres avant", frontieres)
    if debug:
        print("get_frontieres len(frontieres_ff) avant", len(frontieres))
    # On retire les sous-buckets de moins de 12 personnes
    # Comment est-ce que ça peut encore arriver puisqu'on prend un nombre fixe de personne ?
    # Si on a des buckets inégale type [12,11,12] et avec ceux qu'on ajoute à la fin
    i = 0
    while i < len(frontieres) - 1:
        ecart_avec_front_supp = frontieres[i + 1] - frontieres[i]
        # Si on a moins de 12 personnes d'écart entre 2 frontières,
        if ecart_avec_front_supp < nb_respect_secret_statistique:
            # On supprime une frontière pour combiner les 2 buckets mitoyens
            if (  # Si la première fontière repecte le secret
                (i == 0 and frontieres[i] >= nb_respect_secret_statistique)
                or (  # Ou que la frontère précédente le respecte
                    i > 0
                    and frontieres[i] - frontieres[i - 1]
                    >= nb_respect_secret_statistique
                )
                # Et que nous ne sommes pas à la fin
            ) and (i + 1 < len(frontieres) - 1):
                # Alors on supprime la frontière suivante
                if debug:
                    print(
                        f"get_frontieres On supprime la frontière i+1 {i+1} pour combiner les 2 buckets mitoyens : frontieres_ff[i]={frontieres[i]}, frontieres_ff[i+1]={frontieres[i+1]} , frontieres_ff[i+2]={frontieres[i+2]}"
                    )
                frontieres = frontieres[: i + 1] + frontieres[i + 2 :]
                # Sinon on supprime la frontière actuelle
            else:
                if debug:
                    print(
                        f"get_frontieres On supprime la frontière i {i} pour combiner les 2 buckets mitoyens : frontieres_ff[i]={frontieres[i]}, frontieres_ff[i+1]={frontieres[i+1]} "
                    )
                frontieres = frontieres[:i] + frontieres[i + 1 :]
        else:
            i += 1
    if debug:
        print("get_frontieres frontieres apres", frontieres)
    if len(frontieres) == 0:
        if nb_lignes > nb_respect_secret_statistique:
            return [nb_lignes]
        else:
            print("ERROR get_frontieres : On ne devrait jamais arriver ici !")
            return []
    if len(frontieres) == 1:
        if frontieres[0] > nb_respect_secret_statistique:
            return [nb_lignes]
        else:
            print(
                "ERROR get_frontieres : On ne devrait jamais arriver ici ! Voici les frontières",
                frontieres,
            )
            return []
    if debug:
        print("get_frontieres frontieres avant fin", frontieres)
    # On ajoute la dernière frontière si nécessaire
    if (len(frontieres) > 0) and (frontieres[-1] != nb_lignes):
        # frontieres += [nb_lignes]
        print(
            "ERROR get_frontieres : On ne devrait jamais arriver ici ! Voici les frontières",
            frontieres,
        )
        return []

    return frontieres


# Cell
list_revenus_each_bucket = [[]]
frontieres_valeurs = None


def prepare_tranche_rfr_vaex(
    vdx_sort: vaex.dataframe.DataFrameLocal,
    nb_bucket_rfr: int,
    nb_respect_secret_statistique=12,
    debug=False,
) -> Dict:
    """
    Objectif: Splitter le RFR en m buckets
    Dans chaque bucket on stocke toutes les valeurs non nulles de "variable"
    ::vdx_sort:: Le dataset, trié par RFR
    ::nb_bucket_rfr:: Nombre de tranches souhaitées
    ::debug:: Pour activer un mode debug, qui affiche des traces
    """
    nb_lignes = vdx_sort.shape[0]  # Nb de foyers fiscaux

    # On vérifie que le dataset est bien trié
    previous = vdx_sort[["revkire"]][nb_lignes - 1][0]
    for i in range(1, 1000):
        idx = nb_lignes // i
        idx = idx if idx != nb_lignes else nb_lignes - 1
        if previous < vdx_sort[["revkire"]][idx][0]:
            raise DatasetNotSorted("Your dataset is not sorted on revkire!")
        previous = vdx_sort[["revkire"]][idx][0]

    # Découpage du RFR en buckets:
    frontieres_ff = get_frontieres(
        nb_lignes=nb_lignes,
        nb_bucket=nb_bucket_rfr,
        nb_respect_secret_statistique=nb_respect_secret_statistique,
        debug=debug,
    )

    # On retire la dernière frontière pour éviter des tests (index out of range), on la remetra après
    frontieres_ff = frontieres_ff[:-1]
    i = 0
    # On supprime les frontières qui n'auraient que du 0
    while i < len(frontieres_ff):
        if vdx_sort[["revkire"]][frontieres_ff[i]][0] < 1:
            if debug:
                print(
                    f"WARNING: On efface la frontière d'index {i} : {frontieres_ff[i]} inutile car valeur de la borne haute est {vdx_sort[['revkire']][frontieres_ff[i]][0]}"
                )
            frontieres_ff = frontieres_ff[:i] + frontieres_ff[i + 1 :]
        else:
            i += 1
    frontieres_valeurs = [0] + [
        vdx_sort[["revkire"]][frontiere][0] for frontiere in frontieres_ff
    ]
    # On ajoute une valeur de fin treès (10^15€)
    frontieres_valeurs += [10**15]
    # On remet la dernière frontière
    frontieres_ff += [nb_lignes]
    dic = {"frontieres_RFR": frontieres_valeurs, "frontieres_ff": frontieres_ff}
    return dic


# Cell
def compute_copule_vaex(
    vdf: vaex.dataframe.DataFrameLocal,
    variable: str,
    nb_bucket_var: int,
    tranche_RFR: List,
    debug=False,
    nb_respect_secret_statistique=12,
):
    """
    On nous donne des tranches de RFR, en nombre de personne, et en valeur de RFR
    Pour chacune de ses tranches on doit extraire les valeurs de 'variable'
    On ne garde que celle supérieure à 0 et on les envoie à DistribDeVarVaex
    ::vdf:: Le jeux de données
    ::variable:: Nom de la variable secondaire.
    ::nb_bucket_var:: Nombre de tranches de variable secondaire souhaités.
    ::tranche_RFR:: La liste des tranches de RFR.
    ::debug:: Pour activer un mode debug, qui affiche des traces.
    ::nb_respect_secret_statistique:: Nombre minimal d'individus pour respecter le secret statistique.
    """
    controle = []
    copules = []
    frontieres_valeurs = tranche_RFR["frontieres_RFR"]
    frontieres_ff = tranche_RFR["frontieres_ff"]

    # On vérifie que le dataset est bien trié
    nb_lignes = vdf.shape[0]  # Nb de foyers fiscaux
    previous = vdf[["revkire"]][nb_lignes - 1][0]
    for i in range(1, 1000):
        idx = nb_lignes // i
        idx = idx if idx != nb_lignes else nb_lignes - 1
        if previous < vdf[["revkire"]][idx][0]:
            raise DatasetNotSorted("Your dataset is not sorted on revkire !")
        previous = vdf[["revkire"]][idx][0]

    # On parcourt les frontières de FF (= les index dans le tableau)
    idx_inf = 0

    debut = time()
    # On ne peut malheureusement pas filtrer par > 0 avant extraction car cela fausserait le nombre de valeur
    variable_all_values = vdf.to_arrays(
        column_names=[variable], selection=False, array_type="python"
    )[0]

    if "revkire" in vdf.get_column_names():
        col_rfr = "revkire"
    else:
        col_rfr = vdf.get_column_names()[0]
    # On fait l'hypothèse que c'est bien trié par ordre croissant
    lower_bound = vdf[[col_rfr]][idx_inf][0]
    if debug:
        print(f"Temps d'extraction par to_arrays  {time()-debut}")
    for i, idx_sup in enumerate(frontieres_ff):
        starttime = time()
        upper_bound = frontieres_valeurs[i + 1]  # Car frontieres_valeurs contient 0
        variable_values = variable_all_values[idx_inf:idx_sup]
        # nb_ff = vdf_tmp.shape[0]
        nb_ff = len(variable_values)
        if debug:
            print(f"-----------------Temps après slice {time()-starttime}")
        assert nb_ff == idx_sup - idx_inf
        # Quand il y a beaucoup de personne ayant le même revenu on peut avec des tranches avec lower_bound=upper_bound, mais ce n'est pas gênant
        if vdf[[col_rfr]][idx_inf][0] != lower_bound and lower_bound != upper_bound:
            print(
                f"compute_copule_vaex {i} WARNING: Il y a peut-être un problème car le RFR du premier index (idx_inf={idx_inf}) est {vdf[[col_rfr]][idx_inf][0]} alors que lower_bound vaut {lower_bound}"
            )
        if (
            i != len(frontieres_ff) - 1
            and vdf[[col_rfr]][idx_sup][0] != upper_bound
            and lower_bound != upper_bound
        ):
            print(
                f"compute_copule_vaex {i} WARNING: Il y a peut-être un problème car le RFR du dernier index (idx_sup={idx_sup}) est {vdf[[col_rfr]][idx_sup][0]} alors que upper_bound vaut {upper_bound}"
            )
        # Remove 0
        variable_values = [v for v in variable_values if v > 0]

        if debug:
            print(f"Temps avant sort {time()-starttime}")
        # Tri des variables : sort() est plus rapide que sorted, mais écrase notre liste
        variable_values.sort()
        # variable_values = sorted(variable_values)
        if debug:
            print(f"Temps après sort {time()-starttime}")
        if debug:
            print(
                f"compute_copule_vaex {i} : index entre idx_inf={idx_inf} et idx_sup={idx_sup} - RFR entre lower_bound={lower_bound} et upper_bound={upper_bound} - {len(variable_values)} valeurs différentes de zéro."
            )
            if len(variable_values) > 0:
                print(
                    f"\tmin(variable_values)={min(variable_values)} max(variable_values)={max(variable_values)}"
                )
        if len(variable_values) > idx_sup - idx_inf:
            print(
                f"compute_copule_vaex ERROR i={i} len(variable_values)={len(variable_values)} != {idx_sup - idx_inf}"
            )
        assert len(variable_values) <= (idx_sup - idx_inf)
        if debug:
            DistribDeVar_time = time()
        bdr = DistribDeVarVaex(
            variable_values=variable_values,
            variable=variable,
            nb_ff=nb_ff,
            nb_bucket_var=nb_bucket_var,
            lower_bound=lower_bound,
            upper_bound=upper_bound,
            debug=debug,
            nb_respect_secret_statistique=nb_respect_secret_statistique,
        )
        if debug:
            print(f"Temps de DistribDeVarVaex {time()-DistribDeVar_time}")
        # Et on ajoute ce tableau à la liste des tableaux
        copules += [bdr.to_dict()]

        idx_inf = idx_sup
        lower_bound = upper_bound
        if debug:
            print(f"Temps après fin de la boucle {time()-starttime} --------------")
        if debug and i > 10:
            print("DEBUG EXIT !!!")
            break
    dico = {"controle": controle, "copules": copules}

    return dico


# Cell
class DistribDeVarVaex:
    """
    On créée une classe qui, pour un bucket de RFR donné [lower_bound, upper_bound], va générer la distribution des Rk (ou autre Variable) de ce bucket (que l'on a dans liste_des_rk).
    Cette distribution est retournée sous la forme:
        resultat = [ [Nb de gens1,Somme des Rk 1],[Nb2, Sum2], [], ... , [Nb N, Sum N]]
            avec N le nb de buckets de Rk
    """

    def __init__(
        self,
        variable_values: List,
        variable: str,
        nb_ff: int,
        nb_bucket_var=10,
        lower_bound=0,
        upper_bound=10 ^ 15,
        nb_respect_secret_statistique=12,
        debug=False,
    ):
        """
        ::variable_values:: Liste de toutes les valeurs de la variable secondaire, dajà triée par ordre croissant.
        ::variable:: Nom de la variable secondaire.
        ::nb_ff:: Nombre de foyers fiscaux dans cette tranche de la variable secondaire.
        ::nb_bucket_var:: Nombre de tranche souhaité.
        ::lower_bound:: Borne inférieure de RFR.
        ::upper_bound:: Borne supérieure de RFR.
        ::debug:: Pour activer un mode debug, qui affiche des traces
        """
        if debug:
            print(f"DistribDeVarVaex - RFR entre {lower_bound} et {upper_bound}")
        self.lower_bound = lower_bound
        self.upper_bound = upper_bound
        self.nbpeople = nb_ff
        self.nbnz = len(variable_values)
        self.nbzero = max(self.nbpeople - self.nbnz, 0)
        self.sum_variable = sum(variable_values)
        default_tranche = {
            "seuil_var_inf": 0,
            "seuil_var_supp": 0,
            "nombre_ff_tranche": 0,
            "sum_tranche_var": 0,
            "mean_tranche_var": 0,
            "stdev_tranche_var": 0,
            "nb_above_seuil": 0,
            "sum_var_above_seuil": 0,
            "ratio_nb_above_seuil": 0,
            "mean_var_above_seuil": 0,
        }

        # Si nous n'avons pas assez de personnes au dessus de zéro, on arrête là.
        if self.nbnz < nb_respect_secret_statistique:
            print(
                f"DistribDeVar : WARNING !!!!, moins de {nb_respect_secret_statistique} éléments => On retourne une liste vide. !!!!!!!!!!"
            )
            default_tranche["nombre_ff_tranche"] = self.nbnz
            self.buckets = "Error, moins de 12 éléments !"
            return

        # On vérifie que le dataset est bien trié
        nb_lignes = len(variable_values)
        previous = variable_values[nb_lignes - 1]
        for i in range(1, 100):
            idx = nb_lignes // i
            idx = idx if idx != nb_lignes else nb_lignes - 1
            # < because we iterate from the end with idx = nb_lignes // i
            if previous < variable_values[idx]:
                raise DatasetNotSorted(
                    f"Your dataset is not sorted on {variable}! i={i} idx={idx},  variable_values[idx]={variable_values[idx]} >= previous={previous}"
                )
            previous = variable_values[idx]

        premiere_tranche = {
            "seuil_var_inf": 0,
            "seuil_var_supp": 0,
            "nombre_ff_tranche": self.nbzero,
            "sum_tranche_var": 0,
            "mean_tranche_var": 0,
            "stdev_tranche_var": 0,
            "nb_above_seuil": self.nbnz,
            "sum_var_above_seuil": self.sum_variable,
            "ratio_nb_above_seuil": self.nbnz / self.nbpeople,
            "mean_var_above_seuil": self.sum_variable / self.nbnz,
        }
        ajout_ratio_supp = [0.1, 1e-2, 1e-3, 1e-4, 1e-5, 1e-6]
        secret_statistique_respecte = True
        while nb_bucket_var > 0:
            # On récupère les frontières avec ajout_ratio_supp et nb_bucket_var qui peuvent diminuer
            # si on ne respecte pas le secrect statistique
            self.frontieres_ff = get_frontieres(
                nb_lignes=self.nbnz,
                nb_bucket=nb_bucket_var,
                ajout_ratio_supp=ajout_ratio_supp,
                nb_respect_secret_statistique=nb_respect_secret_statistique,
                debug=False,
            )
            if debug:
                print("frontieres_ff:", self.frontieres_ff)

            if self.frontieres_ff == []:
                self.buckets = [default_tranche]
                return

            # Calcul de la distribution dans ces buckets
            currbuck = 0
            buckets = [premiere_tranche]
            max_ff_bucket = 0
            sum_var_bucket = 0
            tranche_courante = deepcopy(default_tranche)
            values = []
            # On parcourt les var >0
            for idrk, current_value in enumerate(variable_values):
                #                 if debug:
                #                     print(f"idrk={idrk}, current_value={current_value} tranche_courante:{tranche_courante}, filtered_above_seuil:{self.rknonzero[idrk:]}")
                # Si on est dans la première itération d'un nouveau bucket
                if tranche_courante["nombre_ff_tranche"] == 0:
                    tranche_courante["seuil_var_inf"] = current_value

                # Si le nb d'observations est supérieur au plafond du current bucket,
                # On est à la fin du bucket
                if (
                    idrk >= self.frontieres_ff[currbuck] or idrk == self.nbnz - 1
                ):  # or idrk==self.nbnz
                    if idrk == self.nbnz - 1:
                        # On est à la fin, donc on inclue current_value dans le bucket en cours
                        # La regle des 85% c'est  en incluant l'élément concerné dans la somme :
                        # Aucune  case  du  tableau  ne  doit  contenir  de  données  pour  lesquelles
                        # une personne représente plus de 85% du total
                        sum_var_bucket += current_value
                        values.append(current_value)
                        tranche_courante["nombre_ff_tranche"] += 1
                        if current_value > max_ff_bucket:
                            max_ff_bucket = current_value
                    # Vérification que la personne la plus riche du bucket ne représente pas plus de 85%
                    if sum_var_bucket > 0 and max_ff_bucket / sum_var_bucket > 0.85:
                        secret_statistique_respecte = False
                        # On réduit le nombre de bucket
                        if ajout_ratio_supp != []:
                            # On retire le ratio le plus haut
                            ajout_ratio_supp = ajout_ratio_supp[:-1]
                        else:
                            # Si on n'a plus de ratio d'affinage des hauts revenus,
                            # On réduit le nombre de bucket
                            nb_bucket_var -= 1
                        if debug:
                            print(
                                f"DistribDeVarVaex : Warning SECRET STATISTIQUE > 0.85 NON RESPECTE (idrk={idrk}), on refait une passe avec moins de frontières"
                            )
                            print(
                                f"sum_var_bucket={sum_var_bucket}, max_ff_bucket={max_ff_bucket},currbuck={currbuck}, nb_bucket_var={nb_bucket_var}, ajout_ratio_supp={ajout_ratio_supp}"
                            )
                        buckets = ["SECRET STATISTIQUE NON RESPECTE"]
                        # On relance la boucle principale au début
                        break
                    else:
                        secret_statistique_respecte = True
                        # On calcul les metadata de notre bucket
                        tranche_courante["sum_tranche_var"] = sum_var_bucket
                        tranche_courante["mean_tranche_var"] = (
                            sum_var_bucket / tranche_courante["nombre_ff_tranche"]
                        )
                        # TODO : calculer incrémentatlement l'écart-type, mais c'est compliquer https://math.stackexchange.com/questions/102978/incremental-computation-of-standard-deviation
                        # Ca permetrait de réduire l'emprunte mémoire, mais elle est acceptable : 300 000 valeurs pour des centiles.
                        if tranche_courante["nombre_ff_tranche"] > 1:
                            tranche_courante["stdev_tranche_var"] = statistics.stdev(
                                values
                            )
                        # Si on n'est pas dans le dernier bucket
                        if idrk != self.nbnz - 1:
                            filtered_above_seuil = variable_values[idrk:]
                            nb_above_seuil = len(filtered_above_seuil)
                            sum_above_seuil = sum(filtered_above_seuil)
                            tranche_courante["nb_above_seuil"] = nb_above_seuil
                            tranche_courante["sum_var_above_seuil"] = sum_above_seuil
                            if self.nbpeople > 0:
                                tranche_courante["ratio_nb_above_seuil"] = (
                                    nb_above_seuil / self.nbpeople
                                )
                            if nb_above_seuil > 0:
                                tranche_courante["mean_var_above_seuil"] = (
                                    sum_above_seuil / nb_above_seuil
                                )
                        else:
                            tranche_courante["nb_above_seuil"] = 0
                            tranche_courante["sum_var_above_seuil"] = 0
                            tranche_courante["ratio_nb_above_seuil"] = 0
                            tranche_courante["mean_var_above_seuil"] = 0

                        buckets.append(tranche_courante.copy())
                        #                         if debug:
                        #                             print(f"idrk={idrk} On change de frontière current_value={current_value} tranche_courante:{tranche_courante}")
                        # On passe au bucket suivant

                        tranche_courante = deepcopy(default_tranche)
                        tranche_courante["nombre_ff_tranche"] = 1
                        tranche_courante["seuil_var_inf"] = current_value
                        currbuck += 1
                        max_ff_bucket = current_value
                        sum_var_bucket = current_value
                        values = [current_value]
                else:
                    # We continue to iterate into the values in the same bucket
                    sum_var_bucket += current_value
                    values.append(current_value)
                    tranche_courante["seuil_var_supp"] = current_value
                    tranche_courante["nombre_ff_tranche"] += 1
                    if current_value > max_ff_bucket:
                        max_ff_bucket = current_value
            if secret_statistique_respecte:
                # On quitte la boucle seulement quand le secret statistique est respecté
                break
        # Sanity check
        prev_b = buckets[0]
        for b in buckets:
            if b["seuil_var_inf"] > b["seuil_var_supp"]:
                print(b)
                raise DatasetNotSorted(
                    "DistribDeVarVaex: The resuling buckets are not sorted: seuil_var_inf > seuil_var_supp!"
                )
            if prev_b["seuil_var_inf"] > b["seuil_var_inf"]:
                print(prev_b)
                print(b)
                raise DatasetNotSorted(
                    "DistribDeVarVaex: The resuling buckets are not sorted prev seuil_var_inf > seuil_var_inf!"
                )
            if prev_b["seuil_var_supp"] > b["seuil_var_inf"]:
                print(prev_b)
                print(b)
                raise DatasetNotSorted(
                    "DistribDeVarVaex: The resuling buckets are not sorted prev seuil_var_supp > seuil_var_inf!"
                )
            if prev_b["seuil_var_supp"] < b["seuil_var_inf"]:
                # We will compute the middle to have continuous bucket
                prev_b["seuil_var_supp"] = (
                    prev_b["seuil_var_supp"]
                    + (b["seuil_var_inf"] - prev_b["seuil_var_supp"]) / 2
                )
                b["seuil_var_inf"] = prev_b["seuil_var_supp"]
            prev_b = b
        # On sauvegarde la distribution de ce bucket
        self.buckets = buckets

    def to_dict(self):
        """
        Exporte le résultat dans un dictionnaire Python
        """
        d = {
            "lower_bound": self.lower_bound,
            "upper_bound": self.upper_bound,
            "nb_foyer": {"zero": self.nbzero, "nonzero": self.nbnz},
            "buckets": self.buckets,
        }
        return d


# Cell
def get_fake_data(
    nb_echantillon_zero=1_000,
    nb_echantillon=10_000,
    var_name="var",
    set_some_var_to_zero=False,
    exponent=1.5,
    divider=15,
):
    """
    Génération d'un faux jeu de données.
    """
    jeux = []

    # On commence par en ajouter 1 000 à 0

    for i in range(nb_echantillon_zero):
        foyer = {
            "idfoy": i,
            "revkire": 0,
            # "var": random.randint(0,i),
            var_name: 0,
        }
        jeux.append(foyer)

    for i in range(nb_echantillon):
        x = i * 100
        foyer = {
            "idfoy": i + nb_echantillon_zero,
            "revkire": x,
            # "var": random.randint(0,x ** 1.5 // (nb_echantillon // 15)) if random.random() > 0.5 else 0,
            var_name: x**exponent // (nb_echantillon // divider),
        }
        if set_some_var_to_zero and random.random() > 0.8:
            foyer[var_name] = 0
        jeux.append(foyer)
    df = pd.DataFrame(jeux)
    return df


# Cell
def pandas_to_vaex(df):
    return vaex.from_pandas(df)


# Cell
def get_calib(vdf, variable, nb_bucket_var, nb_respect_secret_statistique=12):
    """
    ::vdf:: Vaex DataFrame
    ::variable:: Column name to calibrate
    ::nb_bucket_var:: Number of bucket in wich to split the dataframe
    ::nb_respect_secret_statistique:: Minimal number of sample in a bucket
    """
    une_tranche_rfr = prepare_tranche_rfr_vaex(
        vdf, 1, nb_respect_secret_statistique=nb_respect_secret_statistique, debug=True
    )
    out = compute_copule_vaex(
        vdf,
        variable,
        nb_bucket_var,
        une_tranche_rfr,
        nb_respect_secret_statistique=nb_respect_secret_statistique,
        debug=False,
    )

    return out["copules"][0]


# Cell


def bucket_merge_with_above(calib_in, id_rm: int):
    """
    This method merge two bucket together.
    ::calib:: The buckets list
    ::id_rm:: The index of the bucket to merge with the bucket above
    """

    new_calib = copy.deepcopy(calib_in)
    # On supprime le bucket id_rm
    buck_removed = new_calib["buckets"].pop(id_rm)

    # On remplace les valeurs de celui qui est devenu le suivant
    new_calib["buckets"][id_rm]["seuil_var_inf"] = buck_removed["seuil_var_inf"]
    # new_calib["buckets"][id_rm]["seuil_var_supp"] ne change pas
    new_calib["buckets"][id_rm]["nombre_ff_tranche"] = (
        buck_removed["nombre_ff_tranche"]
        + new_calib["buckets"][id_rm]["nombre_ff_tranche"]
    )

    new_calib["buckets"][id_rm]["sum_tranche_var"] = (
        buck_removed["sum_tranche_var"] + new_calib["buckets"][id_rm]["sum_tranche_var"]
    )

    # new_calib["buckets"][id_rm]["nb_above_seuil"] Ne change pas
    new_calib["buckets"][id_rm]["mean_tranche_var"] = (
        new_calib["buckets"][id_rm]["sum_tranche_var"]
        / new_calib["buckets"][id_rm]["nombre_ff_tranche"]
    )
    new_calib["buckets"][id_rm]["stdev_tranche_var"] = 0

    # On verifie qu'on ne perd personne en cours de route
    tot_av = 0
    tot_ap = 0
    for i in range(len(calib_in["buckets"])):
        tot_av += calib_in["buckets"][i]["nombre_ff_tranche"]
    for j in range(len(new_calib["buckets"])):
        tot_av
        tot_ap += new_calib["buckets"][j]["nombre_ff_tranche"]

    tc.assertEqual(tot_av, tot_ap)

    return new_calib


# Cell


def reduce_bucket_number(calib, max_gap: int):
    """
    This method scans a bucket list and merges all buckets where
    ::calib:: The buckets list
    ::max_gap:: The ratio below which the bucket will be merged
    """
    calib_new = {}
    calib_new = copy.deepcopy(calib)
    last = False

    # On décide de recommencer à parcourir la nouvelle liste dès qu'on a fusionné un bucket
    while last is False:
        buckets = calib_new["buckets"]
        for idx, bucket in enumerate(buckets):
            # On s'arrete 2 buckets avant la fin
            if idx + 2 == len(buckets):
                last = True
                break
            # Bucket vide
            elif buckets[idx]["mean_tranche_var"] == 0.0:
                continue
            # Sinon:
            else:
                ecart = (
                    buckets[idx + 1]["mean_tranche_var"] - bucket["mean_tranche_var"]
                )
                moyenne = (
                    bucket["mean_tranche_var"] + buckets[idx + 1]["mean_tranche_var"]
                ) / 2
                if (ecart < max_gap * moyenne) and (
                    buckets[idx + 1]["mean_tranche_var"] != 0
                ):
                    # print("to remove", idx)
                    calib_new = bucket_merge_with_above(calib_new, idx)
                    break
    print(
        "On a fusionné les buckets, passant de ",
        len(calib["buckets"]),
        " à ",
        len(calib_new["buckets"]),
        "buckets \n",
    )

    return calib_new


# Cell
def get_copules(
    vdf, nb_bucket_rfr, variable, nb_bucket_var, nb_respect_secret_statistique=12
):
    tranche_rfr = prepare_tranche_rfr_vaex(
        vdf,
        nb_bucket_rfr,
        nb_respect_secret_statistique=nb_respect_secret_statistique,
        debug=False,
    )
    copules = compute_copule_vaex(
        vdf,
        variable,
        nb_bucket_var,
        tranche_rfr,
        debug=False,
        nb_respect_secret_statistique=nb_respect_secret_statistique,
    )
    return copules


# Cell
def compute_pop_copules(copules):
    nb_pop = 0
    for c in copules["copules"]:
        nb_pop += c["nb_foyer"]["zero"] + c["nb_foyer"]["nonzero"]
    return nb_pop


# Cell
def copules_to_df(copules):
    copules_flat = []
    # nb_pop = compute_pop_copules(copules)
    # On vérifie que l'on est au bon niveau
    if not isinstance(copules, list):
        if copules.get("copules"):
            copules = copules["copules"]
    tc.assertIsNotNone(copules[0].get("buckets"))
    for cop in copules:
        for bucket in cop["buckets"]:
            un_copule = {
                "lower_bound": cop["lower_bound"],
                "upper_bound": cop["upper_bound"],
                "nb_foyer_zero": cop["nb_foyer"]["zero"],
                "nb_foyer_nonzero": cop["nb_foyer"]["nonzero"],
                "seuil_var_inf": bucket["seuil_var_inf"],
                "seuil_var_supp": bucket["seuil_var_supp"],
                "nombre_ff_tranche": bucket["nombre_ff_tranche"],
                "sum_tranche_var": bucket["sum_tranche_var"],
                "mean_tranche_var": bucket["mean_tranche_var"],
                "nb_above_seuil": bucket["nb_above_seuil"],
                "sum_var_above_seuil": bucket["sum_var_above_seuil"],
                "ratio_nb_above_seuil": bucket["ratio_nb_above_seuil"],
                "mean_var_above_seuil": bucket["mean_var_above_seuil"],
            }
            copules_flat.append(un_copule)
    return pd.DataFrame(copules_flat)


def calib_to_df(calib):
    return copules_to_df([calib])


# Attention, toutes les colonnes "*_above_seuil" sont calculées par rapport à l'intérieur du bucket de RFR, pas de la population entière
# On va donc les cumuler
# ########################################
# ######### TODO
# ########################################
# def cumul_above
# if len(copules_flat)>0:
#     un_copule["nb_above_seuil"] = bucket["nb_above_seuil"]+copules_flat[-1]["nb_above_seuil"]
#     un_copule["sum_var_above_seuil"] = bucket["sum_var_above_seuil"]+copules_flat[-1]["sum_var_above_seuil"]
#     un_copule["ratio_nb_above_seuil"] = bucket["nb_above_seuil"]/nb_pop
#     un_copule["mean_var_above_seuil"] = bucket["sum_var_above_seuil"]/bucket["nb_above_seuil"]

# Cell
def get_ecart_frontiere(frontieres, nb_respect_secret_statistique=12):
    prec = 0
    ecart = []
    for f in frontieres:
        e = f - prec
        if e < nb_respect_secret_statistique:
            return False
        ecart.append(e)
        prec = f
    return ecart
