"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitConfig = exports.CloudFormationInit = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const machine_image_1 = require("./machine-image");
const cfn_init_internal_1 = require("./private/cfn-init-internal");
/**
 * A CloudFormation-init configuration.
 *
 * @stability stable
 */
class CloudFormationInit {
    constructor(configSets, configs) {
        this._configSets = {};
        this._configs = {};
        Object.assign(this._configSets, configSets);
        Object.assign(this._configs, configs);
    }
    /**
     * Build a new config from a set of Init Elements.
     *
     * @stability stable
     */
    static fromElements(...elements) {
        return CloudFormationInit.fromConfig(new InitConfig(elements));
    }
    /**
     * Use an existing InitConfig object as the default and only config.
     *
     * @stability stable
     */
    static fromConfig(config) {
        return CloudFormationInit.fromConfigSets({
            configSets: {
                default: ['config'],
            },
            configs: { config },
        });
    }
    /**
     * Build a CloudFormationInit from config sets.
     *
     * @stability stable
     */
    static fromConfigSets(props) {
        return new CloudFormationInit(props.configSets, props.configs);
    }
    /**
     * Add a config with the given name to this CloudFormationInit object.
     *
     * @stability stable
     */
    addConfig(configName, config) {
        if (this._configs[configName]) {
            throw new Error(`CloudFormationInit already contains a config named '${configName}'`);
        }
        this._configs[configName] = config;
    }
    /**
     * Add a config set with the given name to this CloudFormationInit object.
     *
     * The new configset will reference the given configs in the given order.
     *
     * @stability stable
     */
    addConfigSet(configSetName, configNames = []) {
        if (this._configSets[configSetName]) {
            throw new Error(`CloudFormationInit already contains a configSet named '${configSetName}'`);
        }
        const unk = configNames.filter(c => !this._configs[c]);
        if (unk.length > 0) {
            throw new Error(`Unknown configs referenced in definition of '${configSetName}': ${unk}`);
        }
        this._configSets[configSetName] = [...configNames];
    }
    /**
     * Attach the CloudFormation Init config to the given resource.
     *
     * As an app builder, use `instance.applyCloudFormationInit()` or
     * `autoScalingGroup.applyCloudFormationInit()` to trigger this method.
     *
     * This method does the following:
     *
     * - Renders the `AWS::CloudFormation::Init` object to the given resource's
     *    metadata, potentially adding a `AWS::CloudFormation::Authentication` object
     *    next to it if required.
     * - Updates the instance role policy to be able to call the APIs required for
     *    `cfn-init` and `cfn-signal` to work, and potentially add permissions to download
     *    referenced asset and bucket resources.
     * - Updates the given UserData with commands to execute the `cfn-init` script.
     *
     * @stability stable
     */
    attach(attachedResource, attachOptions) {
        var _c, _d, _e;
        if (attachOptions.platform === machine_image_1.OperatingSystemType.UNKNOWN) {
            throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
        }
        const CFN_INIT_METADATA_KEY = 'AWS::CloudFormation::Init';
        if (attachedResource.getMetadata(CFN_INIT_METADATA_KEY) !== undefined) {
            throw new Error(`Cannot bind CfnInit: resource '${attachedResource.node.path}' already has '${CFN_INIT_METADATA_KEY}' attached`);
        }
        // Note: This will not reflect mutations made after attaching.
        const bindResult = this.bind(attachedResource.stack, attachOptions);
        attachedResource.addMetadata(CFN_INIT_METADATA_KEY, bindResult.configData);
        // Need to resolve the various tokens from assets in the config,
        // as well as include any asset hashes provided so the fingerprint is accurate.
        const resolvedConfig = attachedResource.stack.resolve(bindResult.configData);
        const fingerprintInput = { config: resolvedConfig, assetHash: bindResult.assetHash };
        const fingerprint = contentHash(JSON.stringify(fingerprintInput)).substr(0, 16);
        attachOptions.instanceRole.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['cloudformation:DescribeStackResource', 'cloudformation:SignalResource'],
            resources: [core_1.Aws.STACK_ID],
        }));
        if (bindResult.authData) {
            attachedResource.addMetadata('AWS::CloudFormation::Authentication', bindResult.authData);
        }
        // To identify the resources that have the metadata and where the signal
        // needs to be sent, we need { region, stackName, logicalId }
        let resourceLocator = `--region ${core_1.Aws.REGION} --stack ${core_1.Aws.STACK_NAME} --resource ${attachedResource.logicalId}`;
        // If specified in attachOptions, include arguments in cfn-init/cfn-signal commands
        if (attachOptions.includeUrl) {
            resourceLocator = `${resourceLocator} --url https://cloudformation.${core_1.Aws.REGION}.${core_1.Aws.URL_SUFFIX}`;
        }
        if (attachOptions.includeRole) {
            resourceLocator = `${resourceLocator} --role ${attachOptions.instanceRole.roleName}`;
        }
        const configSets = ((_c = attachOptions.configSets) !== null && _c !== void 0 ? _c : ['default']).join(',');
        const printLog = (_d = attachOptions.printLog) !== null && _d !== void 0 ? _d : true;
        if ((_e = attachOptions.embedFingerprint) !== null && _e !== void 0 ? _e : true) {
            // It just so happens that the comment char is '#' for both bash and PowerShell
            attachOptions.userData.addCommands(`# fingerprint: ${fingerprint}`);
        }
        if (attachOptions.platform === machine_image_1.OperatingSystemType.WINDOWS) {
            const errCode = attachOptions.ignoreFailures ? '0' : '$LASTEXITCODE';
            attachOptions.userData.addCommands(...[
                `cfn-init.exe -v ${resourceLocator} -c ${configSets}`,
                `cfn-signal.exe -e ${errCode} ${resourceLocator}`,
                ...printLog ? ['type C:\\cfn\\log\\cfn-init.log'] : [],
            ]);
        }
        else {
            const errCode = attachOptions.ignoreFailures ? '0' : '$?';
            attachOptions.userData.addCommands(...[
                // Run a subshell without 'errexit', so we can signal using the exit code of cfn-init
                '(',
                '  set +e',
                `  /opt/aws/bin/cfn-init -v ${resourceLocator} -c ${configSets}`,
                `  /opt/aws/bin/cfn-signal -e ${errCode} ${resourceLocator}`,
                ...printLog ? ['  cat /var/log/cfn-init.log >&2'] : [],
                ')',
            ]);
        }
    }
    bind(scope, options) {
        const nonEmptyConfigs = mapValues(this._configs, c => c.isEmpty() ? undefined : c);
        const configNameToBindResult = mapValues(nonEmptyConfigs, c => c._bind(scope, options));
        return {
            configData: {
                configSets: mapValues(this._configSets, configNames => configNames.filter(name => nonEmptyConfigs[name] !== undefined)),
                ...mapValues(configNameToBindResult, c => c.config),
            },
            authData: Object.values(configNameToBindResult).map(c => c.authentication).reduce(deepMerge, undefined),
            assetHash: combineAssetHashesOrUndefined(Object.values(configNameToBindResult).map(c => c.assetHash)),
        };
    }
}
exports.CloudFormationInit = CloudFormationInit;
_a = JSII_RTTI_SYMBOL_1;
CloudFormationInit[_a] = { fqn: "@aws-cdk/aws-ec2.CloudFormationInit", version: "1.125.0" };
/**
 * A collection of configuration elements.
 *
 * @stability stable
 */
class InitConfig {
    /**
     * @stability stable
     */
    constructor(elements) {
        this.elements = new Array();
        this.add(...elements);
    }
    /**
     * Whether this configset has elements or not.
     *
     * @stability stable
     */
    isEmpty() {
        return this.elements.length === 0;
    }
    /**
     * Add one or more elements to the config.
     *
     * @stability stable
     */
    add(...elements) {
        this.elements.push(...elements);
    }
    /**
     * Called when the config is applied to an instance.
     * Creates the CloudFormation representation of the Init config and handles any permissions and assets.
     * @internal
     */
    _bind(scope, options) {
        const bindOptions = {
            instanceRole: options.instanceRole,
            platform: this.initPlatformFromOSType(options.platform),
            scope,
        };
        const packageConfig = this.bindForType(cfn_init_internal_1.InitElementType.PACKAGE, bindOptions);
        const groupsConfig = this.bindForType(cfn_init_internal_1.InitElementType.GROUP, bindOptions);
        const usersConfig = this.bindForType(cfn_init_internal_1.InitElementType.USER, bindOptions);
        const sourcesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SOURCE, bindOptions);
        const filesConfig = this.bindForType(cfn_init_internal_1.InitElementType.FILE, bindOptions);
        const commandsConfig = this.bindForType(cfn_init_internal_1.InitElementType.COMMAND, bindOptions);
        // Must be last!
        const servicesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SERVICE, bindOptions);
        const allConfig = [packageConfig, groupsConfig, usersConfig, sourcesConfig, filesConfig, commandsConfig, servicesConfig];
        const authentication = allConfig.map(c => c === null || c === void 0 ? void 0 : c.authentication).reduce(deepMerge, undefined);
        const assetHash = combineAssetHashesOrUndefined(allConfig.map(c => c === null || c === void 0 ? void 0 : c.assetHash));
        return {
            config: {
                packages: packageConfig === null || packageConfig === void 0 ? void 0 : packageConfig.config,
                groups: groupsConfig === null || groupsConfig === void 0 ? void 0 : groupsConfig.config,
                users: usersConfig === null || usersConfig === void 0 ? void 0 : usersConfig.config,
                sources: sourcesConfig === null || sourcesConfig === void 0 ? void 0 : sourcesConfig.config,
                files: filesConfig === null || filesConfig === void 0 ? void 0 : filesConfig.config,
                commands: commandsConfig === null || commandsConfig === void 0 ? void 0 : commandsConfig.config,
                services: servicesConfig === null || servicesConfig === void 0 ? void 0 : servicesConfig.config,
            },
            authentication,
            assetHash,
        };
    }
    bindForType(elementType, renderOptions) {
        var _c;
        const elements = this.elements.filter(elem => elem.elementType === elementType);
        if (elements.length === 0) {
            return undefined;
        }
        const bindResults = elements.map((e, index) => e._bind({ index, ...renderOptions }));
        return {
            config: (_c = bindResults.map(r => r.config).reduce(deepMerge, undefined)) !== null && _c !== void 0 ? _c : {},
            authentication: bindResults.map(r => r.authentication).reduce(deepMerge, undefined),
            assetHash: combineAssetHashesOrUndefined(bindResults.map(r => r.assetHash)),
        };
    }
    initPlatformFromOSType(osType) {
        switch (osType) {
            case machine_image_1.OperatingSystemType.LINUX: {
                return cfn_init_internal_1.InitPlatform.LINUX;
            }
            case machine_image_1.OperatingSystemType.WINDOWS: {
                return cfn_init_internal_1.InitPlatform.WINDOWS;
            }
            default: {
                throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
            }
        }
    }
}
exports.InitConfig = InitConfig;
_b = JSII_RTTI_SYMBOL_1;
InitConfig[_b] = { fqn: "@aws-cdk/aws-ec2.InitConfig", version: "1.125.0" };
/**
 * Deep-merge objects and arrays
 *
 * Treat arrays as sets, removing duplicates. This is acceptable for rendering
 * cfn-inits, not applicable elsewhere.
 */
function deepMerge(target, src) {
    var _c, _d;
    if (target == null) {
        return src;
    }
    if (src == null) {
        return target;
    }
    for (const [key, value] of Object.entries(src)) {
        if (Array.isArray(value)) {
            if (target[key] && !Array.isArray(target[key])) {
                throw new Error(`Trying to merge array [${value}] into a non-array '${target[key]}'`);
            }
            target[key] = Array.from(new Set([
                ...(_c = target[key]) !== null && _c !== void 0 ? _c : [],
                ...value,
            ]));
            continue;
        }
        if (typeof value === 'object' && value) {
            target[key] = deepMerge((_d = target[key]) !== null && _d !== void 0 ? _d : {}, value);
            continue;
        }
        if (value !== undefined) {
            target[key] = value;
        }
    }
    return target;
}
/**
 * Map a function over values of an object
 *
 * If the mapping function returns undefined, remove the key
 */
function mapValues(xs, fn) {
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        const mapped = fn(v);
        if (mapped !== undefined) {
            ret[k] = mapped;
        }
    }
    return ret;
}
// Combines all input asset hashes into one, or if no hashes are present, returns undefined.
function combineAssetHashesOrUndefined(hashes) {
    const hashArray = hashes.filter((x) => x !== undefined);
    return hashArray.length > 0 ? hashArray.join('') : undefined;
}
function contentHash(content) {
    return crypto.createHash('sha256').update(content).digest('hex');
}
//# sourceMappingURL=data:application/json;base64,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