import copy as _copy

import numpy as _np
import pytest
import xarray as _xr

# From OceanSpy
from oceanspy import open_oceandataset
from oceanspy.llc_rearrange import Dims
from oceanspy.llc_rearrange import LLCtransformation as LLC
from oceanspy.llc_rearrange import (
    _edge_arc_data,
    _edge_facet_data,
    arc_limits_mask,
    arct_connect,
    combine_list_ds,
    mask_var,
    mates,
    rotate_dataset,
    rotate_vars,
    shift_dataset,
    shift_list_ds,
    slice_datasets,
)
from oceanspy.utils import _reset_range, get_maskH

Datadir = "./oceanspy/tests/Data/"
ECCO_url = "{}catalog_ECCO.yaml".format(Datadir)
od = open_oceandataset.from_catalog("LLC", ECCO_url)

Nx = od._ds.dims["X"]
Ny = od._ds.dims["Y"]

_datype = _xr.core.dataarray.DataArray
_dstype = _xr.core.dataset.Dataset

# WOCE standard sections
A01_lon = [
    -54.49,
    -54.285,
    -54.1317,
    -53.945,
    -53.8267,
    -53.6083,
    -53.3767,
    -49.175,
    -49.995,
    -49.4533,
    -48.8967,
    -48.6883,
    -48.57,
    -48.5317,
    -48.4217,
    -48.225,
    -42.505,
    -42.1667,
    -41.86,
    -41.4967,
    -41.2233,
    -40.8917,
    -40.0617,
    -38.9133,
    -37.64,
    -37.6433,
    -36.3667,
    -35.0817,
    -33.7883,
    -33.0333,
    -32.3017,
    -31.5633,
    -30.8067,
    -30.2733,
    -29.7267,
    -29.1883,
    -28.6667,
    -28.125,
    -28.1333,
    -27.8033,
    -27.4933,
    -27.15,
    -26.8017,
    -26.4683,
    -26.13,
    -25.875,
    -25.6333,
    -25.4133,
    -24.6867,
    -23.5333,
    -22.39,
    -21.2433,
    -20.01,
    -18.8817,
    -17.8283,
    -16.99,
    -16.1833,
    -15.7633,
    -15.4933,
    -15.215,
    -14.925,
    -14.6417,
    -14.2583,
]

A01_lat = [
    54.7533,
    54.955,
    55.1083,
    55.24,
    55.4133,
    55.615,
    55.8433,
    59.7467,
    59.0433,
    59.4833,
    59.9917,
    60.1833,
    60.3,
    60.3517,
    60.4067,
    60.5633,
    59.9933,
    59.9567,
    59.92,
    59.875,
    59.8667,
    59.8483,
    59.7417,
    59.6467,
    59.52,
    59.5233,
    59.3917,
    59.2567,
    59.1317,
    59.0333,
    58.9717,
    58.88,
    58.795,
    58.5083,
    58.24,
    57.9617,
    57.6783,
    57.3817,
    57.3733,
    56.915,
    56.4533,
    55.9833,
    55.53,
    55.0717,
    54.5983,
    54.2933,
    53.9683,
    53.67,
    53.4483,
    53.1767,
    52.87,
    52.56,
    52.33,
    52.345,
    52.3233,
    52.3333,
    52.3283,
    52.3317,
    52.3333,
    52.33,
    52.3333,
    52.3333,
    52.3333,
]

A03_lon = [
    -8.5263,
    -8.7913,
    -9.2707,
    -11.056,
    -12.2588,
    -13.4383,
    -15.2052,
    -16.409,
    -17.5902,
    -18.7667,
    -20.5403,
    -21.7258,
    -23.5075,
    -24.709,
    -25.8893,
    -27.0733,
    -28.2547,
    -30.0337,
    -31.2247,
    -32.406,
    -33.5903,
    -34.7717,
    -35.955,
    -37.1423,
    -38.9333,
    -40.1187,
    -41.8923,
    -43.0833,
    -44.2413,
    -46.0227,
    -48.3923,
    -49.59,
    -50.7863,
    -52.0168,
    -53.2757,
    -54.5547,
    -55.8257,
    -57.0927,
    -58.355,
    -59.5913,
    -60.8432,
    -62.792,
    -64.0677,
    -65.3348,
    -66.6078,
    -68.5217,
    -69.3757,
    -69.802,
    -70.205,
    -70.8197,
    -71.2227,
    -71.5228,
    -71.9333,
    -72.2883,
    -72.6343,
    -72.9763,
    -73.3187,
]

A03_lat = [
    36.8758,
    36.4247,
    36.2353,
    36.255,
    36.2562,
    36.253,
    36.2532,
    36.2667,
    36.239,
    36.2363,
    36.2667,
    36.2532,
    36.2542,
    36.2257,
    36.252,
    36.2412,
    36.2502,
    36.2417,
    36.2373,
    36.25,
    36.243,
    36.2383,
    36.2377,
    36.256,
    36.2402,
    36.2423,
    36.2403,
    36.2342,
    36.25,
    36.2257,
    36.2363,
    36.234,
    36.2333,
    36.25,
    36.254,
    36.2355,
    36.2343,
    36.2573,
    36.252,
    36.2757,
    36.2417,
    36.2505,
    36.251,
    36.2342,
    36.235,
    36.2363,
    36.3227,
    36.507,
    36.7197,
    36.9893,
    37.1833,
    37.3583,
    37.4757,
    37.6227,
    37.7747,
    37.9255,
    38.0863,
]

A05_lon = [
    -15.9717,
    -17.0942,
    -18.7502,
    -21.3338,
    -23.9965,
    -26.6713,
    -29.3292,
    -31.9947,
    -34.6752,
    -37.3267,
    -39.9963,
    -42.3343,
    -44.6697,
    -46.9997,
    -49.3288,
    -51.6598,
    -53.9998,
    -56.6703,
    -59.3328,
    -61.9872,
    -64.6655,
    -67.3227,
    -70.0017,
    -72.5737,
    -74.822,
    -79.2333,
    -79.5667,
    -79.9333,
]

A05_lat = [
    24.5,
    24.4987,
    24.5023,
    24.5003,
    24.5007,
    24.507,
    24.5032,
    24.5018,
    24.514,
    24.5087,
    24.507,
    24.5017,
    24.501,
    24.4997,
    24.5012,
    24.5083,
    24.4987,
    24.4948,
    24.5003,
    24.4888,
    24.502,
    24.5245,
    24.5007,
    24.7493,
    25.0975,
    26.0667,
    26.0547,
    26.0452,
]


A10_lon = [
    -47.3867,
    -47.0263,
    -46.6698,
    -46.3083,
    -45.9388,
    -45.4082,
    -44.2217,
    -42.9098,
    -41.7367,
    -40.5867,
    -39.5303,
    -38.9165,
    -38.0033,
    -37.1695,
    -36.4998,
    -35.8357,
    -35.1665,
    -34.5015,
    -33.509,
    -32.4993,
    -31.504,
    -30.5015,
    -29.5093,
    -28.4203,
    -26.719,
    -25.0178,
    -23.3153,
    -21.6217,
    -19.9187,
    -18.3845,
    -17.0133,
    -15.6625,
    -14.3347,
    -12.9893,
    -11.6763,
    -10.3297,
    -8.9983,
    -7.3322,
    -5.6678,
    -4.0005,
    -2.3233,
    -0.7297,
    0.5665,
    1.7007,
    2.8298,
    3.7778,
    5.0995,
    6.78,
    8.4653,
    10.1477,
    11.8303,
    12.4668,
    13.1142,
    13.7347,
    14.3665,
    14.999,
]

A10_lat = [
    -27.6833,
    -27.816,
    -27.9052,
    -27.9917,
    -28.0885,
    -28.2238,
    -28.615,
    -29.0362,
    -29.4217,
    -29.8073,
    -30.0027,
    -30.0038,
    -30.0027,
    -30.0017,
    -30.006,
    -30.0042,
    -29.9963,
    -29.9975,
    -29.992,
    -29.9965,
    -30.0002,
    -29.9995,
    -29.9983,
    -29.9977,
    -29.998,
    -29.9963,
    -29.9998,
    -29.9983,
    -29.9988,
    -30.008,
    -29.9987,
    -30.0,
    -30.0058,
    -30.0128,
    -30.0023,
    -30.0007,
    -30.0,
    -30.0017,
    -29.9988,
    -29.9975,
    -29.995,
    -30.0163,
    -29.8583,
    -29.6038,
    -29.3373,
    -29.6085,
    -29.7517,
    -29.754,
    -29.7457,
    -29.7478,
    -29.7517,
    -29.4972,
    -29.2425,
    -29.0012,
    -28.7517,
    -28.5028,
]


A12_lon = [
    1.79667e01,
    1.75500e01,
    1.69333e01,
    1.58000e01,
    1.46667e01,
    1.35500e01,
    1.19833e01,
    1.07833e01,
    9.40000e00,
    7.73330e00,
    6.05000e00,
    4.36670e00,
    2.66670e00,
    1.83330e00,
    1.23330e00,
    9.16700e-01,
    2.83300e-01,
    3.33000e-02,
    0.00000e00,
    1.67000e-02,
    0.00000e00,
    0.00000e00,
    0.00000e00,
    1.67000e-02,
    0.00000e00,
    0.00000e00,
    0.00000e00,
    0.00000e00,
    0.00000e00,
    0.00000e00,
    0.00000e00,
    1.67000e-02,
    1.67000e-02,
    1.67000e-02,
    -3.66700e-01,
    -9.15000e00,
    -9.91670e00,
    -2.50000e01,
    -3.35167e01,
    -3.64500e01,
    -4.02167e01,
    -4.19500e01,
    -4.35833e01,
    -4.35500e01,
    -4.42833e01,
    -4.29833e01,
    -4.71833e01,
    -4.79500e01,
    -4.93167e01,
    -5.02667e01,
]

A12_lat = [
    -34.1833,
    -34.55,
    -34.1167,
    -36.1,
    -37.05,
    -38.05,
    -39.35,
    -40.3167,
    -41.05,
    -41.5167,
    -42.0,
    -42.45,
    -42.9333,
    -44.25,
    -45.5,
    -46.1333,
    -47.4,
    -49.9833,
    -50.9833,
    -54.0,
    -56.0,
    -58.0,
    -59.0,
    -60.0,
    -61.0,
    -62.0,
    -62.9833,
    -64.0,
    -65.0,
    -66.0,
    -67.0,
    -68.0,
    -68.5,
    -69.0,
    -69.5,
    -70.5167,
    -70.4833,
    -69.2167,
    -65.9667,
    -65.55,
    -65.1333,
    -64.6833,
    -63.4167,
    -62.6667,
    -62.0,
    -60.9667,
    -60.2167,
    -60.1667,
    -60.1667,
    -60.2167,
]


A16_lon = [
    -30.8953,
    -30.9118,
    -33.6328,
    -34.7378,
    -36.1152,
    -35.233,
    -34.0001,
    -32.7767,
    -31.513,
    -30.2636,
    -29.0328,
    -27.8001,
    -26.4387,
    -25.3001,
    -25.0005,
    -25.0001,
    -24.9975,
    -25.0002,
    -25.0001,
    -25.0002,
    -25.0001,
    -24.9979,
    -25.0001,
    -25.0015,
    -25.0002,
    -24.9999,
    -25.0001,
    -24.9992,
    -24.9998,
    -25.0,
    -25.0002,
    -25.0002,
    -25.0022,
    -25.0003,
    -25.0,
    -25.7479,
    -26.7509,
    -27.4999,
    -28.7487,
    -29.0002,
    -28.9944,
    -29.0001,
    -28.7185,
    -27.3157,
    -26.1834,
    -25.3517,
    -24.2167,
    -23.0993,
    -21.9664,
    -20.8503,
    -19.9994,
    -19.9994,
    -20.0015,
    -20.0013,
    -19.9967,
    -20.0005,
    -19.9996,
    -19.9919,
    -20.0009,
    -20.0005,
    -19.9997,
    -19.9983,
    -20.0003,
    -19.9991,
    -20.0014,
]

A16_lat = [
    -60.013,
    -58.0281,
    -55.9992,
    -55.2302,
    -53.4317,
    -52.0,
    -50.0008,
    -48.0071,
    -45.9991,
    -44.0015,
    -41.9999,
    -39.9986,
    -38.0,
    -36.0,
    -34.0002,
    -32.0005,
    -30.0002,
    -28.0001,
    -26.0001,
    -23.9999,
    -21.9998,
    -19.9995,
    -18.0003,
    -16.0005,
    -14.0001,
    -11.9984,
    -10.0004,
    -7.9989,
    -6.0016,
    -4.4989,
    -2.6669,
    -1.3326,
    -0.3323,
    1.3336,
    2.6675,
    4.5017,
    6.5006,
    8.0,
    10.5009,
    13.0023,
    15.6684,
    18.9999,
    20.5043,
    23.0006,
    25.0004,
    26.5015,
    28.4996,
    30.5005,
    32.5018,
    34.5008,
    35.9998,
    38.4999,
    40.5002,
    43.9989,
    44.4983,
    46.0023,
    48.4993,
    50.498,
    52.4999,
    54.4991,
    56.4998,
    58.4995,
    60.4994,
    61.8329,
    63.2173,
]


A24_lon = [
    -27.035,
    -25.766,
    -24.318,
    -22.8665,
    -21.3673,
    -19.8657,
    -18.3167,
    -16.7487,
    -15.1642,
    -13.5245,
    -12.4062,
    -11.2297,
    -14.3157,
    -14.7447,
    -16.1748,
    -16.87,
    -9.3343,
    -9.9807,
    -11.8422,
    -12.7628,
    -13.6292,
    -16.0682,
    -18.9,
    -21.1902,
    -23.953,
    -26.3023,
    -27.9245,
    -29.6483,
    -32.2612,
    -34.946,
    -36.2038,
    -36.7672,
    -38.9553,
    -42.714,
    -41.9523,
    -40.7612,
    -39.169,
    -37.6393,
    -35.9328,
    -34.7423,
    -34.2147,
    -33.35,
    -32.5005,
    -31.6378,
    -30.777,
    -30.013,
    -29.2635,
    -28.6958,
    -28.231,
    -27.7658,
    -27.3005,
    -27.0322,
]

A24_lat = [
    38.8193,
    39.7163,
    40.7328,
    41.7383,
    42.7665,
    43.7868,
    44.7995,
    45.8168,
    46.8375,
    47.8335,
    48.5373,
    49.1817,
    53.6897,
    53.9355,
    54.584,
    54.8878,
    57.7497,
    57.7478,
    57.7503,
    57.7697,
    57.8957,
    58.35,
    58.9113,
    59.3512,
    59.8672,
    60.5817,
    61.1753,
    61.7505,
    62.6343,
    63.5038,
    63.9125,
    64.0837,
    64.7667,
    59.5618,
    58.847,
    57.741,
    56.5162,
    55.2828,
    53.8827,
    52.7967,
    51.9222,
    50.5167,
    49.1048,
    47.626,
    46.085,
    44.6842,
    43.2815,
    42.2018,
    41.2505,
    40.3173,
    39.3832,
    38.817,
]


P01_lon = [
    166.7448,
    176.0947,
    178.3045,
    179.4483,
    -179.4295,
    -178.3077,
    -177.2105,
    -176.0447,
    -174.9547,
    -173.796,
    -172.706,
    -171.5572,
    -170.4295,
    -169.3425,
    -168.2105,
    -167.083,
    -165.9802,
    -164.9857,
    -163.7218,
    -162.6202,
    -161.4863,
    -160.3612,
    -159.2583,
    -158.1392,
    -157.026,
    -155.863,
    -154.7653,
    -153.6277,
    -152.5307,
    -151.9977,
    -151.4153,
    -150.295,
    -149.1445,
    -148.0368,
    -146.9277,
    -145.8067,
    -144.4363,
    -143.4973,
    -142.4382,
    -141.353,
    -140.226,
    -139.0657,
    -137.9633,
    -136.8538,
    -135.7345,
    -134.6172,
    -133.4665,
    -132.3657,
    -131.2317,
    -130.031,
    -129.3797,
    -128.6433,
    -127.917,
    -127.199,
    -126.4712,
    -126.0028,
    -125.5103,
    -125.058,
    -124.9837,
]

P01_lat = [
    46.9913,
    46.9973,
    47.0028,
    47.0125,
    46.9947,
    46.9968,
    46.9993,
    47.0123,
    47.0052,
    47.0008,
    46.9957,
    47.0082,
    46.9992,
    46.9945,
    46.9998,
    47.0022,
    47.0118,
    46.9983,
    47.0093,
    47.0025,
    47.0,
    46.9975,
    46.9885,
    47.0043,
    47.0068,
    46.9943,
    46.994,
    47.0,
    46.9903,
    46.9965,
    47.0018,
    46.9892,
    46.9917,
    47.0085,
    46.9982,
    46.9938,
    46.9,
    47.0152,
    47.0082,
    46.9892,
    47.0267,
    46.9952,
    46.993,
    46.9923,
    47.0,
    47.0212,
    46.992,
    46.997,
    47.0065,
    46.9802,
    46.9965,
    47.003,
    46.9958,
    46.9942,
    47.005,
    47.0052,
    47.0042,
    46.999,
    46.9383,
]

P02_lon = [
    133.0297,
    133.4301,
    133.7585,
    134.3366,
    136.32,
    138.7937,
    141.1255,
    143.1775,
    146.0208,
    150.0987,
    154.72,
    159.1331,
    161.7337,
    165.1952,
    168.0658,
    170.74,
    173.9782,
    176.5319,
    179.0847,
    -178.1333,
    -174.3811,
    -169.7624,
    -166.3097,
    -161.7357,
    -157.1623,
    -152.8654,
    -150.267,
    -147.9584,
    -144.886,
    -141.8127,
    -138.7416,
    -135.6686,
    -132.5963,
    -129.5237,
    -126.4515,
    -123.7607,
    -121.7718,
    -119.7702,
    -119.1495,
    -117.8245,
]

P02_lat = [
    32.5068,
    32.124,
    31.6349,
    30.6143,
    29.9997,
    29.9998,
    30.0001,
    30.0002,
    30.0016,
    30.0001,
    29.9999,
    30.0002,
    30.0017,
    30.0001,
    29.9999,
    30.0004,
    30.0003,
    30.0004,
    29.9996,
    29.9999,
    30.0,
    30.0002,
    30.0,
    30.0001,
    30.0002,
    30.0001,
    30.0003,
    30.0002,
    30.0008,
    30.0004,
    29.9997,
    29.9998,
    29.9998,
    30.0004,
    29.9995,
    30.0012,
    31.0267,
    32.0002,
    32.309,
    32.643,
]

P04_lon = [
    126.5483,
    126.645,
    127.0833,
    127.665,
    128.505,
    129.4983,
    130.8383,
    132.4383,
    134.085,
    134.7867,
    135.3317,
    136.2483,
    136.6417,
    137.835,
    138.195,
    138.6783,
    139.58,
    140.4067,
    141.6183,
    142.84,
    144.5033,
    145.8333,
    146.9967,
    148.0867,
    148.755,
    149.9933,
    151.6717,
    153.3267,
    154.995,
    156.6633,
    158.3233,
    159.9967,
    161.6517,
    163.3333,
    165.0017,
    166.4167,
    167.9483,
    169.7517,
    170.4167,
    169.28,
    171.33,
    172.0,
    173.3333,
    174.6667,
    176.3267,
    177.9967,
    179.6617,
    -178.67,
    -177.0017,
    -175.3183,
    -173.6667,
    -172.0033,
    -170.335,
    -168.6717,
    -167.01,
    -165.3367,
    -163.6717,
    -162.005,
    -161.1683,
    -160.3383,
    -158.6633,
    -157.0133,
    -155.345,
    -153.67,
    -152.0067,
    -150.365,
    -148.6667,
    -147.0017,
    -145.3333,
    -143.67,
    -142.0,
    -140.3333,
    -138.6667,
    -137.0033,
    -135.3333,
    -133.6667,
    -132.0,
    -130.3317,
    -128.6667,
    -127.0033,
    -125.335,
    -123.6667,
    -122.0017,
    -120.3333,
    -118.67,
    -117.0017,
    -115.3317,
    -113.6683,
    -112.0033,
    -110.335,
    -108.67,
    -107.0017,
    -105.33,
    -104.165,
    -103.1683,
    -101.5,
    -99.86,
    -98.1667,
    -96.5,
    -94.8283,
    -93.1667,
    -91.4617,
    -89.995,
    -88.6667,
    -87.3317,
    -86.3317,
    -86.2617,
    -85.9833,
]

P04_lat = [
    7.9917,
    7.9733,
    8.0017,
    8.0017,
    8.0067,
    7.9967,
    8.1733,
    8.5017,
    8.8333,
    8.97,
    8.995,
    9.0017,
    9.0333,
    9.0033,
    8.9983,
    8.98,
    8.9917,
    9.085,
    9.1967,
    9.2833,
    9.4483,
    9.4983,
    9.4983,
    9.5,
    9.4967,
    9.4917,
    9.4867,
    9.4967,
    9.49,
    9.4917,
    9.4833,
    9.4883,
    9.4917,
    9.4917,
    9.4917,
    9.6133,
    9.6,
    9.6633,
    9.6133,
    9.6117,
    9.4933,
    9.4933,
    9.4867,
    9.49,
    9.4967,
    9.49,
    9.4983,
    9.4933,
    9.4933,
    9.5017,
    9.4933,
    9.495,
    9.4967,
    9.495,
    9.4967,
    9.4983,
    9.4933,
    9.4967,
    9.5,
    9.4983,
    9.4867,
    9.4933,
    9.4917,
    9.4917,
    9.5,
    9.4933,
    9.4983,
    9.49,
    9.4967,
    9.5033,
    9.4983,
    9.49,
    9.4967,
    9.4917,
    9.4933,
    9.5017,
    9.4883,
    9.5,
    9.4917,
    9.5,
    9.4917,
    9.4933,
    9.5,
    9.4967,
    9.4933,
    9.4917,
    9.4983,
    9.4983,
    9.49,
    9.5,
    9.5017,
    9.4983,
    9.5067,
    9.5,
    9.4983,
    9.4983,
    9.5083,
    9.495,
    9.4983,
    9.4933,
    9.5,
    9.5017,
    9.5033,
    9.4983,
    9.4983,
    9.4967,
    9.62,
    9.7033,
]

P06_lon = [
    -178.9095,
    -178.4699,
    -177.9998,
    -177.2492,
    -176.2493,
    -175.2594,
    -174.3296,
    -173.3827,
    -172.4188,
    -171.9197,
    -171.5896,
    -170.7493,
    -169.7504,
    -168.4447,
    -167.1415,
    -165.8159,
    -164.4883,
    -163.1627,
    -161.8331,
    -160.4581,
    -158.9765,
    -157.6164,
    -156.275,
    -154.9369,
    -153.5987,
    -152.2584,
    -150.9182,
    -149.5785,
    -148.9698,
    -145.7093,
    -142.2516,
    -138.5881,
    -135.028,
    -131.467,
    -127.9686,
    -125.1713,
    -122.216,
    -118.8953,
    -115.5757,
    -112.3954,
    -109.5899,
    -106.8758,
    -104.1169,
    -101.3555,
    -97.9953,
    -94.4335,
    -90.8756,
    -87.3154,
    -83.7564,
    -80.1931,
    -76.6361,
    -73.1964,
    -72.6822,
    -72.3354,
    -71.7483,
]

P06_lat = [
    -32.5,
    -32.4997,
    -32.4998,
    -32.499,
    -32.4996,
    -32.5,
    -32.5001,
    -32.5003,
    -32.5003,
    -32.4994,
    -32.4998,
    -32.4997,
    -32.4996,
    -32.5002,
    -32.4999,
    -32.5007,
    -32.4998,
    -32.4994,
    -32.4994,
    -32.4999,
    -32.4997,
    -32.4999,
    -32.5,
    -32.4999,
    -32.5002,
    -32.4997,
    -32.5006,
    -32.4999,
    -32.4997,
    -32.5008,
    -32.5051,
    -32.4989,
    -32.5004,
    -32.4996,
    -32.5003,
    -32.4998,
    -32.4999,
    -32.5001,
    -32.4995,
    -32.5001,
    -32.5004,
    -32.5004,
    -32.5001,
    -32.5004,
    -32.4999,
    -32.5003,
    -32.4997,
    -32.4998,
    -32.5005,
    -32.5003,
    -32.4998,
    -32.4995,
    -32.4999,
    -32.4999,
    -32.5004,
]


P10_lon = [
    144.8112,
    144.8917,
    145.15,
    145.3583,
    145.4972,
    145.6417,
    145.7862,
    145.9278,
    146.0722,
    146.2138,
    146.3583,
    146.5,
    146.6445,
    146.792,
    146.9305,
    147.2278,
    147.6528,
    148.0667,
    149.3333,
    148.6972,
    149.1667,
    149.3305,
    149.3278,
    149.3333,
    149.3388,
    149.3278,
    149.3417,
    149.3305,
    149.3305,
    149.3362,
    149.3333,
    149.3305,
    149.3417,
    149.3333,
    149.3333,
    149.3362,
    149.3222,
    148.6888,
    147.4028,
    146.1195,
    144.8888,
    143.6638,
    142.7112,
    141.9305,
    141.4583,
    140.9833,
    140.8305,
]


P10_lat = [
    -4.0153,
    -3.8888,
    -3.3083,
    -2.75,
    -2.2528,
    -1.7555,
    -1.25,
    -0.7472,
    -0.25,
    0.25,
    0.75,
    1.25,
    1.75,
    2.25,
    2.7472,
    3.4917,
    4.4945,
    5.4862,
    8.4972,
    7.0,
    7.9945,
    9.45,
    9.9972,
    11.1612,
    12.5028,
    13.8362,
    15.1695,
    16.5055,
    17.8333,
    19.1695,
    20.5112,
    21.8362,
    23.1833,
    24.5055,
    25.8333,
    27.1667,
    28.5055,
    29.6805,
    30.7,
    31.7167,
    32.6862,
    33.6612,
    34.1888,
    34.5972,
    34.8055,
    35.0388,
    35.125,
]

P11_lon = [
    153.934,
    154.0515,
    154.1707,
    154.2882,
    154.395,
    154.5135,
    154.634,
    154.756,
    154.8805,
    155.0008,
    154.8728,
    155.1183,
    155.1173,
    155.3745,
    155.6343,
    155.8993,
    156.1703,
    156.0035,
    155.1658,
    154.9988,
    154.9977,
    155.0015,
    155.003,
    155.0008,
    155.0023,
    154.9995,
    154.9995,
    155.0003,
    154.9983,
    155.0135,
    155.0045,
    154.9985,
    154.9917,
    154.9917,
    155.0025,
    155.0087,
    155.0133,
    154.978,
    155.0222,
    155.0377,
    154.9322,
    155.044,
    154.996,
    154.478,
    153.4507,
    152.1733,
    151.1578,
    150.0918,
    149.2747,
    148.6738,
    146.515,
    146.4507,
    143.948,
]

P11_lat = [
    -12.0483,
    -13.1648,
    -14.4717,
    -15.6478,
    -16.72,
    -17.8868,
    -19.0457,
    -20.2098,
    -21.3758,
    -22.6663,
    -23.3698,
    -23.3573,
    -23.9985,
    -25.1655,
    -26.32,
    -27.473,
    -28.627,
    -29.7785,
    -30.8657,
    -32.1243,
    -33.4743,
    -34.5757,
    -35.763,
    -36.9378,
    -38.1152,
    -39.2905,
    -40.4662,
    -41.6507,
    -42.8185,
    -44.2385,
    -45.2415,
    -46.2542,
    -47.2507,
    -48.2515,
    -49.2578,
    -50.2232,
    -51.2552,
    -52.2193,
    -53.2637,
    -54.2618,
    -55.2582,
    -56.424,
    -57.5855,
    -58.868,
    -60.0807,
    -61.6012,
    -62.8678,
    -64.057,
    -64.9925,
    -65.5783,
    -65.795,
    -65.7735,
    -65.4475,
]


P14_lon = [
    175.0345,
    175.2432,
    175.5537,
    175.9343,
    176.2968,
    176.6637,
    177.0322,
    177.4007,
    177.6367,
    177.5775,
    177.8275,
    178.9968,
    179.0085,
    178.9983,
    178.994,
    179.0052,
    179.0032,
    178.999,
    179.0007,
    178.9972,
    178.9983,
    178.999,
    179.0067,
    178.9938,
    178.9962,
    179.0115,
    179.0003,
    178.9943,
    179.0002,
    178.9737,
    178.994,
    179.0048,
    178.9998,
    179.0025,
    178.9968,
    178.9988,
    179.7522,
    179.0023,
    179.0085,
    179.0055,
    178.9993,
    178.9932,
    178.9998,
    178.9923,
    179.0037,
    178.9955,
    178.9993,
    178.9955,
    179.0012,
    178.9975,
    179.0047,
    179.0007,
    179.0673,
    179.0163,
    178.9987,
    178.9988,
    179.0033,
    178.9943,
    178.9945,
    179.0052,
    178.9955,
    179.4168,
    -179.7845,
    -178.2538,
    -176.4993,
    -174.4137,
]

P14_lat = [
    -35.6343,
    -34.4658,
    -32.6027,
    -30.4468,
    -28.2725,
    -26.1042,
    -23.9583,
    -21.7682,
    -20.3723,
    -19.2115,
    -18.751,
    -14.5,
    -12.0043,
    -9.5005,
    -7.0008,
    -4.7552,
    -3.4955,
    -2.2512,
    -0.9942,
    0.2505,
    1.505,
    2.7468,
    3.9923,
    5.2465,
    6.4938,
    7.7487,
    -15.4975,
    -13.0002,
    -10.5048,
    -8.6143,
    -8.5005,
    -6.009,
    -4.2565,
    -2.996,
    -1.7455,
    -0.4957,
    0.0,
    1.0038,
    2.2468,
    3.4965,
    4.7447,
    6.0,
    7.2452,
    8.4963,
    10.506,
    13.0017,
    15.5057,
    18.0007,
    20.5048,
    23.0057,
    25.4945,
    27.9988,
    29.814,
    31.4983,
    33.9918,
    36.5023,
    38.9932,
    41.4898,
    43.9988,
    46.4992,
    48.9953,
    50.6997,
    51.8588,
    53.4963,
    56.0013,
    58.4998,
]

P16_lon = [
    -153.2202,
    -153.0598,
    -152.95,
    -152.2202,
    -151.9998,
    -151.9987,
    -151.9997,
    -152.0,
    -152.0005,
    -152.0,
    -152.0003,
    -152.0,
    -152.0003,
    -152.0,
    -152.0003,
    -152.0,
    -152.0,
    -152.0007,
    -151.5002,
    -151.0,
    -151.0,
    -151.0012,
    -151.0,
    -151.0005,
    -151.0,
    -151.0003,
    -151.0008,
    -150.8,
    -149.9925,
    -150.2452,
    -150.0002,
    -150.0,
    -149.9998,
    -149.9997,
    -149.9997,
    -150.0003,
    -150.0,
    -150.0003,
    -150.0,
    -150.0,
    -150.0008,
    -150.0,
    -150.0,
    -150.0,
    -150.0,
    -149.9998,
    -149.9982,
    -150.0003,
    -150.0003,
    -149.9985,
    -149.9997,
    -149.9968,
    -149.9997,
    -150.0017,
    -149.9967,
    -149.997,
    -150.0022,
    -150.0017,
    -150.0002,
    -149.9998,
    -150.0377,
    -149.997,
    -150.0,
    -150.1025,
    -149.9997,
    -150.001,
    -150.0025,
]

P16_lat = [
    56.2802,
    55.9197,
    55.67,
    54.0,
    50.9998,
    48.0017,
    44.9998,
    41.0,
    37.9997,
    35.0,
    31.9998,
    29.0,
    26.0002,
    23.0,
    20.0002,
    17.0,
    14.0,
    11.0003,
    8.0002,
    5.0,
    2.0,
    0.5,
    -1.0,
    -2.9997,
    -6.0,
    -9.0,
    -12.0003,
    -15.0,
    -16.0,
    -17.4998,
    -18.9997,
    -20.0,
    -21.5002,
    -22.9997,
    -24.4992,
    -26.0002,
    -27.5,
    -29.0,
    -30.5,
    -31.5,
    -32.9988,
    -34.5,
    -35.9997,
    -37.4993,
    -38.9997,
    -40.5,
    -42.0,
    -43.4997,
    -45.0002,
    -45.9995,
    -47.4998,
    -48.9995,
    -50.4993,
    -52.0005,
    -53.4995,
    -55.0005,
    -56.5002,
    -57.9993,
    -59.501,
    -60.9997,
    -62.0,
    -63.499,
    -64.9995,
    -66.5,
    -67.9987,
    -69.4998,
    -70.9995,
]


I05_lon = [
    30.3437,
    30.5138,
    31.1739,
    32.3194,
    33.7612,
    35.5866,
    36.4947,
    38.6804,
    40.9864,
    42.8326,
    43.9076,
    46.2329,
    48.6105,
    50.9774,
    53.1249,
    54.8836,
    56.4469,
    57.0411,
    58.1793,
    60.5499,
    62.8397,
    65.1243,
    67.412,
    69.6983,
    71.9849,
    73.2475,
    75.0492,
    76.9342,
    78.7907,
    80.1515,
    82.564,
    84.9734,
    86.8619,
    88.6982,
    90.823,
    92.8975,
    94.9896,
    97.2751,
    99.3639,
    100.2442,
    101.5723,
    102.4311,
    104.3666,
    105.292,
    107.2215,
    109.5514,
    111.7592,
    113.9755,
    114.4665,
]

I05_lat = [
    -31.0777,
    -31.1679,
    -31.6528,
    -32.1799,
    -32.6207,
    -32.9969,
    -32.9997,
    -32.9973,
    -32.997,
    -32.998,
    -32.9997,
    -33.2258,
    -33.5341,
    -33.8367,
    -34.0001,
    -33.9975,
    -33.9985,
    -33.5577,
    -33.9982,
    -34.0008,
    -34.0008,
    -33.9986,
    -34.0014,
    -33.9995,
    -33.9993,
    -34.4025,
    -32.8138,
    -31.5341,
    -30.6105,
    -31.1868,
    -31.1923,
    -31.1971,
    -31.5129,
    -31.8206,
    -32.1759,
    -33.0831,
    -33.9966,
    -33.997,
    -33.6276,
    -32.7515,
    -32.0069,
    -32.0028,
    -31.4089,
    -31.0672,
    -31.2926,
    -31.5946,
    -32.0392,
    -32.4885,
    -32.5865,
]


I06S_lon = [
    28.0952,
    29.9871,
    29.9609,
    30.0005,
    30.0016,
    29.9962,
    30.0016,
    30.0013,
    30.0003,
    29.9983,
    30.0009,
    30.0015,
    29.9958,
    30.0029,
    30.0002,
    29.9983,
    30.0024,
    30.0026,
    30.0012,
    30.0023,
    29.9978,
    30.0096,
    30.0008,
    29.9988,
    29.9997,
    30.0002,
    30.0,
    29.9997,
    29.9994,
    30.001,
    30.0004,
    30.0003,
    30.0008,
    29.9989,
    30.0011,
    30.0003,
    30.0018,
    29.9986,
    30.3813,
    30.7556,
    30.9434,
    31.0967,
    31.2087,
    31.3218,
    31.395,
    31.535,
]

I06S_lat = [
    -33.2343,
    -49.9879,
    -50.5386,
    -51.0003,
    -51.4999,
    -51.9996,
    -52.5006,
    -52.9994,
    -53.4999,
    -54.0004,
    -54.4994,
    -55.0013,
    -55.504,
    -56.0035,
    -56.5029,
    -57.0048,
    -57.5023,
    -58.0019,
    -58.5002,
    -59.0069,
    -59.5004,
    -60.0064,
    -60.4999,
    -60.9996,
    -61.5002,
    -61.9998,
    -62.5004,
    -63.0004,
    -63.5012,
    -64.0,
    -64.4995,
    -64.9999,
    -65.4994,
    -65.9999,
    -66.4999,
    -66.9994,
    -67.4998,
    -67.9998,
    -68.102,
    -68.1945,
    -68.2426,
    -68.2795,
    -68.3074,
    -68.3366,
    -68.354,
    -68.3422,
]


I08S09N_lon = [
    78.3815,
    78.1253,
    80.3926,
    82.0002,
    82.0002,
    82.0101,
    82.5226,
    84.2612,
    86.1421,
    87.954,
    89.83,
    91.6778,
    93.5393,
    95.0001,
    95.0002,
    95.002,
    94.988,
    94.9909,
    95.001,
    94.9999,
    95.0059,
    95.0004,
    94.9997,
    94.9996,
    94.9999,
    94.9999,
    94.9999,
    95.0001,
    95.0001,
    94.9999,
    95.0,
    95.0001,
    94.8683,
    94.3327,
    93.9318,
    93.5521,
    92.7335,
    92.02,
    91.3261,
    90.1934,
    89.0549,
    88.2016,
    87.0693,
]

I08S09N_lat = [
    -66.6027,
    -66.2999,
    -64.5799,
    -62.5003,
    -60.5001,
    -58.6101,
    -57.5131,
    -56.058,
    -54.367,
    -52.531,
    -50.5401,
    -48.5469,
    -46.5551,
    -44.351,
    -41.7851,
    -39.507,
    -37.4941,
    -35.4981,
    -33.5083,
    -31.702,
    -29.515,
    -27.7113,
    -25.3351,
    -23.0157,
    -20.7633,
    -19.0833,
    -16.838,
    -14.6328,
    -12.4844,
    -10.3476,
    -8.2181,
    -6.0683,
    -4.5335,
    -2.8333,
    -1.5643,
    -0.3161,
    0.8182,
    2.201,
    3.9002,
    5.394,
    6.8731,
    7.9831,
    9.4582,
]


YRange_test1 = [35, 40]  # degrees latitude
XRange_test1 = [-80, 0]  # longitude


YRange_test2 = [35, 40]  # degrees latitude
XRange_test2 = [-170, -100]  # longitude


@pytest.mark.parametrize(
    "od, var, expected",
    [
        (od, "T", ("X", "Y", "face", "Z", "time")),
        (od, "U", ("Xp1", "Y", "face", "Z", "time")),
        (od, "V", ("X", "Yp1", "face", "Z", "time")),
    ],
)
def test_original_dims(od, var, expected):
    """test original dimensions"""
    dims = Dims([dim for dim in od._ds[var].dims][::-1])
    assert dims == expected


faces = [k for k in range(13)]


expected = [2, 5, 7, 10]  # most faces that connect with arctic cap face=6
acshape = (Nx // 2, Ny)
cuts = [[0, 28], [0, 0], [0, 0], [0, 0]]


@pytest.mark.parametrize(
    "od, faces, expected, atype, XRange, YRange, opt, cuts, masking, size",
    [
        (od, faces, expected, _datype, None, None, False, 0, False, None),
        (od, faces[:2], [0, 0, 0, 0], int, None, None, False, 0, False, None),
        (od, faces[:6], [0, 0, 0, 0], int, None, None, False, 0, False, None),
        (od, [0, 1, 2, 6], [2, 0, 0, 0], _datype, None, None, False, 0, False, None),
        (od, faces[:7], [2, 5, 0, 0], _datype, None, None, False, 0, False, None),
        (od, faces[6:], [0, 0, 7, 10], int, None, None, False, 0, False, None),
        (
            od,
            [2, 6],
            [2, 0, 0, 0],
            _datype,
            [-30, 22],
            [60.0, 80.2],
            True,
            cuts,
            False,
            (90, 29),
        ),
        (
            od,
            [2, 6],
            [2, 0, 0, 0],
            _datype,
            [-30, 22],
            [60.0, 80.2],
            True,
            cuts,
            True,
            (90, 29),
        ),
    ],
)
def test_arc_connect(
    od, faces, expected, atype, XRange, YRange, opt, cuts, masking, size
):
    # need to improve, this behavior, by adding ranges=cuts, opt=true
    # (i.e. transpose) and masking of variable (XG). when cuts is given.
    ds = od._ds
    if XRange is not None and YRange is not None:
        arc_faces, *nnn, DS = arct_connect(
            ds, "YG", faces=faces, masking=masking, opt=opt, ranges=cuts
        )
        for i in range(len(DS)):
            if type(DS[i]) == _datype:
                assert _np.shape(DS[i]) == size
    else:
        arc_faces, *a, DS = arct_connect(ds, "YG", faces)
    assert arc_faces == expected
    assert type(DS[0]) == atype


varList = ["T", "U", "V", "XG", "YG", "XC", "YC"]
XRanges_A = [A01_lon, A03_lon, A05_lon, A10_lon, A12_lon, A16_lon, A24_lon]
YRanges_A = [A01_lat, A03_lat, A05_lat, A10_lat, A12_lat, A16_lat, A24_lat]
XRanges_P = [P01_lon, P02_lon, P04_lon, P06_lon, P10_lon, P11_lon, P14_lon, P16_lon]
YRanges_P = [P01_lat, P02_lat, P04_lat, P06_lat, P10_lat, P11_lat, P14_lat, P16_lat]
XRanges_I = [I05_lon, I06S_lon, I08S09N_lon]
YRanges_I = [I05_lat, I06S_lat, I08S09N_lat]

_XRanges_ = XRanges_A + XRanges_P + XRanges_I
_YRanges_ = YRanges_A + YRanges_P + YRanges_I


@pytest.mark.parametrize(
    "od, faces, varList, XRange, YRange, X0, X1, Y0, Y1",
    [
        (od, None, varList, None, None, 0, 359, 0, 314),
        (od, [2, 5, 6, 7, 10], varList, None, None, 0, 359, 0, 134),
        (od, [2, 5, 7, 10], varList, None, None, 0, 359, 0, 89),
        (od, [1, 4, 8, 11], varList, None, None, 0, 359, 0, 89),
        (od, [0, 3, 9, 12], varList, None, None, 0, 359, 0, 89),
        (od, [6, 7, 8, 9], varList, None, None, 0, 89, 0, 314),
        (od, [6, 10, 11, 12], varList, None, None, 0, 89, 0, 314),
        (od, [0, 1, 2, 6], varList, None, None, 0, 89, 0, 314),
        (od, [3, 4, 5, 6], varList, None, None, 0, 89, 0, 314),
        (od, [2, 6, 10], varList, None, None, 0, 179, 0, 134),
        (od, [6, 7, 10], varList, None, None, 0, 179, 0, 134),
        (od, [6, 7, 8, 10, 11], varList, None, None, 0, 179, 0, 224),
        (
            od,
            [4, 5, 6, 7, 8, 10, 11],
            varList,
            None,
            None,
            0,
            269,
            0,
            224,
        ),
        (od, [7, 8, 10, 11], varList, None, None, 0, 179, 0, 179),
        (od, [1, 2, 10, 11], varList, None, None, 0, 179, 0, 179),
        (od, [8, 9, 11, 12], varList, None, None, 0, 179, 0, 179),
        (od, [0, 1, 11, 12], varList, None, None, 0, 179, 0, 179),
        (od, [0, 1, 3, 4], varList, None, None, 0, 179, 0, 179),
        (od, [9, 12], varList, None, None, 0, 179, 0, 89),
        (od, [0, 12], varList, None, None, 0, 179, 0, 89),
        (od, [0, 3], varList, None, None, 0, 179, 0, 89),
        (od, [0, 9, 12], varList, None, None, 0, 269, 0, 89),
        (od, [0, 3, 12], varList, None, None, 0, 269, 0, 89),
        (od, [0], varList[0], None, None, 0, 89, 0, 89),
        (od, [1], varList[0], None, None, 0, 89, 0, 89),
        (od, [2], varList[0], None, None, 0, 89, 0, 89),
        (od, [3], varList[0], None, None, 0, 89, 0, 89),
        (od, [4], varList[0], None, None, 0, 89, 0, 89),
        (od, [5], varList[0], None, None, 0, 89, 0, 89),
        (od, [7], varList[0], None, None, 0, 89, 0, 89),
        (od, [8], varList[0], None, None, 0, 89, 0, 89),
        (od, [9], varList[0], None, None, 0, 89, 0, 89),
        (od, [10], varList[0], None, None, 0, 89, 0, 89),
        (od, [11], varList[0], None, None, 0, 89, 0, 89),
        (od, [12], varList[0], None, None, 0, 89, 0, 89),
        (od, None, varList, _XRanges_[0], _YRanges_[0], 0, 44, 0, 24),
        (od, None, varList, _XRanges_[1], _YRanges_[1], 0, 68, 0, 8),
        (od, None, varList, _XRanges_[2], _YRanges_[2], 0, 67, 0, 6),
        (od, None, varList, _XRanges_[3], _YRanges_[3], 0, 66, 0, 7),
        (od, None, varList, _XRanges_[4], _YRanges_[4], 0, 71, 0, 72),
        (od, None, varList, _XRanges_[5], _YRanges_[5], 0, 21, 0, 179),
        (od, None, varList, _XRanges_[6], _YRanges_[6], 0, 39, 0, 55),
        (od, None, varList, _XRanges_[7], _YRanges_[7], 0, 72, 0, 6),
        (od, None, varList, _XRanges_[8], _YRanges_[8], 0, 112, 0, 8),
        (od, None, varList, _XRanges_[9], _YRanges_[9], 0, 151, 0, 6),
        (od, None, varList, _XRanges_[10], _YRanges_[10], 0, 109, 0, 5),
        (od, None, varList, _XRanges_[11], _YRanges_[11], 0, 12, 0, 56),
        (od, None, varList, _XRanges_[12], _YRanges_[12], 0, 16, 0, 84),
        (od, None, varList, _XRanges_[13], _YRanges_[13], 0, 13, 0, 127),
        (od, None, varList, _XRanges_[14], _YRanges_[14], 0, 7, 0, 191),
        (od, None, varList, _XRanges_[15], _YRanges_[15], 0, 87, 0, 9),
        (od, None, varList, _XRanges_[16], _YRanges_[16], 0, 6, 0, 68),
        (od, None, varList, _XRanges_[17], _YRanges_[17], 0, 20, 0, 118),
        (od, None, varList, [-31, 25], [58, 85.2], 0, 60, 0, 71),
        (od, None, varList, [-120, -60], [58, 85.2], 0, 64, 0, 70),
        (od, None, varList, [160, -150], [58, 85.2], 0, 54, 0, 70),
        (od, None, varList, [60, 130], [58, 85.2], 0, 74, 0, 70),
    ],
)
def test_transformation(od, faces, varList, XRange, YRange, X0, X1, Y0, Y1):
    """Test the transformation fn by checking the final dimensions."""
    ds = od._ds.reset_coords().drop_vars({"hFacC", "hFacS", "hFacW"})
    args = {
        "ds": ds,
        "varList": varList,
        "XRange": XRange,
        "YRange": YRange,
        "faces": faces,
    }

    ds = LLC.arctic_crown(**args)
    xi, xf = int(ds["X"][0].values), int(ds["X"][-1].values)
    yi, yf = int(ds["Y"][0].values), int(ds["Y"][-1].values)
    assert xi == X0
    assert xf == X1
    assert yi == Y0
    assert yf == Y1


DIMS_c = [dim for dim in od.dataset["XC"].dims if dim not in ["face"]]
DIMS_g = [dim for dim in od.dataset["XG"].dims if dim not in ["face"]]
dims_c = Dims(DIMS_c[::-1])
dims_g = Dims(DIMS_g[::-1])

ds2 = []
ds5 = []
ds7 = []
ds10 = []
ARCT = [ds2, ds5, ds7, ds10]
varList = ["T", "U", "V"]
# create dataset
for var_name in varList:
    *nnn, DS = arct_connect(od.dataset, var_name, faces=None)  # horizontal only
    ARCT[0].append(DS[0])
    ARCT[1].append(DS[1])
    ARCT[2].append(DS[2])
    ARCT[3].append(DS[3])
for i in range(len(ARCT)):  # Not all faces survive the cutout
    if type(ARCT[i][0]) == _datype:
        ARCT[i] = _xr.merge(ARCT[i])

ds2, ds5, ds7, ds10 = ARCT


@pytest.mark.parametrize(
    "ds, dimc, dimg, init_c, final_c, init_g, final_g",
    [
        (ds2, dims_c.X, dims_g.X, [0, 44], [0, 44], [0, 44], [0, 44]),
        (ds7, dims_c.X, dims_g.X, [45, 89], [0, 44], [45, 89], [0, 44]),
        (ds5, dims_c.Y, dims_g.Y, [0, 44], [0, 44], [0, 44], [0, 44]),
        (ds10, dims_c.Y, dims_g.Y, [45, 89], [0, 44], [45, 89], [0, 44]),
    ],
)
def test_shift_dataset(ds, dimc, dimg, init_c, final_c, init_g, final_g):
    nds = shift_dataset(ds, dimc, dimg)
    assert [int(ds[dimc][0].values), int(ds[dimc][-1].values)] == init_c
    assert [int(ds[dimg][0].values), int(ds[dimg][-1].values)] == init_g

    assert [int(nds[dimc][0].values), int(nds[dimc][-1].values)] == final_c
    assert [int(nds[dimg][0].values), int(nds[dimg][-1].values)] == final_g


@pytest.mark.parametrize(
    "ds, var, dimc, dimg, rot_dims",
    [
        (od.dataset.isel(face=6), "T", dims_c, dims_g, ("time", "Z", "X", "Y")),
        (od.dataset.isel(face=6), "U", dims_c, dims_g, ("time", "Z", "X", "Yp1")),
        (od.dataset.isel(face=6), "V", dims_c, dims_g, ("time", "Z", "Xp1", "Y")),
        (0, "T", dims_c, dims_g, ("time", "Z", "X", "Y")),
        ("string", "T", dims_c, dims_g, ("time", "Z", "X", "Y")),
    ],
)
def test_rotate_dataset(ds, var, dimc, dimg, rot_dims):
    nds = rotate_dataset(ds, dimc, dimg)
    if type(ds) == _dstype:
        nvar = nds[var]
        assert nvar.dims == rot_dims


@pytest.mark.parametrize(
    "ds, var, dims0, rot_dims",
    [
        (
            od.dataset.isel(face=2),
            "T",
            ("time", "Z", "Y", "X"),
            ("time", "Z", "Y", "X"),
        ),
        (
            od.dataset.isel(face=2),
            "U",
            ("time", "Z", "Y", "Xp1"),
            ("time", "Z", "Yp1", "X"),
        ),
        (
            od.dataset.isel(face=2),
            "V",
            ("time", "Z", "Yp1", "X"),
            ("time", "Z", "Y", "Xp1"),
        ),
        (mates(ds2), "V", ("time", "Z", "Yp1", "X"), ("time", "Z", "Y", "Xp1")),
    ],
)
def test_rotate_vars(ds, var, dims0, rot_dims):
    nds = rotate_vars(ds)
    if type(ds) == _dstype:
        nvar = nds[var]
        assert nvar.dims == rot_dims


list1 = [od.dataset.isel(face=0), od.dataset.isel(face=1), od.dataset.isel(face=2)]
list2 = [0, od.dataset.isel(face=1), od.dataset.isel(face=2)]
list3 = [0, 0, od.dataset.isel(face=3)]
list4 = [0, 0, 0]
list5 = [0, od.dataset.isel(face=1), 0]

Np = len(od.dataset[dims_c.X])


@pytest.mark.parametrize(
    "DSlist, dimsc, dimsg, Np, facet, expX",
    [
        (
            list1,
            dims_c.X,
            dims_g.X,
            Np,
            3,
            [[0, Np - 1], [Np, int(2 * Np) - 1], [int(2 * Np), int(3 * Np) - 1]],
        ),
        (list2, dims_c.X, dims_g.X, Np, 3, [[Np, 2 * Np - 1], [2 * Np, 3 * Np - 1]]),
        (list3, dims_c.X, dims_g.X, Np, 3, [[2 * Np, 3 * Np - 1]]),
        (
            list1,
            dims_c.X,
            dims_g.X,
            Np,
            1,
            [[0, Np - 1], [Np, 2 * Np - 1], [2 * Np, 3 * Np - 1]],
        ),
        (
            list2,
            dims_c.X,
            dims_g.X,
            Np,
            1,
            [[int(Np / 2), int(1.5 * Np) - 1], [int(1.5 * Np), int(2.5 * Np) - 1]],
        ),
        (list3, dims_c.X, dims_g.X, Np, 1, [[int(1.5 * Np), int(2.5 * Np) - 1]]),
        (
            list1,
            dims_c.Y,
            dims_g.Y,
            Np,
            3,
            [[0, Np - 1], [Np, 2 * Np - 1], [2 * Np, 3 * Np - 1]],
        ),
        (list2, dims_c.Y, dims_g.Y, Np, 3, [[Np, 2 * Np - 1], [2 * Np, 3 * Np - 1]]),
        (list3, dims_c.Y, dims_g.Y, Np, 3, [[2 * Np, 3 * Np - 1]]),
        (
            list1,
            dims_c.Y,
            dims_g.Y,
            Np,
            1,
            [[0, Np - 1], [Np, 2 * Np - 1], [2 * Np, 3 * Np - 1]],
        ),
        (
            list2,
            dims_c.Y,
            dims_g.Y,
            Np,
            1,
            [[int(Np / 2), int(1.5 * Np) - 1], [int(1.5 * Np), int(2.5 * Np) - 1]],
        ),
        (list3, dims_c.Y, dims_g.Y, Np, 1, [[int(1.5 * Np), int(2.5 * Np) - 1]]),
        (list4, dims_c.Y, dims_g.Y, Np, 1, [0, 0, 0]),
        (list5, dims_c.Y, dims_g.Y, Np, 3, [[int(Np), int(2 * Np) - 1]]),
        (list5, dims_c.X, dims_g.X, Np, 3, [[int(Np), int(2 * Np) - 1]]),
        (list5, dims_c.X, dims_g.X, Np, 2, [[int(Np / 2), int(1.5 * Np) - 1]]),
    ],
)
def test_shift_list_ds(DSlist, dimsc, dimsg, Np, facet, expX):
    nDSlist = shift_list_ds(DSlist, dimsc, dimsg, Np, facet)
    if len(nDSlist) > 0:
        assert len(nDSlist) == len(expX)
        assert [
            int(nDSlist[0][dimsc][0].values),
            int(nDSlist[0][dimsc][-1].values),
        ] == expX[0]
    else:
        assert type(nDSlist) == list


list1 = [od.dataset.isel(face=0), od.dataset.isel(face=1), od.dataset.isel(face=2)]
list2 = [0, od.dataset.isel(face=1), od.dataset.isel(face=2)]
list3 = [0, 0, od.dataset.isel(face=3)]
list4 = [0, od.dataset.isel(face=1), 0]

nlist1x = shift_list_ds(_copy.copy(list1), dims_c.X, dims_g.X, Np, facet=3)
nlist1y = shift_list_ds(_copy.copy(list1), dims_c.Y, dims_g.Y, Np, facet=3)
nlist2x1 = shift_list_ds(_copy.copy(list2), dims_c.X, dims_g.X, Np, facet=1)
nlist2x2 = shift_list_ds(_copy.copy(list2), dims_c.X, dims_g.X, Np, facet=3)
nlist2y = shift_list_ds(_copy.copy(list2), dims_c.Y, dims_g.Y, Np, facet=3)
nlist3x = shift_list_ds(_copy.copy(list3), dims_c.X, dims_g.X, Np, facet=2)
nlist3y = shift_list_ds(_copy.copy(list3), dims_c.Y, dims_g.Y, Np, facet=4)
nlist4x1 = shift_list_ds(_copy.copy(list4), dims_c.X, dims_g.X, Np, facet=1)
nlist4y1 = shift_list_ds(_copy.copy(list4), dims_c.Y, dims_g.Y, Np, facet=1)
nlist4x4 = shift_list_ds(_copy.copy(list4), dims_c.X, dims_g.X, Np, facet=1234)
nlist4y4 = shift_list_ds(_copy.copy(list4), dims_c.Y, dims_g.Y, Np, facet=4)


@pytest.mark.parametrize(
    "DSlist, lenX, lenY, x0, x1, y0, y1",
    [
        (list1, int(Np), int(Np), 0, 89, 0, 89),
        (nlist1x, int(3 * Np), int(Np), 0, 269, 0, 89),
        (nlist1y, int(Np), int(3 * Np), 0, 89, 0, 269),
        (nlist2x1, int(2 * Np), int(Np), 45, 224, 0, 89),
        (nlist2x2, int(2 * Np), int(Np), 90, 269, 0, 89),
        (nlist2y, int(Np), int(2 * Np), 0, 89, 90, 269),
        (nlist3x, int(Np), int(Np), 135, 224, 0, 89),
        (nlist3y, int(Np), int(Np), 0, 89, 180, 269),
        (nlist4x1, int(Np), int(Np), 45, 134, 0, 89),
        (nlist4y1, int(Np), int(Np), 0, 89, 45, 134),
        (nlist4x4, int(Np), int(Np), 90, 179, 0, 89),
        (nlist4y4, int(Np), int(Np), 0, 89, 90, 179),
    ],
)
def test_combine_list_ds(DSlist, lenX, lenY, x0, x1, y0, y1):
    nDSlist = combine_list_ds(DSlist)
    assert len(nDSlist.X) == lenX
    assert len(nDSlist.Y) == lenY
    assert [int(nDSlist.X[0].values), int(nDSlist.X[-1].values)] == [x0, x1]
    assert [int(nDSlist.Y[0].values), int(nDSlist.Y[-1].values)] == [y0, y1]


@pytest.mark.parametrize(
    "od, XRange, YRange",
    [
        (od, None, None),
    ],
)
def test_mask_var(od, XRange, YRange):
    ds = od._ds
    ds = mask_var(ds, XRange, YRange)
    assert _np.isnan(ds["nYG"].data).all()


_zeros = [0, 0]


@pytest.mark.parametrize(
    "od, XRange, YRange, A, B, C, D",
    [
        (od, A01_lon, A01_lat, [0, 0], [0, 0], [0, 0], [0, 0]),
        (od, P01_lon, P01_lat, [0, 0], [0, 0], [0, 0], [0, 0]),
        (od, P06_lon, P06_lat, [0, 0], [0, 0], [0, 0], [0, 0]),
        (od, [-31, -2], [58, 68.2], [0, 3], [0, 0], [0, 0], [0, 0]),
        (od, [160, -160], [58, 85.2], [0, 0], [0, 0], [52, 89], [0, 0]),
        (od, [160, 100], [58, 85.2], [0, 0], [0, 39], [51, 89], [0, 0]),
    ],
)
def test_arc_limits_mask(od, XRange, YRange, A, B, C, D):
    # test that cuts is a list of zeros for each side of the arctic cap
    # if there is no data surviving the cutout there. Also, test the
    # correct range of data when data in the arctic cap is retained.
    ds = od._ds
    XRange = _np.array(XRange)
    YRange = _np.array(YRange)
    XRange, ref_lon = _reset_range(XRange)
    add_Hbdr = 2
    maskH, dmaskH, XRange, YRange = get_maskH(
        ds, add_Hbdr, XRange, YRange, ref_lon=ref_lon
    )
    _faces = list(dmaskH["face"].values)
    ds = mask_var(ds, XRange, YRange, ref_lon=ref_lon)
    _var_ = "nYG"
    cuts = arc_limits_mask(ds, _var_, _faces, dims_g, XRange, YRange)

    assert cuts[0] == A
    assert cuts[1] == B
    assert cuts[2] == C
    assert cuts[3] == D


_nans = [[_np.nan, _np.nan], [_np.nan, _np.nan], [_np.nan, _np.nan], [_np.nan, _np.nan]]
_nans1 = _copy.deepcopy(_nans)
_nans2 = _copy.deepcopy(_nans)
_nans3 = _copy.deepcopy(_nans)
_nans4 = _copy.deepcopy(_nans)

_nans1[1] = [72, 89]
_nans2[2] = [48, 72]
_nans3[1] = [18, 42]
_nans4[2] = [0, 26]


@pytest.mark.parametrize(
    "od, XRange, YRange, axis, A, B, C, D",
    [
        (od, A01_lon, A01_lat, 0, _nans, _nans1, _nans2, _nans),
        (od, A01_lon, A01_lat, 1, _nans, _nans3, _nans4, _nans),
    ],
)
def test_edge_facet_data(od, XRange, YRange, axis, A, B, C, D):
    # identifies the ranges where there is data within each faces
    # facet is a list with a dataset as its elements.
    # test for each facet the expected behavior. When there is no
    # data it returns nans
    ds = od._ds
    XRange = _np.array(XRange)
    YRange = _np.array(YRange)
    XRange, ref_lon = _reset_range(XRange)
    add_Hbdr = 2
    maskH, dmaskH, XRange, YRange = get_maskH(
        ds, add_Hbdr, XRange, YRange, ref_lon=ref_lon
    )
    _faces = list(dmaskH["face"].values)
    ds = mask_var(ds, XRange, YRange, ref_lon=ref_lon)

    DSa2 = 0
    DSa5 = 0
    DSa7 = 0
    DSa10 = 0

    # define Facets as lists
    _facet1 = [k for k in range(7, 10)]
    _facet2 = [k for k in range(10, 13)]
    _facet3 = [k for k in range(3)]
    _facet4 = [k for k in range(3, 6)]

    Facet1 = []
    Facet2 = []
    Facet3 = []
    Facet4 = []

    for k in _np.arange(13):
        if k in _faces:
            if k in _facet1:
                Facet1.append(ds.isel(face=k))  #
            elif k in _facet2:
                Facet2.append(ds.isel(face=k))
            elif k in _facet3:
                Facet3.append(ds.isel(face=k))
            elif k in _facet4:
                Facet4.append(ds.isel(face=k))
        else:
            if k in _facet1:
                Facet1.append(0)
            elif k in _facet2:
                Facet2.append(0)
            elif k in _facet3:
                Facet3.append(0)
            elif k in _facet4:
                Facet4.append(0)

    Facet1 = [DSa7] + Facet1
    Facet2 = [DSa10] + Facet2
    Facet3.append(DSa2)
    Facet4.append(DSa5)

    edges1 = _edge_facet_data(Facet1, "nYG", dims_g, axis)
    edges2 = _edge_facet_data(Facet2, "nYG", dims_g, axis)
    edges3 = _edge_facet_data(Facet3, "nYG", dims_g, axis)
    edges4 = _edge_facet_data(Facet4, "nYG", dims_g, axis)

    assert edges1 == A
    assert edges2 == B
    assert edges3 == C
    assert edges4 == D


ds = od._ds
XRange = _np.array(A01_lon)
YRange = _np.array(A01_lat)
XRange, ref_lon = _reset_range(XRange)
add_Hbdr = 2
maskH, dmaskH, XRange, YRange = get_maskH(ds, add_Hbdr, XRange, YRange, ref_lon=ref_lon)
_faces = list(dmaskH["face"].values)
ds = mask_var(ds, XRange, YRange, ref_lon=ref_lon)

DSa2 = 0
DSa5 = 0
DSa7 = 0
DSa10 = 0

# define Facets as lists
_facet1 = [k for k in range(7, 10)]
_facet2 = [k for k in range(10, 13)]
_facet3 = [k for k in range(3)]
_facet4 = [k for k in range(3, 6)]

Facet1 = []
Facet2 = []
Facet3 = []
Facet4 = []

for k in _np.arange(13):
    if k in _faces:
        if k in _facet1:
            Facet1.append(ds.isel(face=k))  #
        elif k in _facet2:
            Facet2.append(ds.isel(face=k))
        elif k in _facet3:
            Facet3.append(ds.isel(face=k))
        elif k in _facet4:
            Facet4.append(ds.isel(face=k))
    else:
        if k in _facet1:
            Facet1.append(0)
        elif k in _facet2:
            Facet2.append(0)
        elif k in _facet3:
            Facet3.append(0)
        elif k in _facet4:
            Facet4.append(0)

Facet1 = [DSa7] + Facet1
Facet2 = [DSa10] + Facet2
Facet3.append(DSa2)
Facet4.append(DSa5)


@pytest.mark.parametrize(
    "Facet, axis, i, Nx, Ny",
    [
        (Facet1, 0, 0, None, None),
        (Facet2, 0, 1, 90, 18),
        (Facet3, 0, 2, 90, 25),
        (Facet4, 0, 0, None, None),
        (Facet1, 1, 0, None, None),
        (Facet2, 1, 1, 25, 90),
        (Facet3, 1, 2, 27, 90),
        (Facet4, 1, 0, None, None),
    ],
)
def test_slice_datasets(Facet, axis, i, Nx, Ny):
    """Test fn that slices the datasets within each facets."""
    edge = _edge_facet_data(Facet, "nYG", dims_g, axis)
    Facet = slice_datasets(Facet, dims_c, dims_g, edge, axis)

    if Nx is None:
        assert Facet == [0, 0, 0, 0]
    else:
        assert len(Facet[i].X) == Nx
        assert len(Facet[i].Y) == Ny


@pytest.mark.parametrize(
    "od, XRange, YRange, F_indx, Nx",
    [
        (od, [-31, 25], [58, 87], 2, 42),
        (od, [-31, 25], [58, 85.2], 2, 39),
        (od, [-31, 25], [58, 80], 2, 28),
        (od, [-31, 25], [58, 75], 2, 18),
        (od, [-31, 25], [58, 70], 2, 7),
        (od, [-120, -60], [58, 85.2], 10, 52),
        (od, [160, -150], [58, 85.2], 7, 52),
        (od, [60, 130], [58, 85.2], 5, 38),
    ],
)
def test_edge_arc_data(od, XRange, YRange, F_indx, Nx):
    """tests edge_arc_data, which find the northtern most
    point within the arctic cap that survives the cutout, taking
    into account that the Facet index with which the arctic
    exchanges data ("triangle of influence").
    F_indx = {2, 5, 7, 10} (Face number).
    """
    ds = od._ds
    XRange = _np.array(XRange)
    YRange = _np.array(YRange)
    XRange, ref_lon = _reset_range(XRange)
    add_Hbdr = 2
    _var_ = "nYG"
    maskH, dmaskH, XRange, YRange = get_maskH(
        ds, add_Hbdr, XRange, YRange, ref_lon=ref_lon
    )
    _faces = list(dmaskH["face"].values)
    ds = mask_var(ds, XRange, YRange, ref_lon=ref_lon)

    dsa2 = []
    dsa5 = []
    dsa7 = []
    dsa10 = []
    ARCT = [dsa2, dsa5, dsa7, dsa10]

    *nnn, DS = arct_connect(ds, _var_, faces=_faces, masking=True, opt=False)
    ARCT[0].append(DS[0])
    ARCT[1].append(DS[1])
    ARCT[2].append(DS[2])
    ARCT[3].append(DS[3])

    for i in range(len(ARCT)):  # Not all faces survive the cutout
        if type(ARCT[i][0]) == _datype:
            ARCT[i] = _xr.merge(ARCT[i])

    face_order = _np.array([2, 5, 7, 10])
    ll = _np.where(face_order == F_indx)[0][0]

    DSa = ARCT[ll]  # Extract the dataset with arctic data in a triangle.

    Xf = _edge_arc_data(DSa[_var_], F_indx, dims_g)

    assert Xf == Nx
