#ifndef DUNE_FEM_BASEFUNCTIONSETS_CODEGEN_HH
#define DUNE_FEM_BASEFUNCTIONSETS_CODEGEN_HH

#include <cassert>
#include <cstdlib>
#include <iostream>
#include <fstream>
#include <sstream>
#include <vector>
#include <set>
#include <map>

#include <dune/common/exceptions.hh>
#include <dune/fem/io/io.hh>
#include <dune/fem/io/parameter.hh>

#include <dune/fem/quadrature/cachingquadrature.hh>
#include <dune/fem/storage/singleton.hh>

namespace Dune
{

  namespace Fem
  {

    namespace Codegen
    {

    class CodegenInfoFinished : public Dune :: Exception {};

    class CodegenInfo
    {
      std::string path_;
      std::string outFileName_;

      int nopMax_;
      int nopMin_;

      int baseMax_;
      int baseMin_;

      typedef std::set< int > DimRangeSetType;
      typedef std::set< void* > BaseSetPointerType;
      DimRangeSetType savedimRanges_;
      mutable DimRangeSetType dimRanges_;
      BaseSetPointerType savedBaseSets_;

      typedef void codegenfunc_t (std::ostream& out,
                                  const int dim,
                                  const int dimRange,
                                  const size_t numRows,
                                  const size_t numCols );

      typedef std::pair< std::string, int > EntryType;
      std::vector< EntryType > filenames_;

      typedef std::vector< int > KeyType;
      typedef std::set< KeyType > KeyStorageType;

      typedef std::pair< int, std::pair< int, int > > EvalPairType ;
      typedef std::set< EvalPairType > EvalSetType ;

      std::map< codegenfunc_t* , KeyStorageType > entryMap_;
      EvalSetType  evalSet_;

    public:
      CodegenInfo()
        : path_( Dune::Fem::Parameter::commonInputPath() + "/autogeneratedcode"),
          outFileName_( "autogeneratedcode.hh" ),
          nopMax_(0), nopMin_(0), baseMax_(0), baseMin_(0),
          dimRanges_(), savedBaseSets_(), filenames_(),
          entryMap_(),
          evalSet_()
      {
      }

      static CodegenInfo& instance()
      {
        return Singleton< CodegenInfo > :: instance();
      }

      //! clear all registered entries
      void clear() {
        savedBaseSets_.clear();
        filenames_.clear();
        dimRanges_.clear();
        entryMap_.clear();
        evalSet_.clear();
        nopMax_ = 0;
        nopMin_ = 0;
        baseMax_ = 0;
        baseMin_ = 0;
      }

      //! overwrite path
      void setPath(const std::string& path ) { path_ = path ; }

      //! overwrite filename
      void setFileName(const std::string& filename ) { outFileName_ = filename ; }

      template <class BaseSet>
      void addDimRange(const BaseSet* baseSet,
                       const int dimRange)
      {
        void* ptr = (void *) baseSet;
        if( savedBaseSets_.find( ptr ) == savedBaseSets_.end() )
        {
          savedBaseSets_.insert( ptr );
#ifndef NDEBUG
          std::cout << "Add dimRange " << dimRange << std::endl;
#endif
          dimRanges_.insert( dimRange ) ;
        }
      }

      void addEntry(const std::string& fileprefix,
                    codegenfunc_t* codegenfunc,
                    const int dim, const int dimRange, const int quadNop, const int numBase )
      {
        KeyStorageType& keyMap = entryMap_[ codegenfunc ];

        typedef KeyStorageType :: iterator iterator;
        KeyType key( 4 );
        key[ 0 ] = dim;
        key[ 1 ] = dimRange;
        key[ 2 ] = quadNop;
        key[ 3 ] = numBase;

        // search for key, if already exists, then do nothing
        iterator it = keyMap.find( key );
        if( it != keyMap.end() ) return ;

        // make sure dimRange was registered
        assert( dimRanges_.find( dimRange ) != dimRanges_.end() );

        // create directory to store files
        createDirectory( path_ );

        std::stringstream filename;
        filename << fileprefix << dimRange << "_" << quadNop << "_" << numBase << ".hh";

        // second check, if file exists, do nothing
        int pos = exists( filename.str() );
        if( pos >= 0 ) return;

        std::string filenameToWrite( path_ + "/" + filename.str() );

        // write code to string stream and compare with existing file
        // if file does not exist, then write code to newly generated file
        std::stringstream code;
        // call code generation function
        codegenfunc( code, dim, dimRange, quadNop, numBase );

        bool writeCode = false ;

        {
          std::ifstream infile( filenameToWrite );
          if( infile )
          {
            std::stringstream checkstr;
            checkstr << infile.rdbuf();

            // if both string are identical we can stop here
            // and don't write the header thus avoiding recompilation
            if( checkstr.str().compare( code.str() ) != 0 )
            {
              // if strings differ then write code
              writeCode = true;
            }
            infile.close();
          }
          else
          {
            // if file does not exist, then write code
            writeCode = true;
          }
        }

        if( writeCode )
        {
          // if file does not exist, then write code to newly generated file
          std::ofstream file( filenameToWrite, std::ios::out );
          file << code.str();
          file.close();
#ifndef NDEBUG
          std::cout << "Generate code " << fileprefix << " for (" << dimRange << ","
                    << quadNop << "," << numBase << ")" << std::endl;
#endif
        }

        // insert evaluate pair in any case
        // otherwise the lists with include files is wrong
        EvalPairType evalPair ( dimRange, std::make_pair(quadNop, numBase) );
        evalSet_.insert( evalPair );

        if( baseMin_ == 0 ) baseMin_ = numBase;
        if( nopMin_  == 0 ) nopMin_  = quadNop;

        EntryType entry ( filename.str() , dimRange );
        filenames_.push_back( entry );
        nopMax_ = std::max( quadNop, nopMax_ );
        nopMin_ = std::min( quadNop, nopMin_ );
        baseMax_ = std::max( numBase, baseMax_ );
        baseMin_ = std::min( numBase, baseMin_ );
      }

      void finalize () const
      {
        if( checkAbort() )
        {
          dumpInfo();
          std::cerr << "All automated code generated, bye, bye !! " << std::endl;
          DUNE_THROW(CodegenInfoFinished,"All automated code generated, bye, bye !!");
        }
      }


      bool dumpInfo(const bool writeToCurrentDir = false ) const
      {
        if( writeToCurrentDir )
        {
          // write file with correct include to current directory
          // this is usually not needed if include paths are correct
          std::ofstream file( outFileName_.c_str() );
          file << "#include \"" <<path_<< "/" << outFileName_ << "\"";
        }

        {
          std::string filename( path_ + "/" + outFileName_ );
          std::stringstream file;

          file << "#ifdef CODEGEN_INCLUDEMAXNUMS" << std::endl;
          file << "#ifndef CODEGEN_INCLUDEMAXNUMS_INCLUDED" << std::endl;
          file << "#define CODEGEN_INCLUDEMAXNUMS_INCLUDED" << std::endl << std::endl;
          file << "#define MAX_NUMBER_OF_QUAD_POINTS " << nopMax_ << std::endl;
          file << "#define MIN_NUMBER_OF_QUAD_POINTS " << nopMin_ << std::endl;
          file << "#define MAX_NUMBER_OF_BASE_FCT    " << baseMax_ << std::endl;
          file << "#define MIN_NUMBER_OF_BASE_FCT    " << baseMin_ << std::endl << std::endl;
#if 0
          file << "/* include all headers with inner loop extern declarations */" << std::endl;
          file << "#define CODEGEN_COMPILE_INNERLOOPS 1" << std::endl;
          file << "namespace Dune {" << std::endl;
          file << "namespace Fem {" << std::endl;
          file << "namespace Codegen {" << std::endl;
          for( size_t i = 0; i < filenames_.size(); ++i )
          {
            file << "#include \""<< filenames_[ i ].first << "\"" << std::endl;
          }
          file << "}}} // end namespaces" << std::endl;
          file << "#undef CODEGEN_COMPILE_INNERLOOPS" << std::endl << std::endl;
          file << "#include \"" << filename << ".c\"" <<std::endl;
#endif

          file << "#endif // CODEGEN_INCLUDEMAXNUMS_INCLUDED" << std::endl << std::endl;
          file << "#elif defined CODEGEN_INCLUDEEVALCALLERS" << std::endl;
          file << "#ifndef CODEGEN_EVALCALLERS_INCLUDED" << std::endl;
          file << "#define CODEGEN_EVALCALLERS_INCLUDED" << std::endl << std::endl;
          file << "namespace Dune {"<< std::endl;
          file << "namespace Fem {"<< std::endl;
          file << "namespace Codegen {"<< std::endl;
          typedef EvalSetType :: iterator iterator ;
          const iterator endit = evalSet_.end();
          for( iterator it = evalSet_.begin(); it != endit; ++it )
          {
            int dimRange = it->first;
            int quadNop  = it->second.first;
            int numBase  = it->second.second;
            file << "  template <class Traits>" << std::endl;
            file << "  struct EvaluateImplementation< Traits, " << dimRange << " , " << quadNop << " , " << numBase << " >" << std::endl;
            file << "    : public EvaluateRealImplementation< Traits, " << dimRange << " , " << quadNop << " , " << numBase << " >" << std::endl;
            file << "  {" << std::endl;
            file << "    typedef EvaluateRealImplementation< Traits, " << dimRange << " , " << quadNop << " , " << numBase << " >  BaseType;" << std::endl;
            file << "    typedef typename BaseType :: RangeVectorType  RangeVectorType;" << std::endl;
            file << "    EvaluateImplementation( const RangeVectorType& rv ) : BaseType ( rv ) {}" << std::endl;
            file << "  };"<< std::endl;
            file << std::endl;
          }
          file << "}}} // end namespaces"<< std::endl;
          file << "#endif // CODEGEN_EVALCALLERS_INCLUDED" << std::endl << std::endl;
          file << "#else" << std::endl << std::endl ;
          file << "#ifndef CODEGEN_INCLUDE_IMPLEMENTATION" << std::endl;
          file << "#define CODEGEN_INCLUDE_IMPLEMENTATION" << std::endl;
          //file << "#undef CODEGEN_COMPILE_INNERLOOPS" << std::endl;
          file << "namespace Dune {" << std::endl;
          file << "namespace Fem {" << std::endl;
          file << "namespace Codegen {" << std::endl;
          for( size_t i = 0; i < filenames_.size(); ++i )
          {
            file << "#include \""<< filenames_[ i ].first << "\"" << std::endl;
          }
          file << "}}} // end namespaces" << std::endl;
          file << "#endif  // CODEGEN_INCLUDE_IMPLEMENTATION" << std::endl << std::endl;
          file << "#endif // CODEGEN_INCLUDEMAXNUMS" << std::endl;

          std::ifstream infile( filename );
          if( infile )
          {
            std::stringstream checkstr;
            checkstr << infile.rdbuf();

            // if both string are identical we can stop here
            // and don't write the header thus avoiding recompilation
            if( checkstr.str().compare( file.str() ) == 0 )
              return false;
          }

          std::ofstream outfile( filename );
          outfile << file.str();
          outfile.close();

#if 0
          // write C file with implementation of inner loop functions
          filename += ".c";
          std::ofstream Cfile( filename.c_str() );

          Cfile << "#include <stdlib.h>" << std::endl;
          Cfile << "/* include all headers with inner loop implementation */" << std::endl;
          Cfile << "#define CODEGEN_COMPILE_INNERLOOPS 2" << std::endl;
          for( size_t i = 0; i < filenames_.size(); ++i )
          {
            Cfile << "#include \""<< filenames_[ i ].first << "\"" << std::endl;
          }
#endif
          return true;
        }
      }
    protected:
      int exists( const std::string& filename ) const
      {
        for( size_t i = 0; i < filenames_.size(); ++i )
        {
          if( filename == filenames_[ i ].first )
          {
            return i;
          }
        }
        return -1;
      }

      bool checkAbort() const
      {
        DimRangeSetType found ;
        bool canAbort = true ;
        for( size_t i = 0; i < filenames_.size(); ++i )
        {
          found.insert( filenames_[ i ].second );
          if ( filenames_[ i ].second > 0 )
          {
            canAbort = false ;
          }
        }
        typedef DimRangeSetType :: iterator iterator ;
        for( iterator it = found.begin(); it != found.end(); ++it )
        {
          dimRanges_.erase( *it );
        }

        if( canAbort && dimRanges_.size() == 0 )
          return true ;
        else
          return false ;
      }
    };

    /** \brief default code generator methods */
    template < int simdWidth = 4 >
    struct DefaultCodeGenerator
    {
      static const char* restrictKey()
      {
        return "__restrict__";
      }

      static const char* doubletype()
      {
#ifdef COUNT_FLOPS
        return "Dune::Fem::Double";
#else
        return "double";
#endif
      }

      static void writePreCompHeader(std::ostream& out, const int stage )
      {
        const char* codegenPreCompVar = "CODEGEN_COMPILE_INNERLOOPS";
        if( stage == -1 )
        {
          out << "#ifndef HEADERCHECK" << std::endl;
          //out << "#if ! " << codegenPreCompVar << std::endl;
        }
        else if( stage == 0 )
        {
          out << "#else" << std::endl;
          out << "#if " << codegenPreCompVar << " == 1" << std::endl;
          out << "extern \"C\" {" << std::endl
              << "  extern inline" << std::endl;
          out << "#endif" << std::endl;
        }
        else if( stage == 1 )
        {
          out << "#if  " << codegenPreCompVar << " == 1" << std::endl;
          out << "  ;" << std::endl;
          out << "}" << std::endl;
          out << "#else" << std::endl;
        }
        else
        {
          //out << "#endif" << std::endl;
          //out << "#endif" << std::endl;
          out << "#endif" << std::endl;
        }
      }

      // generate inner loop function name
      static std::string generateFunctionName( const std::string& prefix,
                                        const int simdW, const int dimRange,
                                        const size_t numRows, const size_t numCols )
      {
        std::stringstream funcName;
        funcName << prefix << "_" << simdWidth << "_" << dimRange << "_" << numRows << "_" << numCols ;
        return funcName.str();
      }

      static void writeInnerLoopEval(std::ostream& out, const int simdW, const int dimRange, const size_t numRows, const size_t numCols )
      {
        out << "      // Loop over all quadrature points" << std::endl;
        out << "      for(int qp = 0; qp < " << numRows << " ; ++qp )" << std::endl;
        out << "      {" << std::endl;
        /*
        if( simdW == 1 )
        {
            out << "        const " << doubletype() << " phi0 = rangeStorage[ quad.cachingPoint( row ) * " << numCols << " + col ][ 0 ];" << std::endl;
        }
        else
        {
          for( int i = 0 ; i< simdW ; ++ i )
            out << "        const " << doubletype() << " phi" << i << " = base" << i << "[ row ];" << std::endl;
        }
        */
        for( int i = 0 ; i< simdW ; ++ i )
          out << "        const " << doubletype() << " phi" << i << " = base" << i << "[ qp ];" << std::endl;
        for(int r = 0; r < dimRange; ++ r )
        {
          out << "        result" << r << "[ qp ] += phi0 * dof0" << r;
          for( int i=1; i<simdW; ++i )
            out << " + phi" << i << " * dof" << i << r;
          out << " ;" << std::endl;
        }
        out << "      }" << std::endl;
      }

      static void evaluateCodegen(std::ostream& out,
                                  const int dim,
                                  const int dimRange,
                                  const size_t numRows,
                                  const size_t numCols )
      {
        const std::string funcName =
              generateFunctionName( "evalRangesLoop", simdWidth, dimRange, numRows, numCols );

        writePreCompHeader( out, -1 );

        out << "template <class BaseFunctionSet> // dimRange = "<< dimRange << ", quadNop = " << numRows << ", scalarBasis = " << numCols << std::endl;
        out << "struct EvaluateRanges<BaseFunctionSet, EmptyGeometry, " << dimRange << ", " << numRows << ", " << numCols << ">" << std::endl;
        out << "{" << std::endl;
        out << "  template< class QuadratureType,"<< std::endl;
        out << "            class RangeVectorType," << std::endl;
        out << "            class RangeFactorType," << std::endl;
        out << "            class LocalDofVectorType>" << std::endl;
        out << "  static void eval( const QuadratureType& quad," << std::endl;
        out << "                    const RangeVectorType& rangeStorageTransposed," << std::endl;
        out << "                    const LocalDofVectorType& dofs," << std::endl;
        out << "                    RangeFactorType &rangeVector)" << std::endl;
        out << "  {" << std::endl;
        //out << "    typedef typename ScalarRangeType :: field_type field_type;" << std::endl;
        //out << "    typedef Field field_type;" << std::endl;
        //out << "    typedef typename RangeVectorType :: value_type value_type;" << std::endl;
        //out << "    typedef RangeType value_type;" << std::endl;

        out << "    typedef " << doubletype() << " Field;" << std::endl;
        //out << "    static Dune::Fem::ThreadSafeValue< std::vector< Field > > memory( " << numRows * dimRange << " );" << std::endl;
        out << "    static thread_local std::vector< Field > memory( " << numRows * dimRange << " );" << std::endl;

        // make length simd conform
        //out << "    Field* resultTmp = (*memory).data();" << std::endl;
        out << "    Field* resultTmp = memory.data();" << std::endl;
        out << "    for( int i=0; i < " << numRows * dimRange << "; ++ i ) resultTmp[ i ] = 0;" << std::endl <<std::endl;

        for(int r=0; r<dimRange ; ++r )
        {
          out << "    Field* result" << r << " = resultTmp + " << r * numRows << " ;" << std::endl;
        }
        out << std::endl;

        out << "    const Field* baseData = rangeStorageTransposed.data();" << std::endl;
        //for( int i=0; i< simdWidth; ++ i )
        //{
        //  out << "    Field base" << i << "[ " << numRows << " ];" << std::endl;
        //}
        out << std::endl;
        const size_t simdCols = simdWidth * ( numCols / simdWidth );
        if( simdCols > 0 )
        {
          out << "    for( int col = 0, dof = 0 ; col < "<< simdCols <<" ; col += " << simdWidth << ", dof += " << simdWidth * dimRange<< " )"<<std::endl;
          out << "    {" << std::endl;

          //out << "        const value_type& rangeStorageRow = rangeStorage[ rowMap[ row ] ];" << std::endl;
          //out << "      for( int row = 0 ; row < " << numRows << " ; ++ row )" << std::endl;
          //out << "      {" << std::endl;

          //out << "        int idx = quad.cachingPoint( row ) * " << numCols << " + col;" << std::endl;
          //for( int i=0; i< simdWidth; ++ i )
          //{
          //  const char* plusplus = (i == simdWidth-1) ? "  " : "++";
          //  out << "        base" << i << "[ row ] = rangeStorage[ idx" << plusplus << " ][ 0 ];" << std::endl;
          //}
          //out << "      }" << std::endl;


          out << "      " << funcName << "(" << std::endl;
          out << "                ";
          for( int i = 0; i< simdWidth * dimRange; ++i )
            out << " dofs[ dof + " << i << " ],";
          out << std::endl;
          for( int i=0; i< simdWidth; ++i )
            out << "                 baseData + ((col + " << i << ") * " << numRows << ")," << std::endl;
          out << "                 ";
          for( int r = 0; r < dimRange-1; ++r)
            out << "result" << r << ", ";
          out << "result" << dimRange-1 << " );" << std::endl;
          out << "    }"<< std::endl;
          out << std::endl;
        }

        if( numCols > simdCols )
        {
          out << "    // remainder iteration" << std::endl;
          out << "    for( int col = " << simdCols << ", dof = " << simdCols * dimRange << " ; col < " << numCols << " ; ++col )" << std::endl;
          out << "    {" << std::endl;
          for( int r=0; r<dimRange; ++r )
            out << "      const " << doubletype() << " dof0" << r << " = dofs[ dof++ ];" << std::endl;
          out << "      const Field* base0 = baseData + (col * " << numRows << ");" << std::endl;

          writeInnerLoopEval( out, 1, dimRange, numRows, numCols );
          out << "    }" << std::endl;
          out << std::endl;
        }

        out << "    // store result" << std::endl;
        out << "    for(int row = 0; row < " << numRows << " ; ++row )" << std::endl;
        out << "    {" << std::endl;
        out << "      auto& result = rangeVector[ row ];" << std::endl;
        for( int r = 0 ; r < dimRange; ++ r )
        {
          out << "      result[ " << r << " ] = result" << r << "[ row ];" << std::endl;
        }
        out << "    }" << std::endl;
        out << "  }" << std::endl << std::endl;
        //out << "};" << std::endl;

        //writePreCompHeader( out, 0 );
        out << "  static void " << funcName << "(" << std::endl;
        for( int i=0; i<simdWidth; ++i )
        {
          out << "        ";
          for( int r=0; r<dimRange; ++ r )
          {
            if( r > 0 ) out << " ";
            out << "const " << doubletype() << " dof"<< i << r << ",";
          }
          out << std::endl;
        }
        for( int i=0; i<simdWidth; ++ i )
          out << "        const " << doubletype() << "* " << restrictKey() << " base" << i << "," << std::endl;
        for( int r=0; r<dimRange; ++ r )
        {
          out << "        " << doubletype() << "* "<< restrictKey() << " result" << r;
          if( r == dimRange-1 ) out << " )" << std::endl;
          else out << "," << std::endl;
        }

        //writePreCompHeader( out, 1 );
        out << "  {" << std::endl;
        writeInnerLoopEval( out, simdWidth, dimRange, numRows, numCols );
        out << "  }" << std::endl;
        out << "};" << std::endl;
        writePreCompHeader( out, 2 );
      }

      static void writeInnerLoop(std::ostream& out, const int simdW, const int dimRange, const size_t numCols )
      {
        for( int i=0; i< simdW; ++i )
        {
          for( int r=0; r< dimRange; ++r )
          {
            out << "      const " << doubletype() << " fac" << i << r << " = rangeFactor" << i << "[ " << r << " ];" << std::endl;
          }
        }
        if( simdW == 1 )
          out << "      int rowCol = quad.cachingPoint( row ) * " << numCols <<";" << std::endl;
        out << "      for(int col = 0 ; col < " << numCols << " ; ++ col";
        if( simdW == 1 )
          out << ", ++rowCol";
        out << " )" << std::endl;
        out << "      {" << std::endl;
        for( int i = 0 ; i< simdW ; ++ i )
        {
          if( simdW == 1 )
            out << "        const " << doubletype() << " phi" << i << " = rangeStorage[ rowCol ][ 0 ];" << std::endl;
          else
            out << "        const " << doubletype() << " phi" << i << " = base" << i  << " [ col ];" << std::endl;
        }
        for(int r = 0; r < dimRange; ++ r )
        {
          out << "        dofs" << r << "[ col ] += phi0 * fac0" << r;
          for( int i=1; i<simdW; ++i )
          {
            out << " + phi" << i << " * fac" << i << r ;
          }
          out << " ;" << std::endl;
        }
        out << "      }" << std::endl;
      }

      static void axpyCodegen(std::ostream& out,
              const int dim, const int dimRange, const size_t numRows, const size_t numCols )
      {
        const std::string funcName =
          generateFunctionName( "axpyRangesLoop", simdWidth, dimRange, numRows, numCols );

        writePreCompHeader( out, -1 );

        out << "template <class BaseFunctionSet> // dimRange = "<< dimRange << ", quadNop = " << numRows << ", scalarBasis = " << numCols << std::endl;
        out << "struct AxpyRanges<BaseFunctionSet, EmptyGeometry, " << dimRange << ", " << numRows << ", " << numCols << ">" << std::endl;
        out << "{" << std::endl;

        out << std::endl;
        out << "  template< class QuadratureType,"<< std::endl;
        out << "            class RangeVectorType," << std::endl;
        out << "            class RangeFactorType," << std::endl;
        out << "            class LocalDofVectorType>" << std::endl;
        out << "  static void axpy( const QuadratureType& quad," << std::endl;
        out << "                    const RangeVectorType& rangeStorage," << std::endl;
        out << "                    const RangeFactorType& rangeFactors," << std::endl;
        out << "                    LocalDofVectorType& dofs)" << std::endl;
        out << "  {" << std::endl;

        ////////////////////////////////////////////////////
        // axpy
        ////////////////////////////////////////////////////

        //out << "    typedef RangeType value_type;" << std::endl;
        //out << "    typedef typename ScalarRangeType :: field_type field_type;" << std::endl;

        out << "    typedef " << doubletype() << " Field;" << std::endl;
        out << "    static thread_local std::vector< Field > memory( " << numCols * dimRange << " );" << std::endl;
        //out << "    static Dune::Fem::ThreadSafeValue< std::vector< Field > > memory( " << numCols * dimRange << " );" << std::endl;

        //out << "    " << doubletype() << "* dofResult = (*memory).data();" << std::endl;
        out << "    " << doubletype() << "* dofResult = memory.data();" << std::endl;
        out << "    for( int i=0; i < " << numCols * dimRange << "; ++i ) dofResult[ i ] = 0;" << std::endl << std::endl;

        out << "    " << doubletype() << "* dofs0 = dofResult;" << std::endl;
        for( int r = 1; r < dimRange; ++ r )
          out << "    " << doubletype() << "* dofs" << r << " = dofResult + " << r * numCols << ";" << std::endl;
        out << std::endl;

        const size_t simdRows  = simdWidth * (numRows / simdWidth) ;

        if( simdRows > 0 )
        {
          out << "    for( int row = 0; row < "<< simdRows << " ; row += " << int(simdWidth) << " )" << std::endl;
          out << "    {" << std::endl;
          for( int i=0; i<simdWidth; ++ i )
            out << "      const " << doubletype() << "* rangeFactor" << i << " = &rangeFactors[ row + " << i << " ][ 0 ];" << std::endl;
          out << "      " << funcName << "(";
          for( int i = 0; i < simdWidth; ++i )
          {
            if( i>0 )
              out << "                              ";
            out << " &rangeStorage[ quad.cachingPoint( row + " << i << " ) * " << numCols << " ][ 0 ]," << std::endl;
          }
          out << "                               rangeFactor0, ";
          for( int i=1; i<simdWidth; ++ i )
            out << "rangeFactor" << i << ",";
          out << std::endl;
          out << "                              ";
          for( int r = 0; r < dimRange; ++ r )
          {
            out << " dofs" << r ;
            if( r == dimRange-1 )
              out << " );" << std::endl;
            else
              out << ",";
          }
          out << std::endl;
          out << "    }" << std::endl;
          out << std::endl;
        }

        if( numRows > simdRows )
        {
          out << "    // remainder iteration" << std::endl;
          out << "    for( int row = " << simdRows << " ; row < " << numRows << " ; ++row )" << std::endl;
          out << "    {" << std::endl;
          out << "      const " << doubletype() << "* rangeFactor0 = &rangeFactors[ row ][ 0 ];" << std::endl;
          writeInnerLoop( out, 1, dimRange, numCols );
          out << "    }" << std::endl;
          out << std::endl;
        }

        out << "    // sum up results (transform from variable based to point based layout)" << std::endl;
        out << "    for( int col = 0, dof = 0 ; col < "<< numCols << " ; ++col )" << std::endl;
        out << "    {" << std::endl;
        for( int r = 0 ; r < dimRange; ++ r )
          out << "      dofs[ dof++ ] += dofs" << r << "[ col ];" << std::endl;
        out << "    }" << std::endl;

        out << "  }" << std::endl << std::endl;
        //out << "};" << std::endl;

        ///////////////////////////////////
        //  inner loop
        ///////////////////////////////////
        //writePreCompHeader( out, 0 );
        out << "  static void " << funcName << "(" << std::endl;
        out << "       const " << doubletype() << "* " << restrictKey() << " base0," << std::endl;
        for( int i=1; i<simdWidth; ++ i )
          out << "       const " << doubletype() << "* " << restrictKey() << " base" << i << "," << std::endl;
        for( int i=0; i<simdWidth; ++ i )
          out << "       const " << doubletype() << "* " << restrictKey() << " rangeFactor" << i << "," << std::endl;
        for( int r = 0; r < dimRange; ++r )
        {
          out << "       " << doubletype() << "* " << restrictKey() << " dofs" << r;
          if( r == dimRange-1 )
            out << " )" << std::endl;
          else
            out << "," << std::endl;
        }
        //writePreCompHeader( out, 1 );
        out << "  {" << std::endl;
        writeInnerLoop( out, simdWidth, dimRange, numCols );
        out << "  }" << std::endl;
        out << "};" << std::endl;
        writePreCompHeader( out, 2 );
      }

      static void writeInnerJacEvalLoop(std::ostream& out, const int simdW, const int dim,
                                        const int dimRange, const size_t numRows, const size_t numCols  )
      {
        out << "      for(int row = 0; row < " << numRows << " ; ++row )" << std::endl;
        out << "      {" << std::endl;
        if( simdW == 1 )
        {
          /*
          out << "        int idx = quad.cachingPoint( row ) * " << numCols << " + col ;" << std::endl;
          out << "        const GeometryJacobianType& gjit = geometry.jacobianInverseTransposed( quad.point( row ) );" << std::endl << std::endl;
          for( int i = 0 ; i< simdW ; ++ i )
          {
            const char* plusplus = (i == simdW-1) ? "  " : "++";
            out << "        gjit.mv( jacStorage[ idx" << plusplus << " ][ 0 ], gradPhi" << i << " );" << std::endl;
            for( int d = 0 ; d < dim; ++ d )
              out << "        const " << doubletype() << " phi" << i << d << " = gradPhi" << i << "[ " << d << " ];" << std::endl;
          }
          */
          /*
          for( int i = 0 ; i< simdW ; ++ i )
          {
            const char* plusplus = (i == simdW-1) ? "  " : "++";
            out << "      gradPhi" << i << " = jacStorage[ idx" << plusplus << " ][ 0 ];" << std::endl;
            //out << "        gjit.mv( jacStorage[ idx" << plusplus << " ][ 0 ], gradPhi" << i << " );" << std::endl;
            for( int d = 0 ; d < dim; ++ d )
              out << "        const " << doubletype() << " phi" << i << d << " = gradPhi" << i << "[ " << d << " ];" << std::endl;
          }
          */
          //  out << "        const "<< doubletype() << "* base" << i << " = jacobians + (" << dim * numRows << " * (col + "<< i << "));" << std::endl;
          for( int i = 0 ; i< simdW ; ++ i )
          {
            //out << "        gjit.mv( jacStorage[ idx" << plusplus << " ][ 0 ], gradPhi" << i << " );" << std::endl;
            for( int d = 0 ; d < dim; ++ d )
              out << "        const " << doubletype() << " phi" << i << d << " = base" << i << "[ row * " << dim << " + " << d << " ];" << std::endl;
          }
          out << std::endl;
        }
        else
        {
          out << "        const int idx = row * " << dim<< ";" << std::endl;
          for( int d = 0; d < dim ; ++ d )
          {
            for( int i = 0 ; i< simdW ; ++ i )
              out << "        const " << doubletype() << " phi" << i << d << " = base" << i << "[ idx + " << d << " ];" << std::endl;
          }
        }
        for( int d = 0; d < dim ; ++ d )
        {
          for(int r = 0; r < dimRange; ++ r )
          {
            out << "        result" << r << d << "[ row ] += phi0" << d << " * dof0" << r;
            for( int i=1; i<simdW; ++i )
              out << " + phi" << i << d << " * dof" << i << r;
            out << " ;" << std::endl;
          }
        }
        out << "      }" << std::endl;
      }

      static void evaluateJacobiansCodegen(std::ostream& out,
              const int dim, const int dimRange, const size_t numRows, const size_t numCols )
      {
        const std::string funcName =
              generateFunctionName( "evalJacobianLoop", simdWidth, dimRange, numRows, numCols );

        writePreCompHeader( out, -1 );

        out << "template <class BaseFunctionSet> // dimRange = "<< dimRange << ", quadNop = " << numRows << ", scalarBasis = " << numCols << std::endl;
        out << "struct EvaluateJacobians<BaseFunctionSet, EmptyGeometry, " << dimRange << ", " << numRows << ", " << numCols << ">" << std::endl;
        out << "{" << std::endl;
        out << "  template< class QuadratureType,"<< std::endl;
        out << "            class JacobianRangeVectorType," << std::endl;
        out << "            class LocalDofVectorType," << std::endl;
        out << "            class JacobianRangeFactorType>" << std::endl;
        out << "  static void eval( const QuadratureType&," << std::endl;
        out << "                    const EmptyGeometry&," << std::endl;
        out << "                    const JacobianRangeVectorType&," << std::endl;
        out << "                    const LocalDofVectorType&," << std::endl;
        out << "                    JacobianRangeFactorType &)" << std::endl;
        out << "  {" << std::endl;
        out << "    std::cerr << \"ERROR: wrong code generated for VectorialBaseFunctionSet::axpyJacobians\" << std::endl;" << std::endl;
        out << "    abort();" << std::endl;
        out << "  }" << std::endl;
        out << "};" << std::endl << std::endl;
        out << "template <class BaseFunctionSet, class Geometry> // dimRange = "<< dimRange << ", quadNop = " << numRows << ", scalarBasis = " << numCols << std::endl;
        out << "struct EvaluateJacobians<BaseFunctionSet, Geometry, " << dimRange << ", " << numRows << ", " << numCols << ">" << std::endl;
        out << "{" << std::endl;
        out << "  template< class QuadratureType,"<< std::endl;
        out << "            class JacobianRangeVectorType," << std::endl;
        out << "            class LocalDofVectorType," << std::endl;
        out << "            class JacobianRangeFactorType>" << std::endl;
        out << "  static void eval( const QuadratureType& quad," << std::endl;
        out << "                    const Geometry& geometry," << std::endl;
        out << "                    const JacobianRangeVectorType& jacStorage," << std::endl;
        out << "                    const LocalDofVectorType& dofs," << std::endl;
        out << "                    JacobianRangeFactorType& jacFactors)" << std::endl;
        out << "  {" << std::endl;
        out << "    evalJac( quad, geometry, jacStorage, dofs, jacFactors, jacFactors[ 0 ] );" << std::endl;
        out << "  }" << std::endl;
        out << "private:" << std::endl;
        out << "  template< class QuadratureType,"<< std::endl;
        out << "            class JacobianRangeVectorType," << std::endl;
        out << "            class LocalDofVectorType," << std::endl;
        out << "            class JacobianRangeFactorType," << std::endl;
        out << "            class GlobalJacobianRangeType>" << std::endl;
        out << "  static void evalJac( const QuadratureType& quad," << std::endl;
        out << "                       const Geometry& geometry," << std::endl;
        out << "                       const JacobianRangeVectorType& jacStorage," << std::endl;
        out << "                       const LocalDofVectorType& dofs," << std::endl;
        out << "                       JacobianRangeFactorType& jacVector," << std::endl;
        out << "                       const GlobalJacobianRangeType&)" << std::endl;
        out << "  {" << std::endl;
        //out << "    typedef typename JacobianRangeVectorType :: value_type  value_type;" << std::endl;
        //out << "    typedef JacobianRangeType value_type;" << std::endl;
        //out << "    typedef typename JacobianRangeType :: field_type field_type;" << std::endl;
        out << "    typedef typename Geometry::JacobianInverseTransposed GeometryJacobianType;" << std::endl;

        const size_t nDofs = numRows * dimRange * dim ;
        out << "    typedef " << doubletype() << " Field;" << std::endl;
        out << "    static thread_local std::vector< Field > memory( " << nDofs << " );" << std::endl;
        //out << "    static Dune::Fem::ThreadSafeValue< std::vector< Field > > memory( " << nDofs << " );" << std::endl;

        for( int d = 0; d < dim ; ++ d )
        {
          // make length simd conform
          //out << "    Field* resultTmp" << d << " = (*memory).data() + " << d * numRows * dimRange << ";" << std::endl;
          out << "    Field* resultTmp" << d << " = memory.data() + " << d * numRows * dimRange << ";" << std::endl;
        }
        out << "    for( int i=0; i<" << numRows * dimRange << "; ++i ) " << std::endl;
        out << "    {" << std::endl;
        for( int d = 0; d < dim ; ++ d )
          out << "      resultTmp" << d << "[ i ] = 0;" << std::endl;
        out << "    }" << std::endl << std::endl;

        for( int d = 0; d < dim ; ++ d )
        {
          for(int r=0; r<dimRange ; ++r )
          {
            out << "    Field* result" << r << d <<" = resultTmp" << d << " + " << r * numRows << " ;" << std::endl;
          }
        }
        out << std::endl;

        const size_t simdNumCols = simdWidth * ( numCols / simdWidth );
        out << "    typedef typename GlobalJacobianRangeType :: row_type JacobianRangeType;" << std::endl;
        out << "    const " << doubletype() << "* jacobians = jacStorage.data();" << std::endl;
        if( simdNumCols > 0 )
        {
          /*
          for( int d = 0; d < dim; ++d )
          {
            for( int i=0; i< simdWidth; ++ i )
            {
              out << "    Field* base" << i << d << " = memory.data() + " << nDofs + ((d * simdWidth) + i) * numRows << ";" << std::endl;
            }
          }
          */

          //for( int i=0; i< simdWidth; ++ i )
          //out << "    JacobianRangeType gradPhi" << i << ";" << std::endl;

          out << "    for( int col = 0, dof = 0 ; col < "<< simdNumCols <<" ; col += " << simdWidth << ", dof += " << simdWidth * dimRange<< " )"<<std::endl;
          out << "    {" << std::endl;
          /*
          out << "      for( int row = 0; row < " << numRows << " ; ++ row )" << std::endl;
          out << "      {" << std::endl;
          out << "        int idx = quad.cachingPoint( row ) * " << numCols << " + col ;" << std::endl;
          out << "        // use reference to GeometryJacobianType to make code compile with SPGrid Geometry" << std::endl;
          */
          //out << "        const GeometryJacobianType& gjit = geometry.jacobianInverseTransposed( quad.point( row ) );" << std::endl << std::endl;
          /*
          for( int i=0; i< simdWidth; ++ i )
          {
            const char* plusplus = (i == simdWidth-1) ? "  " : "++";
            out << "        gjit.mv( jacStorage[ idx" << plusplus << " ][ 0 ], gradPhi" << i << " );" << std::endl;
          }
          */
          /*
          for( int i=0; i< simdWidth; ++ i )
          {
            const char* plusplus = (i == simdWidth-1) ? "  " : "++";
            out << "        gradPhi" << i << " = jacStorage[ idx" << plusplus << " ][ 0 ];" << std::endl;
          }
          out << std::endl;
          */
          /*
          for( int d = 0; d < dim; ++ d )
          {
            for( int i=0; i< simdWidth; ++ i )
            {
              out << "        base" << i << d << "[ row ] = gradPhi"<< i << "[ " << d << " ];" << std::endl;
            }
          }
          out << "      }" << std::endl << std::endl;
          */
          for( int i=0; i< simdWidth; ++ i )
          {
            out << "      const "<< doubletype() << "* base" << i << " = jacobians + (" << dim * numRows << " * (col + "<< i << "));" << std::endl;
          }

          out << "      " << funcName << "(";
          for( int i = 0; i< simdWidth * dimRange; ++i )
            out << " dofs[ dof + " << i << " ],";
          out << std::endl << "                 ";
          for( int i=0; i< simdWidth; ++i )
            out << "base" << i << ", ";
          out << std::endl << "                 ";
          for( int d = 0; d < dim; ++ d )
          {
            for( int r = 0; r < dimRange; ++r)
            {
              out << "result" << r << d;
              if( (r == dimRange - 1) && (d == dim-1 ) ) out << std::endl;
              else out << ", ";
            }
          }
          out << "      );" << std::endl;
          out << "    }"<< std::endl;
          out << std::endl;
        }

        // remainder iteration
        if( numCols > simdNumCols )
        {
          out << "    // remainder iteration" << std::endl;
          out << "    for( int col = " << simdNumCols << ", dof = " << simdNumCols * dimRange << " ; col < " << numCols << " ; ++col )" << std::endl;
          out << "    {" << std::endl;
          out << "      const "<< doubletype() << "* base0" << " = jacobians + (" << dim * numRows << " * col);" << std::endl;
          for( int r=0; r<dimRange; ++r )
            out << "      const " << doubletype() << " dof0" << r << " = dofs[ dof++ ];" << std::endl;
          writeInnerJacEvalLoop( out, 1, dim, dimRange, numRows, numCols );
          out << "    }" << std::endl;
          out << std::endl;
        }

        out << "    // store result" << std::endl;
        out << "    JacobianRangeType tmp;" << std::endl;
        out << "    for(int row = 0; row < " << numRows << " ; ++row )" << std::endl;
        out << "    {" << std::endl;
        out << "      const GeometryJacobianType& gjit = geometry.jacobianInverseTransposed( quad.point( row ) );" << std::endl << std::endl;
        out << "      GlobalJacobianRangeType& result = jacVector[ row ];" << std::endl;
        for( int r = 0 ; r < dimRange; ++ r )
        {
          for( int d = 0 ; d < dim; ++ d )
          {
            out << "      tmp[ " << d << " ] = result" << r << d << "[ row ];" << std::endl;
            /*
            {
              out << "      result[ " << r << " ][ " << d <<" ] = result" << r << d << "[ row ];" << std::endl;
            }
            */
          }
          out << "      gjit.mv( tmp, result[ "<< r << " ] );" << std::endl;
        }
        out << "    }" << std::endl;
        out << "  }" << std::endl << std::endl;
        //out << "};" << std::endl;

        //writePreCompHeader( out, 0 );
        out << "  static void " << funcName << "(" << std::endl;
        for( int i=0; i<simdWidth; ++i )
        {
          out << "                        ";
          for( int r=0; r<dimRange; ++ r )
            out << " const " << doubletype() << " dof"<< i << r << ",";
          out << std::endl;
        }
        for( int i=0; i<simdWidth; ++ i )
          out << "                         const " << doubletype() << "* " << restrictKey() << " base" << i << "," << std::endl;
        for( int d=0; d<dim; ++ d )
        {
          for( int r=0; r<dimRange; ++ r )
          {
            out << "                         " << doubletype() << "* "<< restrictKey() << " result" << r << d;
            if( (r == dimRange - 1) && (d == dim-1 ) ) out << " )" << std::endl;
            else out << "," << std::endl;
          }
        }

        //writePreCompHeader( out, 1 );
        out << "  {" << std::endl;
        writeInnerJacEvalLoop( out, simdWidth, dim, dimRange, numRows, numCols );
        out << "  }" << std::endl;
        out << "};" << std::endl;
        writePreCompHeader( out, 2 );
      }

      static void writeInnerLoopAxpyJac(std::ostream& out, const int dim, const int dimRange, const size_t numCols )
      {
        out << "      for( int col = 0; col < " << numCols << " ; ++col )" << std::endl;
        out << "      {" << std::endl;
        for( int d =0; d < dim; ++d )
          out << "        const " << doubletype() << " phi" << d << " = base[ (col * " << dim << ") + " << d << " ];" << std::endl;

        for( int r = 0; r < dimRange; ++r )
        {
          out << "        result" << r << "[ col ]  +=  phi0 * jacFactorInv0" << r;
          for( int d=1; d < dim; ++d )
            out << "  +  phi" << d << " * jacFactorInv" << d << r;
          out << ";" << std::endl;
        }
        out << "      }" << std::endl;
      }

      static void axpyJacobianCodegen(std::ostream& out,
              const int dim, const int dimRange, const size_t numRows, const size_t numCols )
      {
        const std::string funcName =
              generateFunctionName( "axpyJacobianLoop", simdWidth, dimRange, numRows, numCols );

        writePreCompHeader( out, -1 );

        out << "template <class BaseFunctionSet> // dimRange = "<< dimRange << ", quadNop = " << numRows << ", scalarBasis = " << numCols << std::endl;
        out << "struct AxpyJacobians<BaseFunctionSet, EmptyGeometry, " << dimRange << ", " << numRows << ", " << numCols << ">" << std::endl;
        out << "{" << std::endl;
        out << "  template< class QuadratureType,"<< std::endl;
        out << "            class JacobianRangeVectorType," << std::endl;
        out << "            class JacobianRangeFactorType," << std::endl;
        out << "            class LocalDofVectorType>" << std::endl;
        out << "  static void axpy( const QuadratureType&," << std::endl;
        out << "                    const EmptyGeometry&," << std::endl;
        out << "                    const JacobianRangeVectorType&," << std::endl;
        out << "                    const JacobianRangeFactorType&," << std::endl;
        out << "                    LocalDofVectorType&)" << std::endl;
        out << "  {" << std::endl;
        out << "    std::cerr << \"ERROR: wrong code generated for VectorialBaseFunctionSet::axpyJacobians\" << std::endl;" << std::endl;
        out << "    abort();" << std::endl;
        out << "  }" << std::endl;
        out << "};" << std::endl << std::endl;
        out << "template <class BaseFunctionSet, class Geometry>" << std::endl;
        out << "struct AxpyJacobians<BaseFunctionSet, Geometry, " << dimRange << ", " << numRows << ", " << numCols << ">" << std::endl;
        out << "{" << std::endl;

        out << "  template< class QuadratureType,"<< std::endl;
        out << "            class JacobianRangeVectorType," << std::endl;
        out << "            class JacobianRangeFactorType," << std::endl;
        out << "            class LocalDofVectorType>" << std::endl;
        out << "  static void axpy( const QuadratureType& quad," << std::endl;
        out << "                    const Geometry& geometry," << std::endl;
        out << "                    const JacobianRangeVectorType& jacStorage," << std::endl;
        out << "                    const JacobianRangeFactorType& jacFactors," << std::endl;
        out << "                    LocalDofVectorType& dofs)" << std::endl;
        out << "  {" << std::endl;
        out << "    typedef typename JacobianRangeFactorType :: value_type JacobianRangeType;" << std::endl << std::endl;

        const size_t dofs = dimRange * numCols ;
        out << "    typedef " << doubletype() << " Field;" << std::endl;
        out << "    static thread_local std::vector< Field > memory( " << dofs << " );" << std::endl;
        //out << "    static Dune::Fem::ThreadSafeValue< std::vector< Field > > memory( " << dofs << " );" << std::endl;

        //out << "    Field* result = (*memory).data();" << std::endl;
        out << "    Field* result = memory.data();" << std::endl;
        out << "    for( int i = 0 ; i < " << dofs << "; ++i ) result[ i ] = 0;" << std::endl << std::endl;

        for( int r=0; r<dimRange; ++r )
          out << "    Field* result" << r << " = result + " << r * numCols << ";" << std::endl;
        out << std::endl;

        //for( int d =0; d < dim; ++d )
        //{
        //  out << "    Field* base"<< d << " = result + " << dofs + d * numCols << ";" << std::endl;
        //}

        out << "    const Field* base = jacStorage.data();" << std::endl << std::endl;
        out << "    JacobianRangeType jacFactorTmp;" << std::endl;
        out << "    for( int row = 0; row < " << numRows << " ; ++ row )" << std::endl;
        out << "    {" << std::endl;
        out << "      typedef typename Geometry::JacobianInverseTransposed GeometryJacobianType;" << std::endl;
        out << "      // use reference to GeometryJacobianType to make code compile with SPGrid Geometry" << std::endl;
        out << "      const GeometryJacobianType& gjit = geometry.jacobianInverseTransposed( quad.point( row ) );" << std::endl << std::endl;
        out << "      for( int r = 0; r < " << dimRange << " ; ++r )" << std::endl;
        out << "      {"<<std::endl;
        out << "        gjit.mtv( jacFactors[ row ][ r ], jacFactorTmp[ r ] );" << std::endl;
        out << "      }" << std::endl << std::endl;

        /*
        out << "      for( int col = 0, rowCol = quad.cachingPoint( row ) * " << numCols << "; col < " << numCols << " ; ++ col, ++rowCol )" << std::endl;
        out << "      {" << std::endl;
        for( int d =0; d < dim; ++d )
          out << "        base"<< d << "[ col ] = jacStorage[ rowCol ][ 0 ][ " << d << " ];" << std::endl;

        out << "      }" << std::endl;
        */

        out << "      // calculate updates" << std::endl;
        out << "      " << funcName << "(";
        //for( int d =0; d < dim; ++d ) out << "base" << d << ", ";
        out << " base + ( quad.localCachingPoint( row ) * " << numCols * dim << " )," << std::endl;
        for( int i =0; i < dim; ++i )
        {
          out << "                               ";
          for( int r = 0; r < dimRange; ++ r )
            out << "   jacFactorTmp[ " << r  << " ][ " << i << " ], ";
          out << std::endl;
        }
        out << "                               ";
        for( int r = 0; r < dimRange; ++ r )
        {
          out << "   result" << r;
          if( r == dimRange -1 )
            out << " );" << std::endl ;
          else
            out << ", ";
        }
        out << "    }" << std::endl << std::endl;

        out << "    // sum up results (transform from variable based to point based layout)" << std::endl;
        out << "    for( int col = 0, dof = 0 ; col < "<< numCols << " ; ++col )" << std::endl;
        out << "    {" << std::endl;
        for( int r = 0 ; r < dimRange; ++ r )
          out << "      dofs[ dof++ ]  +=  result" << r << "[ col ];" << std::endl;
        out << "    }" << std::endl;

        out << "  }" << std::endl << std::endl;
        //out << "};" << std::endl;


        ///////////////////////////////////
        //  inner loop
        ///////////////////////////////////
        //writePreCompHeader( out, 0 );

        out << "  static void " << funcName << "(" << std::endl;
        out << "        const " << doubletype() << "* " << restrictKey() << " base," << std::endl;
        //for( int i=1; i<dim; ++ i )
        //  out << "        const " << doubletype() << "* " << restrictKey() << " base" << i << "," << std::endl;
        for( int i=0; i<dim; ++i )
        {
          out << "        ";
          for( int r=0; r<dimRange; ++ r )
            out << "const " << doubletype() << " jacFactorInv"<< i << r << ", ";
          out << std::endl;
        }
        for( int r = 0; r < dimRange; ++r )
        {
          out << "        " << doubletype() << "* " << restrictKey() << " result" << r;
          if( r == dimRange-1 )
            out << " )" << std::endl;
          else
            out << "," << std::endl;
        }
        //writePreCompHeader( out, 1 );
        out << "  {" << std::endl;
        writeInnerLoopAxpyJac( out, dim, dimRange, numCols );
        out << "  }" << std::endl;
        out << "};" << std::endl;
        writePreCompHeader( out, 2 );
      }
    };

    // if this pre processor variable is defined then
    // we assume that CODEGENERATOR_REPLACEMENT is CodeGenerator of choice
#ifndef FEM_CODEGENERATOR_REPLACEMENT
    typedef DefaultCodeGenerator< 4 > CodeGeneratorType;
#else
    typedef FEM_CODEGENERATOR_REPLACEMENT CodeGeneratorType;
#endif

    inline std::string autoFilename(const int dim, const int polynomialOrder )
    {
      std::stringstream name;
      name << "autogeneratedcode_" << dim << "d_k" << polynomialOrder << ".hh";
      return name.str();
    }

    } // namespace Codegen


    template <class DiscreteFunctionSpace, class Vector>
    inline void generateCode (const DiscreteFunctionSpace& space,
                              const Vector& elemQuadOrders,
                              const Vector& faceQuadOrders,
                              const std::string& outpath = "./",
                              const std::string& filename = "autogeneratedcode.hh" )
    {
      using namespace Codegen;

      const int dimRange  = DiscreteFunctionSpace :: dimRange;
      const int dimDomain = DiscreteFunctionSpace :: dimDomain;
      const int dimGrad   = dimRange*dimDomain;

      typedef typename DiscreteFunctionSpace :: GridPartType GridPartType;

      typedef CachingQuadrature< GridPartType, 0 > ElementQuadratureType;
      typedef CachingQuadrature< GridPartType, 1 > FaceQuadratureType;


      std::set< int > sizes;
      std::set< int > quadNops;
      for( const auto& entity : space )
      {
        // only use size of scalar basis function set, i.e. dimRange = 1
        const int scalarSize = space.basisFunctionSet( entity ).size() / dimRange ;
        sizes.insert( scalarSize );

        const auto iend = space.gridPart().iend( entity );
        for( auto it = space.gridPart().ibegin( entity ); it != iend; ++it )
        {
          for( const auto& quadOrd : faceQuadOrders )
          {
            FaceQuadratureType quad( space.gridPart(), *it, quadOrd, FaceQuadratureType::INSIDE );
            quadNops.insert( quad.nop() );
          }
        }
      }

      for( const auto& type : space.indexSet().types(0))
      {
        for( const auto& quadOrd : elemQuadOrders )
        {
          ElementQuadratureType quad( type, quadOrd );
          quadNops.insert( quad.nop() );
        }
      }

      std::string path ( outpath );
      path += "/autogeneratedcode";

      // set output path
      CodegenInfo& info = CodegenInfo::instance();
      info.setPath( path );

      // add my dimrange
      info.addDimRange( &space, dimRange );
      int gradSpace;
      info.addDimRange( &gradSpace, dimGrad );

      for( const auto& size : sizes )
      {
        for( const auto& quadNop : quadNops )
        {
          info.addEntry( "evalranges",
                CodeGeneratorType :: evaluateCodegen, dimDomain, dimRange, quadNop, size );
          info.addEntry( "evaljacobians",
                CodeGeneratorType :: evaluateJacobiansCodegen, dimDomain, dimRange, quadNop, size );
          info.addEntry( "axpyranges",
                CodeGeneratorType :: axpyCodegen, dimDomain, dimRange, quadNop, size );
          info.addEntry( "axpyjacobians",
                CodeGeneratorType :: axpyJacobianCodegen, dimDomain, dimRange, quadNop, size );

          info.addEntry( "evalranges",
                CodeGeneratorType :: evaluateCodegen, dimDomain, dimGrad, quadNop, size );
          info.addEntry( "evaljacobians",
                CodeGeneratorType :: evaluateJacobiansCodegen, dimDomain, dimGrad, quadNop, size );
          info.addEntry( "axpyranges",
                CodeGeneratorType :: axpyCodegen, dimDomain, dimGrad, quadNop, size );
          info.addEntry( "axpyjacobians",
                CodeGeneratorType :: axpyJacobianCodegen, dimDomain, dimRange, quadNop, size );
        }
      }

      //std::cerr << "Code for k="<< MAX_POLORD << " generated!! " << std::endl;
      //std::remove( autoFilename( CODEDIM, MAX_POLORD ).c_str() );

      info.setFileName( filename );
      bool written = info.dumpInfo();

      if( written )
      {
#ifndef NDEBUG
        std::cout << "Written code to " << filename << std::endl;
#endif
        //////////////////////////////////////////////////
        //  write include header
        //////////////////////////////////////////////////
        std::ofstream file( outpath + "/" + filename );

        if( file )
        {
          std::string header( filename );
          size_t size = header.size();
          // replace all . with _
          for( size_t i=0; i<size; ++i )
          {
            if( header[ i ] == '.' )
              header[ i ] = '_';
          }

          file << "#ifndef " << header << "_INCLUDED" << std::endl;
          file << "#define " << header << "_INCLUDED" << std::endl;
          file << "#ifndef USE_BASEFUNCTIONSET_CODEGEN" << std::endl;
          file << "#define USE_BASEFUNCTIONSET_CODEGEN" << std::endl;
          file << "#endif" << std::endl;
          file << "// this is the file containing the necessary includes for the specialized codes" << std::endl;
          file << "#define DUNE_FEM_INCLUDE_AUTOGENERATEDCODE_FILENAME_SPEC \"" << path << "/" << filename << "\"" << std::endl;
          file << "#endif" << std::endl;
        }
      }
      else
      {
#ifndef NDEBUG
        std::cout << "No changes written to " << filename << std::endl << std::endl;
#endif
      }
    }

  } // namespace Fem

} // namespace Dune

#endif // #ifndef DUNE_FEM_BASEFUNCTIONSETS_CODEGEN_HH
