# -*- coding: utf-8 -*-
import click
import json
from ..utils.spinner import (
    init_spinner,
    start_spinner,
    stop_spinner,
)
from ..utils.print import (
    tbprint,
    eprint,
    oprint,
    opprint,
)


@click.group()
@click.pass_obj
@click.pass_context
def fabric_wired(ctx, obj):
    """DNA Center Fabric Wired API (version: 1.2.10).

    Wraps the DNA Center Fabric Wired API and exposes the API as native Python commands.

    """
    ctx.obj = obj.fabric_wired


@fabric_wired.command()
@click.option('--sda_border_device', type=str,
              help='''sda/border-device path parameter.''',
              required=True,
              show_default=True)
@click.option('--device_ip_address', type=str,
              help='''device-ip-address path parameter.''',
              required=True,
              show_default=True)
@click.option('--headers', type=str, help='''Dictionary of HTTP Headers to send with the Request.''',
              default=None,
              show_default=True)
@click.option('-pp', '--pretty_print', type=int, help='''Pretty print indent''',
              default=None,
              show_default=True)
@click.option('--beep', is_flag=True, help='''Spinner beep (on)''')
@click.pass_obj
def gets_border_device_detail(obj, pretty_print, beep,
                              sda_border_device,
                              device_ip_address,
                              headers):
    """**Beta** - Gets border device detail from SDA Fabric.
    """
    spinner = init_spinner(beep=beep)
    start_spinner(spinner)
    try:
        if headers is not None:
            headers = json.loads(headers)
        result = obj.gets_border_device_detail(
            sda_border_device=sda_border_device,
            device_ip_address=device_ip_address,
            headers=headers)
        stop_spinner(spinner)
        opprint(result, indent=pretty_print)
    except Exception as e:
        stop_spinner(spinner)
        tbprint()
        eprint('Error:', e)
        click.Context.exit(-1)


@fabric_wired.command()
@click.option('--sda_border_device', type=str,
              help='''sda/border-device path parameter.''',
              required=True,
              show_default=True)
@click.option('--headers', type=str, help='''Dictionary of HTTP Headers to send with the Request.''',
              default=None,
              show_default=True)
@click.option('--payload', type=str, help='''A JSON serializable Python object to send in the body of the Request.''',
              default=None,
              show_default=True)
@click.option('--active_validation', type=bool, help='''Enable/Disable payload validation.''',
              default=True,
              show_default=True)
@click.option('-pp', '--pretty_print', type=int, help='''Pretty print indent''',
              default=None,
              show_default=True)
@click.option('--beep', is_flag=True, help='''Spinner beep (on)''')
@click.pass_obj
def adds_border_device(obj, pretty_print, beep,
                       sda_border_device,
                       headers,
                       payload,
                       active_validation):
    """Adds border device in SDA Fabric.
    """
    spinner = init_spinner(beep=beep)
    start_spinner(spinner)
    try:
        if headers is not None:
            headers = json.loads(headers)
        if payload is not None:
            payload = json.loads(payload)
        result = obj.adds_border_device(
            sda_border_device=sda_border_device,
            headers=headers,
            payload=payload,
            active_validation=active_validation)
        stop_spinner(spinner)
        opprint(result, indent=pretty_print)
    except Exception as e:
        stop_spinner(spinner)
        tbprint()
        eprint('Error:', e)
        click.Context.exit(-1)


@fabric_wired.command()
@click.option('--sda_border_device', type=str,
              help='''sda/border-device path parameter.''',
              required=True,
              show_default=True)
@click.option('--device_ip_address', type=str,
              help='''device-ip-address path parameter.''',
              required=True,
              show_default=True)
@click.option('--headers', type=str, help='''Dictionary of HTTP Headers to send with the Request.''',
              default=None,
              show_default=True)
@click.option('-pp', '--pretty_print', type=int, help='''Pretty print indent''',
              default=None,
              show_default=True)
@click.option('--beep', is_flag=True, help='''Spinner beep (on)''')
@click.pass_obj
def deletes_border_device(obj, pretty_print, beep,
                          sda_border_device,
                          device_ip_address,
                          headers):
    """Deletes border device from SDA Fabric.
    """
    spinner = init_spinner(beep=beep)
    start_spinner(spinner)
    try:
        if headers is not None:
            headers = json.loads(headers)
        result = obj.deletes_border_device(
            sda_border_device=sda_border_device,
            device_ip_address=device_ip_address,
            headers=headers)
        stop_spinner(spinner)
        opprint(result, indent=pretty_print)
    except Exception as e:
        stop_spinner(spinner)
        tbprint()
        eprint('Error:', e)
        click.Context.exit(-1)
