# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['the_census',
 'the_census._api',
 'the_census._data_transformation',
 'the_census._geographies',
 'the_census._persistence',
 'the_census._stats',
 'the_census._utils',
 'the_census._utils.log',
 'the_census._variables',
 'the_census._variables.repository',
 'the_census._variables.search']

package_data = \
{'': ['*']}

install_requires = \
['pandas>=1.2.0,<2.0.0',
 'punq>=0.4.1,<0.5.0',
 'python-dotenv>=0.15.0,<0.16.0',
 'requests>=2.25.1,<3.0.0',
 'tqdm>=4.55.1,<5.0.0']

setup_kwargs = {
    'name': 'the-census',
    'version': '2.1.0',
    'description': 'Tool to get and query US Census data',
    'long_description': '[![Imports: isort](https://img.shields.io/badge/%20imports-isort-%231674b1?style=flat&labelColor=ef8336)](https://pycqa.github.io/isort/)\n[![pre-commit](https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white)](https://github.com/pre-commit/pre-commit)\n\n![Release](https://github.com/drawjk705/the_census/workflows/Release/badge.svg)\n![Staging](https://github.com/drawjk705/the_census/workflows/Staging/badge.svg)\n![CI](https://github.com/drawjk705/the_census/workflows/CI/badge.svg)\n\n# The Census\n\nWant to work with US Census data? Look no further.\n\n## Getting started\n\n### View all datasets\n\nIf you you\'re not sure what Census dataset you\'re interested in, the following code will take care of you:\n\n```python\nfrom the_census import Census\n\nCensus.list_available_datasets()\n```\n\nThis will present you with a pandas DataFrame listing all available datasets from the US Census API. (This includes only aggregate datasets, as they other types [of which there are very few] don\'t play nice with the client).\n\n### Help with terminology\n\nSome of the terms used in the data returned can be a bit opaque. To get a clearer sense of what some of those mean, run this:\n\n```python\nCensus.help()\n```\n\nThis will print out links to documentation for various datasets, along with what their group/variable names mean, and how statistics were calculated.\n\n### Selecting a dataset\n\nBefore getting started, you need to [get a Census API key](https://api.census.gov/data/key_signup.html), and set the following the environment variable `CENSUS_API_KEY` to whatever that key is, either with\n\n```bash\nexport CENSUS_API_KEY=<your key>\n```\n\nor in a `.env` file:\n\n```bash\nCENSUS_API_KEY=<your key>\n```\n\nSay you\'re interested in the American Community Survey 1-year estimates for 2019. Look up the dataset and survey name in the table provided by `list_available_datasets`, and execute the following code:\n\n```python\n>>> from the_census import Census\n>>> Census(year=2019, dataset="acs", survey="acs1")\n\n<Census year=2019 dataset=acs survey=acs1>\n```\n\nThe `dataset` object will now let you query any census data for the the ACS 1-year estimates of 2019. We\'ll now dive into how to query this dataset with the tool. However, if you aren\'t familiar with dataset "architecture", check out [this](#dataset-architecture) section.\n\n### Arguments to `Census`\n\nThis is the signature of `Census`:\n\n```python\nclass Census\n    def __init__(self,\n                 year: int,\n                 dataset: str = "acs",\n                 survey: str = "acs1",\n                 cache_dir: str = CACHE_DIR,        # cache\n                 should_load_from_existing_cache: bool = False,\n                 should_cache_on_disk: bool = False,\n                 replace_column_headers: bool = True,\n                 log_file: str = DEFAULT_LOG_FILE): # census.log\n        pass\n```\n\n-   `year`: the year of the dataset\n-   `dataset`: type of the dataset, specified by [`list_available_datasets`](#view-all-datasets)\n-   `survey`: type of the survey, specified by [`list_available_datasets`](#view-all-datasets)\n-   `cache_dir`: if you opt in to on-disk caching (more on this below), the name of the directory in which to store cached data\n-   `should_load_from_existing_cache`: if you have cached data from a previous session, this will reload cached data into the `Census` object, instead of hitting the Census API when that data is queried\n-   `should_cache_on_disk`: whether or not to cache data on disk, to avoid repeat API calls. The following data will be cached:\n    -   Supported Geographies\n    -   Group codes\n    -   Variable codes\n-   `replace_column_headers`: whether or not to replace column header names for variables with more intelligible names instead of their codes\n-   `log_file`: name of the file in which to store logging information\n\n###### A note on caching\n\nWhile on-disk caching is optional, this tool, by design, performs in-memory caching. So a call to `census.get_groups()` will hit the Census API one time at most. All subsequent calls will retrieve the value cached in-memory.\n\n## Making queries\n\n### Supported geographies\n\nGetting the [supported geographies](#supported-geographies) for a dataset as as simple as this:\n\n```python\ncensus.get_supported_geographies()\n```\n\nThis will output a DataFrame will all possible supported geographies (e.g., if I can query all school districts across all states).\n\nIf you don\'t want to have to keep on typing supported geographies after this, you can use tab-completion in Jupyter by typing:\n\n```python\ncensus.supported_geographies.<TAB>\n```\n\n### Geography codes\n\nIf you decide you want to query a particular geography (e.g., a particular school district within a particular state), you\'ll need the [FIPS](https://en.wikipedia.org/wiki/Federal_Information_Processing_Standard_state_code#FIPS_state_codes) codes for that school district and state.\n\nSo, if you\'re interested in all school districts in Colorado, here\'s what you\'d do:\n\n1. Get FIPS codes for all states:\n\n```python\nfrom the_census import GeoDomain\n\ncensus.get_geography_codes(GeoDomain("state", "*"))\n```\n\nOr, if you don\'t want to import `GeoDomain`, and prefer to use tuples:\n\n```python\ncensus.get_geography_codes(("state", "*"))\n```\n\n2. Get FIPS codes for all school districts within Colorado (FIPS code `08`):\n\n```python\ncensus.get_geography_codes(GeoDomain("school district", "*"),\n                           GeoDomain("state", "08"))\n```\n\nOr, if you don\'t want to import `GeoDomain`, and prefer to use tuples:\n\n```python\ncensus.get_geography_codes(("school district", "*"),\n                           ("state", "08"))\n```\n\nNote that geography code queries must follow supported geography guidelines.\n\n### Groups\n\nWant to figure out what groups are available for your dataset? No problem. This will do the trick for ya:\n\n```python\ncensus.get_groups()\n```\n\n...and you\'ll get a DataFrame with all groups for your census.\n\n#### Searching groups\n\n`census.get_groups()` will return a lot of data that might be difficult to slog through. In that case, run this:\n\n```python\ncensus.search_groups(regex=r"my regex")\n```\n\nand you\'ll get a filtered DataFrame with matches to your regex.\n\n#### Groups autocomplete\n\nIf you\'re working in a Jupyter notebook and have autocomplete enabled, running `census.groups.`, followed by a tab, will trigger an autocomplete menu for possible groups by their name (as opposed to their code, which doesn\'t have any inherent meaning in and of itself).\n\n```python\ncensus.groups.SexByAge   # code for this group\n```\n\n### Variables\n\nYou can either get a DataFrame of variables based on a set of groups:\n\n```python\ncensus.get_variables_by_group(census.groups.SexByAge,\n                              census.groups.MedianAgeBySex)\n```\n\nOr, you can get a DataFrame with all variables for a given dataset:\n\n```python\ncensus.get_all_variables()\n```\n\nThis second operation, can, however, take a lot of time.\n\n#### Searching variables\n\nSimilar to groups, you can search variables by regex:\n\n```python\ncensus.search_variables(r"my regex")\n```\n\nAnd, you can limit that search to variables of a particular group or groups:\n\n```python\ncensus.search_variables(r"my regex", census.groups.SexByAge)\n```\n\n#### Variables autocomplete\n\nVariables also support autocomplete for their codes, as with groups.\n\n```python\ncensus.variables.EstimateTotal_B01001  # code for this variable\n```\n\n(These names must be suffixed with the group code, since, while variable codes are unique across groups, their names are not unique across groups.)\n\n### Statistics\n\nOnce you have the variables you want to query, along with the geography you\'re interested in, you can now make statistics queries from your dataset:\n\n```python\nfrom the_census import GeoDomain\n\nvariables = census.get_variables_for_group(census.groups.SexByAge)\n\ncensus.get_stats(variables["code"].tolist(),\n                 GeoDomain("school district", "*"),\n                 GeoDomain("state", "08"))\n```\n\nOr, if you\'d rather use tuples instead of `GeoDomain`:\n\n```python\nvariables = census.get_variables_for_group(census.groups.SexByAge)\n\ncensus.get_stats(variables["code"].tolist(),\n                 ("school district", "*"),\n                 ("state", "08"))\n```\n\n## Dataset "architecture"\n\nUS Census datasets have 3 primary components:\n\n1.  [Groups](#groups)\n2.  [Variables](#variables)\n3.  [Supported Geographies](#supported-geographies)\n\n### Groups\n\nA group is a "category" of data gathered for a particular census. For example, the `SEX BY AGE` group would provide breakdowns of gender and age demographics in a given region in the United States.\n\nSome of these groups\' names, however, are a not as clear as `SEX BY AGE`. In that case, I recommend heading over to the survey in question\'s [technical documentation](https://www2.census.gov/programs-surveys/) which elaborates on what certain terms mean with respect to particular groups. Unfortunately, the above link might be complicated to navigate, but if you\'re looking for ACS group documentation, [here\'s](https://www2.census.gov/programs-surveys/acs/tech_docs/subject_definitions/2019_ACSSubjectDefinitions.pdf) a handy link.\n\n(You can also get these links by running `Census.help()`.)\n\n### Variables\n\nVariables measure a particular data-point. While they have their own codes, you might find variables which share the same name (e.g., `Estimate!!:Total:`). This is because each variable belongs to a [group](#group). So, the `Estimate!!:Total` variable for `SEX BY AGE` group is the total of all queried individuals in that group; but the `Estimate!!:Total` variable for `POVERTY STATUS IN THE PAST 12 MONTHS BY AGE` group is the total of queried individuals for _that_ group. (It\'s important when calculating percentages that you work within the same group. So if I want the percent of men in the US, whose total number I got from `SEX BY AGE` I should use the `Estimate!!:Total:` of that group as my denominator, and not the `Estimate!!:Total:` of the `POVERTY STATUS` group).\n\nVariables on their own, however, do nothing. They mean something only when you query a particular [geography](#supported-geographies) for them.\n\n### Supported Geographies\n\nSupported geographies dictate the kinds of queries you can make for a given census. For example, in the ACS-1, I might be interested in looking at stats across all school districts. The survey\'s supported geographies will tell me if I can actually do that; or, if I need to refine my query to look at school districts in a given state or smaller region.\n',
    'author': 'Joel Krim',
    'author_email': 'drawjk705@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/drawjk705/us-stats/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
