# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['plutto', 'plutto.managers', 'plutto.mixins', 'plutto.resources']

package_data = \
{'': ['*']}

install_requires = \
['httpx>=0.16,<1.0']

setup_kwargs = {
    'name': 'plutto',
    'version': '0.2.0',
    'description': 'The official Python client for Plutto API',
    'long_description': '<h1 align="center">Plutto Python Client</h1>\n\n<p align="center">\n  <a href="https://pypi.org/project/plutto" target="_blank">\n      <img src="https://img.shields.io/pypi/v/plutto?label=version&logo=python&logoColor=%23fff&color=306998" alt="PyPI - Version">\n  </a>\n\n  <a href="https://github.com/plutto-labs/plutto-python/actions?query=workflow%3Atests" target="_blank">\n      <img src="https://img.shields.io/github/workflow/status/plutto-labs/plutto-python/tests?label=tests&logo=python&logoColor=%23fff" alt="Tests">\n  </a>\n\n  <a href="https://codecov.io/gh/plutto-labs/plutto-python" target="_blank">\n      <img src="https://img.shields.io/codecov/c/gh/plutto-labs/plutto-python?label=coverage&logo=codecov&logoColor=ffffff" alt="Coverage">\n  </a>\n\n  <a href="https://github.com/plutto-labs/plutto-python/actions?query=workflow%3Alinters" target="_blank">\n      <img src="https://img.shields.io/github/workflow/status/plutto-labs/plutto-python/linters?label=linters&logo=github" alt="Linters">\n  </a>\n</p>\n\nThis library will help you easily integrate Plutto API to your software, making your developer life a little bit more enjoyable.\n\n\n---\n\n## Installation\nInstall using pip\n\n```bash\n$ pip3 install plutto\n```\n*Note:* This SDK requires [Python 3.7+](https://www.python.org/downloads/release/python-370/)\n\n## Usage\nWith this SDK we want to provide a wrapper to the [Plutto API](https://docs.getplutto.com/reference) with a very intuitive way of use. All the methods were implemented as in the API documentation, we strongly recommend to read it before using this SDK\n\n### Quickstart\nFirst of all, you will need a [Plutto] account. After creating it, you can get your API key, which will let you to use the `Plutto` object. Then, you\'re ready to use this awesome SDK!\n\n```python\nfrom plutto import Plutto\n\nclient = Plutto("your_api_key")\n```\n\n### Managers\nTo manage the resources retrieved by the SDK we use managers. They are python objects that let you with any object inside Plutto API.. All the existing **managers** are inside the `Plutto` object. These are:\n- `customers`\n- `invoices`\n- `meter_events`\n- `permission_groups`\n- `products`\n- `subscriptions`\n\n#### `all`\n_Note_: this method is only available in `customers`, `invoices`, `permission_groups` and `products` managers\n\n```python\ncustomers = client.customers.all()\n```\n\nThis method returns a **a generator** with all the instances of the customers resource. But, what if the API can recive more params? `kwargs` to the rescue! This way you can pass params like `q[status_eq]` and `q[customer_eq]` to filter the `Invoices`. If you want to get `invoices` with an specific _status_ and _customer_, you need to pass those params to the request\n```python\nparams = {\n    "q[status_eq]": "paid",\n    "q[customer_eq]": "customer_id"\n}\ninvoices = client.invoices.all(**params)\n```\n\nAlso, if you pass the `lazy=False` parameter, this will force the method of the SDK to return a list of the instances of the resource, instead of the generetors of them. **Disclaimer**: This could take very long if you have a lot of instances to be retrieved.\n\n```python3\ncustomers = client.customers.all(lazy=False)\nisinstance(customers, list) # True\n```\n\n#### `get`\n_Note_: this method is only available in `customers` and `invoices` managers\n\nThis method returns an instance of a resource using it\'s identifier to find it\n\n```python3\ncustomer = client.customers.get("customer_id")\nisinstance(customer, Customer) # True\n```\n\n#### `create`\n_Note_: this method is only available in `customers`, `meter_events` and `subscriptions` managers\n\nThis method creates and returns a new instance of the resource. The attributes of the resource to be created must be passed as `kwargs`. This parameters are specified in the API documentation of the correspondant resource\n\n```python3\npayload = {\n    "identifier": "your-id_12885305",\n    "email": "donald@getplutto.com",\n    "name": "Donald",\n    "billing_information": {\n        "city": "Santiago",\n        "country_iso_code": "CL",\n        "state": "Metropolitana",\n        "address": "Av. Las Condes",\n        "zip": "12345",\n        "tax_id": "73245432-1",\n        "legal_name": "Plutto Inc",\n        "activity": "Software Development",\n        "phone": "+56912345678"\n    }\n}\n\ncustomer = client.customer.create(**payload)\n```\n\n#### `update`\n_Note_: this method is only available in `customers` manager\n\n```python3\ncustomer = client.cuestomers.update(\n    "user_id",\n    email="goofy@getplutto.com",\n    name="Goofy"\n)\n```\nThis is an example of how can be used the update method. The first parameter corresponds to the id of the customer, this way the manager can find the existing resource. Then, the attributes that want to be modified are passed as `kwargs`, this ones are specified in the API in the correspondant resource update method.\nThis manager method is making two calls to the Plutto API, the first, to get the resource, and the second to update it. Therefore, if you only want to make one API call and already got the reource python object, you can call update directly on the object\n\n```python3\n# Get the object\nexample_customer = client.customers.get("customer_id")\n\n# Update the customer\nexample_customer.update(\n    email="goofy@getplutto.com",\n    name="Goofy"\n)\n```\n\nThis way, you can call `update` on the objects you are already working with, evitating to make an innescesary API call and saving some words\n\n\n#### `delete`\n_Note_: this method is only available in `customers` manager\n\n```python3\ndeleted_customer_id = client.customers.delete("customer_id")\n```\nThis method deletes an existing instance of a resource by it\'s identifier, and returns it. As in the `update` method, you can call `delete` on an resource object, for the same reasons explained in the previous method\n\n```python3\n# Get the resource\ncustomer = client.customers.get("customer_id")\n\n# Delete de resource\ndeleted_customer_id = customer.delete()\n```\n\n\n### How to use this SDK\nThe way to use is very similar to the API. For all the methods you need the `Plutto` object\n\n#### `Plutto` object\nInstantiate the object using your secret API key\n```python3\nfrom plutto import Plutto\n\nclient = Plutto("secret_api_key")\n```\nThis client will give you access to all the managers that are available in this SDK. That means you can work with any [manager](#managers) you want from this object\n\n#### `customers` manager\n\n_Available methods_: `all`, `get`, `create`, `update` and `delete`\n\nFrom the `Plutto` object you can manage your `customers` easily. You can get all the customers that have been created\n\n```python3\ncustomers = client.customers.all()\nfor customer in customers:\n    print(customer.name)\n```\n\nAlso, if you have the id, you can get a singular customer\n\n```python3\ncustomer = client.customers.get("id_of_the_customer")\n```\n\nNeed a new customer? Creating it is very intuitive, just pass the [parameters specified in the docs](https://docs.getplutto.com/reference/post_customers) as kwargs\n\n```python3\npayload = {\n    "email": "donald@getplutto.com",\n    "name": "donald",\n    "billing_information": {\n        "city": "Santiago",\n        "country_iso_code": "CL",\n        "legal_name": "Plutto",\n        "activity": "Software Development"\n    }\n}\n\nnew_customer = client.customers.create(**payload)\n```\n\nIf any customers need to be updated, you can do it with it\'s id and pass the params you want to update as kwargs\n\n```python3\nupdate_params = {\n    "name": "Goofy",\n    "email": "goofy@getplutto.com"\n}\n\nupdated_customer = client.customers.update("update_customer_id", **update_params)\n\n# It can also be done this way\nupdate_params = {\n    "name": "Goofy",\n    "email": "goofy@getplutto.com"\n}\n\ncustomer = client.customers.get("update_customer_id")\nupdated_customer = customer.update(**update_params)\n```\n\nDelete a customer can be done by passing it\'s id to the `delete` method\n\n```python3\ndeleted_customer_id = client.customers.delete("delete_customer_id")\n\n# It can also be done this way\ncustomer = client.customers.get("delete_customer_id")\ndeleted_customer_id = customer.delete()\n```\n\n\n#### `invoices` manager\n\n_Available methods_: `all`, `get` and `mark_as`\n\nTo get all the existing invoices you can do the following\n\n```python3\ninvoices = client.invoices.all()\n\nfor invoice in invoices:\n    print(invoice.id)\n\n# If filtering by params is needed\nparams = {\n    "q[status_eq]": "paid",\n    "q[customer_eq]": "customer_id"\n}\ninvoices = client.invoices.all(**params)\n```\n\nTo get a specific invoice, you can use it\'s id\n\n```python3\ninovice = client.invoices.get("invoice_id")\n```\n\nIt\'s possible to change an invoice status through the manager. The possible status are [specified in the docs](https://docs.getplutto.com/reference/patch_invoices-id-mark-as)\n\n```python3\ninvoice = client.invoices.mark_as("invoice_id", status="paid")\n```\n\n#### `meter_events` manager\n_Available methods_: Only `create`\n\nIt\'s possible to create a meter event this way\n\n```python3\npayload = {\n    "customer_id": "example_customer_id",\n    "meter_id": "example_meter_id",\n    "amout": 100,\n    "action": "set"\n}\nmeter_event = client.meter_events.create(**payload)\n```\n\n\n#### `permission_groups` manager\n_Available methods_: Only `all`\n\nYou can get all the permission groups as follows\n\n```python3\npermission_groups = client.permission_groups.all()\nfor permission_group in permission_groups:\n    print(permission_group.name)\n```\n\n\n#### `products` manager\n_Available methods_: Only `all`\n\nYou can get all the products as follows\n\n```python3\nproducts = client.products.all()\nfor product in products:\n    print(product.name)\n```\n\n\n#### `subscriptions` manager\n\nTo create a new subscription through the manager you can do the following. The attributes must be passed as kwargs. Required and optional ones are [specified in the docs](https://docs.getplutto.com/reference/post_subscriptions)\n\n```python3\npayload = {\n    "customer_id": "example_customer_id",\n    "pricing_ids": ["example_pricing_id_1", "example_pricing_id_2"]\n    "bills_at": "start",\n    "billing_period_duration": "P0Y1M0DT0H0M0S"\n}\nsubscription = client.subscriptions.create(**payload)\n```\n\nTo end a subscription you only need the id\n\n```python3\nsubscription = client.subscriptions.end("subscription_id")\n```\n\nAdding a product pricing to a subscription\n\n```python3\n# pricing_ids can have only one item, but it must be a list\npayload = {\n    "pricing_ids": ["pricing_id_1", "pricing_id_2", "pricing_id_3"]\n}\n\nsubscription = client.subscriptions.add_pricings("subscription_id", **payload)\n```\n\nTo remove a pricing, you can do the following\n\n```python3\n# pricing_ids can have only one item, but it must be a list\npayload = {\n    "pricings_id": ["pricing_id_1", "pricing_id_2", "pricing_id_3"]\n}\n\nsubscription = client.subscriptions.remove_pricings("subscription_id", **payload)\n```\n\n\n## Serialization\nAny resource retrieved by the SDK can be serialize, and it\'s super easy to do it. You just need to call the `serialize` method, and it\'s available in any resource\n\n```python3\ncustomer = client.customers.get("customer_id")\nserialized_customer = customer.serialize()\n```\n\n`serialized_customer` corresponds to a dictionary with only the attributes of the retrieved resource. It can be JSON-serialized\n\n\n## Testing\nAll the tests must be added in the `tests/` directory. To run the tests you nedd to execute the following command on the root path of the plutto library\n\n```bash\npytest .\n```\n\n\nEvery piece of code modified or added must be tested. The coverage always have to be increased or maintained, this will be checked in all PR\n\n## Publishing\n\nOn master/main branch...\n\n1. Change `VERSION` in `plutto/version.py`.\n2. Change `Unreleased` title to current version in `CHANGELOG.md`.\n3. Commit new release. For example: `Releasing v0.1.0`.\n4. Create tag. For example: `git tag v0.1.0`.\n5. Push tag. For example: `git push origin v0.1.0`.\n\n## Contributing\n\n1. Fork it\n2. Create your feature branch (`git checkout -b my-new-feature`)\n3. Commit your changes (`git commit -am \'Add some feature\'`)\n4. Push to the branch (`git push origin my-new-feature`)\n5. Create new Pull Request\n\n## Credits\n\nThank you [contributors](https://github.com/plutto-labs/plutto-python/graphs/contributors)!\n\nPlutto Ruby SDK is maintained by [Plutto](https://getplutto.com).\n\n\n## Acknowledgments\n\nThis SDK was strongly based on the [Fintoc python\'s SDK](https://github.com/fintoc-com/fintoc-python), designed by [Daniel Leal](https://github.com/daleal)\n## License\n\nPlutto Python SDK is © 2021 plutto, spa. It is free software and may be redistributed under the terms specified in the LICENSE file.',
    'author': 'Nicolas Massardo',
    'author_email': 'nmassardot@uc.cl',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://getplutto.com/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
