"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.compactFilterPreset = exports.rerootFilter = exports.verifyFilterable = void 0;
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const constructs_1 = require("constructs");
const core_1 = require("../core");
/**
 * Verify that store is filterable, meaning it allows destructive mutations.
 * @throws Error if store is not filterable
 * @internal
 */
function verifyFilterable(store) {
    if (!store.allowDestructiveMutations) {
        throw new Error("Store must allow destructive mutations to perform filtering; clone the store before applying filters using `store.clone(true)` operation and passing the cloned store to filtering operation.");
    }
}
exports.verifyFilterable = verifyFilterable;
/**
 * Changes the root of the store based on filter plan.
 * @throws Error if store is not filterable
 * @internal
 * @destructive
 */
function rerootFilter(store, plan) {
    verifyFilterable(store);
    if (plan.root == null)
        return; // noop
    const planRoot = typeof plan.root === "function" ? plan.root(store) : plan.root;
    if (planRoot === store.root)
        return; // noop
    const hoist = !!plan.hoistRoot;
    const ancestors = planRoot.scopes.slice();
    // remove the actual store.root from ancestors (we can't destroy that)
    const rootAncestor = ancestors.shift();
    if (rootAncestor !== store.root) {
        throw new Error(`${planRoot} is not within the store root graph: it has root of ${rootAncestor}`);
    }
    if (hoist) {
        // Move plan root as direct child of store root and prune all other ancestors
        planRoot.mutateHoist(store.root);
        // Only need to destroy to first non-root ancestor to prune the ancestral tree
        if (ancestors.length) {
            ancestors[0].mutateDestroy();
        }
        // prune all other root children
        store.root.children.forEach((child) => {
            if (child !== planRoot) {
                child.mutateDestroy();
            }
        });
    }
    else {
        // keep the plan root in place, but prune non-direct ancestor chain nodes
        // the direct ancestor chain is only the nodes scopes
        ancestors.reverse().forEach((ancestor) => {
            ancestor.siblings.forEach((ancestorSibling) => {
                ancestorSibling.mutateDestroy();
            });
        });
        // prune all planRoot siblings
        planRoot.siblings.forEach((sibling) => {
            sibling.mutateDestroy();
        });
    }
}
exports.rerootFilter = rerootFilter;
/**
 * Performs **compact** filter preset to store.
 * @throws Error if store is not filterable
 * @internal
 * @destructive
 */
function compactFilterPreset(store) {
    verifyFilterable(store);
    const extraneousNodes = store.root.findAll({
        order: constructs_1.ConstructOrder.POSTORDER,
        predicate: (node) => node.isExtraneous,
    });
    // collapse all extraneous nodes to nearest non-extraneous parent, or prune the node
    for (const extraneousNode of extraneousNodes) {
        const nonExtraneousAncestor = extraneousNode.findAncestor((node) => !node.isExtraneous);
        if (nonExtraneousAncestor && !nonExtraneousAncestor.isGraphContainer) {
            extraneousNode.mutateCollapseTo(nonExtraneousAncestor);
        }
        else {
            extraneousNode.mutateDestroy();
        }
    }
    const cdkOwnedContainers = store.root.findAll({
        order: constructs_1.ConstructOrder.POSTORDER,
        predicate: (node) => node.hasFlag(core_1.FlagEnum.CDK_OWNED) &&
            !node.parent?.hasFlag(core_1.FlagEnum.CDK_OWNED),
    });
    // collapse all cdk owned containers
    // NB: collapses the graph more closely mirror what developer writes, not what is auto created by cdk
    for (const cdkOwnedContainer of cdkOwnedContainers) {
        cdkOwnedContainer.mutateCollapse();
    }
    const cdkResources = store.root.findAll({
        order: constructs_1.ConstructOrder.POSTORDER,
        predicate: (node) => core_1.Graph.ResourceNode.isResourceNode(node),
    });
    // collapse all cfnResource wrapped by cdk resource
    for (const cdkResource of cdkResources) {
        if (cdkResource.isResourceWrapper) {
            cdkResource.mutateCollapse();
        }
        else if (cdkResource.cfnResource) {
            cdkResource.cfnResource.mutateCollapseToParent();
        }
    }
    store.edges.forEach((edge) => {
        if (edge.isExtraneous) {
            edge.mutateDestroy();
        }
    });
}
exports.compactFilterPreset = compactFilterPreset;
//# sourceMappingURL=data:application/json;base64,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