"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.inferFlags = exports.extractUnresolvedReferences = exports.IGNORE_REF_PATTERN = exports.extractInspectableAttributes = exports.inferNodeProps = exports.tryGetLogicalId = exports.getConstructUUID = exports.generateConsistentUUID = void 0;
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloneDeep = require("lodash.clonedeep"); // eslint-disable-line @typescript-eslint/no-require-imports
const shorthash = require("shorthash2"); // eslint-disable-line @typescript-eslint/no-require-imports
const traverse = require("traverse"); // eslint-disable-line @typescript-eslint/no-require-imports
const cdk_internals_1 = require("../cdk-internals");
const types_1 = require("./types");
/**
 * Generate deterministic UUID based on given value and prefix.
 * @param value The value to hash as UUID
 * @param {string} [prefix=""] Optional prefix used to prevent value conflicts
 */
function generateConsistentUUID(value, prefix = "") {
    return prefix + shorthash(JSON.stringify(value));
}
exports.generateConsistentUUID = generateConsistentUUID;
/** Get UUID for a given construct */
function getConstructUUID(construct) {
    return aws_cdk_lib_1.Names.uniqueResourceName(construct, {});
}
exports.getConstructUUID = getConstructUUID;
/** Try to get *logicalId* for given construct */
function tryGetLogicalId(construct) {
    if (aws_cdk_lib_1.CfnElement.isCfnElement(construct)) {
        const stack = aws_cdk_lib_1.Stack.of(construct);
        return stack.resolve(stack.getLogicalId(construct));
    }
    return undefined;
}
exports.tryGetLogicalId = tryGetLogicalId;
/** Infer node props from construct */
function inferNodeProps(construct) {
    const uuid = getConstructUUID(construct);
    const logicalId = tryGetLogicalId(construct);
    const metadata = construct.node.metadata.filter((entry) => {
        if (entry.type === types_1.MetadataTypeEnum.LOGICAL_ID)
            return false;
        return true;
    });
    const attributes = cloneDeep(extractInspectableAttributes(construct) || {});
    const cfnType = attributes[types_1.CfnAttributesEnum.TYPE];
    if (cfnType) {
        // @ts-ignore
        delete attributes[types_1.CfnAttributesEnum.TYPE];
    }
    const cfnProps = attributes[types_1.CfnAttributesEnum.PROPS] || {};
    let tags = {};
    // normalize tags
    if (typeof cfnProps === "object" && "tags" in cfnProps) {
        const _tags = cfnProps.tags;
        // remove the tags from the attributes since we normalize
        // @ts-ignore
        delete cfnProps.tags;
        if (Array.isArray(_tags)) {
            tags = Object.fromEntries(_tags.map(({ key, value }) => [key, value]));
        }
        else {
            tags = _tags;
        }
    }
    const constructInfo = cdk_internals_1.constructInfoFromConstruct(construct);
    const flags = inferFlags(construct, constructInfo);
    return {
        uuid,
        attributes,
        metadata,
        tags,
        logicalId,
        cfnType,
        constructInfo,
        dependencies: construct.node.dependencies.map(getConstructUUID),
        unresolvedReferences: extractUnresolvedReferences(uuid, attributes),
        flags,
    };
}
exports.inferNodeProps = inferNodeProps;
/** Extract inspectable attributes from construct */
function extractInspectableAttributes(construct) {
    // check if a construct implements IInspectable
    function canInspect(inspectable) {
        return inspectable.inspect !== undefined;
    }
    const inspector = new aws_cdk_lib_1.TreeInspector();
    // get attributes from the inspector
    if (canInspect(construct)) {
        construct.inspect(inspector);
        return aws_cdk_lib_1.Stack.of(construct).resolve(inspector.attributes);
    }
    return undefined;
}
exports.extractInspectableAttributes = extractInspectableAttributes;
/** Pattern of ignored references. Those which are resolved during deploy-time. */
exports.IGNORE_REF_PATTERN = /^AWS::/;
/** Extract unresolved references from attributes for a given source */
function extractUnresolvedReferences(source, from) {
    const references = [];
    traverse(from).forEach(function () {
        switch (this.key) {
            case types_1.ReferenceTypeEnum.ATTRIBUTE: {
                const [logicalId, attribute] = this.node;
                references.push({
                    source,
                    referenceType: types_1.ReferenceTypeEnum.ATTRIBUTE,
                    target: logicalId,
                    value: attribute,
                });
                break;
            }
            case types_1.ReferenceTypeEnum.REF: {
                if (typeof this.node === "string") {
                    if (!exports.IGNORE_REF_PATTERN.test(this.node)) {
                        references.push({
                            source,
                            referenceType: types_1.ReferenceTypeEnum.REF,
                            target: this.node,
                        });
                    }
                }
                else {
                    console.warn(`Found non-string "Ref"`, this.node);
                }
                break;
            }
            case types_1.ReferenceTypeEnum.IMPORT: {
                // "Fn::ImportValue": "Ada:ExportsOutputFnGetAttCommonStackA8F9EE77OutputsAdaCommonStackCounterTable5D6ADA16ArnED1AF27F"
                // "Fn::ImportValue": "Stage-Ada:ExportsOutputFnGetAttCommonStackA8F9EE77OutputsAdaCommonStackCounterTable5D6ADA16ArnED1AF27F"
                references.push({
                    source,
                    referenceType: types_1.ReferenceTypeEnum.IMPORT,
                    // NB: remove stage - separator
                    target: this.node.replace("-", ""),
                });
                break;
            }
        }
    });
    return references;
}
exports.extractUnresolvedReferences = extractUnresolvedReferences;
/** Infer construct flags  */
function inferFlags(construct, constructInfo) {
    const flags = new Set();
    const fqn = constructInfo?.fqn;
    if (fqn && types_1.ExtraneousFqnEnum.includes(fqn)) {
        flags.add(types_1.FlagEnum.EXTRANEOUS);
    }
    if (fqn && types_1.AssetFqnEnum.includes(fqn)) {
        flags.add(types_1.FlagEnum.ASSET);
    }
    if (construct.node.children.length === 1) {
        const onlyChild = construct.node.children[0];
        if ([types_1.CdkConstructIds.RESOURCE, types_1.CdkConstructIds.DEFAULT].includes(onlyChild.node.id)) {
            flags.add(types_1.FlagEnum.RESOURCE_WRAPPER);
        }
    }
    if (construct.node.id === "Exports" && aws_cdk_lib_1.Stack.isStack(construct.node.scope)) {
        flags.add(types_1.FlagEnum.EXTRANEOUS);
    }
    if (construct.node.id.startsWith("SsmParameterValue:")) {
        flags.add(types_1.FlagEnum.EXTRANEOUS);
    }
    return Array.from(flags.values());
}
exports.inferFlags = inferFlags;
//# sourceMappingURL=data:application/json;base64,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