"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.deserializeStore = exports.Graph = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const console_1 = require("console");
const constructs_1 = require("constructs");
const cloneDeep = require("lodash.clonedeep"); // eslint-disable-line @typescript-eslint/no-require-imports
const isEmpty = require("lodash.isempty"); // eslint-disable-line @typescript-eslint/no-require-imports
const omit = require("lodash.omit"); // eslint-disable-line @typescript-eslint/no-require-imports
const uniq = require("lodash.uniq"); // eslint-disable-line @typescript-eslint/no-require-imports
const counter_1 = require("./counter");
const types_1 = require("./types");
/** Public cdk-graph interface */
var Graph;
(function (Graph) {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s;
    /** Store class provides the in-memory database-like interface for managing all entities in the graph */
    class Store {
        constructor(allowDestructiveMutations = false) {
            /** Current SemVer version of the store */
            this.version = "0.0.0";
            /** @internal */
            this._edges = new Map();
            /** @internal */
            this._nodes = new Map();
            /** @internal */
            this._stacks = new Map();
            /** @internal */
            this._stages = new Map();
            /** @internal */
            this._logicalIdLookup = new Map();
            /** @internal */
            this._counters = {
                cfnResources: new counter_1.Counter(),
                nodeTypes: new counter_1.Counter(),
                edgeTypes: new counter_1.Counter(),
            };
            this._root = new RootNode(this);
            this.allowDestructiveMutations = allowDestructiveMutations;
        }
        /** Builds store from serialized store data */
        static fromSerializedStore(serializedStore) {
            return deserializeStore(serializedStore);
        }
        /**
         * Root node in the store. The **root** node is not the computed root, but the graph root
         * which is auto-generated and can not be mutated.
         */
        get root() {
            return this._root;
        }
        /**
         * Gets all stored **edges**
         * @type ReadonlyArray<Edge>
         */
        get edges() {
            return Array.from(this._edges.values());
        }
        /**
         * Gets all stored **nodes**
         * @type ReadonlyArray<Node>
         */
        get nodes() {
            return Array.from(this._nodes.values());
        }
        /**
         * Gets all stored **stack** nodes
         * @type ReadonlyArray<StackNode>
         */
        get stacks() {
            return Array.from(this._stacks.values());
        }
        /**
         * Gets all stored **stage** nodes
         * @type ReadonlyArray<StageNode>
         */
        get stages() {
            return Array.from(this._stages.values());
        }
        /**
         * Gets all stored **root stack** nodes
         * @type ReadonlyArray<StackNode>
         */
        get rootStacks() {
            return this.stacks.filter((stack) => StackNode.isStackNode(stack));
        }
        /** Get record of all store counters */
        get counts() {
            return {
                nodes: this._nodes.size,
                edges: this._edges.size,
                stacks: this._stacks.size,
                stages: this._stages.size,
                nodeTypes: this._counters.nodeTypes.counts,
                edgeTypes: this._counters.edgeTypes.counts,
                cfnResources: this._counters.cfnResources.counts,
            };
        }
        /** Add **edge** to the store */
        addEdge(edge) {
            this._edges.set(edge.uuid, edge);
            this._counters.edgeTypes.add(edge.edgeType);
        }
        /** Get stored **edge** by UUID */
        getEdge(uuid) {
            const edge = this._edges.get(uuid);
            if (edge != null) {
                return edge;
            }
            throw new Error(`Edge ${uuid} is not defined`);
        }
        /** Add **node** to the store */
        addNode(node) {
            this._nodes.set(node.uuid, node);
            this._counters.nodeTypes.add(node.nodeType);
            if (CfnResourceNode.isCfnResourceNode(node) && node.cfnType) {
                this._counters.cfnResources.add(node.cfnType);
            }
        }
        /** Get stored **node** by UUID */
        getNode(uuid) {
            const node = this._nodes.get(uuid);
            if (node != null) {
                return node;
            }
            throw new Error(`Node ${uuid} is not defined`);
        }
        /** Add **stack** node to the store */
        addStack(stack) {
            this._stacks.set(stack.uuid, stack);
        }
        /** Get stored **stack** node by UUID */
        getStack(uuid) {
            const stack = this._stacks.get(uuid);
            if (stack != null) {
                return stack;
            }
            throw new Error(`Stack ${uuid} is not defined`);
        }
        /** Add **stage** to the store */
        addStage(stage) {
            this._stages.set(stage.uuid, stage);
        }
        /** Get stored **stage** node by UUID */
        getStage(uuid) {
            const stage = this._stages.get(uuid);
            if (stage != null) {
                return stage;
            }
            throw new Error(`Stage ${uuid} is not defined`);
        }
        /**
         * Compute **universal** *logicalId* based on parent stack and construct *logicalId* (`<stack>:<logicalId>`).
         *
         * Construct *logicalIds are only unique within their containing stack, so to use *logicalId*
         * lookups universally (like resolving references) we need a universal key.
         */
        computeLogicalUniversalId(stack, logicalId) {
            return `${stack.uuid}:${logicalId}`;
        }
        /** Find node by **universal** *logicalId* (`<stack>:<logicalId>`) */
        findNodeByLogicalUniversalId(uid) {
            const [stackUUID, logicalId] = uid.split(":");
            const stack = this.getStack(stackUUID);
            return this.findNodeByLogicalId(stack, logicalId);
        }
        /** Find node within given **stack** with given *logicalId* */
        findNodeByLogicalId(stack, logicalId) {
            const uid = this.computeLogicalUniversalId(stack, logicalId);
            const nodeUUID = this._logicalIdLookup.get(uid);
            if (nodeUUID == null) {
                if (stack instanceof NestedStackNode && stack.parentStack) {
                    return this.findNodeByLogicalId(stack.parentStack, logicalId);
                }
                throw new Error(`Failed to find node by logicalId: ${uid}`);
            }
            const node = this._nodes.get(nodeUUID);
            if (node != null) {
                return node;
            }
            throw new Error(`Unable to find node mapped to logical id ${logicalId}`);
        }
        /** Record a **universal** *logicalId* to node mapping in the store */
        recordLogicalId(stack, logicalId, resource) {
            const uid = this.computeLogicalUniversalId(stack, logicalId);
            this._logicalIdLookup.set(uid, resource.uuid);
        }
        /** Serialize the store */
        serialize() {
            return {
                version: this.version,
                tree: this.root._serialize(),
                edges: Array.from(this.edges).map((edge) => edge._serialize()),
            };
        }
        /**
         * Clone the store to allow destructive mutations.
         * @param allowDestructiveMutations Indicates if destructive mutations are allowed; defaults to `true`
         * @returns {Store} Returns a clone of the store that allows destructive mutations
         */
        clone(allowDestructiveMutations = true) {
            return deserializeStore(this.serialize(), allowDestructiveMutations);
        }
        /**
         * Verifies that the store allows destructive mutations.
         * @throws Error is store does **not** allow mutations
         */
        verifyDestructiveMutationAllowed() {
            if (!this.allowDestructiveMutations) {
                throw new Error("GraphStore must be a clone to perform destructive mutations");
            }
        }
        /**
         * Remove **edge** from the store
         * @destructive
         */
        mutateRemoveEdge(edge) {
            const deleted = this._edges.delete(edge.uuid);
            if (deleted) {
                this._counters.edgeTypes.subtract(edge.edgeType);
            }
            return deleted;
        }
        /**
         * Remove **node** from the store
         * @destructive
         */
        mutateRemoveNode(node) {
            if (node.logicalId && node.stack) {
                this._logicalIdLookup.delete(this.computeLogicalUniversalId(node.stack, node.logicalId));
            }
            if (StackNode.isStackNode(node)) {
                this._stacks.delete(node.uuid);
            }
            const deleted = this._nodes.delete(node.uuid);
            if (deleted) {
                this._counters.nodeTypes.subtract(node.nodeType);
                if (CfnResourceNode.isCfnResourceNode(node) && node.cfnType) {
                    this._counters.cfnResources.subtract(node.cfnType);
                }
            }
            return deleted;
        }
    }
    _a = JSII_RTTI_SYMBOL_1;
    Store[_a] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.Store", version: "0.12.10" };
    Graph.Store = Store;
    /** Base class for all store entities (Node and Edges) */
    class BaseEntity {
        constructor(props) {
            /** @internal */
            this._destroyed = false;
            this.store = props.store;
            this.uuid = props.uuid;
            this._attributes = props.attributes || {};
            this._metadata = props.metadata || [];
            this._tags = new Map(Object.entries(props.tags || {}));
            this._flags = new Set(props.flags);
        }
        /**
         * Get *readonly* record of all attributes
         * @type Readonly<SerializedGraph.Attributes>
         */
        get attributes() {
            return cloneDeep(this._attributes);
        }
        /**
         * Get *readonly* list of all metadata entries
         * @type Readonly<SerializedGraph.Metadata>
         */
        get metadata() {
            return cloneDeep(this._metadata);
        }
        /**
         * Get *readonly* record of all tags
         * @type Readonly<SerializedGraph.Tags>
         */
        get tags() {
            return Object.fromEntries(this._tags);
        }
        /**
         * Get *readonly* list of all flags
         * @type ReadonlyArray<FlagEnum>
         */
        get flags() {
            return Array.from(this._flags);
        }
        /** Indicates if the entity has been destroyed (eg: removed from store) */
        get isDestroyed() {
            return this._destroyed;
        }
        /** Indicates if the entity has had destructive mutations applied */
        get isMutated() {
            return this.hasFlag(types_1.FlagEnum.MUTATED);
        }
        /** Indicates if entity has a given attribute defined, and optionally with a specific value */
        hasAttribute(key, value) {
            if (key in this._attributes) {
                if (value !== undefined) {
                    return this._attributes[key] === value;
                }
                return true;
            }
            return false;
        }
        /**
         * Add attribute.
         *
         * @throws Error if attribute for key already exists
         */
        addAttribute(key, value) {
            if (this.hasAttribute(key)) {
                throw new Error(`Entity ${String(this)} already has attribute ${key}; use setAttribute to override`);
            }
            this.setAttribute(key, value);
        }
        /** Set attribute. This will overwrite existing attribute. */
        setAttribute(key, value) {
            // @ts-ignore
            this._attributes[key] = value;
        }
        /** Get attribute by key */
        getAttribute(key) {
            return this._attributes[key];
        }
        /** Add metadata entry */
        addMetadata(metadataType, data) {
            this._metadata.push({
                type: metadataType,
                data,
            });
        }
        /** Indicates if entity has matching metadata entry */
        hasMetadata(metadataType, data) {
            return !!this._metadata.find((metadata) => {
                if (metadata.type !== metadataType)
                    return false;
                if (metadata.data !== data)
                    return false;
                return true;
            });
        }
        /**
         * Retrieves all metadata entries of a given type
         * @type Readonly<SerializedGraph.Metadata>
         */
        findMetadata(metadataType) {
            return this._metadata.filter((entry) => entry.type === metadataType);
        }
        /**
         * Add tag.
         * @throws Throws Error is tag for key already exists
         */
        addTag(key, value) {
            if (this.hasTag(key)) {
                throw new Error(`Entity ${String(this)} already has tag ${key}; use setTag to override`);
            }
            this.setTag(key, value);
        }
        /** Set tag. Will overwrite existing tag. */
        setTag(key, value) {
            this._tags.set(key, value);
        }
        /** Indicates if entity has tag, optionally verifying tag value */
        hasTag(key, value) {
            if (!this._tags.has(key))
                return false;
            if (value !== undefined && this._tags.get(key) !== value)
                return false;
            return true;
        }
        /** Get tag by key */
        getTag(key) {
            return this._tags.get(key);
        }
        /** Add flag */
        addFlag(flag) {
            this._flags.add(flag);
        }
        /** Indicates if entity has a given flag */
        hasFlag(flag) {
            return this._flags.has(flag);
        }
        /**
         * Applies data (attributes, metadata, tags, flag) to entity.
         *
         * Generally used only for mutations such as collapse and consume to retain data.
         * @param data - The data to apply
         * @param {boolean} [applyFlags=false] - Indicates if data is overwritten
         * @param {boolean} [applyFlags=false] - Indicates if flags should be applied
         */
        applyData(data, overwrite = false, applyFlags = false) {
            if (data.attributes) {
                Object.entries(data.attributes).forEach(([key, value]) => {
                    if (overwrite || !this.hasAttribute(key)) {
                        this.setAttribute(key, value);
                    }
                });
            }
            if (data.metadata) {
                data.metadata.forEach((v) => {
                    if (!this.hasMetadata(v.type, v.data)) {
                        this.addMetadata(v.type, v.data);
                    }
                });
            }
            if (data.tags) {
                Object.entries(data.tags).forEach(([key, value]) => {
                    if (overwrite || !this.hasTag(key)) {
                        this.setTag(key, value);
                    }
                });
            }
            if (applyFlags && data.flags) {
                data.flags.forEach((flag) => {
                    this.addFlag(flag);
                });
            }
        }
        /**
         * Performs pre-mutate operations on entity and store
         * @internal
         */
        _preMutate() {
            this.store.verifyDestructiveMutationAllowed();
            this.addFlag(types_1.FlagEnum.MUTATED);
        }
        /**
         * Serialize entity
         * @internal
         */
        _serialize() {
            return {
                uuid: this.uuid,
                attributes: isEmpty(this._attributes) ? undefined : this._attributes,
                metadata: isEmpty(this._metadata) ? undefined : this._metadata,
                tags: this._tags.size ? Object.fromEntries(this._tags) : undefined,
                flags: this._flags.size ? Array.from(this._flags) : undefined,
            };
        }
    }
    _b = JSII_RTTI_SYMBOL_1;
    BaseEntity[_b] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.BaseEntity", version: "0.12.10" };
    Graph.BaseEntity = BaseEntity;
    /** Edge class defines a link (relationship) between nodes, as in standard [graph theory](https://en.wikipedia.org/wiki/Graph_theory) */
    class Edge extends BaseEntity {
        constructor(props) {
            super(props);
            this.edgeType = props.edgeType;
            this._direction = props.direction;
            this._source = props.source;
            this._target = props.target;
            // Do not change original closed edge flag from a mutation.
            if (this._target === this._source && this.hasFlag(types_1.FlagEnum.MUTATED)) {
                this.addFlag(types_1.FlagEnum.CLOSED_EDGE);
            }
            // wire up links
            this._source.addLink(this);
            this._target.addReverseLink(this);
            this.store.addEdge(this);
        }
        /** Find first edge matching predicate within an EdgeChain */
        static findInChain(chain, predicate) {
            for (const entry of chain) {
                if (Array.isArray(entry)) {
                    const edge = Edge.findInChain(entry, predicate);
                    if (edge)
                        return edge;
                }
                else {
                    if (predicate(entry))
                        return entry;
                }
            }
            return undefined;
        }
        /** Find all matching edges based on predicate within an EdgeChain */
        static findAllInChain(chain, predicate) {
            const edges = [];
            for (const entry of chain) {
                if (Array.isArray(entry)) {
                    const edge = Edge.findInChain(entry, predicate);
                    if (edge) {
                        edges.push(edge);
                    }
                }
                else {
                    if (predicate(entry)) {
                        edges.push(entry);
                    }
                }
            }
            return edges;
        }
        /** Edge **source** is the node that defines the edge (tail) */
        get source() {
            return this._source;
        }
        /** Edge **target** is the node being referenced by the **source** (head) */
        get target() {
            return this._target;
        }
        /** Indicates the direction in which the edge is directed */
        get direction() {
            return this._direction;
        }
        /** Indicates if **source** and **target** nodes reside in different *root* stacks */
        get isCrossStack() {
            return this._source.rootStack !== this._target.rootStack;
        }
        /**
         * Indicates if the Edge's **source** and **target** are the same, or were the same
         * when it was created (prior to mutations).
         *
         * To check whether it was originally closed, use `hasFlag(FlagEnum.CLOSED_EDGE)` instead.
         */
        get isClosed() {
            return (this._source === this._target || this.hasFlag(types_1.FlagEnum.CLOSED_EDGE));
        }
        /**
         * Indicates if edge is extraneous which is determined by explicitly having *EXTRANEOUS* flag
         * added and/or being a closed loop (source===target).
         */
        get isExtraneous() {
            return this.hasFlag(types_1.FlagEnum.EXTRANEOUS) || this.isClosed;
        }
        /**
         * Indicates if this edge is equivalent to another edge.
         *
         * Edges are considered equivalent if they share same type, source, and target.
         */
        isEquivalent(edge) {
            if (edge.edgeType !== this.edgeType)
                return false;
            if (edge.source !== this.source)
                return false;
            if (edge.target !== this.target)
                return false;
            return true;
        }
        /** Indicates if edge allows destructive mutations */
        get allowDestructiveMutations() {
            return this.store.allowDestructiveMutations;
        }
        /**
         * Change the edge **direction**
         * @destructive
         */
        mutateDirection(direction) {
            this._preMutate();
            this._direction = direction;
        }
        /**
         * Change the edge **source**
         * @destructive
         */
        mutateSource(node) {
            this._preMutate();
            this._source.mutateRemoveLink(this);
            this._source = node;
            this._source.addLink(this);
        }
        /**
         * Change the edge **target**
         * @destructive
         */
        mutateTarget(node) {
            this._preMutate();
            this._target.mutateRemoveReverseLink(this);
            this._target = node;
            this._target.addReverseLink(this);
        }
        /**
         * Destroy the edge. Remove all references and remove from store.
         * @destructive
         */
        mutateDestroy(_strict = false) {
            this._preMutate();
            this.source.mutateRemoveLink(this);
            this.target.mutateRemoveReverseLink(this);
            this.store.mutateRemoveEdge(this);
            this._destroyed = true;
        }
        /**
         * Merge an equivalent edge's data into this edge and destroy the other edge.
         *
         * Used during filtering operations to consolidate equivalent edges.
         * @param edge - The edge to consume
         * @throws Error is edge is not *equivalent*
         * @destructive
         */
        mutateConsume(edge) {
            this._preMutate();
            if (!this.isEquivalent(edge)) {
                throw new Error(`Only equivalent edges can be consumed: ${edge} > ${this}`);
            }
            // propagate edge data
            this.applyData(edge);
            // destroy the consumed edge
            edge.mutateDestroy();
        }
        /** Get string representation of this edge */
        toString() {
            return `Edge:${this.edgeType}::${this.uuid}::${this.direction}(${this.source}->${this.target})`;
        }
        /** @internal */
        _serialize() {
            return {
                ...super._serialize(),
                edgeType: this.edgeType,
                direction: this.direction,
                source: this.source.uuid,
                target: this.target.uuid,
            };
        }
    }
    _c = JSII_RTTI_SYMBOL_1;
    Edge[_c] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.Edge", version: "0.12.10" };
    Graph.Edge = Edge;
    /** Dependency edge class defines CloudFormation dependency between resources */
    class Dependency extends Edge {
        constructor(props) {
            super({
                ...props,
                edgeType: types_1.EdgeTypeEnum.DEPENDENCY,
                direction: types_1.EdgeDirectionEnum.FORWARD,
            });
            this.addFlag(types_1.FlagEnum.EXTRANEOUS);
        }
        /** Indicates if given edge is a {@link Dependency} edge */
        static isDependency(edge) {
            return edge.edgeType === types_1.EdgeTypeEnum.DEPENDENCY;
        }
    }
    _d = JSII_RTTI_SYMBOL_1;
    Dependency[_d] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.Dependency", version: "0.12.10" };
    /** Edge prefix to denote dependency edge  */
    Dependency.PREFIX = "DEP:";
    Graph.Dependency = Dependency;
    /** Reference edge class defines a directed relationship between nodes  */
    class Reference extends Edge {
        constructor(props) {
            super({
                edgeType: types_1.EdgeTypeEnum.REFERENCE,
                direction: types_1.EdgeDirectionEnum.FORWARD,
                ...props,
            });
            this.setAttribute(Reference.ATT_TYPE, props.referenceType || types_1.ReferenceTypeEnum.REF);
        }
        /** Indicates if edge is a {@link Reference} */
        static isReference(edge) {
            return edge.edgeType === types_1.EdgeTypeEnum.REFERENCE;
        }
        /** Indicates if edge is a **Ref** based {@link Reference} edge */
        static isRef(edge) {
            return edge.referenceType === types_1.ReferenceTypeEnum.REF;
        }
        /** Get type of reference */
        get referenceType() {
            return this.getAttribute(Reference.ATT_TYPE);
        }
        /** Resolve reference chain */
        resolveChain() {
            if (OutputNode.isOutputNode(this.target)) {
                function _resolveChain(_ref) {
                    if (OutputNode.isOutputNode(_ref.target)) {
                        return [
                            _ref,
                            ..._ref.target.referenceLinks.map(_resolveChain),
                        ];
                    }
                    return [_ref];
                }
                return [
                    this,
                    ...this.target.referenceLinks.map(_resolveChain),
                ];
            }
            return [this];
        }
        /**
         * Resolve targets by following potential edge chain.
         *
         * @see {@link EdgeChain}
         */
        resolveTargets() {
            if (OutputNode.isOutputNode(this.target)) {
                function resolveOutputTarget(_target) {
                    if (OutputNode.isOutputNode(_target))
                        return resolveOutputTarget(_target);
                    return [_target];
                }
                return this.target.referenceLinks.flatMap((ref) => resolveOutputTarget(ref.target));
            }
            return [this.target];
        }
    }
    _e = JSII_RTTI_SYMBOL_1;
    Reference[_e] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.Reference", version: "0.12.10" };
    /** Edge prefix to denote **Ref** type reference edge  */
    Reference.PREFIX = "REF:";
    /** Attribute defining the type of reference */
    Reference.ATT_TYPE = "graph:reference:type";
    Graph.Reference = Reference;
    /** Attribute type reference edge */
    class AttributeReference extends Reference {
        constructor(props) {
            super({
                ...props,
                referenceType: types_1.ReferenceTypeEnum.ATTRIBUTE,
            });
            this.setAttribute(AttributeReference.ATT_VALUE, props.value);
        }
        /** Indicates if edge in an **Fn::GetAtt** {@link Reference} */
        static isAtt(edge) {
            return edge.referenceType === types_1.ReferenceTypeEnum.ATTRIBUTE;
        }
        /** Get the resolved attribute value */
        get value() {
            return this.getAttribute(AttributeReference.ATT_VALUE);
        }
    }
    _f = JSII_RTTI_SYMBOL_1;
    AttributeReference[_f] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.AttributeReference", version: "0.12.10" };
    /** Edge prefix to denote **Fn::GetAtt** type reference edge  */
    AttributeReference.PREFIX = "ATT:";
    /** Attribute key for resolved value of attribute reference */
    AttributeReference.ATT_VALUE = "graph:reference:attribute:value";
    Graph.AttributeReference = AttributeReference;
    /** Import reference defines **Fn::ImportValue** type reference edge. */
    class ImportReference extends Reference {
        constructor(props) {
            super({
                ...props,
                referenceType: types_1.ReferenceTypeEnum.IMPORT,
            });
        }
        /** Indicates if edge is **Fn::ImportValue** based {@link Reference} */
        static isImport(edge) {
            return edge.referenceType === types_1.ReferenceTypeEnum.IMPORT;
        }
    }
    _g = JSII_RTTI_SYMBOL_1;
    ImportReference[_g] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.ImportReference", version: "0.12.10" };
    /** Edge prefix to denote **Fn::ImportValue** type reference edge */
    ImportReference.PREFIX = "IMP:";
    Graph.ImportReference = ImportReference;
    /** Node class is the base definition of **node** entities in the graph, as in standard [graph theory](https://en.wikipedia.org/wiki/Graph_theory) */
    class Node extends BaseEntity {
        constructor(props) {
            super(props);
            /** @internal */
            this._children = new Map();
            /** @internal */
            this._links = new Map();
            /** @internal */
            this._reverseLinks = new Map();
            this.nodeType = props.nodeType;
            this.id = props.id;
            this.path = props.path;
            this.constructInfo = props.constructInfo;
            this._cfnType = props.cfnType;
            this._parent = props.parent;
            this.depth = this.parent ? this.parent.depth + 1 : 0;
            this._stack =
                props.stack || (this instanceof StackNode ? this : undefined);
            this.logicalId = props.logicalId;
            if (this.logicalId) {
                if (this.stack == null) {
                    throw new Error(`LogicalId defined outside of stack: ${this.logicalId} - ${String(this)}`);
                }
                this.store.recordLogicalId(this.stack, this.logicalId, this);
            }
            if (this.parent) {
                this.parent.addChild(this);
            }
            this.store.addNode(this);
        }
        /** Stack the node is contained in */
        get stack() {
            return this._stack;
        }
        /** Parent node. Only the root node should not have parent. */
        get parent() {
            return this._parent;
        }
        /** Gets descending ordered list of ancestors from the root */
        get scopes() {
            if (this.parent) {
                return [...this.parent.scopes, this.parent];
            }
            return [];
        }
        /** Get **root** stack */
        get rootStack() {
            if (StackNode.isStackNode(this))
                return this;
            return this.scopes.find((scope) => StackNode.isStackNode(scope));
        }
        /** Get all direct child nodes */
        get children() {
            return Array.from(this._children.values());
        }
        /** Indicates if this node is a *leaf* node, which means it does not have children */
        get isLeaf() {
            return this._children.size === 0;
        }
        /** Gets all links (edges) in which this node is the **source** */
        get links() {
            return Array.from(this._links.values());
        }
        /** Gets all links (edges) in which this node is the **target** */
        get reverseLinks() {
            return Array.from(this._reverseLinks.values());
        }
        /** Synthesized construct information defining jii resolution data */
        get constructInfoFqn() {
            return this.constructInfo?.fqn;
        }
        /** Indicates if node is a *Custom Resource* */
        get isCustomResource() {
            return types_1.ConstructInfoFqnEnum.CUSTOM_RESOURCE === this.constructInfoFqn;
        }
        /** Gets CloudFormation properties for this node */
        get cfnProps() {
            return this.attributes[types_1.CfnAttributesEnum.PROPS];
        }
        /** Get the CloudFormation resource type for this node */
        get cfnType() {
            return this._cfnType;
        }
        /** Gets list of {@link Dependency} links (edges) where this node is the **source** */
        get dependencyLinks() {
            return Array.from(this._links.values()).filter((link) => {
                return link.edgeType === types_1.EdgeTypeEnum.DEPENDENCY;
            });
        }
        /** Gets list of {@link Dependency} links (edges) where this node is the **target** */
        get reverseDependencyLinks() {
            return Array.from(this._links.values()).filter((link) => {
                return link.edgeType === types_1.EdgeTypeEnum.DEPENDENCY;
            });
        }
        /** Gets list of {@link Reference} links (edges) where this node is the **source** */
        get referenceLinks() {
            return Array.from(this._links.values()).filter((link) => {
                return link.edgeType === types_1.EdgeTypeEnum.REFERENCE;
            });
        }
        /** Gets list of {@link Reference} links (edges) where this node is the **target** */
        get reverseReferenceLinks() {
            return Array.from(this._links.values()).filter((link) => {
                return link.edgeType === types_1.EdgeTypeEnum.REFERENCE;
            });
        }
        /**
         * Get list of **Nodes** that *this node references*
         * @see {@link Node.referenceLinks}
         */
        get references() {
            return uniq(this.referenceLinks.flatMap((link) => link.resolveTargets()));
        }
        /**
         * Get list of **Nodes** that *reference this node*
         * @see {@link Node.reverseReferenceLinks}
         */
        get referencedBy() {
            return uniq(this.reverseReferenceLinks.flatMap((link) => link.resolveTargets()));
        }
        /**
         * Get list of **Nodes** that *this node depends on*
         * @see {@link Node.dependencyLinks}
         */
        get dependencies() {
            return uniq(this.dependencyLinks.flatMap((link) => link.target));
        }
        /**
         * Get list of **Nodes** that *depend on this node*
         * @see {@link Node.reverseDependencyLinks}
         */
        get dependedOnBy() {
            return uniq(this.reverseDependencyLinks.flatMap((link) => link.target));
        }
        /** Indicates if this node is considered a {@link FlagEnum.GRAPH_CONTAINER} */
        get isGraphContainer() {
            return this.hasFlag(types_1.FlagEnum.GRAPH_CONTAINER);
        }
        /** Indicates if this node is considered a {@link FlagEnum.CLUSTER} */
        get isCluster() {
            return this.hasFlag(types_1.FlagEnum.CLUSTER);
        }
        /** Indicates if this node is considered a {@link FlagEnum.EXTRANEOUS} */
        get isExtraneous() {
            return this.hasFlag(types_1.FlagEnum.EXTRANEOUS);
        }
        /** Indicates if this node is considered a {@link FlagEnum.RESOURCE_WRAPPER} */
        get isResourceWrapper() {
            return this.hasFlag(types_1.FlagEnum.RESOURCE_WRAPPER);
        }
        /** Indicates if this node is considered a {@link FlagEnum.ASSET} */
        get isAsset() {
            return this.hasFlag(types_1.FlagEnum.ASSET);
        }
        /** Get list of *siblings* of this node. */
        get siblings() {
            if (this.parent) {
                return this.parent.children.filter((child) => child !== this);
            }
            return [];
        }
        /** Get specific CloudFormation property */
        getCfnProp(key) {
            return this.cfnProps && this.cfnProps[key];
        }
        /** Add *link* to another node */
        addLink(edge) {
            this._links.set(edge.uuid, edge);
        }
        /** Add *link* from another node */
        addReverseLink(edge) {
            this._reverseLinks.set(edge.uuid, edge);
        }
        /** Add *child* node */
        addChild(node) {
            this._children.set(node.id, node);
        }
        /** Indicates if specific *node* is a *child* of *this node* */
        isChild(node) {
            for (const child of this._children.values()) {
                if (child === node)
                    return true;
            }
            return false;
        }
        /** Indicates if a specific *node* is an *ancestor* of *this node* */
        isAncestor(ancestor) {
            return this.scopes.includes(ancestor);
        }
        /** Find nearest *ancestor* of *this node* matching given predicate */
        findAncestor(predicate) {
            return this.scopes.slice().reverse().find(predicate);
        }
        /**
         * Gets the nearest **common** *ancestor* shared between *this node* and another *node*.
         * @throws Error if *node* does not share a **common** *ancestor*
         */
        getNearestAncestor(node) {
            if (node === this)
                throw new Error("Node is the current node");
            const aScopes = this.scopes;
            const bScopes = node.scopes;
            for (const aScope of aScopes) {
                for (const bScope of bScopes) {
                    if (aScope === bScope)
                        return aScope;
                }
            }
            throw new Error(`Nodes do not share common ancestor: ${String(this)} ^ ${String(node)}`);
        }
        /**
         * Return this construct and all of its sub-nodes in the given order.
         *
         * Optionally filter nodes based on predicate.
         */
        findAll(options) {
            const { predicate, order = constructs_1.ConstructOrder.PREORDER } = options || {};
            const all = new Array();
            function visit(c) {
                if (order === constructs_1.ConstructOrder.PREORDER) {
                    all.push(c);
                }
                for (const child of c.children) {
                    visit(child);
                }
                if (order === constructs_1.ConstructOrder.POSTORDER) {
                    all.push(c);
                }
            }
            visit(this);
            if (predicate) {
                return all.filter(predicate);
            }
            return all;
        }
        /** Recursively find the nearest sub-node matching predicate */
        find(predicate) {
            if (predicate(this))
                return this;
            for (const child of this.children) {
                const node = child.find(predicate);
                if (node != null)
                    return node;
            }
            return undefined;
        }
        /**
         * Get *child* node with given *id*.
         *
         * @throws Error if no child with given id
         */
        getChild(id) {
            const child = this._children.get(id);
            if (child == null) {
                throw new Error(`${String(this)} does not have child with id "${id}"`);
            }
            return child;
        }
        /** Find child with given *id*. Similar to `find` but does not throw error if no child found. */
        findChild(id) {
            return this._children.get(id);
        }
        /**
         * Return all direct links of this node and that of all sub-nodes.
         *
         * Optionally filter links based on predicate.
         */
        findAllLinks(options) {
            const { predicate, order = constructs_1.ConstructOrder.PREORDER, reverse, } = options || {};
            const all = new Array();
            visit(this);
            if (predicate) {
                return all.filter(predicate);
            }
            return all;
            function visit(c) {
                if (order === constructs_1.ConstructOrder.PREORDER) {
                    all.push(...c[reverse ? "reverseLinks" : "links"]);
                }
                for (const child of c.children) {
                    visit(child);
                }
                if (order === constructs_1.ConstructOrder.POSTORDER) {
                    all.push(...c[reverse ? "reverseLinks" : "links"]);
                }
            }
        }
        /**
         * Resolve all link chains
         * @see {@link EdgeChain}
         */
        getLinkChains(reverse = false) {
            let links = this[reverse ? "reverseLinks" : "links"];
            return links.map((link) => {
                if (Reference.isReference(link)) {
                    return link.resolveChain();
                }
                return [link];
            });
        }
        /**
         * Find link of this node based on predicate. By default this will follow link
         * chains to evaluate the predicate against and return the matching direct link
         * of this node.
         *
         * @param predicate Edge predicate function to match edge
         * @param reverse Indicates if links are search in reverse order
         * @param follow Indicates if link chain is followed
         * @param direct Indicates that only *direct* links should be searched
         * @returns
         */
        findLink(predicate, reverse = false, follow = true, direct = true) {
            if (follow) {
                const chains = this.getLinkChains(reverse);
                for (const chain of chains) {
                    const edge = Edge.findInChain(chain, predicate);
                    if (edge) {
                        if (direct)
                            return chain[0];
                        return edge;
                    }
                }
                return undefined;
            }
            return this[reverse ? "reverseLinks" : "links"].find(predicate);
        }
        /**
         * Find all links of this node based on predicate. By default this will follow link
         * chains to evaluate the predicate against and return the matching direct links
         * of this node.
         *
         * @param predicate Edge predicate function to match edge
         * @param reverse Indicates if links are search in reverse order
         * @param follow Indicates if link chain is followed
         * @param direct Indicates that only *direct* links should be searched
         * @returns
         */
        findLinks(predicate, reverse = false, follow = true, direct = true) {
            if (follow) {
                return this.getLinkChains(reverse).flatMap((chain) => {
                    const edges = Edge.findAllInChain(chain, predicate);
                    if (direct) {
                        return edges.length ? [chain[0]] : [];
                    }
                    return edges;
                });
            }
            return this[reverse ? "reverseLinks" : "links"].filter(predicate);
        }
        /** Indicates if *this node* references *another node* */
        doesReference(node) {
            return this.references.includes(node);
        }
        /** Indicates if *this node* depends on *another node* */
        doesDependOn(node) {
            return this.dependencies.includes(node);
        }
        /**
         * Indicates if this node allows destructive mutations
         * @see {@link Store.allowDestructiveMutations}
         */
        get allowDestructiveMutations() {
            return this.store.allowDestructiveMutations;
        }
        /**
         * Collapses all sub-nodes of *this node* into *this node*.
         * @destructive
         */
        mutateCollapse() {
            this._preMutate();
            this.children.forEach((child) => child.mutateCollapseToParent());
            this._mutateReconcileLinks();
        }
        /**
         * Collapses *this node* into *it's parent node*
         * @destructive
         */
        mutateCollapseToParent() {
            this._preMutate();
            if (this.parent == null) {
                throw new Error(`${this} does not have parent to collapse to.`);
            }
            return this.mutateCollapseTo(this.parent);
        }
        /**
         * Collapses *this node* into *an ancestor*
         * @destructive
         */
        mutateCollapseTo(ancestor) {
            this._preMutate();
            if (!this.isAncestor(ancestor)) {
                throw new Error(`${ancestor} is not an ancestor of ${this}`);
            }
            // TODO: should we retain the child attributes somewhere?
            this.children.forEach((child) => {
                if (child.isDestroyed)
                    return;
                child.mutateCollapseToParent();
            });
            // redirect all links to parent
            // while also deleting links to parent
            this.links.forEach((link) => {
                if (link.isDestroyed)
                    return;
                if (link.target === ancestor) {
                    link.mutateDestroy();
                }
                else {
                    link.mutateSource(ancestor);
                }
            });
            // redirect all "reverse" links to parent
            // while also deleting links from parent
            this.reverseLinks.forEach((link) => {
                if (link.isDestroyed)
                    return;
                if (link.source === ancestor) {
                    link.mutateDestroy();
                }
                else {
                    link.mutateTarget(ancestor);
                }
            });
            this.mutateDestroy(true);
            ancestor._mutateReconcileLinks();
            return ancestor;
        }
        /**
         * Destroys this node by removing all references and removing this node from the store.
         * @param {boolean} [strict=false] - Indicates that this node must not have references
         * @destructive
         */
        mutateDestroy(strict = false) {
            this._preMutate();
            if (strict) {
                if (this.children.length) {
                    throw new Error(`[strict] ${this} can not destroys because it has children`);
                }
                if (this.links.length || this.reverseLinks.length) {
                    throw new Error(`[strict] ${this} can not destroys because there are links referencing it`);
                }
            }
            if (strict && (this.links.length || this.reverseLinks.length)) {
                throw new Error(`[strict] ${this} can not destroys because there are links referencing it`);
            }
            this.children.forEach((child) => {
                child.mutateDestroy();
            });
            this.links.forEach((link) => {
                link.mutateDestroy();
            });
            this.reverseLinks.forEach((link) => {
                link.mutateDestroy();
            });
            if (this.parent) {
                this.parent.mutateRemoveChild(this);
            }
            this.store.mutateRemoveNode(this);
            this._destroyed = true;
        }
        /**
         * Reconciles links defined by this node. During mutations, multiple *equivalent* links may exist and should be
         * consolidated into a single link. This operation should be called after collapsing children to remove duplicates.
         * @internal
         * @destructive
         */
        _mutateReconcileLinks() {
            this._preMutate();
            const links = this.links;
            for (const a of links) {
                if (a.isDestroyed)
                    continue;
                for (const b of links) {
                    if (a === b || b.isDestroyed)
                        continue;
                    if (a.isEquivalent(b)) {
                        a.mutateConsume(b);
                    }
                }
            }
            const reverseLinks = this.reverseLinks;
            for (const a of reverseLinks) {
                if (a.isDestroyed)
                    continue;
                for (const b of reverseLinks) {
                    if (a === b || b.isDestroyed)
                        continue;
                    if (a.isEquivalent(b)) {
                        a.mutateConsume(b);
                    }
                }
            }
        }
        /**
         * Remove a *child* node from *this node*
         * @destructive
         */
        mutateRemoveChild(node) {
            this._preMutate();
            if (!this.isChild(node)) {
                throw new Error(`${node} is not a child of ${this}`);
            }
            // NB: children are stored by "id" not "uuid"
            return this._children.delete(node.id);
        }
        /**
         * Remove a *link* from *this node*
         * @destructive
         */
        mutateRemoveLink(link) {
            this._preMutate();
            return this._links.delete(link.uuid);
        }
        /**
         * Remove a *link* to *this node*
         * @destructive
         */
        mutateRemoveReverseLink(link) {
            this._preMutate();
            return this._reverseLinks.delete(link.uuid);
        }
        /**
         * Hoist *this node* to an *ancestor* by removing it from its current parent node and
         * in turn moving it to the ancestor.
         * @destructive
         */
        mutateHoist(newParent) {
            this._preMutate();
            if (!this.isAncestor(newParent)) {
                throw new Error(`${newParent} is not an ancestor of ${this}`);
            }
            if (this.parent) {
                this.parent.mutateRemoveChild(this);
            }
            this._parent = newParent;
            newParent.addChild(this);
            if (this.stack &&
                this.stack !== this &&
                !this.isAncestor(this.stack)) {
                this._stack = this.findAncestor((node) => StackNode.isStackNode(node) ||
                    NestedStackNode.isNestedStackNode(node));
            }
        }
        /** Get string representation of this node */
        toString() {
            return `Node:${this.nodeType}::${this.uuid}`;
        }
        /**
         * Serialize this node
         * @internal
         */
        _serialize() {
            return {
                ...super._serialize(),
                nodeType: this.nodeType,
                stack: this.stack?.uuid,
                parent: this.parent?.uuid,
                id: this.id,
                path: this.path,
                constructInfo: this.constructInfo,
                logicalId: this.logicalId,
                cfnType: this.cfnType,
                edges: this._links.size
                    ? Array.from(this._links.values()).map(({ uuid }) => uuid)
                    : undefined,
                children: this._children.size
                    ? Object.fromEntries(Array.from(this._children.entries()).map(([key, node]) => [
                        key,
                        node._serialize(),
                    ]))
                    : undefined,
            };
        }
    }
    _h = JSII_RTTI_SYMBOL_1;
    Node[_h] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.Node", version: "0.12.10" };
    Graph.Node = Node;
    /** ResourceNode class defines a L2 cdk resource construct */
    class ResourceNode extends Node {
        constructor(props) {
            super({
                nodeType: types_1.NodeTypeEnum.RESOURCE,
                ...props,
            });
            if (props.cdkOwned) {
                this.addFlag(types_1.FlagEnum.CDK_OWNED);
            }
        }
        /** Indicates if node is a {@link ResourceNode} */
        static isResourceNode(node) {
            return node.nodeType === types_1.NodeTypeEnum.RESOURCE;
        }
        /** Get the CloudFormation resource type for this L2 resource or for the L1 resource is wraps. */
        get cfnType() {
            return (super.cfnType ||
                this.getAttribute(ResourceNode.ATT_WRAPPED_CFN_TYPE) ||
                this.cfnResource?.cfnType);
        }
        /** Indicates if this resource is owned by cdk (defined in cdk library) */
        get isCdkOwned() {
            return this.hasFlag(types_1.FlagEnum.CDK_OWNED);
        }
        /** Get the L1 cdk resource that this L2 resource wraps */
        get cfnResource() {
            if (this._cfnResource !== undefined) {
                if (this._cfnResource && this._cfnResource.isDestroyed)
                    return undefined;
                return this._cfnResource || undefined;
            }
            const resourceNode = this.findChild(types_1.CdkConstructIds.RESOURCE);
            if (resourceNode) {
                this._cfnResource = resourceNode;
                return resourceNode;
            }
            const defaultNode = this.findChild(types_1.CdkConstructIds.DEFAULT);
            if (defaultNode) {
                this._cfnResource = defaultNode;
                return defaultNode;
            }
            const childNode = this.children.filter((node) => CfnResourceNode.isCfnResourceNode(node));
            if (childNode.length === 1) {
                this._cfnResource = childNode[0];
                return childNode[0];
            }
            // prevent looking up again by setting to `null`
            this._cfnResource = null;
            return undefined;
        }
        /** Get the cfn properties from the L1 resource that this L2 resource wraps */
        get cfnProps() {
            if (this.cfnResource) {
                return this.cfnResource.cfnProps;
            }
            return this.getAttribute(ResourceNode.ATT_WRAPPED_CFN_PROPS);
        }
        /**
         * Modifies the L1 resource wrapped by this L2 resource
         * @param cfnResource
         * @destructive
         */
        mutateCfnResource(cfnResource) {
            this._preMutate();
            this._cfnResource = cfnResource || null;
        }
    }
    _j = JSII_RTTI_SYMBOL_1;
    ResourceNode[_j] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.ResourceNode", version: "0.12.10" };
    /** Attribute key for cfn resource type */
    ResourceNode.ATT_WRAPPED_CFN_TYPE = "graph:resource:cfn-type";
    /** Attribute key for cfn properties */
    ResourceNode.ATT_WRAPPED_CFN_PROPS = "graph:resource:cfn-props";
    Graph.ResourceNode = ResourceNode;
    /** CfnResourceNode defines an L1 cdk resource */
    class CfnResourceNode extends Node {
        constructor(props) {
            super({
                nodeType: types_1.NodeTypeEnum.CFN_RESOURCE,
                ...props,
            });
            if (this.cfnType == null) {
                throw new Error("CfnResourceNode requires `cfnType` property");
            }
        }
        /** Indicates if a node is a {@link CfnResourceNode} */
        static isCfnResourceNode(node) {
            return node.nodeType === types_1.NodeTypeEnum.CFN_RESOURCE;
        }
        /**
         * Finds the near *ancestor* that is a {@link ResourceNode}
         */
        findNearestResource() {
            return this.scopes
                .slice()
                .reverse()
                .find((scope) => ResourceNode.isResourceNode(scope));
        }
        /**
         * @inheritdoc
         */
        mutateDestroy(strict) {
            const resource = this.findNearestResource();
            if (resource?.cfnResource === this) {
                resource.setAttribute(ResourceNode.ATT_WRAPPED_CFN_TYPE, this.cfnType);
                resource.setAttribute(ResourceNode.ATT_WRAPPED_CFN_PROPS, this.cfnProps);
                resource.mutateCfnResource(undefined);
            }
            super.mutateDestroy(strict);
        }
    }
    _k = JSII_RTTI_SYMBOL_1;
    CfnResourceNode[_k] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.CfnResourceNode", version: "0.12.10" };
    Graph.CfnResourceNode = CfnResourceNode;
    /** OutputNode defines a cdk CfnOutput resources */
    class OutputNode extends Node {
        constructor(props) {
            super({
                ...props,
                nodeType: types_1.NodeTypeEnum.OUTPUT,
            });
            /** Indicates if {@link OutputNode} is **exported** */
            this.isExport = false;
            if (this.stack == null) {
                throw new Error(`OutputNode instantiated outside of stack: ${this}`);
            }
            this.addFlag(types_1.FlagEnum.EXTRANEOUS);
            this.setAttribute(OutputNode.ATTR_VALUE, props.value);
            if (props.exportName) {
                this.isExport = true;
                this.setAttribute(OutputNode.ATTR_EXPORT_NAME, props.exportName);
            }
            props.description && this.setAttribute("description", props.description);
            this.stack.addOutput(this);
        }
        /** Indicates if node is an {@link OutputNode} */
        static isOutputNode(node) {
            return node.nodeType === types_1.NodeTypeEnum.OUTPUT;
        }
        /** Get the *value** attribute */
        get value() {
            return this.getAttribute(OutputNode.ATTR_VALUE);
        }
        /** Get the export name attribute */
        get exportName() {
            return this.getAttribute(OutputNode.ATTR_EXPORT_NAME);
        }
        /** @inheritdoc */
        mutateDestroy(strict) {
            super.mutateDestroy(strict);
            this.stack?.mutateRemoveOutput(this);
        }
    }
    _l = JSII_RTTI_SYMBOL_1;
    OutputNode[_l] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.OutputNode", version: "0.12.10" };
    /** Attribute key where output value is stored */
    OutputNode.ATTR_VALUE = "graph:output:value";
    /** Attribute key where output export name is stored */
    OutputNode.ATTR_EXPORT_NAME = "graph:output:export-name";
    Graph.OutputNode = OutputNode;
    /** ParameterNode defines a CfnParameter node */
    class ParameterNode extends Node {
        constructor(props) {
            super({
                ...props,
                nodeType: types_1.NodeTypeEnum.PARAMETER,
            });
            if (this.stack == null) {
                throw new Error(`ParameterNode instantiated outside of stack: ${this}`);
            }
            this.addFlag(types_1.FlagEnum.EXTRANEOUS);
            this.setAttribute(ParameterNode.ATTR_VALUE, props.value);
            this.setAttribute(ParameterNode.ATTR_TYPE, props.parameterType);
            props.description && this.setAttribute("description", props.description);
            this.isStackReference = this.id.startsWith("reference-to-");
            this.stack.addParameter(this);
        }
        /** Indicates if node is a {@link ParameterNode} */
        static isParameterNode(node) {
            return node.nodeType === types_1.NodeTypeEnum.PARAMETER;
        }
        /** Get the value attribute */
        get value() {
            return this.getAttribute(ParameterNode.ATTR_VALUE);
        }
        /** Get the parameter type attribute */
        get parameterType() {
            return this.getAttribute(ParameterNode.ATTR_TYPE);
        }
        /** @inheritdoc */
        mutateDestroy(strict) {
            super.mutateDestroy(strict);
            this.stack?.mutateRemoveParameter(this);
        }
    }
    _m = JSII_RTTI_SYMBOL_1;
    ParameterNode[_m] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.ParameterNode", version: "0.12.10" };
    /** Attribute key where parameter value is store */
    ParameterNode.ATTR_VALUE = "graph:parameter:value";
    /** Attribute key where parameter type is stored */
    ParameterNode.ATTR_TYPE = "graph:parameter:type";
    Graph.ParameterNode = ParameterNode;
    /** StackNode defines a cdk Stack */
    class StackNode extends Node {
        constructor(props) {
            super({
                nodeType: types_1.NodeTypeEnum.STACK,
                ...props,
            });
            /** @internal */
            this._outputs = new Set();
            /** @internal */
            this._parameters = new Set();
            if (this.stack !== this) {
                throw new Error(`Stack.stack is not self: ${this.uuid}`);
            }
            this.addFlag(types_1.FlagEnum.CLUSTER);
            this.store.addStack(this);
            const stage = this.findAncestor(StageNode.isStageNode);
            if (stage) {
                this._stage = stage;
                stage.addStack(this);
            }
        }
        /** Indicates if node is a {@link StackNode} */
        static isStackNode(node) {
            return node.nodeType === types_1.NodeTypeEnum.STACK;
        }
        /**
         * Gets the {@link StackNode} containing a given resource
         * @throws Error is node is not contained in a stack
         */
        static of(node) {
            const stack = node.stack;
            if (stack == null) {
                throw new Error(`${String(node)} is not within StackNode`);
            }
            return stack;
        }
        /** Get {@link StageNode} containing this stack */
        get stage() {
            return this._stage;
        }
        /** Get all {@link OutputNode}s defined by this stack */
        get outputs() {
            return Array.from(this._outputs);
        }
        /** Get all {@link ParameterNode}s defined by this stack */
        get parameters() {
            return Array.from(this._parameters);
        }
        /** Get all **exported** {@link OutputNode}s defined by this stack */
        get exports() {
            return this.outputs.filter((node) => node.isExport);
        }
        /** Associate {@link OutputNode} with this stack */
        addOutput(node) {
            this._outputs.add(node);
        }
        /**
         * Find {@link OutputNode} with *logicalId* defined by this stack
         * @throws Error is no output found matching *logicalId*
         */
        findOutput(logicalId) {
            const output = this.outputs.find((_output) => _output.logicalId === logicalId);
            if (output == null) {
                console.debug(`${this}.Outputs: [logicalId]`, this.outputs.map((n) => n.logicalId));
                throw new Error(`Output ${logicalId} does not exist in ${this}`);
            }
            return output;
        }
        /** Associate {@link ParameterNode} with this stack */
        addParameter(node) {
            this._parameters.add(node);
        }
        /**
         * Find {@link ParameterNode} with *parameterId* defined by this stack
         * @throws Error is no parameter found matching *parameterId*
         */
        findParameter(parameterId) {
            const parameter = this.parameters.find((_parameter) => _parameter.id === parameterId);
            if (parameter == null) {
                console.debug(`${this}.Parameters: [id]`, this.parameters.map((n) => n.id));
                throw new Error(`Parameter ${parameterId} does not exist in ${this}`);
            }
            return parameter;
        }
        /**
         * Disassociate {@link OutputNode} from this stack
         * @destructive
         */
        mutateRemoveOutput(node) {
            this._preMutate();
            return this._outputs.delete(node);
        }
        /**
         * Disassociate {@link ParameterNode} from this stack
         * @destructive
         */
        mutateRemoveParameter(node) {
            this._preMutate();
            return this._parameters.delete(node);
        }
        /** @inheritdoc */
        mutateDestroy(strict) {
            super.mutateDestroy(strict);
            this.stage?.mutateRemoveStack(this);
        }
        /** @inheritdoc */
        mutateHoist(newParent) {
            super.mutateHoist(newParent);
            if (this.stage && this.isAncestor(this.stage)) {
                this.stage.mutateRemoveStack(this);
                this._stage = this.findAncestor((node) => StageNode.isStageNode(node));
                if (this._stage) {
                    this._stage.addStack(this);
                }
            }
        }
    }
    _o = JSII_RTTI_SYMBOL_1;
    StackNode[_o] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.StackNode", version: "0.12.10" };
    Graph.StackNode = StackNode;
    /** NestedStackNode defines a cdk NestedStack */
    class NestedStackNode extends StackNode {
        constructor(props) {
            super({
                ...props,
                nodeType: types_1.NodeTypeEnum.NESTED_STACK,
            });
            this._parentStack = props.parentStack;
        }
        /** Indicates if node is a {@link NestedStackNode} */
        static isNestedStackNode(node) {
            return node.nodeType === types_1.NodeTypeEnum.NESTED_STACK;
        }
        /** Get parent stack of this nested stack */
        get parentStack() {
            return this._parentStack;
        }
        /** @inheritdoc */
        mutateHoist(newParent) {
            super.mutateHoist(newParent);
            if (this.parentStack && this.isAncestor(this.parentStack)) {
                this._parentStack = this.findAncestor((node) => StackNode.isStackNode(node));
            }
        }
    }
    _p = JSII_RTTI_SYMBOL_1;
    NestedStackNode[_p] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.NestedStackNode", version: "0.12.10" };
    Graph.NestedStackNode = NestedStackNode;
    /** StageNode defines a cdk Stage */
    class StageNode extends Node {
        constructor(props) {
            super({
                ...props,
                nodeType: types_1.NodeTypeEnum.STAGE,
            });
            /** @internal */
            this._stacks = new Set();
            this.store.addStage(this);
            this.addFlag(types_1.FlagEnum.CLUSTER);
        }
        /** Indicates if node is a {@link StageNode} */
        static isStageNode(node) {
            return node.nodeType === types_1.NodeTypeEnum.STAGE;
        }
        /**
         * Gets the {@link StageNode} containing a given resource
         * @throws Error is node is not contained in a stage
         */
        static of(node) {
            const stage = node.rootStack?.stage;
            if (stage == null) {
                throw new Error(`${node} is not within a stage`);
            }
            return stage;
        }
        /** Gets all stacks contained by this stage */
        get stacks() {
            return Array.from(this._stacks);
        }
        /** Associate a {@link StackNode} with this stage */
        addStack(stack) {
            this._stacks.add(stack);
        }
        /**
         * Disassociate {@link StackNode} from this stage
         * @destructive
         */
        mutateRemoveStack(stack) {
            this._preMutate();
            return this._stacks.delete(stack);
        }
    }
    _q = JSII_RTTI_SYMBOL_1;
    StageNode[_q] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.StageNode", version: "0.12.10" };
    Graph.StageNode = StageNode;
    /** AppNode defines a cdk App */
    class AppNode extends Node {
        constructor(props) {
            super({
                ...props,
                nodeType: types_1.NodeTypeEnum.APP,
                uuid: AppNode.UUID,
                id: AppNode.UUID,
                path: AppNode.PATH,
            });
            this.addFlag(types_1.FlagEnum.GRAPH_CONTAINER);
            this.addFlag(types_1.FlagEnum.CLUSTER);
        }
        /** Indicates if node is a {@link AppNode} */
        static isAppNode(node) {
            return node.nodeType === types_1.NodeTypeEnum.APP;
        }
    }
    _r = JSII_RTTI_SYMBOL_1;
    AppNode[_r] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.AppNode", version: "0.12.10" };
    /** Fixed UUID for App node */
    AppNode.UUID = "App";
    /** Fixed path of the App  */
    AppNode.PATH = "/";
    Graph.AppNode = AppNode;
    /** RootNode represents the root of the store tree */
    class RootNode extends Node {
        constructor(store) {
            super({
                store,
                nodeType: types_1.NodeTypeEnum.ROOT,
                uuid: RootNode.UUID,
                id: RootNode.UUID,
                path: RootNode.PATH,
            });
            this.addFlag(types_1.FlagEnum.GRAPH_CONTAINER);
            this.addFlag(types_1.FlagEnum.CLUSTER);
        }
        /** Indicates if node is a {@link RootNode} */
        static isRootNode(node) {
            return node.nodeType === types_1.NodeTypeEnum.ROOT;
        }
        /**
         * > {@link RootNode} does not support this mutation
         * @throws Error does not support
         * @inheritdoc
         */
        mutateCollapse() {
            throw new Error("Root node can not be collapsed");
        }
        /**
         * > {@link RootNode} does not support this mutation
         * @throws Error does not support
         * @inheritdoc
         */
        mutateCollapseToParent() {
            throw new Error("Root node can not be collapsed to parent");
        }
        /**
         * > {@link RootNode} does not support this mutation
         * @throws Error does not support
         * @inheritdoc
         */
        mutateCollapseTo(_ancestor) {
            throw new Error("Root node can not be collapsed");
        }
        /**
         * > {@link RootNode} does not support this mutation
         * @throws Error does not support
         * @inheritdoc
         */
        mutateDestroy(_strict = false) {
            throw new Error("Root node can not be destroyed");
        }
        /**
         * > {@link RootNode} does not support this mutation
         * @throws Error does not support
         * @inheritdoc
         */
        mutateHoist(_newParent) {
            throw new Error("Root node can not be hoisted");
        }
    }
    _s = JSII_RTTI_SYMBOL_1;
    RootNode[_s] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.RootNode", version: "0.12.10" };
    /** Fixed UUID of root */
    RootNode.UUID = "Root";
    /** Fixed path of root */
    RootNode.PATH = "";
    Graph.RootNode = RootNode;
})(Graph = exports.Graph || (exports.Graph = {}));
/**
 * Deserializes a *serialized store object* into an *in-memory store instance**.
 * @param serializedStore - The serialized store to deserialize
 * @param allowDestructiveMutations - Indicates if the store instance allows destructive mutations.
 * @throws Error if a serialized node's parent does not match visitor parent
 * @throws Error if a serialized node type deserialization mapping is not defined
 * @throws Error if edge type deserialization mapping is not defined
 */
function deserializeStore(serializedStore, allowDestructiveMutations = false) {
    const store = new Graph.Store(allowDestructiveMutations);
    // TODO: ensure store versions are compatible
    function visit(sNode, parent) {
        const nodeProps = {
            ...omit(sNode, [
                "children",
                "parent",
                "stack",
                "nodeType",
            ]),
            parent: sNode.parent ? store.getNode(sNode.parent) : undefined,
            // resolve stack node, unless stack is itself
            stack: sNode.stack && sNode.stack !== sNode.uuid
                ? store.getStack(sNode.stack)
                : undefined,
            store,
        };
        if (nodeProps.parent !== parent) {
            throw new Error(`SerializedNode parent ${sNode.parent} does not match visitor parent ${parent.uuid}`);
        }
        let node = undefined;
        switch (sNode.nodeType) {
            case types_1.NodeTypeEnum.APP: {
                node = new Graph.AppNode({
                    ...nodeProps,
                    parent,
                });
                break;
            }
            case types_1.NodeTypeEnum.STAGE: {
                node = new Graph.StageNode(nodeProps);
                break;
            }
            case types_1.NodeTypeEnum.STACK: {
                node = new Graph.StackNode(nodeProps);
                break;
            }
            case types_1.NodeTypeEnum.NESTED_STACK: {
                node = new Graph.NestedStackNode({
                    ...nodeProps,
                    parentStack: Graph.StackNode.of(parent),
                });
                break;
            }
            case types_1.NodeTypeEnum.OUTPUT: {
                node = new Graph.OutputNode({
                    ...nodeProps,
                    value: nodeProps.attributes[Graph.OutputNode.ATTR_VALUE],
                    exportName: nodeProps.attributes[Graph.OutputNode.ATTR_EXPORT_NAME],
                    description: nodeProps.attributes.description,
                });
                break;
            }
            case types_1.NodeTypeEnum.PARAMETER: {
                node = new Graph.ParameterNode({
                    ...nodeProps,
                    value: nodeProps.attributes[Graph.ParameterNode.ATTR_VALUE],
                    parameterType: nodeProps.attributes[Graph.ParameterNode.ATTR_TYPE],
                    description: nodeProps.attributes.description,
                });
                break;
            }
            case types_1.NodeTypeEnum.CFN_RESOURCE: {
                node = new Graph.CfnResourceNode(nodeProps);
                break;
            }
            case types_1.NodeTypeEnum.RESOURCE: {
                node = new Graph.ResourceNode({
                    ...nodeProps,
                    cdkOwned: !!nodeProps.flags?.includes(types_1.FlagEnum.CDK_OWNED),
                });
                break;
            }
            case types_1.NodeTypeEnum.DEFAULT: {
                node = new Graph.Node({
                    ...nodeProps,
                    nodeType: types_1.NodeTypeEnum.DEFAULT,
                });
                break;
            }
        }
        if (node == null) {
            console.debug(sNode.nodeType, sNode);
            throw new Error(`NodeType ${sNode.nodeType} missing deserialization mapping`);
        }
        // ensure node is registered in store
        console_1.assert(store.getNode(sNode.uuid) === node, `Node ${sNode.uuid} did not register in store`);
        Object.values(sNode.children || {}).forEach((sChild) => {
            visit(sChild, node);
        });
    }
    Object.values(serializedStore.tree.children || {}).forEach((sNode) => {
        visit(sNode, store.root);
    });
    serializedStore.edges.forEach((sEdge) => {
        const edgeProps = {
            ...sEdge,
            store,
            source: store.getNode(sEdge.source),
            target: store.getNode(sEdge.target),
        };
        switch (sEdge.edgeType) {
            case types_1.EdgeTypeEnum.DEPENDENCY: {
                new Graph.Dependency({
                    ...edgeProps,
                });
                break;
            }
            case types_1.EdgeTypeEnum.REFERENCE: {
                const referenceType = sEdge.attributes[Graph.Reference.ATT_TYPE];
                if (referenceType === types_1.ReferenceTypeEnum.ATTRIBUTE) {
                    new Graph.AttributeReference({
                        ...edgeProps,
                        value: sEdge.attributes[Graph.AttributeReference.ATT_VALUE],
                    });
                }
                else if (referenceType === types_1.ReferenceTypeEnum.IMPORT) {
                    new Graph.ImportReference({
                        ...edgeProps,
                    });
                }
                else if (referenceType === types_1.ReferenceTypeEnum.REF) {
                    new Graph.Reference(edgeProps);
                }
                else {
                    throw new Error(`Unsupported reference type of ${referenceType}`);
                }
                break;
            }
            default: {
                // TODO: support custom edge types explicitly
                new Graph.Edge(edgeProps);
            }
        }
    });
    return store;
}
exports.deserializeStore = deserializeStore;
//# sourceMappingURL=data:application/json;base64,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